//
// Copyright (C) 1996 Ben Ross
//
// You may distribute under the terms of the GNU General Public
// License as specified in the COPYING file.
//

// $Id: XeArray.CXX,v 1.2 1997/03/10 11:08:38 ben Exp $

#include <XeArray.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

template <class T>
XeArray<T>::XeArray(int extrabuf, int size)
{
	_bufsize = size + extrabuf;
	_elements = 0;
	_data = (T*)malloc(sizeof(T) * _bufsize);
	_extrabuf = extrabuf;
}


template <class T>
XeArray<T>::~XeArray()
{
	if (_data)
		free(_data);
}

//
// insert() inserts a block of data at the index given.
// The value contained at that index gets moved to the end
// of the given block. To insert after the last index
// simply use an index 1 greater than the last index.
//

template <class T>
void
XeArray<T>::insert(ulong index, const T* array, int len)
{
	ulong newsize = _elements + len;
	
	if(index > _elements) {
		fprintf(stderr, "XeArray::insert - Index out of bounds.\n");
		exit(1);
	}
	if(newsize > _bufsize) {
		_bufsize = newsize + _extrabuf;
		_data = (T *)realloc(_data, sizeof(T) * _bufsize);
	}
	if(index < _elements) {
		memmove(&_data[index+len], &_data[index], sizeof(T)*(_elements - index));
	}
	memcpy(&_data[index], array, sizeof(T) * len);
	_elements = newsize;
}

template <class T>
void
XeArray<T>::insert(ulong index, T item)
{
	insert(index, &item, 1);
}

template <class T>
void
XeArray<T>::deleteRange(ulong startIndex, ulong endIndex)
{
	// inclusive remove.
	// just chop out this range by copying everything after the
	// range to the beginning of the range.
	
	int range = endIndex - startIndex + 1;
	
	if(endIndex >= _elements || startIndex >= _elements) {
		fprintf(stderr, "XeArray::deleteRange - Index out of bounds.\n");
		exit(1);
	}
	
	if(endIndex < _elements - 1) {
		memmove(&_data[startIndex], &_data[endIndex+1],
					sizeof(T) * (_elements - endIndex - 1));
	}
	_elements -= range;
}

template <class T>
void
XeArray<T>::replaceRange(ulong startIndex, ulong endIndex, 
	const T* array, int len)
{
	deleteRange(startIndex, endIndex);
	insert(startIndex, array, len);
}

template <class T>
T&
XeArray<T>::operator[](ulong index)
{
	if(index >= _elements) {
		fprintf(stderr, "XeArray::operator[%lu] - Index out of bounds\n", index);
		exit(1);
	}
	return _data[index];
}


template <class T>
void
XeArray<T>::replaceBuffer(T* newbuf, ulong len)
{
	if (_data)
		free(_data);
	
	_data = newbuf;
	_elements = len;
	_bufsize = len;
}
