#include <Kr/String.h>
#include <Kr/minmax.h>

#include <string.h>
#include <ctype.h>

// $Id: String.C,v 1.14 98/12/02 17:15:51 brian Exp $

const char String::_emptyString = 0;

String::String(const char* str)
{
    if (!str || !*str) {
	_core = NULL;
	return;
    }

    _core = new Core;
    _core->len = ::strlen(str);
    _core->data = new char[_core->len + 1];
    ::strcpy(_core->data, str);
}

String::String(const void* data, int len)
{
    if (len <= 0) {
	_core = NULL;
	return;
    }
	
    _core = new Core;
    _core->len = len;
    _core->data = new char[len + 1];
    _core->data[len] = 0;
    if (data)
	::memcpy(_core->data, data, len);
    else
	_core->data[0] = 0;

// don't necessarily need to clear the data
//	::memset(_core->data, 0, len + 1);
}

String&
String::operator = (const String& str)
{
    if (_core != str._core) {
	disconnect();
	connect(str);
    }

    return *this;
}

//
// Addition (catenation) operators
//

String&
String::operator += (const String& s)
{
    if (!s._core)
	return *this;
    
    if (!_core) {
	// Create a new core with a copy of s._core->data
	//
	_core = new Core;
	_core->len = s._core->len;
	_core->data = new char[_core->len + 1];
	::memcpy(_core->data, s._core->data, _core->len);
    
    } else {
	// Create a new core with copy of _core->data and s._core->data
	// catenated together, then disconnect and replace the core
	//
	Core* core = new Core;
	core->len = _core->len + s._core->len;
	core->data = new char[core->len + 1];
	::memcpy(core->data, _core->data, _core->len);
	::memcpy(core->data + _core->len, s._core->data, s._core->len);
	disconnect();
	_core = core;
    }
    
    (_core->data)[_core->len] = 0;
    return *this;
}

String
operator + (const String& a, const String& b)
{
    if (!a._core && !b._core)
	return String();
    if (!a._core)
	return String(b._core->data, b._core->len);
    if (!b._core)
	return String(a._core->data, a._core->len);
    
    return String::catenate(a._core->data, a._core->len,
				b._core->data, b._core->len);
}

String
operator + (const String& a, const char* b)
{    
    if (!a._core && (!b || !*b))
	return String();
    if (!a._core)
	return String(b, ::strlen(b));
    if (!b || !*b)
	return String(a._core->data, a._core->len);
    
    return String::catenate(a._core->data, a._core->len, b, ::strlen(b));
}

String
operator + (const String& a, char ch)
{
    if (!a._core)
	return String(&ch, 1);
    else
	return String::catenate(a._core->data, a._core->len, &ch, 1);
}

String
operator + (const char* a, const String& b)
{    
    if ((!a || !*a) && !b._core)
	return String();
    if (!a || !*a)
	return String(b._core->data, b._core->len);
    if (!b._core)
	return String(a, ::strlen(a));
    
    return String::catenate(a, ::strlen(a), b._core->data, b._core->len);
}

String
String::catenate(const char* adata, int alen,
		 const char* bdata, int blen)
{
    String str(NULL, alen + blen);
    ::memcpy(str._core->data, adata, alen);
    ::memcpy(str._core->data + alen, bdata, blen);    
    return str;
}

//
// Comparison operators
//

bool
operator == (const String& a, const String& b)
{
    if (a._core == b._core)
	return true;
    
    int len = a.length();
    
    if (len != b.length())
	return false;
    
    if (len == 0 || !memcmp(a._core->data, b._core->data, len))
	return true;

    return false;
}

bool
operator == (const String& a, const char* b)
{
    if (!a._core)
	return (!b || !*b);

    if (!b || !*b)
	return false;

    int len = a._core->len;
    if (strncmp(a._core->data, b, len))
	return false;

    // By this stage, b is at least len chars long.
    // If not exactly len (i.e. zero at b[len]), then not equal.

    return !b[len];
}

int
strcmp(const String& a, const String& b)
{
    if (a._core == b._core)
	return 0;
    
    int alen = a.length();
    int blen = b.length();

    switch (!alen + 2 * !blen) {
	case 1:
	    return -1;	// alen = 0, blen > 0
	case 2:
	    return 1;	// alen > 0, blen = 0
	case 3:
	    return 0;	// alen = 0, blen = 0
    }
    
    int diff = memcmp(a._core->data, b._core->data, min(alen, blen));
    if (diff != 0)
	return diff;

    if (alen == blen)
	return 0;
    return (alen < blen) ? -1 : 1;
}

char*
String::charP()
{
    if (!_core)
	return NULL; // $$$ NULL or ""?

    forkIfShared();
    return _core->data;
}

void
String::forkIfShared()
{
    if (_core && _core->ref != 1) {
	Core* core = new Core;
	core->len = _core->len;
	core->data = new char[_core->len + 1];
	::memcpy(core->data, _core->data, _core->len + 1);
	_core->ref--;
	_core = core;
    }
}

void
String::truncate(int len)
{
    if (!_core || len >= _core->len)
	return;

    forkIfShared();
    _core->len = len;
}

String
int_to_str(int i)
{
    // $$$ a better way of doing this ...
    char buf[100];
    sprintf(buf, "%d", i);
    return String(buf);
}

int String::_dumpPrintableMaxLength = 0;

void
String::dumpPrintable(ostream& ostr) const
{
    int len = length();
    if (len == 0)
	return;

    const char* cp = _core->data;
    const char* firstprintable = NULL;

    char esc[5];
    esc[0] = '\\';
    esc[4] = '\0';
    
    bool truncated = false;
    int plen = 0;

    ostr << '\"';
    
    while (len--) {
	char ch = *cp;
	if (!isprint(ch)) {
	    if (firstprintable) {
		ostr.write(firstprintable, cp - firstprintable);
		firstprintable = NULL;
	    }
	    
	    switch (ch) {
		case '\n': ch = 'n'; goto specialesc;
		case '\t': ch = 't'; goto specialesc;
		case '\v': ch = 'v'; goto specialesc;
		case '\b': ch = 'b'; goto specialesc;
		case '\r': ch = 'r'; goto specialesc;
		case '\f': ch = 'f'; goto specialesc;
		case '\a': ch = 'a'; goto specialesc;

		specialesc:
		    esc[1] = ch;
		    esc[2] = '\0';
		    plen++;
		    break;

		default:
		    esc[1] = '0' + ((ch >> 6) & 0x03);
		    esc[2] = '0' + ((ch >> 3) & 0x07);
		    esc[3] = '0' + (ch & 0x07);
		    plen += 3;
		    break;
	    }
	    
	    ostr << esc;
	    
	} else {
	    if (!firstprintable)
		firstprintable = cp;
	}

	cp++;
	plen++;

	if (_dumpPrintableMaxLength && plen > _dumpPrintableMaxLength) {
	    truncated = true;
	    break;
	}
    }
    
    if (firstprintable)
	ostr.write(firstprintable, cp - firstprintable);
    
    ostr << '\"';
    if (truncated)
	ostr << "...";
}

void
String::dumpHex(ostream& ostr) const
{
    int len = length();
    if (len == 0)
	return;

    const char hex[] = "0123456789abcdef";
    const char* cp = _core->data;
    
    for (;;) {
	unsigned char byte = (unsigned char) *cp++;
	ostr << hex[byte >> 4] << hex[byte & 0xf];

	if (--len == 0)
	    break;
	
	ostr << ' ';
    }
}
