/*
    XCruise - A directory browser
    Copyright (C) 1999  Yusuke Shinyama <euske@cl.cs.titech.ac.jp>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 *  main.c
 *	Main routine
 */


#include "xcruise.h"


/*  Local type definitions
 */
typedef struct _Apparam {
    char    *pname;
    int     needvalue;
    char    **pval;
} Apparam;

typedef struct _Pattern {
    unsigned char pat[8];
} Pattern;


/*  Graphical parameters
 */

/* default colors (used for stars) */
static char* starcolname[TheColors] = {
    /* inside */
    "blue",      /* root */
    "yellow",    /* planet-in */
    "orange",    /* planet-in */
    "violet",    /* planet-in */
    "red",       /* planet-in */
    "magenta",   /* unread-planet-in */
    "cyan",      /* galaxy-in */
    "gray",      /* galaxy-in */
    "magenta",   /* unread-galaxy-in */
    "#20ff00",   /* wormhole-in */
    "magenta",   /* unread-wormhole-in */
    
    /* outside */
    "blue",      /* root */
    "#c0c000",   /* planet-out */
    "darkorange",/* planet-out */
    "darkviolet",/* planet-out */
    "brown",     /* planet-out */
    "#c000c0",   /* unread-planet-in */
    "#008080",   /* galaxy-out */
    "#606060",   /* galaxy-out */
    "#c000c0",   /* unread-galaxy-in */
    "#108000",   /* wormhole-out */
    "#c000c0"    /* unread-wormhole-in */
};

/* default colors (used for information) */
static char* alrtcolname = "red";      
static char* namecolname = "white";
static char* infocolname = "green";
static char* meancolname = "green";
static char* bgcolname = "#000020";

/* default fonts (used for stars' name) */
static char* namefontname[NameFonts] = {
    "*misc-fixed-medium-r*100*",
    "*misc-fixed-medium-r*130*",
    "*misc-fixed-medium-r*140*",
    "*misc-fixed-medium-r*200*"
};

/* default font sizes */
static char* namesizestr[NameFonts] = {
    "9", "12", "18", "24"
};

/* patterns (in monochrome mode) */
static Pattern patWhite = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
static Pattern patBlack = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
static Pattern starpat[TheColors] = {
    /* inside */
    {0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55}, /* root */
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* planet-in */
    {0x22, 0x00, 0x88, 0x00, 0x22, 0x00, 0x88, 0x00}, /* planet-in */
    {0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88}, /* planet-in */
    {0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55}, /* planet-in */
    {0xff, 0x55, 0xff, 0x55, 0xff, 0x55, 0xff, 0x55}, /* noread-planet-in */
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* galaxy-in */
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}, /* galaxy-in */
    {0xff, 0x55, 0xff, 0x55, 0xff, 0x55, 0xff, 0x55}, /* noread-planet-in */
    {0xaa, 0x00, 0xaa, 0x00, 0xaa, 0x00, 0xaa, 0x00}, /* wormhole-in */
    {0xff, 0x55, 0xff, 0x55, 0xff, 0x55, 0xff, 0x55}, /* noread-planet-in */
    /* outside */
    {0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55}, /* root */
    {0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88}, /* planet-out */
    {0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55}, /* planet-out */
    {0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77}, /* planet-out */
    {0xdd, 0x77, 0xdd, 0xff, 0xdd, 0x77, 0xdd, 0xff}, /* planet-out */
    {0xff, 0x55, 0xff, 0x55, 0xff, 0x55, 0xff, 0x55}, /* noread-planet-out */
    {0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55}, /* galaxy-out */
    {0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55}, /* galaxy-out */
    {0xff, 0x55, 0xff, 0x55, 0xff, 0x55, 0xff, 0x55}, /* noread-planet-out */
    {0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55}, /* wormhole-out */
    {0xff, 0x55, 0xff, 0x55, 0xff, 0x55, 0xff, 0x55}  /* noread-planet-out */
};

/* default fonts (used for information) */
static char* alrtfontname = "*adobe-helvetica-bold-r*120*";
static char* infofontname = "*adobe-times-bold-r*120*";
static char* meanfontname = "*adobe-helvetica-medium-r*100*";


/*  Options & Resources
 */

/* X parameters */
static	XClassHint myxch = { "xcruise", "XCruise" };
static	char*	appname;
static	char*	dispname = NULL;
static	char*	mygeom = "1000x800+100+100";

/* options */
static	char*	mydelayticks = "50";
static	char*	mydiraccel = "0.5";
static	char*	myaccel1 = "0.1";
static	char*	myaccel2 = "0.5";
static	int	monoflag = 0;
static	int	titleflag = 0;
	int	viewallflag = 0;
static Apparam myopts[] = {
    {"-display",	true,	&dispname},
    {"-d",		true,	&dispname},
    {"-geometry",	true,	&mygeom},
    {"-g",		true,	&mygeom},
    {"-titlemode",	false,	(char**)&titleflag},
    {"-t",		false,	(char**)&titleflag},
    {"-viewall",	false,	(char**)&viewallflag},
    {"-v",		false,	(char**)&viewallflag},
    {"-monochrome",	false,	(char**)&monoflag},
    {"-m",		false,	(char**)&monoflag},
    {NULL, false, NULL}		/* end */
};

/* resources */
static Apparam myres[] = {
    {"rootin",      true,  &starcolname[0]},
    {"planetin1",   true,  &starcolname[1]},
    {"planetin2",   true,  &starcolname[2]},
    {"planetin3",   true,  &starcolname[3]},
    {"planetin4",   true,  &starcolname[4]},
    {"unrdplanetin",true,  &starcolname[5]},
    {"galaxyin1",   true,  &starcolname[6]},
    {"galaxyin2",   true,  &starcolname[7]},
    {"unrdgalaxyin",true,  &starcolname[8]},
    {"wormholein1", true,  &starcolname[9]},
    {"unrdwormholein",true,&starcolname[10]},
    {"rootout",     true,  &starcolname[11]},
    {"planetout1",  true,  &starcolname[12]},
    {"planetout2",  true,  &starcolname[13]},
    {"planetout3",  true,  &starcolname[14]},
    {"planetout4",  true,  &starcolname[15]},
    {"unrdplanetout",true, &starcolname[16]},
    {"galaxyout1",  true,  &starcolname[17]},
    {"galaxyout2",  true,  &starcolname[18]},
    {"unrdgalaxtyout",true,&starcolname[19]},
    {"wormholeout1",true,  &starcolname[20]},
    {"unrdwormholeout",true,&starcolname[21]},
    {"alertcolor",  true,  &alrtcolname},
    {"namecolor",   true,  &namecolname},
    {"infocolor",   true,  &infocolname},
    {"meancolor",   true,  &meancolname},
    {"bkgndcolor",  true,  &bgcolname},
    {"namefont0",   true,  &namefontname[0]},
    {"namefont1",   true,  &namefontname[1]},
    {"namefont2",   true,  &namefontname[2]},
    {"namefont3",   true,  &namefontname[3]},
    {"namesize0",   true,  &namesizestr[0]},
    {"namesize1",   true,  &namesizestr[1]},
    {"namesize2",   true,  &namesizestr[2]},
    {"namesize3",   true,  &namesizestr[3]},
    {"alertfont",   true,  &alrtfontname},
    {"infofont",    true,  &infofontname},
    {"meanfont",    true,  &meanfontname},
    {"geometry",    true,  &mygeom},
    {"acceleration1", true, &myaccel1},
    {"acceleration2", true, &myaccel2},
    {"delayticks",  true,  &mydelayticks},
    {"diraccel",    true,  &mydiraccel},
    {NULL, false, NULL}		/* end */
};

/* other global variables */
Display* disp;
Colormap cmap;

int	colorize;
int	screenWidth;
int	screenHeight;
int	screenSize;

long	rgbWhite;
long	rgbBlack;
GC	defaultgc;
GC	alrtgc;
GC	stargc[TheColors];
GC	namegc[NameFonts];
GC	infogc;
GC	meangc;
GC	bggc;
GC	cursorgc;
XFontStruct* namefont[NameFonts];
XFontStruct* alrtfont;
XFontStruct* infofont;
XFontStruct* meanfont;
int	namesize[NameFonts];

star	universe;
star*	curzone;
star*	stpswin;
int	lnupdate;
int	numlinks;
wormholeinfo* curlinks[MaxLinks];
star	starbuff[MaxStars];
Point3D	zerop = {0.0, 0.0, 0.0};
double	cirx[GalaxyPoly], ciry[GalaxyPoly];
XRectangle meanhclip;
XRectangle meanvclip;

Point3D	viewp;
Point3D	forwardv, rightv, upperv;
double	vspeed;

double	accel1,	accel2;
double	diraccel;
int	delayticks;

Window	mywin;
Pixmap	myscreen;


/*  Usage & Getopt
 */
static void usage(char* appname)
{
    fprintf(stderr,
	    "usage : %s [-display ...] [-geometry ...] [-monochrome]\n"
	    "                 [-titlemode] [-viewall]\n",
	    appname);
    exit(1);
}

static void getopt2(int argc, char **argv)
{
    int i;
    char **nval = NULL;
    
    for(i = 1; i < argc; i++) {
	if (nval != NULL) {
	    *nval = argv[i];
	    nval = NULL;
	} else {
	    int j;
	    for(j = 0; myopts[j].pname; j++) {
		if (!strcmp(myopts[j].pname, argv[i])) {
		    if (myopts[j].needvalue)
			nval = (char **)myopts[j].pval;
		    else
			*(char *)myopts[j].pval = true;
		    break;
		}
	    }
	    if (!myopts[j].pname)
		usage(appname);
	}
    }
    if (nval)
	usage(appname);
}


/*  Initialize Window
 */
static Window openSolidWindow(int argc, char** argv, 
			      char* geom, long evmask, 
			      XWindowAttributes* xwa)
{
    XWMHints xwmh;
    XSizeHints xsh;
    XSetWindowAttributes xswa;
    XTextProperty xtp;
    Window win;

    (void) XGeometry(disp, DefaultScreen(disp), geom, geom,
		     1, 1, 1, 1, 1, 
		     &xsh.x, &xsh.y, &xsh.width, &xsh.height);
    xsh.min_width = xsh.max_width = xsh.width;
    xsh.min_height = xsh.max_height = xsh.height;
    xsh.flags = (PPosition | PSize | PMinSize | PMaxSize);
    
    win = XCreateSimpleWindow(disp, DefaultRootWindow(disp),
			      xsh.x, xsh.y, xsh.width, xsh.height,
			      1, rgbBlack, rgbWhite);
    
    xwmh.flags = (InputHint | StateHint);
    xwmh.input = True;
    xwmh.initial_state = NormalState;
    
    xtp.value = (unsigned char*)"XCruise";
    xtp.format = 8;
    xtp.nitems = strlen(appname);
    xtp.encoding = XA_STRING;
    XSetWMProperties(disp, win, &xtp, &xtp, argv, argc, &xsh, &xwmh, &myxch);
    
    xswa.colormap = cmap;
    xswa.bit_gravity = NorthWestGravity;
    XChangeWindowAttributes(disp, win, (CWColormap | CWBitGravity), &xswa);
    XSelectInput(disp, win, evmask);
    XMapWindow(disp, win);
    
    if (!XGetWindowAttributes(disp, win, xwa)) {
	fprintf(stderr, "%s : can't open window.\n", appname);
	exit(2);
    }
    return(win);
}


/*  Manage GCs
 */
GC newGC(Drawable d, char* colname, Pattern* pat)
{
    XColor col;
    XGCValues gcv;
    unsigned int vmask;
    if (!colorize || !XParseColor(disp, cmap, colname, &col) ||
	!XAllocColor(disp, cmap, &col)) {
	if (colorize) {
	    fprintf(stderr, "newGC: can't allocate color: %s\n", colname);
	}
	gcv.stipple = XCreateBitmapFromData(disp, d, 
					    (const char*)pat->pat, 8, 8);
	gcv.fill_style = FillOpaqueStippled;
	gcv.background = rgbWhite;
	gcv.foreground = rgbBlack;
	vmask = GCForeground | GCBackground | GCStipple | GCFillStyle;
    } else {
	gcv.foreground = col.pixel;
	gcv.background = rgbBlack;
	vmask = GCForeground | GCBackground;
    }
    return(XCreateGC(disp, d, vmask, &gcv));
}

void disposeGC(GC gc)
{
    XGCValues gcv;
    XGetGCValues(disp, gc, GCFillStyle, &gcv);
    XFreeGC(disp, gc);
}

XFontStruct* setFont(GC gc, char* fontname)
{
    XFontStruct* fs;
    if ((fs = XLoadQueryFont(disp, fontname)) == NULL) {
	fprintf(stderr, "%s : display %s can't load font %s.\n",
		appname, DisplayString(disp), fontname);
	exit(3);
    }
    XSetFont(disp, gc, fs->fid);
    return(fs);
}

void disposeFont(XFontStruct* fs)
{
    XFreeFont(disp, fs);
}


/*  Initialize
 */
void initAppl(int argc, char** argv)
{
    int i,i0,i1;
    unsigned int i2,i3,i4,i5;
    Window dummy;
    XWindowAttributes myxwa;
    char* s;
    
    /* getopt */
    appname = argv[0];
    getopt2(argc, argv);
    
    /* connect X */
    if ((disp = XOpenDisplay(dispname)) == NULL) {
	fprintf(stderr, "%s : can't open display named %s.\n",
		appname, XDisplayName(dispname));
	exit(1);
    }
    
    /* get resources */
    for(i = 0; myres[i].pname; i++) {
	if ((s = XGetDefault(disp, myxch.res_name, myres[i].pname)) != NULL)
	    *myres[i].pval = s;
	if ((s = XGetDefault(disp, myxch.res_class, myres[i].pname)) != NULL)
	    *myres[i].pval = s;
    }
    
    /* get colors */
    colorize = !monoflag;
    cmap = DefaultColormap(disp, DefaultScreen(disp));
    rgbWhite = WhitePixel(disp, DefaultScreen(disp));
    rgbBlack = BlackPixel(disp, DefaultScreen(disp));
    
    /* open a window */
    mywin = openSolidWindow(argc, argv, mygeom,
			    ExposureMask | KeyPressMask | ButtonPressMask,
			    &myxwa);

    /* create a background image buffer */
    myscreen = XCreatePixmap(disp, mywin, myxwa.width, myxwa.height,
			     DefaultDepth(disp, DefaultScreen(disp)));
    if (!XGetGeometry(disp, myscreen, &dummy, &i0, &i1, &i2, &i3, &i4, &i5)) {
	fprintf(stderr, "%s : can't create pixmap.\n", appname);
	exit(4);
    }
    
    /* create GCs */
    defaultgc = XCreateGC(disp, DefaultRootWindow(disp), 0, 0);
    alrtgc = newGC(DefaultRootWindow(disp), alrtcolname, &patWhite);
    alrtfont = setFont(alrtgc, alrtfontname);
    for(i = 0; i < TheColors; i++)
	stargc[i] = newGC(myscreen, starcolname[i], &starpat[i]);
    for(i = 0; i < NameFonts; i++) {
	namegc[i] = newGC(myscreen, namecolname, &patWhite);
	namefont[i] = setFont(namegc[i], namefontname[i]);
	namesize[i] = atoi(namesizestr[i]);
    }
    bggc = newGC(myscreen, bgcolname, &patBlack);
    infogc = newGC(myscreen, infocolname, &patWhite);
    meangc = newGC(myscreen, meancolname, &patWhite);
    infofont = setFont(infogc, infofontname);
    meanfont = setFont(meangc, meanfontname);
    cursorgc = XCreateGC(disp, myscreen, 0, 0);
    XSetForeground(disp, cursorgc, rgbWhite);
    XSetBackground(disp, cursorgc, rgbBlack);
    
    /* initialize variables */
    screenWidth = myxwa.width;
    screenHeight = myxwa.height;
    screenSize = (screenWidth < screenHeight)? screenHeight : screenWidth;
    meanhclip.x = MeasurePosX + MeasureHeight;
    meanhclip.y = MeasurePosY;
    meanhclip.width = MeasureSize;
    meanhclip.height = MeasureHeight;
    meanvclip.x = MeasurePosX;
    meanvclip.y = MeasurePosY + MeasureHeight;
    meanvclip.width = MeasureHeight;
    meanvclip.height = MeasureSize;

    delayticks = atoi(mydelayticks);
    diraccel = atof(mydiraccel);
    accel1 = atof(myaccel1);
    accel2 = atof(myaccel2);
}


/*  Clean up
 */
void exitApp(void)
{
    int i;

    XDestroyWindow(disp, mywin);
    collapseGalaxy(&universe.g);
    XFreePixmap(disp, myscreen);
    disposeGC(defaultgc);
    disposeGC(alrtgc);
    for(i = 0; i < TheColors; i++)
	disposeGC(stargc[i]);
    for(i = 0; i < NameFonts; i++) {
	disposeGC(namegc[i]);
	disposeFont(namefont[i]);
    }
    disposeGC(infogc);
    disposeFont(infofont);
    disposeGC(meangc);
    disposeFont(meanfont);
    disposeGC(bggc);
    disposeGC(cursorgc);
    XCloseDisplay(disp);
    exit(0);
}


/*  Main
 */
int main(int argc, char** argv)
{
    initAppl(argc, argv);
    iniTask();
    eventLoop(titleflag);
    exitApp();
}
