/* winops.c
 * Copyright (C) 2000, 2001 Hidetoshi Ohtomo.  All rights reserved. */

#include <stdio.h>
#include <stdlib.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/keysym.h>

#include "heliwm.h"
#include "config.h"

extern int find_window(Window w, struct heliwm_window **hw);
extern void set_focus(struct heliwm_window *new_focus);
extern void send_cmessage(Window w, Atom atom);

extern Display *d;
extern Window rootwin, indicator;

#ifdef NLS
extern XFontSet fontset;
extern XFontSetExtents *extents;
#else
extern XFontStruct *font_info;
#endif

extern Cursor cursor, fleur;
extern Atom wm_delete_window;
extern GC text_gc, band_gc;
extern int scr, scr_w, scr_h;
extern int ind_w, title_h, bd_w;
#ifdef FRAME
extern int mifs_w;
#endif
extern int h_offset, v_offset;
extern int scale_norm, scale_shift, scale_ctrl, scale_mod1;
extern char options;
extern unsigned long bgc, bdc;
extern struct heliwm_window *list, *last, *focus, *pointer;

static void update_indicator(int operation, int x, int y)
{
	char s[20];
#ifdef NLS
	XRectangle inkbox;
#endif

	if (operation == OP_MOVE)
		sprintf(s, "%d, %d", x, y);
	else
		sprintf(s, "%d x %d", x, y);
	XClearWindow(d, indicator);
#ifdef NLS
	XmbTextExtents(fontset, s, strlen(s), NULL, &inkbox);
	XmbDrawString(d, indicator, fontset, text_gc, (ind_w - inkbox.width) / 2
		- inkbox.x, (title_h - inkbox.height) / 2 - inkbox.y, s,
		strlen(s));
#else
	XDrawString(d, indicator, text_gc, (ind_w - XTextWidth(font_info, s,
		strlen(s))) / 2, font_info->ascent + v_offset, s, strlen(s));
#endif
}

static void draw_band(int x, int y, int width, int height)
{
	XDrawRectangle(d, rootwin, band_gc, x, y, (unsigned int) width - 1,
		(unsigned int) height - 1);
	XDrawLine(d, rootwin, band_gc, x + width / 2, y, x + width / 2, y +
		height);
	XDrawLine(d, rootwin, band_gc, x, y + height / 2, x + width, y + height
		/ 2);
}

#ifdef FRAME
static void move(struct heliwm_window *hw)
{
	XConfigureEvent e;

	XMoveWindow(d, hw->frame, hw->x, hw->y);
#ifdef SHAPE
	if (hw->flags & SHAPED)
		XMoveWindow(d, hw->child, hw->x + mifs_w, hw->y + (mifs_w +
			bd_w) * 2 + title_h);
	else
#endif /* SHAPE */
	{	/* Notify child about the move. */
		e.type = ConfigureNotify;
		e.event = hw->child;
		e.window = hw->child;
		e.x = bd_w + hw->x + mifs_w;
		e.y = bd_w + hw->y + mifs_w + title_h;
		e.width = hw->child_w;
		e.height = hw->child_h;
		e.border_width = hw->child_bw;
		e.above = None;
		e.override_redirect = False;
		XSendEvent(d, hw->child, False, StructureNotifyMask, (XEvent *)
			&e);
	}
}
#else
#define move(h) XMoveWindow(d, h->child, h->x, h->y);
#endif /* FRAME */

void move_or_resize(struct heliwm_window *hw, int operation, int child_or_icon)
{
	Window root, win;
	XEvent e;
	Bool done = False, changed_x = False;
	int ow, oh, px, py, opx, opy, goal_w, goal_h, x, y, width, height, a, b,
		scale = scale_norm;
	unsigned int state;

	if (XGrabPointer(d, rootwin, False, ButtonPressMask, GrabModeAsync,
		GrabModeAsync, None, fleur, CurrentTime) != GrabSuccess)
		return;
	else if (XGrabKeyboard(d, rootwin, True, GrabModeAsync, GrabModeAsync,
		CurrentTime) != GrabSuccess)
		goto ungrab_pointer_and_return;

	if (child_or_icon == ICON) {
		x = hw->icon_x;
		y = hw->icon_y;
		width = hw->icon_w + hw->icon_bw * 2;
		height = hw->icon_h + hw->icon_bw * 2;
	} else {
		x = hw->x, y = hw->y;
#ifdef FRAME
		width = hw->w + bd_w * 2;
		height = hw->child_h + (hw->child_bw + mifs_w + bd_w) * 2 +
			title_h;
#else
		width = hw->child_w + hw->child_bw * 2;
		height = hw->child_h + hw->child_bw * 2;
#endif
	}

	if (operation == OP_MOVE) {
		XQueryPointer(d, rootwin, &root, &win, &px, &py, &opx, &opy,
			&state);
		a = x, b = y;
	} else {
		px = py = 0;
		ow = hw->child_w;
		oh = hw->child_h;
		a = (ow - hw->base_w) / hw->inc_w;
		b = (oh - hw->base_h) / hw->inc_h;
	}

	state = 0;
	XGrabServer(d);
	XMapRaised(d, indicator);
	update_indicator(operation, a, b);
	draw_band(x, y, width, height);

	while (!done) {
		if (operation == OP_MOVE)
			opx = px, opy = py;
		XMaskEvent(d, KeyPressMask | ButtonPressMask, &e);
		if (e.type == ButtonPress) {
			done = True;
			continue;
		}
		if (e.xkey.state != state) {
			if (e.xkey.state & Mod1Mask)
				scale = scale_mod1;
			else if (e.xkey.state & ControlMask)
				scale = scale_ctrl;
			else if (e.xkey.state & ShiftMask)
				scale = scale_shift;
			else
				scale = scale_norm;
			state = e.xkey.state;
		}
		switch (XLookupKeysym(&e.xkey, 0)) {
		case XK_Down:
		case XK_j:
		case XK_n:
			py += scale;
			break;
		case XK_Up:
		case XK_k:
		case XK_p:
			py -= scale;
			break;
		case XK_Left:
		case XK_h:
		case XK_b:
			px -= scale;
			changed_x = True;
			break;
		case XK_Right:
		case XK_l:
		case XK_f:
			px += scale;
			changed_x = True;
			break;
		case XK_Return:
			done = True; /* Fall through. */
		default:
			continue;
		}
		draw_band(x, y, width, height);
		if (changed_x == True) {
			if (operation == OP_MOVE) {
				if (px < 0)
					px = 0;
				else if (px >= scr_w)
					px = scr_w - 1;
				x += px - opx;
				a = x;
			} else {
				goal_w = ow + px;
				if (goal_w <= hw->base_w) {
					px += hw->base_w - goal_w;
					if ((goal_w = hw->base_w) == 0)
						px++, goal_w++;
				} else if ((hw->flags & MAX_SIZE_DEFINED) &&
					goal_w > hw->max_w) {
					px -= goal_w - hw->max_w;
					goal_w = hw->max_w;
				}
				a = (goal_w - hw->base_w) / hw->inc_w;
				hw->child_w = hw->base_w + hw->inc_w * a;
#ifdef FRAME
				width = hw->child_w + (hw->child_bw + mifs_w +
					bd_w) * 2;
#else
				width = hw->child_w + hw->child_bw * 2;
#endif
			}
			changed_x = False;
		} else {
			if (operation == OP_MOVE) {
				if (py < 0)
					py = 0;
				else if (py >= scr_h)
					py = scr_h - 1;
				y += py - opy;
				b = y;
			} else {
				goal_h = oh + py;
				if (goal_h <= hw->base_h) {
					py += hw->base_h - goal_h;
					if ((goal_h = hw->base_h) == 0)
						py++, goal_h++;
				} else if ((hw->flags & MAX_SIZE_DEFINED) &&
					goal_h > hw->max_h) {
					py -= goal_h - hw->max_h;
					goal_h = hw->max_h;
				}
				b = (goal_h - hw->base_h) / hw->inc_h;
				hw->child_h = hw->base_h + hw->inc_h * b;
#ifdef FRAME
				height = hw->child_h + (hw->child_bw + mifs_w +
					bd_w) * 2 + title_h;
#else
				height = hw->child_h + hw->child_bw * 2;
#endif
			}
		}
		update_indicator(operation, a, b);
		draw_band(x, y, width, height);
		if (operation == OP_MOVE)
			XWarpPointer(d, None, rootwin, 0, 0, 0, 0, px, py);
	}
	draw_band(x, y, width, height);
	XUnmapWindow(d, indicator);
	if (operation == OP_MOVE) {
		if (child_or_icon == CHILD) {
			hw->x = (short) x;
			hw->y = (short) y;
			move(hw);
		} else {
			XMoveWindow(d, hw->icon, x, y);
			hw->icon_x = (short) x;
			hw->icon_y = (short) y;
		}
	} else if (hw->child_w != ow || hw->child_h != oh) {
#ifdef MAXIMIZE
		hw->flags &= ~MAXIMIZED;
#endif
		XResizeWindow(d, hw->child, (unsigned int) hw->child_w,
			(unsigned int) hw->child_h);
#ifdef FRAME
		hw->w = hw->child_w + (hw->child_bw + mifs_w) * 2;
		hw->h = title_h + mifs_w * 2 + ((hw->flags & SHAPED) ? 0 :
			hw->child_h + hw->child_bw * 2);
		XResizeWindow(d, hw->frame, (unsigned int) hw->w, (unsigned int)
			hw->h);
#endif
	}
	XUngrabServer(d);
	XUngrabKeyboard(d, CurrentTime);
ungrab_pointer_and_return:
	XUngrabPointer(d, CurrentTime);
}

void iconify(struct heliwm_window *hw)
{
	if (hw->icon == None) {
		XSetWindowAttributes attr;
		unsigned int w, h;
		unsigned long mask = CWBorderPixel;

		attr.border_pixel = bdc;
		if (hw->icon_pm != None) {
			Window win;
			int x, y;
			unsigned int depth;

			XGetGeometry(d, hw->icon_pm, &win, &x, &y, &w, &h,
				(unsigned int *) &hw->icon_bw, &depth);
			if (depth != DefaultDepth(d, scr)) {
				Pixmap tmp;

				tmp = XCreatePixmap(d, rootwin, w, h,
					DefaultDepth(d, scr));
				XCopyPlane(d, hw->icon_pm, tmp, text_gc, 0, 0,
					w, h, 0, 0, 1);
				hw->icon_pm = tmp;
				hw->flags |= CREATED_PIXMAP;
			}
			attr.background_pixmap = hw->icon_pm;
			mask |= CWBackPixmap;
		} else {
			w = (unsigned int)
#ifdef NLS
				XmbTextEscapement(fontset, hw->icon_name,
				strlen(hw->icon_name))
#else
				XTextWidth(font_info, hw->icon_name,
				strlen(hw->icon_name))
#endif
				+ h_offset * 2;
			h = (unsigned int) title_h;
			attr.background_pixel = bgc;
			mask |= CWBackPixel;
			hw->flags |= NEED_TO_DRAW_ICON;
		}
		hw->icon = XCreateWindow(d, rootwin, hw->x, hw->y, w, h, bd_w,
			CopyFromParent, InputOutput, CopyFromParent, mask,
			&attr);
		hw->icon_w = (int) w;
		hw->icon_h = (int) h;
		hw->icon_bw = bd_w;
		XSelectInput(d, hw->icon, ButtonPressMask | ExposureMask);
		XDefineCursor(d, hw->icon, cursor);
		hw->flags |= CREATED_ICON;
	}

#ifdef FRAME
	XUnmapWindow(d, hw->frame);
	if (hw->flags & SHAPED) {
		XUnmapWindow(d, hw->child);
		hw->flags |= IGNORE_UNMAP;
	}
#else
	XUnmapWindow(d, hw->child);
	hw->flags |= IGNORE_UNMAP;
#endif
	if (!(hw->flags & ICON_POS_DEFINED)) {
		hw->icon_x = (hw->x < 0) ? 0 : hw->x;
		if (hw->x >= scr_w)
			hw->icon_x = (short) scr_w - 1;
		hw->icon_y = (hw->y < 0) ? 0 : hw->y;
		if (hw->y >= scr_h)
			hw->icon_y = (short) scr_h - 1;
		XWarpPointer(d, None, rootwin, 0, 0, 0, 0, hw->icon_x,
			hw->icon_y);
		move_or_resize(hw, OP_MOVE, ICON);
		hw->flags |= ICON_POS_DEFINED;
	}
	XMapRaised(d, hw->icon);
	hw->flags |= ICONIFIED;
}

#ifdef MAXIMIZE
#define ADJUST_INC(x, b, i) ((x - b) / i) * i + b
static void maximize(struct heliwm_window *hw)
{
	int w, h;

	w = scr_w -
#ifdef FRAME
		(hw->child_bw + mifs_w + bd_w) * 2;
#else
		hw->child_bw * 2;
#endif
	h = scr_h -
#ifdef FRAME
		(hw->child_bw + mifs_w + bd_w) * 2 - title_h;
#else
		hw->child_bw * 2;
#endif

	if (w < hw->base_w)
		w = hw->base_w;
	else if ((hw->flags & MAX_SIZE_DEFINED) && w > hw->max_w)
		w = hw->max_w;
	else
		w = ADJUST_INC(w, hw->base_w, hw->inc_w);

	if (h < hw->base_h)
		h = hw->base_h;
	else if ((hw->flags & MAX_SIZE_DEFINED) && h > hw->max_h)
		h = hw->max_h;
	else
		h = ADJUST_INC(h, hw->base_h, hw->inc_h);

	if (w != hw->child_w || h != hw->child_h) {
		hw->flags |= MAXIMIZED;
		hw->ow = hw->child_w;
		hw->oh = hw->child_h;
		hw->ox = hw->x;
		hw->oy = hw->y;
		hw->child_w = w;
		hw->child_h = h;
		if (hw->x != 0 || hw->y != 0) {
			hw->x = hw->y = 0;
			move(hw);
		}
		XResizeWindow(d, hw->child, (unsigned int) w, (unsigned int) h);
#ifdef FRAME
		hw->w = (hw->child_bw + mifs_w) * 2 + w;
		hw->h = ((hw->flags & SHAPED) ? 0 : h + hw->child_bw * 2) +
			mifs_w * 2 + title_h;
		XResizeWindow(d, hw->frame, (unsigned int) hw->w,
			(unsigned int) hw->h);
#endif
	}
}
#endif

void exec_operation(Window win, int op_code)
{
	Window w;
	struct heliwm_window *hw;
	int rel;

	if (op_code != OP_NEXT && op_code != OP_PREV) {
		if (win == None || win == rootwin)
			return;
		rel = find_window(win, &hw);
		if (rel == UNDEFINED)
			return;
	}

	switch (op_code) {
	case OP_MOVE:
		if (rel == ICON)
			move_or_resize(hw, OP_MOVE, ICON);
		else
			move_or_resize(hw, OP_MOVE, CHILD);
		break;
	case OP_RESIZE:
		if (rel == CHILD)
			move_or_resize(hw, OP_RESIZE, CHILD);
		break;
	case OP_DELETE:
		if (hw->flags & DELETE_WINDOW_IS_SET)
			send_cmessage(hw->child, wm_delete_window);
		break;
	case OP_ICONIFY:
		if (rel == CHILD)
			iconify(hw);
		else if (rel == ICON) {
			XUnmapWindow(d, hw->icon);
#ifdef FRAME
#ifdef SHAPE
			if (hw->flags & SHAPED)
				XMapRaised(d, hw->child);
#endif
			XMapRaised(d, hw->frame);
#else
			XMapRaised(d, hw->child);
#endif
			hw->flags &= ~ICONIFIED;
		}
		if (focus == hw)
			set_focus(focus);
		break;
#ifdef MAXIMIZE
	case OP_MAXIMIZE:
		if (rel != CHILD)
			break;
		XGrabServer(d);
		if (hw->flags & MAXIMIZED) {
			hw->flags &= ~MAXIMIZED;
			if (hw->x != hw->ox || hw->y != hw->oy) {
				hw->x = hw->ox;
				hw->y = hw->oy;
				move(hw);
			}
			if (hw->child_w != hw->ow || hw->child_h != hw->oh) {
				hw->child_w = hw->ow;
				hw->child_h = hw->oh;
				XResizeWindow(d, hw->child, (unsigned int)
					hw->child_w, (unsigned int)
					hw->child_h);
#ifdef FRAME
				hw->w = (hw->child_bw + mifs_w) * 2 +
					hw->child_w;
				hw->h = ((hw->flags & SHAPED) ? 0 : hw->child_h
					+ hw->child_bw * 2) + mifs_w * 2 +
					title_h;
				XResizeWindow(d, hw->frame, (unsigned int)
					hw->w, (unsigned int) hw->h);
#endif
			}
		} else
			maximize(hw);
		XUngrabServer(d);
		break;
#endif /* MAXIMIZE */
	case OP_NEXT:
	case OP_PREV:
		if (list == NULL)
			break;
		if (op_code == OP_NEXT) {
			if (pointer == NULL || (pointer = pointer->next) ==
				NULL)
				pointer = list;
		} else if (pointer == NULL || (pointer = pointer->prev) == NULL)
			pointer = last;
		hw = pointer;
		if (op_code == OP_NEXT)
			while (!(pointer->flags & MAPPED)) {
				if ((pointer = pointer->next) == NULL)
					pointer = list;
				if (pointer == hw)
					break;
			}
		else
			while (!(pointer->flags & MAPPED)) {
				if ((pointer = pointer->prev) == NULL)
					pointer = last;
				if (pointer == hw)
					break;
			}
		if (pointer->flags & MAPPED) {
			int x, y;

			if (pointer->flags & ICONIFIED) {
				w = pointer->icon;
				x = pointer->icon_x;
				y = pointer->icon_y;
			} else {
#ifdef FRAME
				w = pointer->frame;
#else
				w = pointer->child;
#endif
				x = pointer->x;
				y = pointer->y;
			}
			XWarpPointer(d, None, rootwin, 0, 0, 0, 0, x, y);
			if ((options & FOCUS_WHEN_CIRCULATED) && focus !=
				pointer)
				set_focus(focus = pointer);
			if (options & RAISE_WHEN_CIRCULATED) {
				hw = pointer;
				goto raise_heliwm_window;
			}
		} else {
			pointer = NULL;
			set_focus(focus = NULL);
		}
		break;
	case OP_RAISE:
	case OP_LOWER:
		w = (rel == ICON) ? hw->icon :
#ifdef FRAME
			hw->frame;
#else
			hw->child;
#endif
		if (op_code == OP_RAISE) {
raise_heliwm_window:
			XRaiseWindow(d, w);
#ifdef SHAPE
			if (hw->flags & SHAPED)
				XRaiseWindow(d, hw->child);
#endif
			if ((options & FOCUS_WHEN_RAISED) && focus != hw)
				set_focus(focus = hw);
		} else {
			XLowerWindow(d, w);
#ifdef SHAPE
			if (hw->flags & SHAPED)
				XLowerWindow(d, hw->child);
#endif
		}
		break;
	}
}
