/* heliwm.c
 * Copyright (C) 2000, 2001 Hidetoshi Ohtomo.  All rights reserved. */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#ifdef NLS
#include <X11/Xlocale.h>
#endif

#include "heliwm.h"
#include "config.h"
#include "version.h"

extern int configure_heliwm(char *config_file_name);
extern void move_or_resize(struct heliwm_window *hw, int operation, int
	child_or_icon);
extern void iconify(struct heliwm_window *hw);
extern void exec_operation(Window win, int op_code);

Display *d;
Window rootwin, indicator;

#ifdef NLS
XFontSet fontset;
XFontSetExtents *extents;
#else
XFontStruct *font_info;
#endif

Cursor cursor, fleur;
Colormap cmap, cur_cmap;		/* Default & current colormaps. */
Time ev_time;	/* Time of {Key,Button}Press for XSetInputFocus(3X). */
Atom wm_protocols, wm_take_focus, wm_delete_window;
GC text_gc, band_gc;
#ifdef FRAME
GC iframe_gc;
#endif

int scr, scr_w, scr_h;			/* Screen number, width & height. */
int ind_w;				/* Width of indicator window. */
int title_h;				/* Height of a text line. */
int bd_w = DEF_BORDER_WIDTH;		/* Border (external frame) width. */

#ifdef FRAME
int mf_w = DEF_MIDDLE_FRAME_WIDTH;	/* Middle frame width. */
int if_w = DEF_INTERNAL_FRAME_WIDTH;	/* Internal frame width. */
int mifs_w;				/* Sum of mf_w & if_w. */
#endif

int h_offset = DEF_HORIZONTAL_OFFSET;	/* Offsets for text placement. */
int v_offset = DEF_VERTICAL_OFFSET;
int scale_norm = DEF_SCALE_NORM;	/* Move/resize scales. */
int scale_shift = DEF_SCALE_SHIFT;
int scale_ctrl = DEF_SCALE_CTRL;
int scale_mod1 = DEF_SCALE_MOD1;
int defined_keys = 0;			/* Number of defined keys. */
int defined_buttons = 0;		/* Number of defined buttons. */
unsigned long bgc, bdc;			/* Background & border colors. */
struct keyop keys[MAX_KEY_DEFINITIONS];
struct buttonop buttons[MAX_BUTTON_DEFINITIONS];
struct heliwm_window
	*list = NULL,		/* Child management list. */
	*last = NULL,		/* Last entry of the list. */
	*focus = NULL,		/* Entry with focus. */
	*pointer = NULL;	/* Entry with circulation pointer. */

static char *prog_name;
static char no_name[] = "No name";
char options = DEF_OTHER_OPTIONS;	/* Other options. */

static void version()
{
	printf("heliwm " VERSION
#ifndef FRAME
		" alpha particle"
#else
		" +FRAME"
#endif
#ifdef SHAPE
		" +SHAPE"
#endif
#ifdef NLS
		" +NLS"
#endif
#ifdef MAXIMIZE
		" +MAXIMIZE"
#endif
		"\n");
	exit(1);
}

int err_handler(Display *disp, XErrorEvent *ev)
{
	char code[16], mesg[80], req[80];

	sprintf(code, "%u", ev->request_code);
	XGetErrorDatabaseText(d, "XRequest", code, code, req, sizeof(req));
	XGetErrorText(d, ev->error_code, mesg, sizeof(mesg));
	fprintf(stderr, "\nError: error_code=%u %s;\n request_code=%u (%s); "
		"minor_code=%u; XID=%lx; serial=%ld.\n", ev->error_code, mesg,
		ev->request_code, req, ev->minor_code, ev->resourceid,
		ev->serial);

	return 0;
}

static char *get_wm_name(Window w, int child_or_icon)
{
	XTextProperty prop;

	if (child_or_icon == ICON) {
		if (XGetWMIconName(d, w, &prop) == 0)
			return NULL;
	} else if (XGetWMName(d, w, &prop) == 0)
		return NULL;
	return (char *) prop.value;
}

/* get_wm_hints() calls XGetWMHints(3X) on a window, gets attributes & returns
 * its initial state, which is one of the following: WithdrawnState,
 * NormalState, IconicState. */
static int get_wm_hints(Window win, struct heliwm_window *hw)
{
	XWMHints *hints;
	int ini_state = NormalState;

	if ((hints = XGetWMHints(d, win)) == NULL)
		return NormalState;

	if ((hints->flags & InputHint) && hints->input == True)
		hw->flags |= INPUT;
	if (hw->icon == None)
		if (hints->flags & IconWindowHint) {
			Window root;
			unsigned int depth;

			hw->icon = hints->icon_window;
			XSelectInput(d, hw->icon, ButtonPressMask);
			XDefineCursor(d, hw->icon, cursor);
			XGetGeometry(d, hw->icon, &root, (int *) &hw->icon_x,
				(int *) &hw->icon_y, (unsigned int *)
				&hw->icon_w, (unsigned int *) &hw->icon_h,
				(unsigned int *) &hw->icon_bw, &depth);
		} else if (hints->flags & IconPixmapHint)
			hw->icon_pm = hints->icon_pixmap;
	if (hints->flags & StateHint)
		ini_state = hints->initial_state;
	XFree(hints);
	return ini_state;
}

/* get_wm_normal_hints() calls XGetWMNormalHints(3X) on a window, gets size
 * hints, set *uspos True if it is non-NULL & USPosition is set, & return
 * win_gravity if compiled with FRAME option. */
#ifdef FRAME
int
#else
void
#endif
get_wm_normal_hints(Window win, struct heliwm_window *hw, Bool *uspos)
{
	XSizeHints hints;
	long int supplied;
	int min_w, min_h;

	if (XGetWMNormalHints(d, win, &hints, &supplied) == 0)
		hints.flags = 0;

	if (uspos != NULL)
		*uspos = (hints.flags & USPosition) ? True : False;

	if (hints.flags & PResizeInc)
		hw->inc_w = hints.width_inc, hw->inc_h = hints.height_inc;
	else
		hw->inc_w = hw->inc_h = 1;

	if (hints.flags & PMinSize)
		min_w = hints.min_width, min_h = hints.min_height;
	else
		min_w = min_h = 0;

	if (hints.flags & PBaseSize)
		hw->base_w = hints.base_width, hw->base_h = hints.base_height;
	else
		hw->base_w = min_w, hw->base_h = min_h;

	if ((hints.flags & PMaxSize) && hints.max_width >= hw->base_w &&
		hints.max_height >= hw->base_h) {
		hw->max_w = hints.max_width;
		hw->max_h = hints.max_height;
		hw->flags |= MAX_SIZE_DEFINED;
	}

#ifdef FRAME
	return ((hints.flags & PWinGravity) ? hints.win_gravity :
		NorthWestGravity);
#endif
}

static void get_wm_protocols(Window w, struct heliwm_window *hw)
{
	Atom *protocols;
	int i, n;

	if (XGetWMProtocols(d, w, &protocols, &n)) {
		for (i = 0; i < n; i++) {
			if (protocols[i] == wm_take_focus)
				hw->flags |= TAKE_FOCUS_IS_SET;
			else if (protocols[i] == wm_delete_window)
				hw->flags |= DELETE_WINDOW_IS_SET;
		}
		XFree(protocols);
	}
}

void send_cmessage(Window w, Atom atom)
{
	XClientMessageEvent ev;

	ev.type = ClientMessage;
	ev.window = w;
	ev.message_type = wm_protocols;
	ev.format = 32;
	ev.data.l[0] = atom;
	if (atom == wm_take_focus)
		ev.data.l[1] = ev_time;
	XSendEvent(d, w, False, 0, (XEvent *) &ev);
}

void set_focus(struct heliwm_window *new_focus)
{
	Colormap new;
	Window w = None;

	if (new_focus == NULL) {
		new = cmap;
		w = PointerRoot;
	} else {
		if (new_focus->flags & ICONIFIED) {
			new = cmap;
			w = new_focus->icon;
		} else {
			new = new_focus->cmap;
			if (new_focus->flags & TAKE_FOCUS_IS_SET)
				send_cmessage(new_focus->child, wm_take_focus);
			else
#ifdef FRAME
				w = (new_focus->flags & INPUT) ?
					new_focus->child : new_focus->frame;
#else
				w = new_focus->child;
#endif
		}
		if (options & WARP_POINTER_WHEN_FOCUSED) {
			int x, y;
			if (new_focus->flags & ICONIFIED)
				x = new_focus->icon_x, y = new_focus->icon_y;
			else
				x = new_focus->x, y = new_focus->y;
			XWarpPointer(d, None, rootwin, 0, 0, 0, 0, x, y);
			pointer = new_focus;
		}
	}
	if (w != None)
		XSetInputFocus(d, w, RevertToParent, ev_time);
	if (cur_cmap != new)
		XInstallColormap(d, cur_cmap = new);
}

/* Add a child of root window to the management list if necessary, map when
 * needed, & return pointer to the new entry.  map_request is True when called
 * from MapRequest event, False when called from initialization routine. */
static struct heliwm_window *add_child(Window win, Bool map_request)
{
	struct heliwm_window *hw;
	XWindowAttributes attr;
#ifdef FRAME
	XSetWindowAttributes sattr;
	int win_gravity;
#endif
	Window w;
	Status s;
	Bool pos_decided;
	int ini_state, child_or_icon;

	s = XGetWindowAttributes(d, win, &attr);
	if (s == BadWindow || s == BadDrawable || attr.override_redirect ==
		True || (map_request == False && attr.map_state == IsUnmapped))
		return NULL;

	if ((hw = malloc(sizeof(struct heliwm_window))) == NULL) {
		fputs("Failed to allocate memory.\n", stderr);
		return NULL;
	}
	hw->prev = NULL;
	if ((hw->next = list) == NULL)
		last = hw;
	else
		list->prev = hw;
	list = hw;
	hw->child = win;
	hw->icon = None;
	hw->icon_pm = None;
	hw->flags = MAPPED;
	hw->cmap = attr.colormap;
	hw->x = (short) attr.x;
	hw->y = (short) attr.y;
#ifdef FRAME
	if ((hw->name = get_wm_name(win, CHILD)) == NULL)
		hw->name = no_name;
#endif
	if ((hw->icon_name = get_wm_name(win, ICON)) == NULL)
		hw->icon_name =
#ifdef FRAME
			hw->name;
#else
			no_name;
#endif
	get_wm_protocols(win, hw);
	ini_state = get_wm_hints(win, hw);
#ifdef FRAME
	win_gravity =
#endif
		get_wm_normal_hints(win, hw, &pos_decided);
#ifdef FRAME
	if (win_gravity > NorthWestGravity) {
		int d;
		char m[9][3] = {{1, 0, 0}, {2, 0, 0}, {0, 1, 1}, {1, 1, 1},
			{2, 1, 1}, {0, 2, 2}, {1, 2, 2}, {2, 2, 2}, {1, 1, 2}};

		d = bd_w + mifs_w;
		win_gravity -= 2;
		hw->x -= d * m[win_gravity][0];
		hw->y -= d * m[win_gravity][1] + (title_h * m[win_gravity][2])
			/ 2;
	}
#endif
	if (XGetTransientForHint(d, win, &w) != 0)
		pos_decided = True;

#ifdef SHAPE
	{
		int b_shaped, c_shaped, xbs, ybs, xcs, ycs;
		unsigned int wbs, hbs, wcs, hcs;

		XShapeQueryExtents(d, win, &b_shaped, &xbs, &ybs, &wbs, &hbs,
			&c_shaped, &xcs, &ycs, &wcs, &hcs);
		if (b_shaped)
			hw->flags |= SHAPED;
	}
#endif

	hw->child_w = attr.width;
	hw->child_h = attr.height;
	hw->child_bw = attr.border_width;
	XSelectInput(d, win,
#ifdef FRAME
#ifdef SHAPE
		((hw->flags & SHAPED) ? StructureNotifyMask : 0) |
#endif
#else
		StructureNotifyMask |
#endif /* FRAME */
		PropertyChangeMask | ColormapChangeMask);
	XAddToSaveSet(d, hw->child);

#ifdef FRAME
	hw->w = hw->child_w + (hw->child_bw + mifs_w) * 2;
	hw->h = title_h + mifs_w * 2 +
#ifdef SHAPE
		((hw->flags & SHAPED) ? 0 : hw->child_h + hw->child_bw * 2);
#else
		hw->child_h + hw->child_bw * 2;
#endif

	sattr.background_pixel = bgc;
	sattr.border_pixel = bdc;
	sattr.cursor = cursor;
	sattr.event_mask = ButtonPressMask | ExposureMask |
#ifdef SHAPE
		((hw->flags & SHAPED) ? 0 : (SubstructureNotifyMask |
		SubstructureRedirectMask));
#else
		SubstructureNotifyMask | SubstructureRedirectMask;
#endif

	hw->frame = XCreateWindow(d, rootwin, hw->x, hw->y, (unsigned int)
		hw->w, (unsigned int) hw->h, (unsigned int) bd_w,
		CopyFromParent, InputOutput, CopyFromParent, CWBackPixel |
		CWBorderPixel | CWEventMask | CWCursor, &sattr);

#ifdef SHAPE
	if (hw->flags & SHAPED)
		XMoveWindow(d, win, hw->x + mifs_w, hw->y + (mifs_w + bd_w) * 2
			+ title_h);
	else
#endif
		XReparentWindow(d, win, hw->frame, mifs_w, mifs_w + title_h);

#endif /* FRAME */

	if (ini_state == IconicState) {
		if (!(hw->flags & ICON_POS_DEFINED)) {
			hw->icon_x = hw->x;
			hw->icon_y = hw->y;
			hw->flags |= ICON_POS_DEFINED;
		}
		iconify(hw);
		child_or_icon = ICON;
		w = hw->icon;
	} else {
		child_or_icon = CHILD;
#ifdef FRAME
		w = hw->frame;
#else
		w = hw->child;
#endif
	}

	if (pos_decided == False && map_request == True)
		move_or_resize(hw, OP_MOVE, child_or_icon);

#ifdef FRAME
#ifdef SHAPE
	if (!(hw->flags & SHAPED) || !(hw->flags & ICONIFIED))
#endif
		XMapRaised(d, win);
#endif /* FRAME */
	XMapRaised(d, w);
	return hw;
}

/* Find specified window in the list.
 * Returns CHILD or ICON for heliwm child (or frame) window or associated icon
 * window, respectively.  When w is not in the list, it returns UNDEFINED. */
int find_window(Window w, struct heliwm_window **hw)
{
	for (*hw = list; *hw != NULL; *hw = (*hw)->next)
#ifdef FRAME
		if (w == (*hw)->child || w == (*hw)->frame)
#else
		if (w == (*hw)->child)
#endif
			return CHILD;
		else if (w == (*hw)->icon)
			return ICON;
	return UNDEFINED;
}

/* Process XConfigureRequestEvent(3X). */
static void configure(XConfigureRequestEvent *e)
{
	XWindowChanges wc;
	struct heliwm_window *hw;
	unsigned int mask = 0;

	if (e->value_mask & CWStackMode)
		mask |= CWStackMode, wc.stack_mode = e->detail;

	if (find_window(e->window, &hw) == CHILD) {
		if (e->value_mask & CWSibling) {
#ifdef FRAME
			struct heliwm_window *sibling;

			wc.sibling = (find_window(e->above, &sibling) ==
				UNDEFINED) ? e->above : sibling->frame;
#else
			wc.sibling = e->above;
#endif
			mask |= CWSibling;
		}

		if (e->value_mask & CWBorderWidth) {
			if (hw->child_bw != e->border_width) {
				hw->child_bw = e->border_width;
#ifdef FRAME
				XSetWindowBorderWidth(d, hw->child,
					e->border_width);
#else
				wc.border_width = e->border_width;
				mask |= CWBorderWidth;
#endif
			}
			e->x -= e->border_width;
			e->y -= e->border_width;
		}
#ifdef FRAME
		else
			e->x -= bd_w, e->y -= bd_w;
#endif

		if ((e->value_mask & CWX) && hw->x != e->x) {
			mask |= CWX;
			wc.x = e->x;
			hw->x = (short) e->x;
		}

		if ((e->value_mask & CWY) && hw->y != e->y) {
			mask |= CWY;
			wc.y = e->y;
			hw->y = (short) e->y;
		}

		if ((e->value_mask & CWWidth) && hw->child_w != e->width) {
			mask |= CWWidth;
			hw->child_w = e->width;
#ifdef FRAME
			wc.width = hw->w = hw->child_w + (hw->child_bw +
				mifs_w) * 2;
#else
			wc.width = e->width;
#endif
		}

		if ((e->value_mask & CWHeight) && hw->child_h != e->height) {
			mask |= CWHeight;
			hw->child_h = e->height;
#ifdef FRAME
			wc.height = hw->h = title_h + mifs_w * 2 +
#ifdef SHAPE
				((hw->flags & SHAPED) ? 0 : hw->child_h +
				hw->child_bw * 2);
#else
				hw->child_h + hw->child_bw * 2;
#endif
#else
			wc.height = e->height;
#endif /* FRAME */
		}

#ifdef FRAME
		if (mask & (CWWidth | CWHeight)) {
			XResizeWindow(d, hw->child, hw->child_w, hw->child_h);
#ifdef MAXIMIZE
			hw->flags &= ~MAXIMIZED;
#endif
		}
		if (mask)
			XConfigureWindow(d, hw->frame, mask, &wc);
#else
		XConfigureWindow(d, hw->child, mask, &wc);
#endif
	} else {
		wc.x = e->x;
		wc.y = e->y;
		wc.width = e->width;
		wc.height = e->height;
		wc.border_width = e->border_width;
		wc.sibling = e->above;
		XConfigureWindow(d, e->window, (unsigned int) e->value_mask,
			&wc);
	}
}

static void destroy(struct heliwm_window *hw)
{
#ifdef FRAME
	XDestroyWindow(d, hw->frame);
#endif
	if (hw->icon != None) {
		if (hw->flags & CREATED_ICON)
			XDestroyWindow(d, hw->icon);
		if (hw->flags & CREATED_PIXMAP)
			XFreePixmap(d, hw->icon_pm);
	}
}

static void terminate()
{
	struct heliwm_window *hw, *tmp;

	XGrabServer(d);
	for (hw = list; hw != NULL;) {
#ifdef FRAME
		XReparentWindow(d, hw->child, rootwin, hw->x, hw->y);
#endif
		XRemoveFromSaveSet(d, hw->child);
		if (hw->flags & ICONIFIED) {
			XUnmapWindow(d, hw->icon);
			XMapWindow(d, hw->child);
		}
		destroy(hw);
		tmp = hw;
		hw = hw->next;
		free(tmp);
	}
	XUngrabServer(d);
	set_focus(NULL);
	XCloseDisplay(d);
	exit(0);
}

int main(int argc, char *argv[])
{
	char *cf_name = NULL, *dname = NULL;
	int i, j;
	unsigned int nchildren;
	struct heliwm_window *hw;
	Window root, w, *children;
	XEvent ev;
#ifdef NLS
	XRectangle inkbox;
#endif

	prog_name = argv[0];
	for (i = 1; i < argc; i++)
		if (argv[i][0] != '-' || ++i >= argc)
			version();
		else
			switch (argv[i - 1][1]) {
			case 'd':
				dname = argv[i];
				break;
			case 'f':
				cf_name = argv[i];
				break;
			default:
				version();
			}

#ifdef NLS
	if (setlocale(LC_ALL, "") == NULL) {
		fprintf(stderr, "%s: can't set locale.\n", argv[0]);
		return 1;
	}
#endif

	if ((d = XOpenDisplay(dname)) == NULL) {
		fprintf(stderr, "%s: can't open display %s.\n", argv[0],
			XDisplayName(dname));
		return 1;
	}

#ifdef NLS
	if (XSupportsLocale() == False) {
		fprintf(stderr, "%s: X doesn't support locale.\n", argv[0]);
		return 1;
	}
#endif

	scr = DefaultScreen(d);
	rootwin = RootWindow(d, scr);
	cur_cmap = cmap = DefaultColormap(d, scr);

	if (configure_heliwm(cf_name) != 0) {
		XCloseDisplay(d);
		return 1;
	}

	signal(SIGINT, terminate);
	signal(SIGHUP, terminate);
	signal(SIGQUIT, terminate);
	signal(SIGTERM, terminate);
	XSetErrorHandler(err_handler);

	{
		char *names[] = {"WM_PROTOCOLS", "WM_TAKE_FOCUS",
			"WM_DELETE_WINDOW"};
		Atom atoms[3];

		XInternAtoms(d, names, 3, False, atoms);
		wm_protocols = atoms[0];
		wm_take_focus = atoms[1];
		wm_delete_window = atoms[2];
	}

	scr_w = DisplayWidth(d, scr);
	scr_h = DisplayHeight(d, scr);
	cursor = XCreateFontCursor(d, XC_crosshair);
	fleur = XCreateFontCursor(d, XC_fleur);

	ind_w = h_offset * 2 +
#ifdef NLS
		XmbTextEscapement(fontset, "8888 x 8888", 11);
#else
		XTextWidth(font_info, "8888 x 8888", 11);
#endif
	indicator = XCreateSimpleWindow(d, rootwin, (scr_w - ind_w) / 2,
		(scr_h - title_h) / 2, (unsigned int) ind_w, (unsigned int)
		title_h, (unsigned int) bd_w, bdc, bgc);

	XGrabServer(d);
	XSelectInput(d, rootwin, KeyPressMask | ButtonPressMask |
		SubstructureRedirectMask);
	if (XQueryTree(d, rootwin, &root, &w, &children, &nchildren) != 0) {
		XWMHints *hints;

		/* Look for & mark icon windows in the list. */
		for (i = 0; i < nchildren; i++)
			if (children[i] != None && (hints = XGetWMHints(d,
				children[i])) != NULL) {
				if (hints->flags & IconWindowHint)
					for (j = 0; j < nchildren; j++)
						if (children[j] ==
							hints->icon_window) {
							children[j] = None;
							break;
						}
				XFree(hints);
			}
		for (i = 0; i < nchildren; i++)
			if (children[i] != None)
				add_child(children[i], False);
	}
	XUngrabServer(d);
	if (children != NULL)
		XFree(children);

	for (i = 0; i < defined_keys; i++)
		XGrabKey(d, keys[i].keycode, keys[i].state, rootwin,
			True, GrabModeAsync, GrabModeAsync);

	while (1) {
		XNextEvent(d, &ev);
		if (ev.type == KeyRelease)
			continue;
		switch (ev.type) {
		case KeyPress:
			for (i = 0; i < defined_keys; i++)
				if (keys[i].keycode == ev.xkey.keycode &&
					keys[i].state == ev.xkey.state)
					break;
			if (i == defined_keys)
				continue;
			if (keys[i].op == OP_TERMINATE)
				terminate();
			ev_time = ev.xkey.time;
			exec_operation(ev.xkey.subwindow, keys[i].op);
			continue;
		case ButtonPress:
			for (i = 0; i < defined_buttons; i++)
				if (buttons[i].button == ev.xbutton.button &&
					buttons[i].state == ev.xbutton.state)
					break;
			if (i == defined_buttons) {
				if (ev.xbutton.window == rootwin && focus !=
					NULL)
					set_focus(focus = NULL);
				continue;
			}
			if (buttons[i].op == OP_TERMINATE)
				terminate();
			ev_time = ev.xbutton.time;
			exec_operation(ev.xbutton.window, buttons[i].op);
			continue;
		case Expose:
			if (ev.xexpose.count != 0)
				continue;
			if ((i = find_window(ev.xexpose.window, &hw)) == ICON
				&& (hw->flags & NEED_TO_DRAW_ICON))
#ifdef NLS
				{
				XmbTextExtents(fontset, hw->icon_name,
					strlen(hw->icon_name), NULL, &inkbox);
				XmbDrawString(d, hw->icon, fontset, text_gc,
					h_offset, (title_h - inkbox.height) / 2
					- inkbox.y, hw->icon_name,
					strlen(hw->icon_name));
			}
#else
				XDrawString(d, hw->icon, text_gc, h_offset,
					font_info->ascent + v_offset,
					hw->icon_name, strlen(hw->icon_name));
#endif
#ifdef FRAME
			else if (i == CHILD) {
				XDrawRectangle(d, hw->frame, iframe_gc, mf_w,
					mf_w, (unsigned int) hw->w - mf_w * 2 -
					1, (unsigned int) hw->h - mf_w * 2 - 1);
#ifdef NLS
				XmbTextExtents(fontset, hw->name,
					strlen(hw->name), NULL, &inkbox);
				XmbDrawString(d, hw->frame, fontset, text_gc,
					(hw->w - inkbox.width) / 2 - inkbox.x,
					(title_h - inkbox.height) / 2 -
					inkbox.y + mifs_w, hw->name,
					strlen(hw->name));
#else
				XDrawString(d, hw->frame, text_gc, (hw->w -
					XTextWidth(font_info, hw->name,
					strlen(hw->name))) / 2, mifs_w +
					font_info->ascent + v_offset, hw->name,
					strlen(hw->name));
#endif
#ifdef SHAPE
				if (!(hw->flags & SHAPED))
#endif
					XDrawLine(d, hw->frame, iframe_gc,
						mifs_w, mifs_w + title_h - 1,
						hw->w - mifs_w, mifs_w +
						title_h - 1);
			}
#endif /* FRAME */
			continue;
		case DestroyNotify:
			if (find_window(ev.xdestroywindow.window, &hw) != CHILD)
				continue;
			if (pointer == hw)
				pointer = NULL;
			if (focus == hw)
				set_focus(focus = NULL);
			destroy(hw);
			if (hw == list)
				list = hw->next;
			else
				hw->prev->next = hw->next;
			if (hw == last)
				last = hw->prev;
			else
				hw->next->prev = hw->prev;
			free(hw);
			continue;
		case UnmapNotify:
			if (find_window(ev.xunmap.window, &hw) == UNDEFINED)
				continue;
			if (hw->flags & IGNORE_UNMAP) {
				hw->flags &= ~IGNORE_UNMAP;
				continue;
			}
#ifdef FRAME
			XUnmapWindow(d, (hw->flags & ICONIFIED) ? hw->icon :
				hw->frame);
#else
			if (hw->flags & ICONIFIED)
				XUnmapWindow(d, hw->icon);
#endif
			hw->flags &= ~MAPPED;
			if (pointer == hw)
				pointer = NULL;
			if (focus == hw)
				set_focus(focus = NULL);
			continue;
		case MapRequest:
			if ((i = find_window(ev.xmaprequest.window, &hw)) ==
				CHILD) {
				if (hw->flags & ICONIFIED) {
					XUnmapWindow(d, hw->icon);
					hw->flags &= ~ICONIFIED;
				}
				XMapRaised(d, hw->child);
#ifdef FRAME
				XMapRaised(d, hw->frame);
#endif
				hw->flags |= MAPPED;
			} else if (i == UNDEFINED) {
				XGrabServer(d);
				hw = add_child(ev.xmaprequest.window, True);
				XUngrabServer(d);
			} else
				continue;
			if ((options & FOCUS_WHEN_MAPPED) && hw != NULL)
				set_focus(focus = hw);
			continue;
		case ConfigureRequest:
			configure(&ev.xconfigurerequest);
			continue;
		case PropertyNotify:
			if (ev.xproperty.state == PropertyDelete ||
				find_window(ev.xproperty.window, &hw) != CHILD)
				continue;
			switch (ev.xproperty.atom) {
#ifdef FRAME
			case XA_WM_NAME:
				if (hw->name != no_name && hw->name !=
					hw->icon_name)
					XFree(hw->name);
				if ((hw->name = get_wm_name(ev.xproperty.window,
					CHILD)) == NULL)
					hw->name = no_name;
				if (!(hw->flags & ICONIFIED))
					XClearArea(d, hw->frame, 0, mifs_w +
						v_offset, hw->w,
#ifdef NLS
						extents->max_ink_extent.height
#else
						font_info->ascent +
						font_info->descent
#endif
						, True);
				continue;
#endif
			case XA_WM_ICON_NAME:
				if (hw->icon_name != no_name
#ifdef FRAME
					&& hw->icon_name != hw->name
#endif
					)
					XFree(hw->icon_name);
				if ((hw->icon_name =
					get_wm_name(ev.xproperty.window, ICON))
					== NULL)
					hw->icon_name =
#ifdef FRAME
						hw->name;
#else
						no_name;
#endif
				if (hw->flags & NEED_TO_DRAW_ICON) {
					hw->icon_w = h_offset * 2 +
#ifdef NLS
						XmbTextEscapement(fontset,
						hw->icon_name,
						strlen(hw->icon_name));
#else
						XTextWidth(font_info,
						hw->icon_name,
						strlen(hw->icon_name));
#endif
					XResizeWindow(d, hw->icon, (unsigned
						int) hw->icon_w, (unsigned int)
						hw->icon_h);
					if (hw->flags & ICONIFIED)
						XClearArea(d, hw->icon, 0, 0,
						0, 0, True);
				}
				continue;
			case XA_WM_HINTS:
				get_wm_hints(ev.xproperty.window, hw);
				continue;
			case XA_WM_NORMAL_HINTS:
				get_wm_normal_hints(ev.xproperty.window, hw,
					NULL);
				continue;
			default:
				if (ev.xproperty.atom == wm_protocols)
					get_wm_protocols(ev.xproperty.window,
						hw);
				continue;
			}
			continue;
		case ColormapNotify:
			if (ev.xcolormap.new == True &&
				find_window(ev.xcolormap.window, &hw) !=
				UNDEFINED) {
				hw->cmap = ev.xcolormap.colormap;
				if (focus == hw)
					XInstallColormap(d, cur_cmap =
						hw->cmap);
			}
			continue;
		default:
			continue;
		}
	}
}
