/* config.c
 * Copyright (C) 2000, 2001 Hidetoshi Ohtomo.  All rights reserved. */

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Xos.h>

#include "heliwm.h"
#include "config.h"

#define COLOR_KEY	'c'
#define FOREGROUND	0
#define BACKGROUND	1
#define BORDER		2
#define INTERNAL_FRAME	3
#define BAND		4
#define COLOR_SUB_KEYS	"fboia"
#define NUM_COLOR_SUB_KEYS	5

#define FONT_KEY	'f'
#define FONT		5
#define NUM_VALUES	(FONT + 1)

#define KEY_OP_KEY	'k'
#define BUTTON_OP_KEY	'b'

#define WIDTH_KEY	'w'
#define BORDER_WIDTH_SUB_KEY		'b'
#define MIDDLE_FRAME_WIDTH_SUB_KEY	'm'

#define OTHER_OPTION_KEY	'o'

#define NAME(x) ((given[x] != NULL) ? given[x] : def[x])

extern Display *d;
extern Window rootwin;

#ifdef NLS
extern XFontSet fontset;
extern XFontSetExtents *extents;
#else
extern XFontStruct *font_info;
#endif

extern Colormap cmap;
extern GC text_gc, band_gc;
#ifdef FRAME
extern GC iframe_gc;
#endif
extern int scr;
#ifdef FRAME
extern int mf_w, if_w, mifs_w;
#endif
extern int bd_w, title_h, v_offset;
extern int defined_keys, defined_buttons;
extern char options;
extern unsigned long bgc, bdc;
extern struct keyop keys[];
extern struct buttonop buttons[];

static void store_value(char *src, char **dest)
{
	if (*dest != NULL || (*dest = malloc(strlen(src) + 1)) == NULL)
		return;
	strcpy(*dest, src);
}

static int alloc_color(XColor *col, char *name)
{
	if (XParseColor(d, cmap, name, col) && XAllocColor(d, cmap, col))
		return 0;
	else {
		fprintf(stderr, "Failed to parse or allocate color %s.\n",
			name);
		return 1;
	}
}

static void read_config_file(FILE *config, char **given)
{
	char line[160], *p;
	int i, op, len;
	unsigned int state, xbuttons[] = {Button1, Button2, Button3, Button4,
		Button5};

	while (fgets(line, sizeof(line), config) != NULL) {
		len = strlen(line) - 1;
		if (len < 3)
			continue;
		line[len] = '\0';
		switch (line[0]) {
		case COLOR_KEY:
			if (len < 4)
				break;
			for (i = 0; i < NUM_COLOR_SUB_KEYS; i++)
				if (COLOR_SUB_KEYS[i] == line[1])
					break;
			if (i == NUM_COLOR_SUB_KEYS)
				break;
			store_value(&line[3], &given[i]);
			continue;
		case KEY_OP_KEY:
		case BUTTON_OP_KEY:
			if (len < 6)
				break;
			if (line[0] == KEY_OP_KEY) {
				if (defined_keys == MAX_KEY_DEFINITIONS)
					break;
			} else if (defined_buttons == MAX_BUTTON_DEFINITIONS)
				break;
			for (op = 0; op < NUM_OP_SUB_KEYS; op++)
				if (OP_SUB_KEYS[op] == line[1])
					break;
			if (op == NUM_OP_SUB_KEYS)
				break;
			state = 0;
			i = atoi(&line[3]);
			if (i & 0x1)
				state |= ShiftMask;
			if (i & 0x2)
				state |= ControlMask;
			if (i & 0x4)
				state |= Mod1Mask;
			if (i & 0x8)
				state |= Mod2Mask;
			p = &line[4];
			while (*p != ' ' && *p != '\t' && *p != '\0')
				p++;
			if (p == '\0')
				break;
			p++;
			if (line[0] == KEY_OP_KEY) {
				if ((keys[defined_keys].keycode =
					XKeysymToKeycode(d,
					XStringToKeysym(p))) == 0)
					break;
				keys[defined_keys].op = op;
				keys[defined_keys].state = state;
				defined_keys++;
			} else {
				if ((i = atoi(p)) < 1 || i > 5)
					break;
				buttons[defined_buttons].button = xbuttons[--i];
				buttons[defined_buttons].op = op;
				buttons[defined_buttons].state = state;
				defined_buttons++;
			}
			continue;
		case FONT_KEY:
			store_value(&line[2], &given[FONT]);
			continue;
		case WIDTH_KEY:
			switch (line[1]) {
			case BORDER_WIDTH_SUB_KEY:
				if ((bd_w = atoi(&line[2])) < 0)
					bd_w = DEF_BORDER_WIDTH;
				continue;
#ifdef FRAME
			case MIDDLE_FRAME_WIDTH_SUB_KEY:
				if ((mf_w = atoi(&line[2])) < 0)
					mf_w = DEF_MIDDLE_FRAME_WIDTH;
				continue;
#endif
			}
			break;
		case OTHER_OPTION_KEY:
			options = (char) atoi(&line[2]);
			continue;
		}
	}
}

int configure_heliwm(char *config_file_name)
{
	XColor col;
	XGCValues val;
	FILE *config;
	int specified = 0, i = 0;
	char *home, *name, *given[NUM_VALUES], *def[NUM_VALUES] = {
		DEF_FOREGROUND_COLOR,
		DEF_BACKGROUND_COLOR,
		DEF_BORDER_COLOR,
		DEF_INTERNAL_FRAME_COLOR,
		DEF_BAND_COLOR,
		DEF_FONT
	};

	if (config_file_name == NULL)
		config_file_name = HOME_CF;
	else
		specified++;

	if (strncmp(config_file_name, "~/", 2) == 0) {
		i++;
		if ((home = getenv("HOME")) == NULL)
			home = ".";
		if ((name = malloc(strlen(home) + strlen(config_file_name))) ==
			NULL) {
			fprintf(stderr, "Failed to allocate memory.\n");
			return 1;
		}
		strcpy(name, home);
		strcat(name, "/");
		strcat(name, &config_file_name[2]);
	} else
		name = config_file_name;

	config = fopen(name, "r");
	if (i != 0)
		free(name);
	if (config == NULL) {
		if (specified) {
			fprintf(stderr, "Can't open configuration file %s.\n",
				config_file_name);
			return 1;
		}
		config = fopen(SYSTEM_CF, "r");
	}

	for (i = 0; i < NUM_VALUES; i++)
		given[i] = NULL;

	if (config != NULL) {
		read_config_file(config, given);
		fclose(config);
	}

#ifdef FRAME
	mifs_w = mf_w + if_w;
#endif

#ifdef NLS
	{
		int nmissing;
		char **missing, *defstr;

		fontset = XCreateFontSet(d, NAME(FONT), &missing, &nmissing,
			&defstr);
		if (nmissing > 0) {
			fprintf(stderr, "%d missing charsets:", nmissing);
			for (i = 0; i < nmissing; i++)
				fprintf(stderr, " %s", missing[i]);
			XFreeStringList(missing);
			return 1;
		}
		extents = XExtentsOfFontSet(fontset);
	}
	title_h = extents->max_ink_extent.height + v_offset * 2;
#else
	if ((font_info = XLoadQueryFont(d, NAME(FONT))) == NULL) {
		fprintf(stderr, "Can't open font %s.\n", NAME(FONT));
		return 1;
	}
	title_h = font_info->ascent + font_info->descent + v_offset * 2;
#endif

	bgc = (alloc_color(&col, NAME(BACKGROUND)) == 0) ? col.pixel :
		BlackPixel(d, scr);
	bdc = (alloc_color(&col, NAME(BORDER)) == 0) ? col.pixel :
		WhitePixel(d, scr);

	val.foreground = (alloc_color(&col, NAME(FOREGROUND)) == 0) ?
		col.pixel : WhitePixel(d, scr);
	val.background = bgc;
#ifndef NLS
	val.font = font_info->fid;
#endif
	text_gc = XCreateGC(d, rootwin,
#ifndef NLS
		GCFont |
#endif
		GCForeground | GCBackground, &val);

#ifdef FRAME
	val.line_width = if_w;
	val.foreground = (alloc_color(&col, NAME(INTERNAL_FRAME)) == 0) ?
		col.pixel : WhitePixel(d, scr);
	iframe_gc = XCreateGC(d, rootwin, GCForeground | GCLineWidth, &val);
#endif

	val.line_width = 1;
	val.function = GXxor;
	val.foreground = (alloc_color(&col, NAME(BAND)) == 0) ? col.pixel :
		WhitePixel(d, scr);
	val.subwindow_mode = IncludeInferiors;
	band_gc = XCreateGC(d, rootwin, GCFunction | GCForeground | GCLineWidth
		| GCSubwindowMode, &val);

	for (i = 0; i < NUM_VALUES; i++)
		if (given[i] != NULL)
			free(given[i]);

	return 0;
}
