____________________________________________________________________________________________
;
;                            TechNES source
;                          2009 TechEmporium
;
;   This program is free software: you can redistribute it and/or modify
;   it under the terms of the GNU General Public License as published by
;   the Free Software Foundation, either version 3 of the License, or
;   (at your option) any later version.
;
;   This program is distributed in the hope that it will be useful,
;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;   GNU General Public License for more details.
;
;   You should have received a copy of the GNU General Public License
;   along with this program.  If not, see http://www.gnu.org/licenses.
;
____________________________________________________________________________________________
TITLE Macros
____________________________________________________________________________________________

; asm
[pop  | pop  #1   | #+1]        ; push eax, edx
[push | push #1   | #+1]        ; pop eax, edx
[mov  | mov #1 #2 | #+2]        ; mov eax 0, ebx 3
[call | push #L>2 | call #1]    ; call 'MessageBoxA' &NULL, MessageTitle, Message, &MB_OK
[copy | push #1   | pop #2]     ; copy D$Source D$Dest
[move | lea #1 D$#2]            ; move eax ecx+ebx*2-4
[enumerate | {#2 (#x-1+#F)} | #+1] ; enumerate 1, ONE, TWO, THREE...
[enumerate4 | {#2 (((#x-1) shl 2)+#F)} | #+1] ; enumerate 0, ZERO, FOUR, EIGHT...

; arguments
[arguments | push ebp | mov ebp esp | getArgument #1>L] ; arguments hWindow, Message, wParam, lParam
[getArgument | {#1 ebp+(#x shl 2)+4} | &9=(#x shl 2) | #+1]
[return | pop ebp | ret &9]

; DirectX
[dxCall ; dxCall lpdd,CreateSurface ddsd, lpdds, &NULL
    mov eax D$#1
    mov eax D$eax
    push #L>3
    call D$eax+#2 D$#1]
[dxRelease ; dxRelease lpdd
 mov eax &DD_OK
 pushad
    if D$#1 != &NULL, dxCall #1,Release
 popad
 mov D$#1 &NULL]

; Flow control
[! n  = e  != ne  >= ae  > a  < b  <= be]
[if |        cmp #1 #3 | jn#2 I9> | #4>L | I9:] ; if D$RunRoutine = &TRUE, call Routine
[on |  #=4 | cmp #1 #3 | j#2 #4]                ; on D$Quit = &TRUE, Quit
[if. | #=3 | cmp #1 #3 | jn#2 I8>>]             ; if. D$RunRoutine = &TRUE
                                                ;     mov D$RunRoutine &FALSE
                                                ;     call Routine
[else   | jmp I4>> | I8:]                       ; else
                                                ;     call SomeOtherRoutine
[break  | jmp I4>>]                             ;     if ecx = 0, break
                                                ;     mov D$RunRoutine eax
[endif  | I4: | I8:]                            ; endif
[end I4>>]

[if..        | #=3 | cmp  #1 #3 | jn#2 J8>>]
[ifFlag..    | #=2 | test #1 #2 | jz   J8>>]
[ifNotFlag.. | #=2 | test #1 #2 | jnz  J8>>]
[else..      | jmp J4>> | J8:]
[break..     | jmp J4>>]
[endif..     | J4: | J8:]
[end..         J4>]

[ifFlag     | test #1 #2 | jz  I7> | #3>L | I7:] ; ifFlag D$FlagRegister FLAG, call FlagRoutine
[ifNotFlag  | test #1 #2 | jnz I6> | #3>L | I6:]
[ifFlag.    | test #1 #2 | jz  I8>>]
[ifNotFlag. | test #1 #2 | jnz I8>>]

[ifKeyDown | pushad | call 'User32.GetAsyncKeyState' #1 | and ax 08000 | popad | jz I5> | #2>L | I5:]
[ifKeyUp   | pushad | call 'User32.GetAsyncKeyState' #1 | and ax 08000 | popad | js I4> | #2>L | I4:]

; Message box
[msg | #=3 | pushad | call 'User32.MessageBoxA' D$hMainWindow &0 #1 &MB_APPLMODAL+#3 | popad]
[msgWarning | msg Warning  {#1, 0} &MB_ICONEXCLAMATION]
[msgInfo    | msg Info     {#1, 0} &MB_ICONINFORMATION]
[msgError   | msg Error    {#1, 0} &MB_ICONERROR]
[Warning: B$ 'Warning!',    0
 Info:    B$ 'Information', 0
 Error:   B$ 'Error!',      0]

; Misc
[. | call #3>L | mov #1 eax] ; . D$hMem = 'GlobalAlloc' &GPTR, 01000
[Temp: ?]
____________________________________________________________________________________________
; Useful routines
____________________________________________________________________________________________

ZeroMemory:

    arguments @pDest, @Length
    pushad
        mov edi D@pDest, eax 0, ecx D@Length
        rep stosb
    popad
    return

CopyMemory:

    arguments @pSource, @pDest, @Length
    pushad
        mov esi D@pSource, edi D@pDest, ecx D@Length
        rep movsb
    popad
    return

AddString:

    arguments @pFirst @pSecond

    pushad

        ; Length of second
        mov ecx 0-1, al 0, edi D@pSecond
        repne scasb
        push ecx

            ; End of first
            mov edi D@pFirst
            repne scasb | dec edi

        ; Copy
        pop ecx | not ecx
        mov esi D@pSecond
        repne movsb

    popad
    return

CRC32: ; call CRC32 D$Pointer, D$Length  -->  CRC in eax

    [CRC32_POLY 0EDB8_8320]
    ; Completely unreadable code
    pop eax, esi, ecx | push eax
    mov eax 0FFFF_FFFF
    jecxz L9>
    mov edx 0
L1: xor al, B$esi | inc esi
L2: shr eax, 1 | jnc L3> | xor eax, CRC32_POLY | L3:
    add dl 020 | jnz L2<
    dec ecx | jnz L1<
L9: not eax
    ret
____________________________________________________________________________________________
TITLE WinMain

Main:
____________________________________________________________________________________________
; Load resources
___________________________________________________________________________________________

    ; Load
  . D$hInstance = 'Kernel32.GetModuleHandleA' &NULL  S     | winErrorCheck 'Could not get module handle.'
  . D$hMenu     = 'User32.LoadMenuA' D$hInstance, IDM_MENU | winErrorCheck 'Could not load menu.'

    ; Load icons
    mov ecx NUM_ICONS
L0: push ecx
        call 'User32.LoadIconA' D$hInstance, ecx  | winErrorCheck 'Could not load icon.'
    pop ecx
    dec ecx
    mov D$hIcon+ecx*4 eax
    jnz L0<

    ; Load RC data
  . D$pDatabase  = LoadResource IDR_DATABASE
  . D$pConfig    = LoadResource IDR_CONFIG
  . D$pPalette   = LoadResource IDR_PALETTE
  . D$pLicense   = LoadResource IDR_LICENSE
  . D$pChangelog = LoadResource IDR_CHANGELOG
  . D$pManual    = LoadResource IDR_MANUAL

    ; Set database
    mov eax D$pDatabase
    mov edx D$eax | mov D$nROMs edx | add eax 4
    mov D$pCRC32 eax     | move eax eax+edx*4
    mov D$pPROMCRC32 eax | move eax eax+edx*4
    mov D$pData eax      | move eax eax+edx*8
    mov D$pNames eax

____________________________________________________________________________________________
; Create window
____________________________________________________________________________________________

    ; Register class
  . D$WindowClassEx@hbrBackground = 'GDI32.GetStockObject' &BLACK_BRUSH | winErrorcheck 'Could not get background brush.'
    copy D$hInstance D$WindowClassEx@hInstance
    copy D$hIcon     D$WindowClassEx@hIcon
    call 'User32.RegisterClassExA' WindowClassEx | winErrorCheck 'Could not register class.'

    ; Adjust window coordinates
    call 'User32.AdjustWindowRectEx' WindowRect, WINDOW_STYLE, &FALSE, WINDOW_STYLE_EX
    mov eax D$WindowRect+8  | sub eax D$WindowRect
    mov edx D$WindowRect+12 | sub edx D$WindowRect+4
    add D$WindowRect+00 040
    add D$WindowRect+04 040
    add D$WindowRect+08 040
    add D$WindowRect+0C 040

    ; Create window
  . D$hMainWindow = 'User32.CreateWindowExA' WINDOW_STYLE_EX,
                                             ClassName,
                                             Classname,
                                             WINDOW_STYLE,
                                             &CW_USEDEFAULT, &CW_USEDEFAULT,
                                             eax, edx,
                                             &NULL,
                                             &NULL,
                                             D$hInstance,
                                             &NULL
    winErrorCheck 'Could not create window.'
    call ShowMenu
____________________________________________________________________________________________
; Init other
____________________________________________________________________________________________

    ; DirectX
    call VideoInit
    call AudioInit
    call InputInit

    ; Nessie files
    call ParseCommandLine
    call LoadConfiguration
    if B$Argument != 0, call LoadRemoteFile Argument
____________________________________________________________________________________________
; Main loop
____________________________________________________________________________________________

    [Paused: ?]
L0: on B$Window@Active = &FALSE,   S1>
    on B$Window@Minimized = &TRUE, S1>
    on B$Paused = &TRUE,           S0>
    on D$Power != POWER_ON,        S0>
    on D$pImage = &NULL,           S0>

    call RunOneFrame
S0: call GetDeviceStates
    call TranslateInput
    call HandleInput
    jmp S2>

S1: call 'User32.WaitMessage'

____________________________________________________________________________________________
; Check message queue
____________________________________________________________________________________________

    ; Check for message
    [@Message: ? #7]
S2: call 'User32.PeekMessageA' @Message, 0, 0, 0, &PM_REMOVE
    on eax = 0, L0<<

    ; Handle message
    call 'User32.TranslateMessage' @Message
    call 'User32.DispatchMessageA' @Message
    mov eax &FALSE
    on D@Message+04 != &WM_QUIT, L0<<

____________________________________________________________________________________________
; Shutdown
____________________________________________________________________________________________

    call UnloadUnzipDLL
    call 'User32.UnregisterClassA' ClassName, D$hInstance | winErrorCheck 'Could not unregister class.'
    call 'Kernel32.ExitProcess' 0

____________________________________________________________________________________________
; Windows data
____________________________________________________________________________________________

[hInstance: ?]

; Window class
[ClassName: B$ 'TechNES', 0]
[WindowClassEx:
 @cbSize:        len
 @style:         &CS_HREDRAW+&CS_VREDRAW
 @lpfnWndProc:   MainWindowProc
 @cbClsExtra:    0
 @cbWndExtra:    0
 @hInstance:     0
 @hIcon:         0
 @hCursor:       0
 @hbrBackground: 0
 @lpszMenuName:  0
 @lpszClassName: ClassName
 @hIconSm:       0]

; Window
[hMainWindow: ?]
[Window:
 @Active:       ?
 @Minimized:    ?
 @Existing:     ?]
[hMenu: ?]
[NUM_ICONS 2]
[hIcon:  ? #NUM_ICONS]
[ScreenWidth:  0100
 ScreenHeight: (NTSC_SCREEN_BOTTOM-NTSC_SCREEN_TOP)]
[DisplayMode: ?]
[DM_NOTHING    0
 DM_WINDOWED   1
 DM_FULLSCREEN 2]
[FULLSCREEN_STYLE  &WS_VISIBLE+&WS_POPUP]
[WINDOW_STYLE      &WS_VISIBLE+&WS_OVERLAPPEDWINDOW]
[WINDOW_STYLE_EX   &WS_EX_ACCEPTFILES]
[WindowRect: 0, 0, 512, ((NTSC_SCREEN_BOTTOM-NTSC_SCREEN_TOP) shl 1)]

enumerate 1,
    IDT_MAIN,
    IDT_ABOUT,
    IDT_PRESSNEWBUTTON,
    IDT_AUTOFIRE

____________________________________________________________________________________________
; Resources
____________________________________________________________________________________________

LoadResource:

    arguments @ID
    call 'Kernel32.FindResourceA' D$hInstance, D@ID, &RT_RCDATA
    call 'Kernel32.LoadResource' D$hInstance, eax
    call 'Kernel32.LockResource' eax
    return

enumerate 1,

    IDR_DATABASE,
    IDR_CONFIG,
    IDR_PALETTE,
    IDR_LICENSE,
    IDR_CHANGELOG,
    IDR_MANUAL

; RC data
[pDataBase:  ?
 pConfig:    ?
 pPalette:   ?
 pLicense:   ?
 pChangelog: ?
 pManual:    ?]
; Database
[pCRC32:     ?
 pPROMCRC32: ?
 pData:      ?
 pNames:     ?
 nROMs:      ?]
____________________________________________________________________________________________
; Command line parser
____________________________________________________________________________________________

ParseCommandLine:

    [@pCommandLine: ?
     @nLength:      ?]

    ; Get command line
  . D@pCommandLine = 'Kernel32.GetCommandLineA'
    mov edi eax, ecx 0-1, al 0
    repne scasb | not ecx | mov D@nLength ecx

    ; Application path
    mov edi D@pCommandLine, al ' ', ecx D@nLength
    if. B$edi = '"'
        mov al '"'
        inc edi
    endif
    repne scasb
    mov D@nLength ecx
    jecxz Q0>

    ; Move to first argument
    mov esi edi
L0: lodsb | dec D@nLength
    on al = 0, Q0>
    on al = ' ', L0<
    on al = '"', L0<

    dec esi | inc D@nLength
    mov edi esi, al B$esi-1, ecx D@nLength
    repne scasb
    inc ecx | sub D@nLength ecx

    ; Copy argument
    call CopyMemory esi, Argument, D@nLength


    ; Get application path
Q0: call 'Kernel32.GetModuleFileNameA' D$hInstance, Filename, &MAX_PATH
    mov edi Filename, ecx 0-1, al 0 | repne scasb
    std
        neg ecx
        mov al '\' | repne scasb
    cld
    add edi 2 | sub edi Filename

    ; Set directories
    call CopyMemory Filename, FilesDirectory, edi
    mov B$FilesDirectory+edi 0
    call AddString FilesDirectory, FilesName
    call AddString NessieDirectory, FilesDirectory
    call AddString NessieDirectory, NessieName
    ret

____________________________________________________________________________________________
TITLE WindowProc
____________________________________________________________________________________________
; Winproc
____________________________________________________________________________________________

[MessageTable:

 ; User interaction
 &WM_NCLBUTTONDBLCLK,   WmNCLButtonDblClick
 &WM_NCMBUTTONDBLCLK,   Silence
 &WM_NCRBUTTONDBLCLK,   Silence
 &WM_NCLBUTTONDOWN,     Silence
 &WM_NCMBUTTONDOWN,     Silence
 &WM_NCRBUTTONDOWN,     Silence

 &WM_ENTERSIZEMOVE,     Silence
 &WM_ENTERMENULOOP,     Silence
 &WM_COMMAND,           WmCommand
 &WM_DROPFILES,         WmDropFiles
 &WM_LBUTTONDOWN,       WmLButtonDown
 &WM_LBUTTONUP,         WmLButtonUp
 &WM_RBUTTONDOWN,       WmRButtonDown
 &WM_MOUSEMOVE,         WmMouseMove
 &WM_KEYDOWN,           WmKeyDown

 ; Drawing
 &WM_PAINT,             WmPaint
 &WM_ERASEBKGND,        WmEraseBackground
 &WM_DISPLAYCHANGE      WmDisplayChange

 ; Borders
 &WM_SIZING,            WmSizing
 &WM_SIZE,              WmSize
 &WM_MOVE,              WmMove
 &WM_MOVING,            WmMoving

 ; System notifications
 &WM_CREATE             WmCreate
 &WM_TIMER              WmTimer
 &WM_ACTIVATE,          WmActivate
 &WM_ACTIVATEAPP,       WmActivateApp
 &WM_SYSCOMMAND,        WmSysCommand
 &WM_DESTROY,           WmDestroy

 @EndMessage: 0         Default]

MainWindowProc:

    arguments hWindow, Message, wParam, lParam
    pushad

        ; Use the jump table
        copy D$Message D$MessageTable@EndMessage
        mov esi MessageTable
    L0: lodsd | add esi 4 | on eax != D$Message, L0<
        call D$esi-4

        mov D$esp+28 eax

    popad
    return
____________________________________________________________________________________________

; DefWindowProc
Default: call 'User32.DefWindowProcA' D$hWindow, D$Message, D$wParam, D$lParam | ret
____________________________________________________________________________________________

WmCreate:

    ; Create timer
    call 'User32.SetTimer' D$hWindow, IDT_MAIN, 250, &NULL

    mov D$Window@Existing &TRUE
    mov eax &TRUE
    ret

; Got/lost focus
WmActivateApp:

    ; Application lost focus? Fullscreen --> Minimize
    if W$wParam = &FALSE,
        if D$DisplayMode = DM_FULLSCREEN,
            call 'User32.CloseWindow' D$hWindow
    mov eax 0
    ret

WmActivate:

    ; Memorize window active/inactive
    call ClearAudio
    cmp W$wParam &WA_INACTIVE | setne B$Window@Active
    mov eax 0
    ret

WmTimer:

    [@Icon: 0]
    xor B@Icon 1 | mov ebx D@Icon
    call 'User32.SetClassLongA' D$hWindow, &GCL_HICON, D$hIcon+ebx*4
    ret

WmSysCommand:

    on D$wParam != &SC_SCREENSAVE, Default
    mov eax 0-1
    ret

WmDestroy:

    call UnloadNES

    ; Destroy
    call 'User32.KillTimer' D$hWindow, IDT_MAIN
    call 'User32.DestroyMenu' D$hMenu | winErrorCheck 'Could not destroy menu.'
    mov D$Window@Existing &FALSE

    ; Shut down DirectX
    call InputShutdown
    call AudioShutdown
    call VideoShutdown

    call 'User32.PostQuitMessage' 0
    mov eax 0
    ret

____________________________________________________________________________________________

; Clear sound buffer
Silence: call ClearAudio | call Default |  ret

; Menu
WmCommand:

    if. W$wParam+2 != 0

        mov eax 1

    else

        ; MRU list?
        if.. D$wParam > IDM_RECENT

            call MenuRecentItem

        else..

            ; Find the routine that handles this menu item
            mov eax D$wParam | sub eax IDM_MENU+1
            move eax MenuItemTable+eax*4
            if eax < EndOfMenuItemTable, call D$eax

        endif..
        mov eax 0

    endif
    ret

; Keyboard
WmKeyDown:
;;
    [Debug: 32 0 0 0 0]
    if D$wParam = &VK_F1,  xor B$Emulate@SoundSquare1  &TRUE
    if D$wParam = &VK_F2,  xor B$Emulate@SoundSquare2  &TRUE
    if D$wParam = &VK_F3,  xor B$Emulate@SoundTriangle &TRUE
    if D$wParam = &VK_F4,  xor B$Emulate@SoundNoise    &TRUE
    if D$wParam = &VK_F5,  xor B$Emulate@SoundDMC      &TRUE
    if D$wParam = &VK_6,   xor B$Emulate@SpriteZero    &TRUE
    if D$wParam = &VK_7,   xor B$Emulate@Background    &TRUE
    if D$wParam = &VK_8,   xor B$Emulate@Sprites       &TRUE
    if. D$wParam = &VK_9 | inc D$Debug+00 | outdec D$Debug+00 | endif
    if. D$wParam = &VK_0 | dec D$Debug+00 | outdec D$Debug+00 | endif
    if. D$wParam = &VK_O | inc D$Debug+04 | outdec D$Debug+04 | endif
    if. D$wParam = &VK_P | dec D$Debug+04 | outdec D$Debug+04 | endif
;;
    mov eax 0
    ret

; Mouse messages
WmDropFiles:

    call 'Shell32.DragQueryFile' D$wParam, 0, FullFilename, &MAX_PATH
    call LoadRemoteFile FullFilename
    mov eax 0
    ret

WmMouseMove:

    call ZapperCoordinates D$lParam
    mov eax 0
    ret

WmLButtonDown:

    mov B$Zapper@Triggered &TRUE
    call ZapperCoordinates D$lParam
    mov eax 0
    ret

WmLButtonUp:

    mov B$Zapper@Triggered &FALSE
    call ZapperCoordinates D$lParam
    mov eax 0
    ret

WmRButtonDown:

    call ToggleMenu
    mov eax 0
    ret

WmNCLButtonDblClick:

    if D$Displaymode = DM_FULLSCREEN, call SetWindowed
    jmp Silence
____________________________________________________________________________________________

; Paint messages
WmEraseBackground:

    on D$Window@Active = &FALSE, Default
    on D$DisplayMode = DM_FULLSCREEN, Default
    mov eax &TRUE
    ret

WmPaint:

    call Default
    call UpdateVideo
    mov eax 0
    ret

WmDisplayChange:

    call VideoShutDown
    call VideoInit
    ret

____________________________________________________________________________________________

; Window borders changed?

; Horrible code, but it works.
; Ratio is specified by D$ScreenWidth and D$ScreenHeight
WmSizing:

    [@Rect: ? #4]
    call ZeroMemory @Rect, 010
    call 'User32.AdjustWindowRectEx' @Rect, WINDOW_STYLE, &FALSE, WINDOW_STYLE_EX
    mov ebx D$lParam

    ; Border --> client area
    mov ecx 0
L0: mov eax D$ebx+ecx*4
    sub eax D$@Rect+ecx*4
    mov D$ebx+ecx*4 eax
    inc ecx | on ecx < 4, L0<

    mov eax D$ebx+8 | sub eax D$ebx
    if. eax < D$ScreenWidth
        call 'User32.GetWindowRect' D$hWindow, ebx
        mov eax &TRUE
        ret
    endif

    mov eax D$ebx+12 | sub eax D$ebx+4
    if. eax < D$ScreenHeight
        call 'User32.GetWindowRect' D$hWindow, ebx
        mov eax &TRUE
        ret
    endif

    ; Adjust which edge?
    on D$wParam = &WMSZ_LEFT,  L0>
    on D$wParam = &WMSZ_RIGHT, L0>
    on D$wParam = &WMSZ_BOTTOMLEFT, L1>
    on D$wParam = &WMSZ_TOPLEFT,    L1>

    ; Adjust right
    mov eax D$ebx+12
    sub eax D$ebx+4
    mov ecx D$ScreenWidth | mul ecx

    mov ecx D$ScreenHeight | div ecx
    add eax D$ebx
    mov D$ebx+8 eax
    jmp L2>

    ; Adjust bottom
L0: mov eax D$ebx+8
    sub eax D$ebx
    mov ecx D$ScreenHeight | mul ecx

    mov ecx D$ScreenWidth | div ecx
    add eax D$ebx+4
    mov D$ebx+12 eax
    jmp L2>

    ; Adjust left
L1: mov eax D$ebx+12
    sub eax D$ebx+4
    mov ecx D$ScreenWidth | mul ecx

    mov ecx D$ScreenHeight | div ecx
    mov edx D$ebx+8 | sub edx eax
    mov D$ebx edx

    ; Client area --> border
L2: mov ecx 4
L0: mov eax D$ebx+ecx*4-4
    add eax D$@Rect+ecx*4-4
    mov D$ebx+ecx*4-4 eax
    dec ecx | jnz L0<

    ; Return
    mov eax &TRUE
    ret

WmSize:

    ; Resized -> Save rect
    if D$wParam = &SIZE_RESTORED,
        if D$DisplayMode = DM_WINDOWED,
            call 'User32.GetWindowRect' D$hWindow WindowRect

    ; Maximized -> Fullscreen
    if D$wParam = &SIZE_MAXIMIZED, call SetFullscreen

    ; Minimized -> Set flag
    cmp D$wParam &SIZE_MINIMIZED | sete B$Window@Minimized

    ; Window style
    call 'User32.SetWindowLongA' D$hWindow, &GWL_STYLE, WINDOW_STYLE
    if D$Window@Minimized = &FALSE,
        if D$DisplayMode = DM_FULLSCREEN,
            call 'User32.SetWindowLongA' D$hWindow, &GWL_STYLE, FULLSCREEN_STYLE

    ; No display mode yet?
    if D$DisplayMode = &NULL, call SetWindowed
    call UpdateClientRect
    mov eax 0
    ret

WmMoving:

    call CopyMemory D$lParam, WindowRect, 010
    mov eax &TRUE
    ret

WmMove:

    ; Minimized --> negative coordinates
    ifFlag D$lParam 08000,
        ifNotFlag D$lParam 04000, ret
    call UpdateClientRect

    mov eax &TRUE
    ret

____________________________________________________________________________________________
; Cursor
____________________________________________________________________________________________

UpdateCursor:

    ; Visible?
    call HideCursor
    call 'User32.GetMenu' D$hMainWindow
    if eax != &NULL, call ShowCursor
    if D$DisplayMode != DM_FULLSCREEN, call ShowCursor

    ; Standard arrow
    call 'User32.LoadCursorA' &NULL, &IDC_ARROW
    call 'User32.SetClassLongA' D$hMainWindow, &GCL_HCURSOR, eax

    ; Zapper?
    if B$Port1 != CONTROLLER_ZAPPER,
        if B$Port2 != CONTROLLER_ZAPPER,
            ret
    call ShowCursor
    call 'User32.LoadCursorA' D$hInstance, 1
    call 'User32.SetClassLongA' D$hMainWindow, &GCL_HCURSOR, eax
    ret

[CursorHidden: ?]
HideCursor: if B$CursorHidden = &FALSE, call 'User32.ShowCursor' &FALSE | mov b$CursorHidden &TRUE  | ret
ShowCursor: if B$CursorHidden = &TRUE,  call 'User32.ShowCursor' &TRUE  | mov b$CursorHidden &FALSE | ret

____________________________________________________________________________________________
; OpenFilename data
____________________________________________________________________________________________

[FullFilename:  B$ ? #&MAX_PATH]
[FilterStrings: B$ 'NES ROMs', 0                  '*.nes;*.zip', 0
                   'All supported file types', 0, '*.nes;*.zip;*.pal;*.cfg;*.sst;*.sav;*.ips', 0, 0]
[OpenFilename:
 @lStructSize:         D$ len
 @hwndOwner:           D$ 0
 @hInstance:           D$ 0
 @lpstrFilter:         D$ FilterStrings
 @lpstrCustomFilter:   D$ 0
 @nMaxCustFilter:      D$ 0
 @nFilterIndex:        D$ 0
 @lpstrFile:           D$ FullFilename
 @nMaxFile:            D$ &MAX_PATH
 @lpstrFileTitle:      D$ 0
 @nMaxFileTitle:       D$ 0
 @lpstrInitialDir:     D$ 0
 @lpstrTitle:          D$ 0
 @Flags:               D$ &OFN_FILEMUSTEXIST+&OFN_EXPLORER
 @nFileOffset:         W$ 0
 @nFileExtension:      W$ 0
 @lpstrDefExt:         D$ 0
 @lCustData:           D$ 0
 @lpfnHook:            D$ 0
 @lpTemplateName:      D$ 0]
____________________________________________________________________________________________
TITLE Files

; Here's where all file loading/saving is handled.
; Pass the full file name to SaveRemoteFile or LoadRemoteFile.
; They will use the filename extension to decide what to do with it.
; From there, *.sav files are handled by LoadSAV and SaveSAV, *.cfg by LoadCFG, SaveCFG and so on.
; SaveNessieFile will save .\saves\settings\[argument]
; SaveROMFile will save .\saves\[ROMName]\[argument]
; For example, call LoadROMFile 'palette.pal' will load the palette for this ROM (if any).
; SaveNumberedROMFile is used for screenshots and save games. It saves .\saves\[ROMName]\[filename]xxx.[extension].

____________________________________________________________________________________________
; Save file
____________________________________________________________________________________________

SaveNessieFile:

    arguments @pFilename

    ; .\saves\
    call 'Kernel32.CreateDirectoryA' FilesDirectory, &NULL
    ; .\saves\settings\
    call 'Kernel32.CreateDirectoryA' NessieDirectory, &NULL
    ; .\saves\settings\filename.ext
    mov B$Filename 0
    call AddString Filename, NessieDirectory
    call AddString Filename, D@pFilename
    call SaveRemoteFile Filename
    return

SaveROMFile:

    arguments @pFilename

    ; .\saves\ROMTitle\
    call CreateROMDirectory
    ; .\saves\ROMTitle\filename.ext
    mov B$Filename 0
    call AddString Filename, ROMDirectory
    call AddString Filename, D@pFilename
    call SaveRemoteFile Filename
    return

SaveRemoteFile:

    arguments @pSaveFilename

    ; Extension
    mov edi D@pSaveFilename, al 0, ecx 0-1 | repne scasb
    copy D$edi-5 D$Extension | or D$Extension 020202020

    ; Identify
    if. D$Extension = '.pal' | call SavePAL | else
    if. D$Extension = '.cfg' | call SaveCFG | else
    if. D$Extension = '.gen' | if D$pImage != &NULL, call SaveGEN | else
    if. D$Extension = '.sav' | if D$pImage != &NULL, call SaveSAV | else
    if. D$Extension = '.bmp' | if D$pImage != &NULL, call SaveBMP | else
    if. D$Extension = '.sst' | if D$pImage != &NULL, call SaveSST | endif

    ; Open file
  . D$hFile = 'Kernel32.CreateFileA' D@pSaveFilename, &GENERIC_WRITE, &NULL, &NULL, &CREATE_ALWAYS, &FILE_ATTRIBUTE_NORMAL, &NULL

    if. D$hFile != &INVALID_HANDLE_VALUE
        call 'Kernel32.WriteFile' D$hFile, SaveData, D$SaveSize, Temp, &NULL
        call 'Kernel32.CloseHandle' D$hFile
    endif
    return

SaveNumberedROMFile:

    arguments @pFilename, @pExtension

    call CreateROMDirectory
    mov D$FileNumber 0

    ; - Convert to decimal -
    ; Push numbers
L0: mov eax D$FileNumber, ecx 10
    push 0FF
L1: mov edx 0 | div ecx
    push edx | on eax != 0, L1<
    ; Pop numbers
    mov edi FileNumberString
L2: pop eax
    if. eax != 0FF
        add al '0' | stosb
        jmp L2<
    endif
    mov B$edi 0

    ; Check if file exists
    mov B$Filename 0
    call AddString Filename, ROMDirectory
    call AddString Filename, D@pFilename
    call AddString Filename, FileNumberString
    call AddString Filename, D@pExtension
    call 'Kernel32.FindFirstFileA' Filename, WIN32_FIND_DATA

    ; File exists, try next number
    if. eax != &INVALID_HANDLE_VALUE
        call 'Kernel32.FindClose' eax
        inc D$FileNumber | jnz L0<< ; jnz, if someone would produce > 0FFFFFFFF save files :-D
    endif

    ; Found an unused number
    call SaveRemoteFile Filename
    return

[FileNumber: ?]
[FileNumberString: B$ ? #10]
[WIN32_FIND_DATA: D$ ? #11
 @cFileName:      B$ ? #&MAX_PATH
 @cAlternate:     B$ ? #14]

____________________________________________________________________________________________
; Load file
____________________________________________________________________________________________

LoadNessieFile:

    arguments @pFileName

    mov B$Filename 0
    call AddString Filename, NessieDirectory
    call AddString Filename, D@pFilename
    call LoadRemoteFile Filename
    return

LoadROMFile:

    arguments @pFileName

    if. D$pImage != &NULL
        mov B$Filename 0
        call AddString Filename, ROMDirectory
        call AddString Filename, D@pFilename
        call LoadRemoteFile Filename
    endif
    return

LoadRemoteFile:  ; Sets D$pFile, D$FileSize and Filetitle

    arguments @pLoadFilename

    ; Extension
    mov edi D@pLoadFileName, al 0, ecx 0-1 | repne scasb
    copy D$edi-5 D$Extension | or D$Extension 020202020

    ; Filetitle
    std
        mov al '\' | repne scasb
        add edi 2
    cld
    mov B$Filetitle 0
    call AddString Filetitle, edi
    mov edi Filetitle, ecx 0-1, al 0
    repne scasb
    sub edi 5 | call ZeroMemory edi, 4

    ; Open file
  . D$hFile = 'Kernel32.CreateFileA' D@pLoadFilename, &GENERIC_READ, &NULL, &NULL, &OPEN_EXISTING, &FILE_ATTRIBUTE_NORMAL, &NULL

    if.. D$hFile != &INVALID_HANDLE_VALUE

        ; Allocate space, read file
      . D$FileSize = 'Kernel32.GetFileSize' D$hFile &NULL
        if. eax <= 0800000 ; Files > 8MB are skipped
          . D$pFile = 'Kernel32.GlobalAlloc' &GPTR, D$FileSize                 | winErrorCheck 'Could not allocate memory for file.'
            call 'Kernel32.ReadFile' D$hFile, D$pFile, D$FileSize, Temp, &NULL | winErrorCheck 'Could not read file.'
        endif

        ; Close file
        call 'Kernel32.CloseHandle' D$hFile

        ; Focus on this window
        call 'User32.SetForegroundWindow' D$hMainWindow

        ; Convert to long name
        call 'Kernel32.GetLongPathNameA' D@pLoadFilename, D@pLoadFilename, &MAX_PATH
        mov edi FileTitle

        ; Identify
        if. D$Extension = '.zip' | call AddMRUItem D@pLoadFilename | call LoadZIP D@pLoadFilename | else
        if. D$Extension = '.nes' | call AddMRUItem D@pLoadFilename | call LoadNES | else
        if. D$Extension = '.pal' | call LoadPAL | else
        if. D$Extension = '.cfg' | call LoadCFG | else
        if. D$Extension = '.gen' | if D$pImage != &NULL, call LoadGEN | else
        if. D$Extension = '.ips' | if D$pImage != &NULL, call LoadIPS | else
        if. D$Extension = '.sst' | if D$pImage != &NULL, call LoadSST | else
        if. D$Extension = '.sav' | if D$pImage != &NULL, call LoadSAV | endif

        call 'Kernel32.GlobalFree' D$pFile | mov D$pFile &NULL

    endif..
    return

LoadFirstFile:

    arguments @pDirectory, @pFilename
    mov B$Filename 0
    call AddString Filename, D@pDirectory
    call AddString Filename, D@pFilename
    call 'Kernel32.FindFirstFileA' Filename, WIN32_FIND_DATA
    if. eax != &INVALID_HANDLE_VALUE
        call 'Kernel32.FindClose' eax
        mov B$Filename 0
        call AddString Filename, D@pDirectory
        call AddString Filename, WIN32_FIND_DATA@cFilename
        call LoadRemoteFile Filename
    endif
    return
____________________________________________________________________________________________
; Subdirectory (for saves and screenshots)
____________________________________________________________________________________________

CreateROMDirectory:

    if D$pImage = &NULL, ret

    ; .\
    call 'Kernel32.CreateDirectoryA' FilesDirectory, &NULL

    ; .\saves\RomTitle\
    call 'Kernel32.CreateDirectoryA' ROMDirectory, &NULL
    ret
____________________________________________________________________________________________
; Chunks
____________________________________________________________________________________________

SaveChunks:

    arguments @FileID, @pSavingTable

    mov edi D@pSavingTable, eax 0, ecx 0-1
    repne scasd | not ecx | dec ecx | mov D$MaxChunks ecx

    mov edi SaveData

    ; File ID
    mov eax D@FileID | stosd

    ; Write data chunks
    mov ebx 0
L0: if. ebx < D$MaxChunks
        mov eax ebx | stosd
        pushad
            mov edx D@pSavingTable
            call D$edx+ebx*4
        popad
        call CopyMemory Chunk, edi, D$ChunkSize | add edi D$ChunkSize
        inc ebx | jmp L0<
    endif
    sub edi SaveData | mov D$SaveSize edi
    return

LoadChunks:

    arguments @FileID, @pLoadingTable

    mov edi D@pLoadingTable, eax 0, ecx 0-1
    repne scasd | not ecx | dec ecx | mov D$MaxChunks ecx

    mov esi D$pFile
    on esi = &NULL, Q0>

    ; Verify file ID
    lodsd | on eax != D@FileID, Q0>

    ; Set end
    [@pEnd: ?]
    mov eax D$pFile | add eax D$FileSize
    mov D@pEnd eax

    ; Read block
L0: if. esi < D@pEnd

        ; Load chunk
        lodsd | mov ebx eax
        call CopyMemory esi, Chunk, D$esi

        ; Handle chunk data
        push esi
            mov edx D@pLoadingTable
            if ebx < D$MaxChunks, call D$edx+ebx*4
        pop esi

        ; Next chunk
        add esi D$ChunkSize | jmp L0<

    endif
Q0: return

SaveChunk:

    arguments @pData, @Size
    copy D@Size D$ChunkSize | add D$ChunkSize 4
    call CopyMemory D@pData, ChunkData, D@Size
    return

LoadChunk:

    arguments @pData, @Size
    call CopyMemory ChunkData, D@pData, D@Size
    return

____________________________________________________________________________________________
; Chunk data
____________________________________________________________________________________________

[ChunkID:   ?
 Chunk:
 ChunkSize: ?
 ChunkData: ? #080000]

[FILE_ID_CFG 'cfg0'
 FILE_ID_SST 'sst0']
[MaxChunks: ?]
[CFGSaveTable:
 SaveConfigVersion
 SaveConfigPreferences
 SaveConfigButtonAssignments
 SaveConfigInputSettings
 SaveConfigMRUList
 &NULL

 CFGLoadTable:
 LoadConfigVersion
 LoadConfigPreferences
 LoadConfigButtonAssignments
 LoadConfigInputSettings
 LoadConfigMRUList
 &NULL]

[SSTSaveTable:
 SaveGameDescription
 SaveGameScreenshot
 SaveGameCPU
 SaveGamePPU
 SaveGameAPU
 SaveGameRendering
 SaveGamePorts
 SaveGameMapper
 &NULL

 SSTLoadTable:
 LoadGameDescription
 LoadGameScreenshot
 LoadGameCPU
 LoadGamePPU
 LoadGameAPU
 LoadGameRendering
 LoadGamePorts
 LoadGameMapper
 &NULL]
____________________________________________________________________________________________
; File data
____________________________________________________________________________________________

[SaveSize: ?
 SaveData: ? #080000]
[WindowTitle: B$ ? #&MAX_PATH]

; File data
[hFile:         ?
 pFile:         ?
 FileSize:      ?
 FileTitle:  B$ ? #&MAX_PATH
 Extension:  D$ ?]

; Directories and filenames
[Argument:        B$ ? #&MAX_PATH]
[ROMDirectory:    B$ ? #&MAX_PATH] ; .\saves\[ROM name]\
[NessieDirectory: B$ ? #&MAX_PATH] ; .\saves\settings\
[FilesDirectory:  B$ ? #&MAX_PATH] ; .\saves\
[Filename:        B$ ? #&MAX_PATH]
[Filesname:         'saves\',       0
 NessieName:        'settings\',      0
 ConfigFilename:    'config.cfg',   0
 QuickSaveFilename: 'quick.sst',    0
 BatteryFilename:   'battery.sav',  0
 Backslash:         '\',            0]
____________________________________________________________________________________________

TITLE GameGenie

GameGenieDialogProc:

    arguments @hDialog, @Message, @wParam, @lParam
    pushad

        ; Message?
        if. D@Message = &WM_CLOSE       | call @Close      | else
        if. D@Message = &WM_COMMAND     | call @Command    | else
        if. D@Message = &WM_INITDIALOG  | call @InitDialog | else | mov eax &FALSE | endif

        mov D$esp+28 eax

    popad
    return

@InitDialog:

    call @PutCodesInListBox
    mov eax &TRUE
    ret

@Command:

    ; Escape pressed
    on W@wParam = &IDCANCEL, @Close

    ; Enter in edit box -> add code
    if. W@wParam = &IDOK
        call 'User32.GetFocus'
        call 'User32.GetDlgCtrlID' eax
        if eax = IDC_EDIT+0, mov D@wParam IDC_BUTTON+0
    endif

    ; Add code
    if. D@wParam = IDC_BUTTON+0
        ; Decode
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_EDIT+0, &WM_GETTEXT, 010, GameGenieString
        call GameGenieDecode
        if eax = &INVALID_HANDLE_VALUE, break
        ; Encode, add if not found
        call GameGenieEncode
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_FINDSTRINGEXACT, 0-1, GameGenieString
        if eax != &LB_ERR, break
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_ADDSTRING, 0, GameGenieString
    endif

    ; Remove
    if. D@wParam = IDC_BUTTON+1
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETCURSEL, 0, 0
        if eax = &LB_ERR, break
        push eax
            call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_DELETESTRING, eax, 0
            call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETCOUNT, 0, 0
        pop edx
        dec eax
        if edx <= eax, mov eax edx
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_SETCURSEL, eax, 0
    endif

    ; Close dialog
    if D@wParam = IDC_BUTTON+2, call @Close
    mov eax &TRUE
    ret

@Close:

    call @GetCodesFromListbox
    call 'User32.EndDialog' D@hDialog, 1
    ret
____________________________________________________________________________________________

@PutCodesInListbox:

    ; Fill list box with codes
    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_RESETCONTENT, 0, 0
    mov esi GameGenieCodes
L0: lodsd
    if. eax != &INVALID_HANDLE_VALUE
        push esi
            call GameGenieEncode
            call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_ADDSTRING, 0, GameGenieString
        pop esi | jmp L0<
    endif
    ret

@GetCodesFromListbox:

    mov D$GameGenieCodes &INVALID_HANDLE_VALUE
    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETCOUNT, 0, 0
    mov ebx 0
L0: on ebx >= eax, Q0>
    pushad
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETTEXT, ebx, GameGenieString
        call GameGenieDecode
        call GameGenieAddCode eax
    popad
    inc ebx | jmp L0<
Q0: ret
____________________________________________________________________________________________

GameGenieDecode:

    ; Length valid?
    [@Length: ?]
    mov edi GameGenieString, al 0, ecx 0F | repne scasb
    xor ecx 0F | dec ecx | mov D@Length ecx

    ; Convert characters to binary values, break if invalid character is found
    mov esi GameGenieString, edi GameGenieNumbers
L0: lodsb | or al 020
    mov ebx 0
L1: if. al != B$GameGenieCharacters+ebx
        inc ebx | on ebx <= 0F, L1<
        ; Invalid character found
        mov D@Length 0-1
    else
        mov B$edi bl | inc edi
        on B$esi > 0, L0<
    endif

    ; Extract address
    mov edx 0
    movzx eax B$GameGenieNumbers+04 | and al 07 | shl eax 00 | or edx eax
    movzx eax B$GameGenieNumbers+03 | and al 08 | shl eax 00 | or edx eax
    movzx eax B$GameGenieNumbers+02 | and al 07 | shl eax 04 | or edx eax
    movzx eax B$GameGenieNumbers+01 | and al 08 | shl eax 04 | or edx eax
    movzx eax B$GameGenieNumbers+05 | and al 07 | shl eax 08 | or edx eax
    movzx eax B$GameGenieNumbers+04 | and al 08 | shl eax 08 | or edx eax
    movzx eax B$GameGenieNumbers+03 | and al 07 | shl eax 0C | or edx eax
    movzx eax B$GameGenieNumbers+02 | and al 08 | shl eax 0C | or edx eax
    if D@Length = 8, or edx 08000
    shl edx 8

    ; Extract values
    if. D@Length = 6

        ; 6 character code
        movzx eax B$GameGenieNumbers+00 | and al 07 | shl eax 00 | or edx eax
        movzx eax B$GameGenieNumbers+05 | and al 08 | shl eax 00 | or edx eax
        movzx eax B$GameGenieNumbers+01 | and al 07 | shl eax 04 | or edx eax
        movzx eax B$GameGenieNumbers+00 | and al 08 | shl eax 04 | or edx eax
        shl edx 8

    else
    if. D@Length = 8

        ; 8 character code
        movzx eax B$GameGenieNumbers+00 | and al 07 | shl eax 00 | or edx eax
        movzx eax B$GameGenieNumbers+07 | and al 08 | shl eax 00 | or edx eax
        movzx eax B$GameGenieNumbers+01 | and al 07 | shl eax 04 | or edx eax
        movzx eax B$GameGenieNumbers+00 | and al 08 | shl eax 04 | or edx eax
        shl edx 8

        ; Compare value
        movzx eax B$GameGenieNumbers+06 | and al 07 | shl eax 00 | or edx eax
        movzx eax B$GameGenieNumbers+05 | and al 08 | shl eax 00 | or edx eax
        movzx eax B$GameGenieNumbers+07 | and al 07 | shl eax 04 | or edx eax
        movzx eax B$GameGenieNumbers+06 | and al 08 | shl eax 04 | or edx eax

    else

        ; Invalid length or invalid character
        mov edx &INVALID_HANDLE_VALUE

    endif
    mov eax edx
    ret

; Gets code in eax, returns string in GameGenieString
GameGenieEncode:

    mov edx eax
    call ZeroMemory GameGenieNumbers, 8

    ; Convert value and compare bits
    ifNotFlag. edx 08000_0000

        ; Value
        shr edx 8
        mov eax edx | shr eax 00 | and al 07 | or B$GameGenieNumbers+00 al
        mov eax edx | shr eax 00 | and al 08 | or B$GameGenieNumbers+05 al
        mov eax edx | shr eax 04 | and al 07 | or B$GameGenieNumbers+01 al
        mov eax edx | shr eax 04 | and al 08 | or B$GameGenieNumbers+00 al
        shr edx 8

        ; 6 characters
        mov ecx 6

    else

        ; Compare
        mov eax edx | shr eax 00 | and al 07 | or B$GameGenieNumbers+06 al
        mov eax edx | shr eax 00 | and al 08 | or B$GameGenieNumbers+05 al
        mov eax edx | shr eax 04 | and al 07 | or B$GameGenieNumbers+07 al
        mov eax edx | shr eax 04 | and al 08 | or B$GameGenieNumbers+06 al
        shr edx 8

        ; Value
        mov eax edx | shr eax 00 | and al 07 | or B$GameGenieNumbers+00 al
        mov eax edx | shr eax 00 | and al 08 | or B$GameGenieNumbers+07 al
        mov eax edx | shr eax 04 | and al 07 | or B$GameGenieNumbers+01 al
        mov eax edx | shr eax 04 | and al 08 | or B$GameGenieNumbers+00 al
        shr edx 8

        ; 8 characters
        mov ecx 8

    endif

    ; Convert address bits
    mov eax edx | shr eax 00 | and al 07 | or B$GameGenieNumbers+04 al
    mov eax edx | shr eax 00 | and al 08 | or B$GameGenieNumbers+03 al
    mov eax edx | shr eax 04 | and al 07 | or B$GameGenieNumbers+02 al
    mov eax edx | shr eax 04 | and al 08 | or B$GameGenieNumbers+01 al
    mov eax edx | shr eax 08 | and al 07 | or B$GameGenieNumbers+05 al
    mov eax edx | shr eax 08 | and al 08 | or B$GameGenieNumbers+04 al
    mov eax edx | shr eax 0C | and al 07 | or B$GameGenieNumbers+03 al
    mov eax edx | shr eax 0C | and al 08 | or B$GameGenieNumbers+02 al ; length bit

    ; Convert numbers to characters
    mov esi GameGenieNumbers, edi GameGenieString, eax 0
L0: lodsb | mov al B$GameGenieCharacters+eax | and al 0FF-020
    stosb | dec ecx | jnz L0<
    mov B$edi 0
    ret
____________________________________________________________________________________________

ResetGameGenie:

    mov esi GameGenieCodes
    mov ebx 0
L0: lodsd | if eax = &INVALID_HANDLE_VALUE, ret

    mov edx eax
    shr edx 16

    ; Save previous setting, override
    ifFlag. edx 08000
        copy D$CPUReadTable+edx*4 D$GameGenieOriginal+ebx*4
        CPURead edx, edx, CompareGameGenie
    else
        or edx 08000
        copy D$CPUReadTable+edx*4 D$GameGenieOriginal+ebx*4
        CPURead edx, edx, ReadGameGenie
    endif

    ; Save Game Genie data for this address
    mov W$GameGenieAddress+ebx*2 dx
    mov B$GameGenieCompare+ebx al
    mov B$GameGenieValue+ebx ah
    inc ebx
    jmp L0<<

ReadGameGenie:

    pushad

        ; Find address
        mov edi GameGenieAddress, ecx 0-1
        repne scasw
        mov ebx edi | sub ebx GameGenieAddress+2 | shr ebx 1

        ; Override value
        movzx eax B$GameGenieValue
        mov D$esp+01C eax

    popad
    ret

CompareGameGenie:

    pushad

        ; Find address
        mov edi GameGenieAddress, ecx 0-1
        repne scasw
        mov ebx edi | sub ebx GameGenieAddress+2 | shr ebx 1

        ; Override value
        push ebx
            call D$GameGenieOriginal+ebx*4
        pop ebx
        if al = B$GameGenieCompare+ebx, mov al B$GameGenieValue+ebx
        mov D$esp+01C eax

    popad
    ret
____________________________________________________________________________________________

GameGenieAddCode:

    arguments @Code

    ; Get current length
    mov edi GameGenieCodes, eax &INVALID_HANDLE_VALUE, ecx 0-1
    repne scasd | not ecx

    ; Code already there?
    mov edi GameGenieCodes, eax D@Code
    repne scasd
    sub edi 4
    if. D$edi = &INVALID_HANDLE_VALUE
        ; Append code to end
        stosd | mov D$edi &INVALID_HANDLE_VALUE
    endif
    return
____________________________________________________________________________________________

SaveGameGenie:

    ; If there are codes, save. Otherwise, delete.
    [GameGenieFile: 'Game genie.gen', 0]
    if. D$GameGenieCodes != &INVALID_HANDLE_VALUE
        call SaveROMFile GameGenieFile
    else
        mov B$Filename 0
        call AddString Filename ROMDirectory
        call ADdString Filename GameGenieFile
        call 'Kernel32.DeleteFileA' Filename
    endif
    ret

LoadGameGenie:

    [GENExtension: '*.gen', 0]
    call LoadFirstFile ROMDirectory, GENExtension
    ret

SaveGEN:

    [@NewLine: B$ 13, 10, 0]

    ; Convert all codes to ascii
    mov ebx 0, edi SaveData
L0: mov eax D$GameGenieCodes+ebx*4 | inc ebx
    if. eax != &INVALID_HANDLE_VALUE
        pushad
            call GameGenieEncode
        popad
        mov esi GameGenieString
    L1: if.. B$esi != 0 | movsb | jmp L1< | endif..
        mov ax W@NewLine | stosw
        jmp L0<
    endif

    ; Set size of file
    sub edi SaveData | mov D$SaveSize edi
    ret

LoadGEN:

    ; Reset parser
    [@Comment: ?]
    mov esi D$pFile
    jmp @EndOfLine

    ; Which ascii?
L0: lodsb
    if al = 00,  ret
    on al = 10,  @EndOfLine
    on al = 13,  @EndOfLine
    on al = ';', @StartComment
    on al = ' ', @EndOfWord
    on al = 09,  @EndOfWord

    ; Character
    on edi > GameGenieString+010, L0<
    on B@Comment = &TRUE, L0<
    stosb | jmp L0<

@EndOfWord:

    ; Try to decode
    mov B$edi 0
    pushad
        call GameGenieDecode
        if eax != &INVALID_HANDLE_VALUE,
            call GameGenieAddCode eax
    popad
    mov edi GameGenieString
    jmp L0<

@EndOfLine:    mov B@Comment &FALSE | jmp @EndOfWord
@StartComment: mov B@Comment &TRUE  | jmp @EndOfWord

____________________________________________________________________________________________

[MAX_GAMEGENIE_CODES 0100]

[GameGenieCharacters: 'apzlgityeoxuksvn']
[GameGenieString:   B$ ? #010]
[GameGenieNumbers:  B$ ? #8]

[GameGenieCodes:    D$ ? #MAX_GAMEGENIE_CODES
 GameGenieAddress:  W$ ? #MAX_GAMEGENIECODES
 GameGenieCompare:  B$ ? #MAX_GAMEGENIECODES
 GameGenieValue:    B$ ? #MAX_GAMEGENIECODES
 GameGenieOriginal: D$ ? #MAX_GAMEGENIECODES]
____________________________________________________________________________________________
TITLE Zip

[ListboxClass: 'Listbox', 0]
[SelectAFile: 'Select a file', 0]
[ZIPFilename: B$ ? #&MAX_PATH]
LoadZIP:

    arguments @pFilename
    call 'Kernel32.GlobalFree' D$pFile | mov D$pFile &NULL

    ; Load DLL
    call LoadUnzipDLL

    if. D$pUnzipToMemory != &NULL

        ; Save zip name
        mov B$ZIPFilename 0
        call AddString ZipFilename, D@pFilename

        ; Open unzip dialog
        call 'User32.DialogBoxParamA' D$hInstance, IDD_UNZIP, D$hMainWindow, UnzipDialogProc, &NULL

    else

        ; You need unzip32.dll
        call 'User32.MessageBoxA' D$hMainWindow,
                                  {'To load zip files, download unzip32.dll from www.techemporium.info and place it in the same directory as TechNES.exe.', 0},
                                  {'The file unzip32.dll has not been found.', 0},
                                  &MB_OK+&MB_ICONINFORMATION
    endif

    ; Unload DLL
    call UnloadUnzipDLL
    return

UnzipFile:

    arguments @pFilename

    ; Load DLL
    call LoadUnzipDLL
    on D$pUnzipToMemory = &NULL, Q0>>

    ; Put '\' in front of all special characters
    [@Filename: B$ ? #&MAX_PATH]
    mov esi D@pFilename, edi @Filename
L0: lodsb
    on al = '[', S0>
    on al = ']', S0>
    on al = '*', S0>
    on al = '?', S0>
    on al = '!', S0>
    on al = '^', S0>
    on al = '-', S0>
    on al = '\', S0>
    on al = 0, S1>
    stosb | jmp L0<
S0: mov B$edi '\' | inc edi
    stosb | jmp L0<
S1: stosb

    ; Unzip file
    call D$pUnzipToMemory ZipFilename, @Filename, UserFunctions, UnzipBuffer
    if. eax != 0

        ; Filetitle
        mov B$Filetitle 0
        call AddString Filetitle, D@pFilename
        mov edi Filetitle, ecx 0-1, al 0
        repne scasb
        sub edi 5 | call ZeroMemory edi, 4
        ; Load data
      . D$pFile = 'Kernel32.GlobalAlloc' &GPTR, D$UnzipBuffer@Length
        call CopyMemory D$UnzipBuffer@pData, D$pFile, D$UnzipBuffer@Length
        copy D$UnzipBuffer@Length D$Filesize
        call LoadNES

    endif
    call UnloadUnzipDLL
Q0: return
____________________________________________________________________________________________
; Unzip dialog
____________________________________________________________________________________________

[hUnzipDialog: ?]
[nZippedFiles: ?]

ServiceCallback:

    arguments @pFilename, @FileSize
    pushad

        ; Fill listbox
        call 'User32.SendDlgItemMessageA' D$hUnzipDialog, IDC_LISTBOX+0, &LB_ADDSTRING, 0, D@pFilename
        inc D$nZippedFiles
        if D$nZippedFiles = 10, call ClearAudio

    popad
    mov eax 0
    return

UnzipDialogProc:

    arguments @hDialog, @Message, @wParam, @lParam
    pushad

        ; Message?
        if. D@Message = &WM_INITDIALOG | call @Init    | else
        if. D@Message = &WM_COMMAND    | call @Command | else
        if. D@Message = &WM_CLOSE      | call @Close   | else
        mov eax &FALSE | endif

        mov D$esp+01C eax

    popad
    return

@Init:

    copy D@hDialog D$hUnzipDialog

    ; Check zip file
    [@Extension: '*.nes', 0]
    mov D$nZippedFiles 0
    call D$pUnzipToMemory ZipFilename, @Extension, UserFunctions, UnzipBuffer

    call 'User32.GetDlgItem' D@hDialog, IDC_LISTBOX+0 | call HorizontalScrollbar eax

    ; No files --> kill dialog
    if. D$nZippedFiles = 0
        call @Close
    else
    ; Single file --> load it
    if. D$nZippedFiles = 1
        mov eax 0
        call @LoadFile
    else
    ; Multiple files --> show dialog
    if. D$nZippedFiles > 1
        call 'User32.ShowWindow' D@hDialog, &TRUE
    endif
    mov eax &TRUE
    ret

@Command:

    ; Double-click?
    if. W@wParam+2 = &LBN_DBLCLK
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETCURSEL, 0, 0
        call @LoadFile
    endif
    mov eax &TRUE
    ret

@Close:

    ; Kill dialog
    call 'User32.EndDialog' D@hDialog, 1
    mov eax &TRUE
    ret

____________________________________________________________________________________________

@LoadFile:

    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETTEXT, eax, Filename
    call @Close
    call UnzipFile Filename
    ret

HorizontalScrollbar:

    [@hDC:      ?
     @hFontOld: ?
     @hFontNew: ?
     @nCount:   ?
     @nLength:  ?
     @Size:     ? ?
     @Text:     ? #&MAX_PATH]

    arguments @hWindow
    mov D@nLength 0
    pushad

        ; Retrieve data
      . D@hDC = 'User32.GetDC' D@hWindow
      . D@hFontNew = 'User32.SendMessageA' D@hWindow, &WM_GETFONT, &NULL, &NULL
      . D@hFontOld = 'GDI32.SelectObject'  D@hDC, D@hFontNew

        ; Find longest string
      . D@nCount = 'User32.SendMessageA' D@hWindow, &LB_GETCOUNT, 0, 0
    L0: if. D@nCount > 0
            dec D@nCount
            call 'User32.SendMessageA' D@hWindow, &LB_GETTEXTLEN, D@nCount, &NULL | inc eax
            push eax
                call 'User32.SendMessageA' D@hWindow, &LB_GETTEXT, D@nCount, @Text
            pop eax
            call 'GDI32.GetTextExtentPoint32A' D@hDC, @Text, eax, @Size
            mov eax D@Size
            if eax > D@nLength, mov D@nLength eax
            jmp L0<
        endif

        ; Restore data, set new width
    S0: call 'GDI32.SelectObject' D@hDC, D@hFontOld
        call 'User32.ReleaseDC' D@hWindow, D@hDC
        call 'User32.SendMessageA' D@hWindow, &LB_SETHORIZONTALEXTENT, D@nLength, 0

    popad
    return

____________________________________________________________________________________________
; Load unload DLL
____________________________________________________________________________________________

[UnzipDLL: 'Unzip32', 0]
[UnzipToMemory: 'Wiz_UnzipToMemory', 0]
[hUnzipDLL: ?]
[pUnzipToMemory: ?]
LoadUnzipDLL:

  . D$hUnzipDLL = 'Kernel32.LoadLibraryA' UnzipDLL
  . D$pUnzipToMemory = 'Kernel32.GetProcAddress' eax, UnzipToMemory
    ret

UnloadUnzipDLL:

    call 'Kernel32.FreeLibrary' D$hUnzipDLL
    mov D$pUnzipToMemory &NULL
    ret

____________________________________________________________________________________________
; Unzip32.dll stuff
____________________________________________________________________________________________

PrintRoutine:

    mov eax D$esp+04, ebx D$esp+08
    int 3 | mov eax 0 | ret 8
PasswordRoutine: int 3 | mov eax 0 | ret 8
ReplaceRoutine:  int 3 | mov eax 0 | ret 4
SendMessage:     int 3 | mov eax 0 | ret (13 shl 2)
SoundRoutine:    ret

[UnzipBuffer:
 @Length:  ?
 @pData:   ?]

[UserFunctions:
 PrintRoutine
 SoundRoutine
 ReplaceRoutine
 PasswordRoutine
 SendMessage
 ServiceCallback

 TOtalSizeComp: 0
 TotalSize: 0
 NumMembers: 0
 cchComment: 0]

____________________________________________________________________________________________
TITLE IPS

LoadIPS:

    ; Check file ID
    [IpsId: 'PATCH']
    mov esi D$pFile, edi IpsId, ecx 5
    rep cmpsb | jecxz L0>
    ret

    ; 3 byte offset -> edi
L0: mov eax 0
    lodsb | shl eax 8
    lodsb | shl eax 8
    lodsb | mov edi eax
    add edi D$pImage

    ; EOF?
    on eax = 'FOE', Q0>

    ; 2 byte size -> ecx
    mov eax 0
    lodsb | shl eax 8
    lodsb | mov ecx eax

    ; Patch!
    if. ecx > 0
        rep movsb
    else
        ; RLE encoded patch
        lodsw | xchg al ah
        movzx ecx ax
        lodsb
        rep stosb
    endif
    jmp L0<

    ; Hard reset
Q0: call SetPower POWER_OFF
    call SetPower POWER_ON
    ret
____________________________________________________________________________________________
TITLE Menu

; Handles the main window menu.
; These functions are called upon WM_COMMAND messages

____________________________________________________________________________________________
; Hide/show menu
____________________________________________________________________________________________

Togglemenu:

    call 'User32.GetMenu' D$hMainWindow
    on eax != &NULL, HideMenu

    ShowMenu:

        call 'User32.SetMenu' D$hMainWindow, D$hMenu
        call UpdateCursor
        ret

    HideMenu:

        call 'User32.SetMenu' D$hMainWindow, &NULL
        call UpdateCursor
        ret
____________________________________________________________________________________________
; MRU list
____________________________________________________________________________________________

[LENGTH_MRULIST <(&MAX_PATH shl 4)>] ; Max 16 items
[nMRUItems: 5]
[MRUList: B$ ? #LENGTH_MRULIST]
[hMRUList: ?]
[IDM_RECENT 2000]

MenuRecentItem:

    [@Filename: ? #&MAX_PATH]
    call 'User32.GetMenuStringA' D$hMRUList, D$wParam, @Filename, &MAX_PATH, &MF_BYCOMMAND
    call DeleteMRUItem @Filename
    call LoadRemoteFile @Filename
    ret

AddMRUItem:

    arguments @pFilename

    call DeleteMRUItem D@pFilename

    ; Find an available slot
    mov edx D$nMRUItems
    mov edi MRUList, ecx 0-1, al 0
L0: if. B$edi != 0
        repne scasb
        dec edx | jnz L0<
    endif

    ; Place new string there
    mov B$edi 0
    call AddString edi, D@pFilename

    ; List full?
    if. edx = 0

        ; Find end
        repne scasb
        ; Source?
        push edi
            mov edi MRUList
            repne scasb
            mov esi edi
        ; Length?
        pop ecx
        sub ecx esi
        ; Dest?
        mov edi MRUList
        rep movsb

        ; End of list? Erase everything else
        mov ecx MRUList+LENGTH_MRULIST | sub ecx edi
        dec edi
        mov al 0 | rep stosb

    endif

    call UpdateMRUList
    call SaveConfiguration
    return

DeleteMRUItem:

    arguments @pFilename

    ; Search for filename in MRU list
    mov edi MRUList, ecx LENGTH_MRULIST, al 0
L0: mov esi D@pFilename
L1: if. B$edi != 0
        cmpsb | je L1<
        repne scasb
        on B$edi != 0, L0<
    else

        ; Filename found!
        inc edi
        move ecx edi-MRUList
        push edi
            sub edi 2
            std
                repne scasb | add edi 2
            cld
        pop esi
        mov ecx MRUList+LENGTH_MRULIST | sub ecx esi
        rep movsb

    endif

    call UpdateMRUList
    call SaveConfiguration
    return

UpdateMRUList:

    ; Renew menu
    if. D$hMRUList != &NULL
        call 'User32.DeleteMenu' D$hMenu, D$hMRUList, &MF_BYCOMMAND
        winErrorCheck 'Could not delete MRU list'
    endif
  . D$hMRUList = 'User32.CreatePopupMenu' | winErrorCheck 'Could not create MRU list'

    ; Fill list
    mov edi MRUList, ecx LENGTH_MRULIST, al 0, edx D$nMRUItems
L0: if. edx > 0

        ; Add item
        pushad
            add edx IDM_RECENT
            if B$edi != 0, call 'User32.InsertMenuA' D$hMRUList, 0, &MF_BYPOSITION, edx, edi
        popad
        dec edx
        repne scasb
        jmp L0<

    endif

    ; Insert menu
    call 'User32.GetSubMenu' D$hMenu, 0
    call 'User32.InsertMenuA' eax, IDM_EXIT-IDM_MENU, &MF_BYPOSITION+&MF_POPUP, D$hMRUList, {'Recent', 0}

    ; Empty? --> gray
    if B$MRUList = 0, call 'User32.EnableMenuItem' D$hMenu, D$hMRUList, &MF_BYCOMMAND+&MF_GRAYED
    if B$MRUList != 0, call 'User32.EnableMenuItem' D$hMenu, D$hMRUList, &MF_BYCOMMAND+&MF_ENABLED
    ret
____________________________________________________________________________________________
; Menu items
____________________________________________________________________________________________

; File
____________________________________________________________________________________________

MenuOpen:

    ; Open file dialog
    copy D$hMainWindow D$OpenFilename@hwndOwner
    copy D$hInstance   D$OpenFilename@hInstance
    call 'Comdlg32.GetOpenFileNameA' OpenFilename

    ; Open file
    if eax != 0, call LoadRemoteFile FullFilename
    endif
    ret

MenuClose:

    ; Unload cartridge
    call UnloadNES
    ret

MenuExit:

    ; Kill window
    call 'User32.DestroyWindow' D$hMainWindow
    ret

; NES
____________________________________________________________________________________________

MenuPower:

    ; Toggle NES power
    mov eax D$Power | xor eax POWER_ON
    call SetPower eax
    ret

MenuReset:

    ; Send a reset IRQ
    SetIRQ CPU, IRQ_RESET
    ret

MenuNTSC:

    ; Check radio button
    call 'User32.CheckMenuRadioItem' D$hMenu, IDM_NTSC, IDM_PAL, IDM_NTSC, &MF_BYCOMMAND

    ; Switch to NTSC
    call SetMode NTSC
    ret

MenuPAL:

    ; Check radio button
    call 'User32.CheckMenuRadioItem' D$hMenu, IDM_NTSC, IDM_PAL, IDM_PAL,  &MF_BYCOMMAND

    ; Switch to PAL
    call SetMode PAL
    ret

; Dialogs
____________________________________________________________________________________________

MenuGameGenie:   call LaunchDialog IDD_GAMEGENIE,   GameGenieDialogProc   | ret
MenuSaveGame:    call LaunchDialog IDD_SAVE,        SaveDialogProc        | ret
MenuLoadGame:    call LaunchDialog IDD_LOAD,        LoadDialogProc        | ret
MenuPreferences: call LaunchDialog IDD_PREFERENCES, PreferencesDialogProc | ret
MenuInput:       call LaunchDialog IDD_INPUT,       InputDialogProc       | ret
MenuAbout:       call LaunchDialog IDD_ABOUT,       AboutDialogProc       | ret

LaunchDialog:

    arguments @ID, @DialogProc
    call 'User32.DialogBoxParamA' D$hInstance, D@ID, D$hMainWindow, D@DialogProc, &NULL
    return

; Texts
____________________________________________________________________________________________

MenuLicense:    call LaunchFile {'license.txt',       0}, D$pLicense,   IDR_LICENSE   | ret
MenuChangelog:  call LaunchFile {'changelog.txt',     0}, D$pChangelog, IDR_CHANGELOG | ret
MenuManual:     call LaunchFile {'manual.chm', 0}, D$pManual,    IDR_MANUAL    | ret


LaunchFile:

    arguments @pFilename, @pRCData, @ResourceID

    ; Create file
    call 'Kernel32.GetTempPathA' &MAX_PATH, Filename
    call AddString Filename, D@pFilename
  . D$hFile = 'Kernel32.CreateFileA' Filename, &GENERIC_WRITE, &NULL, &NULL, &CREATE_ALWAYS, &FILE_ATTRIBUTE_NORMAL, &NULL
    if. D$hFile != &INVALID_HANDLE_VALUE

        ; Write and close
        call 'Kernel32.FindResourceA' D$hInstance, D@ResourceID, &RT_RCDATA
        call 'Kernel32.SizeofResource' D$hInstance, eax
        call 'Kernel32.WriteFile' D$hFile, D@pRCData, eax, Temp, &NULL
        call 'Kernel32.CloseHandle' D$hFile

        ; Open file
        call 'Shell32.ShellExecuteA' &NULL, &NULL, Filename, &NULL, &NULL, &TRUE

    endif
    return

____________________________________________________________________________________________
; Menu data
____________________________________________________________________________________________

enumerate 1000,

    IDM_MENU,
    IDM_OPEN,
    IDM_CLOSE,
    IDM_EXIT,

    IDM_POWER,
    IDM_RESET,
    IDM_NTSC,
    IDM_PAL,
    IDM_GAMEGENIE,
    IDM_SAVE,
    IDM_LOAD,

    IDM_PREFERENCES,
    IDM_INPUT,
    IDM_FULLSCREEN,

    IDM_MANUAL,
    IDM_LICENSE,
    IDM_CHANGELOG,
    IDM_ABOUT

[MenuItemTable:
 MenuOpen
 MenuClose
 MenuExit

 MenuPower
 MenuReset
 MenuNTSC
 MenuPAL
 MenuGameGenie
 MenuSaveGame
 MenuLoadGame

 MenuPreferences
 MenuInput
 ToggleFullscreen

 MenuManual
 MenuLicense
 MenuChangelog
 MenuAbout
 EndOfMenuItemTable: 0]
____________________________________________________________________________________________
TITLE Dialog
____________________________________________________________________________________________
; Some equates used
____________________________________________________________________________________________

; Dialog IDs
enumerate 1000,
    IDD_PREFERENCES,
    IDD_INPUT,
    IDD_ABOUT,
    IDD_SAVE,
    IDD_LOAD,
    IDD_GAMEGENIE,
    IDD_UNZIP

; Child IDs
[IDC_CHECKBOX    2000
 IDC_RADIOBUTTON 3000
 IDC_LISTBOX     4000
 IDC_STATIC      5000
 IDC_BUTTON      6000
 IDC_TRACKBAR    7000
 IDC_EDIT        8000
 IDC_OK          IDC_BUTTON+999
 IDC_CANCEL      IDC_BUTTON+998]

____________________________________________________________________________________________
; About dialog
____________________________________________________________________________________________

AboutDialogProc:

    arguments @hDialog, @Message, @wParam, @lParam
    pushad

        ; Message?
        if. D@Message = &WM_CLOSE       | call @Close      | else
        if. D@Message = &WM_TIMER       | call @ToggleIcon | else
        if. D@Message = &WM_COMMAND     | call @Command    | else
        if. D@Message = &WM_INITDIALOG  | call @InitDialog | else | mov eax &FALSE | endif

        mov D$esp+28 eax

    popad
    return

@InitDialog:

    ; Create a timer
    call 'User32.SetTimer' D@hDialog, IDT_ABOUT, 250, &NULL
    call @ToggleIcon
    mov eax &TRUE
    ret

@ToggleIcon:

    [@Icon: ?]
    xor B@Icon 1 | mov ebx D@Icon
    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_STATIC+0, &STM_SETICON, D$hIcon+ebx*4, 0
    mov eax &TRUE
    ret

@Command:

    ; Escape pressed
    on W@wParam = &IDCANCEL, @Close

    ; Contact
    if W@wParam = IDC_BUTTON+0, call 'Shell32.ShellExecuteA' &NULL, &NULL, NessieMail, &NULL, &NULL, &TRUE
    ; Homepage
    if W@wParam = IDC_BUTTON+1, call 'Shell32.ShellExecuteA' &NULL, &NULL, NessieURL,  &NULL, &NULL, &TRUE
    mov eax &TRUE
    ret

@Close:

    ; Kill dialog
    call 'User32.KillTimer' D@hDialog, IDT_ABOUT
    call 'User32.EndDialog' D@hDialog, 1
    mov eax &TRUE
    ret

[NessieURL:    B$ 'http://www.techemporium.info', 0
 NessieMail:   B$ 'mailto:techemporium.info@gmail.com', 0]
____________________________________________________________________________________________
TITLE Config

; Config file handling and the preferences dialog.

____________________________________________________________________________________________
; Config files
____________________________________________________________________________________________

SaveConfiguration:

    call SaveNessieFile ConfigFilename
    ret

LoadConfiguration:

    ; Load internal config
    push D$pFile
    push D$FileSize

        call 'Kernel32.FindResourceA'  D$hInstance, IDR_CONFIG, &RT_RCDATA
        call 'Kernel32.SizeofResource' D$hInstance, eax
        mov D$FileSize eax
        copy D$pConfig D$pFile | call LoadCFG

    pop D$Filesize
    pop D$pFile

    ; Load external config
    call LoadFirstFile NessieDirectory, ConfigFilename
    if D$pImage != &NULL, call LoadFirstFile ROMDirectory, ConfigFilename
    ret

SaveCFG: call SaveChunks FILE_ID_CFG, CFGSaveTable | ret
LoadCFG: call LoadChunks FILE_ID_CFG, CFGLoadTable | ret
____________________________________________________________________________________________
; Chunk saving/loading
; They all use ChunkData and ChunkSize
____________________________________________________________________________________________

SaveConfigMRUList:

    ; Length?
    mov edi MRUList, ecx 0-1, al 0, edx D$nMRUItems
L0: if. edx > 0
        if B$edi = 0, break
        repne scasb
        dec edx | jnz L0<
    endif
    sub edi MRUList

    ; Set data
    call CopyMemory MRUList, ChunkData, edi
    add edi 4 | mov D$ChunkSize edi
    ret

LoadConfigMRUList:

    call ZeroMemory MRUList, LENGTH_MRULIST
    mov ecx D$ChunkSize | sub ecx 4
    if ecx > LENGTH_MRULIST, mov ecx LENGTH_MRULIST
    call CopyMemory ChunkData, MRUList, ecx

    call UpdateMRUList
    ret

____________________________________________________________________________________________

SaveConfigVersion:

    ; Version 0: TechNES 1.00
    ; - Saves/loads 010 preferences bytes, eventhough only 4 are used
    ; - Saves/loads 040 input settings bytes, eventhough only 4 are used
    [NESSIE_VERSION 1]
    [ConfigVersion: ?]
    mov D$ChunkSize 8
    mov D$ChunkData NESSIE_VERSION
    ret

LoadConfigVersion:

    copy D$ChunkData D$ConfigVersion
    ret
____________________________________________________________________________________________

SaveConfigPreferences:

    mov D$ChunkSize LENGTH_PREFERENCES+4
    call CopyMemory Preferences, ChunkData, LENGTH_PREFERENCES
    ret

LoadConfigPreferences:

    mov ecx D$ChunkSize | sub ecx 4
    if ecx > LENGTH_PREFERENCES, mov ecx LENGTH_PREFERENCES
if D$ConfigVersion = 0, mov ecx 4
    call CopyMemory ChunkData, Preferences, ecx
    ret
____________________________________________________________________________________________

SaveConfigInputSettings:

    mov D$ChunkSize LENGTH_INPUTSETTINGS+4
    call CopyMemory InputSettings, ChunkData, LENGTH_INPUTSETTINGS
    ret

LoadConfigInputSettings:

    mov ecx D$ChunkSize | sub ecx 4
    if ecx > LENGTH_INPUTSETTINGS, mov ecx LENGTH_INPUTSETTINGS
if D$ConfigVersion = 0, mov ecx 4
    call CopyMemory ChunkData, InputSettings, ecx

    call UpdateCursor
    ret
____________________________________________________________________________________________

; Button assigments are saved in chunks:
; 00-15: D$ #4 - instance GUID (sys keyboard, joystick 1, etc...)
; 16-??: W$ #2 - A bunch of WORD pairs:
;                W$ Offset in InputStatus array
;                W$ Offset in DeviceState array
; ??:    D$ #1  - A null-DWORD terminating the WORD pairs

SaveConfigButtonAssignments:

    mov edx ChunkData
    mov ebx 0
L0: if. ebx < D$nDevices

        ; Save GUID
        mov eax ebx | shl eax 4 | add eax GUID
        call CopyMemory eax, edx, 010 | add edx 010

        ; Save mappings
        mov edi D$pMappings+ebx*4, eax 0, ecx 0-1 | repne scasd
        not ecx | shl ecx 2
        call CopyMemory D$pMappings+ebx*4, edx, ecx | add edx ecx

        inc ebx | jmp L0<

    endif

    ; Save size
    sub edx Chunk | mov D$ChunkSize edx
    ret

LoadConfigButtonAssignments:

    ; Set start/end
    [@pEnd: ?]
    mov eax D$ChunkSize | add eax Chunk | mov D@pEnd eax
    mov esi ChunkData

L0: if. esi < D@pEnd

        mov ebx 0
    L1: if.. ebx < D$nDevices

            ; Compare GUID
            push esi
                move edi ebx | shl edi 4 | add edi GUID
                mov ecx 010
                repe cmpsb
            pop esi
            jecxz F0>

            ; Check GUID of next device
        S0: inc ebx | jmp L1<<

            ; Load mappings
        F0: add esi 010
            mov edi D$pMappings+ebx*4
        L2: movsd | on D$edi-4 != 0, L2<
            jmp L0<

        endif..

        ; GUID not found, skip
        add esi 010
    L2: lodsd | on eax != 0, L2<
        jmp L0<<

    endif
    ret
____________________________________________________________________________________________
; Preferences dialog
____________________________________________________________________________________________

PreferencesDialogProc:

    arguments @hDialog, @Message, @wParam, @lParam
    pushad

        ; Message?
        mov eax &TRUE
        if. D@Message = &WM_CLOSE      | call @Close      | else
        if. D@Message = &WM_COMMAND    | call @Command    | else
        if. D@Message = &WM_INITDIALOG | call @InitDialog | else | mov eax &FALSE   | endif

        mov D$esp+28 eax

    popad
    return

@InitDialog:

    ; Checkboxes
    movzx eax B$Preferences@UseDatabase  | call 'User32.CheckDlgButton' D@hDialog, IDC_CHECKBOX+0, eax
    movzx eax B$Preferences@AutoPowerOn  | call 'User32.CheckDlgButton' D@hDialog, IDC_CHECKBOX+1, eax
    movzx eax B$Preferences@AutoHideMenu | call 'User32.CheckDlgButton' D@hDialog, IDC_CHECKBOX+2, eax

    ; Radio buttons
    if. B$Preferences@PreferNTSC = &TRUE
        mov eax IDC_RADIOBUTTON+0
    else
        mov eax IDC_RADIOBUTTON+1
    endif
    call 'User32.CheckRadioButton' D@hDialog, IDC_RADIOBUTTON+0, IDC_RADIOBUTTON+1, eax

    mov eax &TRUE
    ret

@Command:

    ; Escape pressed
    on W@wParam = &IDCANCEL, @Close

    ; Which button?
    mov W@wParam+2 0
    on W@wParam = IDC_OK,            @SaveAndExit
    on W@wParam = IDC_CANCEL,        @Close
    on W@wParam = IDC_CHECKBOX+0,    @CheckBox
    on W@wParam = IDC_CHECKBOX+1,    @CheckBox
    on W@wParam = IDC_CHECKBOX+2,    @CheckBox
    ret

@Close:

    call 'User32.EndDialog' D@hDialog, 1
    ret
____________________________________________________________________________________________

@CheckBox:

    call 'User32.IsDlgButtonChecked' D@hDialog, D@wParam | xor eax &BST_CHECKED
    call 'User32.CheckDlgButton' D@hDialog, D@wParam, eax
    ret

@SaveAndExit:

    ; Save new settings
    call 'User32.IsDlgButtonChecked' D@hDialog, IDC_CHECKBOX+0    | mov B$Preferences@UseDatabase  al
    call 'User32.IsDlgButtonChecked' D@hDialog, IDC_CHECKBOX+1    | mov B$Preferences@AutoPowerOn  al
    call 'User32.IsDlgButtonChecked' D@hDialog, IDC_CHECKBOX+2    | mov B$Preferences@AutoHideMenu al
    call 'User32.IsDlgButtonChecked' D@hDialog, IDC_RADIOBUTTON+0 | mov B$Preferences@PreferNTSC   al
    call SaveConfiguration
    call @Close
    ret
____________________________________________________________________________________________
; Preferences
____________________________________________________________________________________________

[LENGTH_PREFERENCES 4]
[Preferences:

 @UseDatabase:    B$ ?
 @AutoPowerOn:    B$ ?
 @AutoHideMenu:   B$ ?
 @PreferNTSC:     B$ ?]
____________________________________________________________________________________________
TITLE SaveGame

; Save game files and the load/save dialog.

____________________________________________________________________________________________
; Save/load game
____________________________________________________________________________________________

SaveGame:

    [@Filename: B$ 'save', 0]
    [@Extension: '.sst', 0]
    if D$pImage = &NULL, ret
    call SaveNumberedROMFile @Filename, @Extension
    ret

LoadGame:

    arguments @pFilename
    if D$pImage != &NULL, call LoadROMFile D@pFilename
    return

SaveQuick:

    [QuickSaveDescription: B$ 'Quick save', 0]
    mov B$GameDescription 0
    call AddString GameDescription, QuickSaveDescription
    call SaveROMFile QuickSaveFilename
    ret

LoadQuick:

    call LoadROMFile QuickSaveFilename
    ret

SaveSST: call SaveChunks FILE_ID_SST, SSTSaveTable | ret
LoadSST: call LoadChunks FILE_ID_SST, SSTLoadTable | ret
____________________________________________________________________________________________

SaveGameDescription:

    mov esi GameDescription, edi ChunkData
L0: movsb | on B$edi-1 != 0, L0<
    sub edi Chunk | mov D$ChunkSize edi
    ret

SaveGameScreenshot:

    call CopyMemory VideoBuffer+0900, ChunkData, (224 shl 8)
    mov D$ChunkSize (224 shl 8)+4
    ret

LoadGameDescription: mov B$GameDescription 0 | call AddString GameDescription ChunkData | ret
LoadGameScreenshot:  call CopyMemory ChunkData, ScreenshotData, (224 shl 8) | ret
____________________________________________________________________________________________

SaveGameCPU:        call SaveChunk CPU,             LENGTH_CPU | ret
SaveGamePPU:        call SaveChunk PPU,             LENGTH_PPU | ret
SaveGameAPU:        call SaveChunk APU,             LENGTH_APU | ret
SaveGameRendering:  call SaveChunk Rendering,       LENGTH_RENDERING | ret
SaveGameMapper:     call SaveChunk MapperData,      LENGTH_MAPPERDATA | ret
SaveGamePorts:      call SaveChunk Ports,           LENGTH_PORTS | ret

LoadGameCPU:        call LoadChunk CPU,             LENGTH_CPU | ret
LoadGamePPU:        call LoadChunk PPU,             LENGTH_PPU | ret
LoadGameAPU:        call LoadChunk APU,             LENGTH_APU | ret
LoadGameRendering:  call LoadChunk Rendering,       LENGTH_RENDERING | ret
LoadGameMapper:     call LoadChunk MapperData,      LENGTH_MAPPERDATA | ret
LoadGamePorts:      call LoadChunk Ports,           LENGTH_PORTS | ret
____________________________________________________________________________________________
; Battery-backed RAM
____________________________________________________________________________________________
[SAVExtension: '*.sav']
SaveBattery: call SaveROMFile BatteryFilename | ret
LoadBattery: call LoadFirstFile ROMDirectory, SAVExtension | ret
SaveSAV: call CopyMemory WRAM, SaveData, 02000 | mov D$SaveSize 02000 | ret
LoadSAV: call CopyMemory D$pFile, WRAM, 02000 | ret
____________________________________________________________________________________________
; Load/save dialog
____________________________________________________________________________________________

SaveDialogProc:

    arguments @hDialog, @Message, @wParam, @lParam

    pushad

        mov eax &FALSE
        if. D@Message = &WM_INITDIALOG  | call @Init    | else
        if. D@Message = &WM_COMMAND     | call @Command | else
        if. D@Message = &WM_CLOSE       | call @Close   | endif

        mov D$esp+28 eax
    popad
    return

@Init:

    ; Edit box (game description)
    [@SaveName: B$ 'Save ', 0]
    call 'Kernel32.GetLocalTime' SystemTime
    call SystemTimeToString
    mov B$GameDescription 0
    call AddString GameDescription, @SaveName
    call AddString GameDescription, SystemTimeString
    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_EDIT+0, &WM_SETTEXT, 0, GameDescription
    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_EDIT+0, &EM_SETSEL, 0, 10
    mov eax &TRUE
    ret

@Command:

    ; Escape pressed
    on W@wParam = &IDCANCEL, @Close

    ; Buttons
    on W@wParam = IDC_CANCEL, @Close
    if. W@wParam = IDC_BUTTON+0 ; Save
        if D$pImage = &NULL, break
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_EDIT+0, &WM_GETTEXT, &MAX_PATH, GameDescription
        call SaveGame
        jmp @Close
    endif
    mov eax &TRUE
    ret

@Close:

    call 'User32.EndDialog' D@hDialog, 1
    mov eax &TRUE
    ret

____________________________________________________________________________________________
; Load dialog
____________________________________________________________________________________________

LoadDialogProc:

    arguments @hDialog, @Message, @wParam, @lParam

    pushad

        mov eax &FALSE
        if. D@Message = &WM_INITDIALOG  | call @Init    | else
        if. D@Message = &WM_COMMAND     | call @Command | else
        if. D@Message = &WM_CLOSE       | call @Close   | endif

        mov D$esp+28 eax
    popad
    return

@Init:

    ; Create bitmap
    call CopyMemory Palette32, ScreenshotPalette, 0100
  . D$hDC = 'User32.GetDC' D@hDialog
  . D$hBMP = 'GDI32.CreateDIBitmap' D$hDC, BMIHeader, &NULL, VideoBuffer+0900, BMIHeader, &DIB_RGB_COLORS

    ; List box
    call @ListSaveGames
    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_SETCURSEL, 0, 0
    call @LoadPreview
    mov eax &TRUE
    ret

@Command:

    ; Escape pressed
    on W@wParam = &IDCANCEL, @Close

    ; List box
    if. W@wParam = IDC_LISTBOX+0
        if W@wParam+2 = &LBN_SELCHANGE, call @LoadPreview
        if W@wParam+2 = &LBN_DBLCLK, mov D@wParam IDC_BUTTON+1
    endif

    ; Buttons
    on W@wParam = IDC_CANCEL, @Close
    if. W@wParam = IDC_BUTTON+0 ; Delete

        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETCURSEL, 0, 0
        if eax = &LB_ERR, break
        push eax

            mov B$Filename 0
            call AddString Filename, ROMDirectory
            call AddString Filename, @Filetitle
            call 'Kernel32.DeleteFileA' Filename
            call @ListSaveGames

            call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETCOUNT, 0, 0

        pop edx
        dec eax
        if edx <= eax, mov eax edx
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_SETCURSEL, eax, 0
        call @LoadPreview

    endif
    if. W@wParam = IDC_BUTTON+1 ; Load
        call LoadGame @FileTitle
        jmp @Close
    endif
    mov eax &TRUE
    ret

@Close:

    call 'GDI32.DeleteObject' D$hBMP
    call 'User32.ReleaseDC' D@hDialog, D$hDC
    call 'User32.EndDialog' D@hDialog, 1
    mov eax &TRUE
    ret
____________________________________________________________________________________________

@ListSaveGames:

    mov B$Filename 0
    call AddString Filename, ROMDirectory
    call AddString Filename, SearchSaveGame
    call 'User32.DlgDirListA' D@hDialog, Filename, IDC_LISTBOX+0, &NULL, &DDL_READWRITE
    ret

@LoadPreview:

    ; Get filename
    [@FileTitle: ? #80]
    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETCURSEL, 0, 0
    call 'User32.SendDlgItemMessageA' D@hDialog, IDC_LISTBOX+0, &LB_GETTEXT, eax, @FileTitle
    mov B$Filename 0
    call AddString Filename, ROMDirectory
    call AddString Filename, @Filetitle

    ; Open file
  . D$hFile = 'Kernel32.CreateFileA' Filename, &GENERIC_READ, &NULL, &NULL, &OPEN_EXISTING, &FILE_ATTRIBUTE_NORMAL, &NULL

    if. eax != &INVALID_HANDLE_VALUE

        ; File time --> string
        call 'Kernel32.GetFileTime' D$hFile, &NULL, &NULL, UTCFileTime
        call 'Kernel32.FileTimeToLocalFileTime' UTCFileTime, FileTime
        call 'Kernel32.FileTimeToSystemTime' FileTime, SystemTime
        call SystemTimeToString

        ; Load file
      . D$Filesize = 'Kernel32.GetFileSize' D$hFile, &NULL
      . D$pFile = 'Kernel32.GlobalAlloc' &GPTR, D$Filesize
        call 'Kernel32.ReadFile' D$hFile, D$pFile, D$Filesize, Temp, &NULL
        call LoadChunks FILE_ID_SST, LoadPreviewTable

        call 'Kernel32.GlobalFree' D$pFile
        call 'Kernel32.CloseHandle' D$hFile

        ; Set static controls
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_STATIC+0, &WM_SETTEXT, 0, GameDescription
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_STATIC+2, &WM_SETTEXT, 0, SystemTimeString
        call 'GDI32.SetDIBits' D$hDC, D$hBMP, 0, 224, ScreenshotData, BMIHeader, &DIB_RGB_COLORS
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_STATIC+1, &STM_SETIMAGE, &IMAGE_BITMAP, D$hBMP
        call 'User32.InvalidateRect' D@hDialog, &NULL, &TRUE

    else

        ; Reset static controls
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_STATIC+0, &WM_SETTEXT, 0, &NULL
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_STATIC+2, &WM_SETTEXT, 0, &NULL
        call 'User32.SendDlgItemMessageA' D@hDialog, IDC_STATIC+1, &STM_SETIMAGE, &IMAGE_BITMAP, &NULL
        call 'User32.InvalidateRect' D@hDialog, &NULL, &TRUE

    endif
    ret

[LoadPreviewTable:
 LoadGameDescription
 LoadGameScreenshot
 &NULL]
[UTCFileTime: ? ?
 FileTime:    ? ?]
[SearchSaveGame:  B$ '*.sst', 0]
[GameDescription: B$ ? #&MAX_PATH]
[FileTimeString:  B$ ? #20]
[ScreenShotData:  B$ ? #LENGTH_VIDEO_BUFFER]
[hDC: ?]
[hBMP: ?]
[BMIHeader:
 @biSize:           D$ 40
 @biWidth:          D$ WIDTH_VIDEO_BUFFER
 @biHeight:         D$ (0-(HEIGHT_VIDEO_BUFFER-17))
 @biPlanes:         W$ 1
 @biBitCount:       W$ 8
 @biCompression:    D$ &BI_RGB
 @biSizeImage:      D$ 0
 @biXPelsPerMeter:  D$ 0
 @biYPelsPerMeter:  D$ 0
 @biClrUsed:        D$ 040
 @biClrImportant:   D$ 040

 ScreenshotPalette: D$ 0 #040]
____________________________________________________________________________________________
; System time to string
____________________________________________________________________________________________

[SystemTime:
 @wYear:         W$ ?
 @wMonth:        W$ ?
 @wDayOfWeek:    W$ ?
 @wDay:          W$ ?
 @wHour:         W$ ?
 @wMinute:       W$ ?
 @wSecond:       W$ ?
 @wMilliseconds: W$ ?]
[SystemTimeString: B$ ? #&MAX_PATH]
SystemTimeToString:

    mov edi SystemTimeString

    ; Year
    movzx eax W$SystemTime@wYear
    mov ebx 100, edx 0 | div ebx
    call BinToDec00
    movzx eax W$SystemTime@wYear
    call BinToDec00
    mov al '-' | stosb

    ; Month
    movzx eax W$SystemTime@wMonth
    call BinToDec00
    mov al '-' | stosb

    ; Day
    movzx eax W$SystemTime@wDay
    call BinToDec00

    ; Hour
    mov al ' ' | stosb
    mov al '(' | stosb
    movzx eax W$SystemTime@wHour
    call BinToDec00
    mov al ':' | stosb

    ; Minute
    movzx eax W$SystemTime@wMinute
    call BinToDec00
    mov al ')' | stosb

    mov B$edi 0
    ret

BinToDec00:

    mov ebx 10
    mov edx 0 | div ebx | push edx
    mov edx 0 | div ebx | push edx

    pop eax | add al '0' | stosb
    pop eax | add al '0' | stosb
    ret
____________________________________________________________________________________________
TITLE NES

; The NES emulator.
; SetMode [PAL/NTSC] - sets the NES to emulate the PAL or NTSC NES.
; SetPower [POWER_ON/POWER_OFF] - toggles NES power
; RunOneFrame - runs the NES for one frame (NTSC: 1/60 second, PAL: 1/50 second)

____________________________________________________________________________________________
; PAL/NTSC differences
____________________________________________________________________________________________

SetMode:

    pop eax, D$CPUMode | push eax

    ; NTSC
    if. D$CPUMode = NTSC
        mov D$CPUCycles          NTSC_CPU_CYCLES
        mov D$PPUCycles          NTSC_PPU_CYCLES
        mov D$CyclesPerFrame     NTSC_CYCLES_PER_FRAME
        mov D$CyclesPerSample    NTSC_APU_CYCLES_PER_SAMPLE
        mov D$PPUCyclesInVBlank  NTSC_PPU_CYCLES_IN_VBLANK

        mov D$ScreenHeight       NTSC_SCREEN_BOTTOM-NTSC_SCREEN_TOP

        mov D$SourceRect+04      NTSC_SCREEN_TOP
        mov D$SourceRect+0C      NTSC_SCREEN_BOTTOM
    endif

    ; PAL
    if. D$CPUMode = PAL
        mov D$CPUCycles          PAL_CPU_CYCLES
        mov D$PPUCycles          PAL_PPU_CYCLES
        mov D$CyclesPerFrame     PAL_CYCLES_PER_FRAME
        mov D$CyclesPerSample    PAL_APU_CYCLES_PER_SAMPLE
        mov D$PPUCyclesInVBlank  PAL_PPU_CYCLES_IN_VBLANK

        mov D$ScreenHeight       PAL_SCREEN_BOTTOM-PAL_SCREEN_TOP

        mov D$SourceRect+04      PAL_SCREEN_TOP
        mov D$SourceRect+0C      PAL_SCREEN_BOTTOM
    endif
    ret

____________________________________________________________________________________________
; Power ON/OFF
____________________________________________________________________________________________

SetPower:

    ; Get argument
    pop eax, edx | push eax

    ; Power changed?
    ; (prevent battery save when power already is off)
    if edx = D$Power, ret
    mov D$Power edx

    ; Power OFF
    if. D$Power = POWER_OFF

        ; Save files
        if B$Cartridge@Battery = &TRUE, call SaveBattery
        call SaveGameGenie

        ; Zero out all RAM
        call ZeroMemory CPUMemory, (SIZE_CPU_MEMORY shl 2)
        call ZeroMemory PPUMemory, (SIZE_PPU_MEMORY shl 2)
        call ZeroMemory CPU (SIZE_CPU_REGS shl 2)
        call ZeroMemory HardResetData, LENGTH_HARDRESET_DATA

        ; Clear output
        call ClearVideo
        call ClearAudio

        ; Uncheck menu
        call 'User32.CheckMenuItem' D$hMenu, IDM_POWER, &MF_BYCOMMAND+&MF_UNCHECKED

    ; Power ON
    else

        ; Load files
        if B$Cartridge@Battery = &TRUE, call LoadBattery
        call LoadGameGenie

        ; Reset the NES
        mov B$S 0FF
        SetIRQ CPU, IRQ_RESET

        ; Check menu
        call 'User32.CheckMenuItem' D$hMenu, IDM_POWER, &MF_BYCOMMAND+&MF_CHECKED

    endif
    ret
____________________________________________________________________________________________
; Run one frame
____________________________________________________________________________________________

RunOneFrame:

    ; Interrupt?
L0: if D$CheckIRQLine != &FALSE, call HandleInterrupts

    ; Execute next opcode
    getNextByte
    copy D$InstructionTable+eax*4 D$Instruction
    call D$AddressingTable+eax*4

    ; Loop
    call PPUSynchronize
    call CartridgeSynchronize
    mov eax D$CyclesPerFrame
    on D$CPUClock < eax, L0<<

    ; Synchronize
    call PPUSynchronize
    call APUSynchronize
    call CartridgeSynchronize
    sub D$APUClock eax
    sub D$CPUClock eax
    sub D$PPUClock eax
    ;ifNotFlag D$IRQClock 08000_0000,
    sub D$IRQClock eax
    sub D$CartridgeClock eax
    ret
____________________________________________________________________________________________
; Data
____________________________________________________________________________________________

[SIZE_NES 7]
[NES:
 Power:              ?
 CPUMode:            ?
 CPUCycles:          ?
 PPUCycles:          ?
 CyclesPerFrame:     ?
 CyclesPerSample:    ?
 PPUCyclesInVBlank:  ?]

[NTSC 0  PAL 1]

[PAL_SCREEN_TOP             2
 PAL_SCREEN_BOTTOM          1+238
 PAL_CPU_CYCLES             16
 PAL_PPU_CYCLES             5
 PAL_CYCLES_PER_FRAME       531960  ; 312 scanlines * 341 cc * 5 PPU cc per Xtal cc
 PAL_APU_CYCLES_PER_SAMPLE  38      ; 50 frames * 531960 cc per frame / 16 cc per APU cc / 44100
 PAL_CPU_CYCLES_PER_SECOND  1662607
 PAL_PPU_CYCLES_IN_VBLANK   23870   ; 70 scanlines * 341 cc

NTSC_SCREEN_TOP             1+8
NTSC_SCREEN_BOTTOM          1+232
NTSC_CPU_CYCLES             12
NTSC_PPU_CYCLES             4
NTSC_CYCLES_PER_FRAME       357368  ; 262 scanlines * 341 cc * 4 PPU cc per Xtal cc
NTSC_APU_CYCLES_PER_SAMPLE  41      ; 60 frames * 357368 cc per frame / 12 cc per APU cc / 44100
NTSC_CPU_CYCLES_PER_SECOND  1789772
NTSC_PPU_CYCLES_IN_VBLANK   6820]   ; 20 scanlines * 341 cc

[POWER_OFF 0
 POWER_ON  1]
____________________________________________________________________________________________
TITLE CPU

; This is used from the NES title to emulate the Motorola 6502.
; SetIRQ [IRQ_NMI/IRQ_RESET/...] - triggers an interrupt
; ClearIRQ [IRQ_NMI/IRQ_RESET/...] - clears an interrupt
; HandleInterrupts - handles any pending interrupts (should be called after every instruction)
; To execute an opcode, see RunOneFrame

____________________________________________________________________________________________
; Interrupts
____________________________________________________________________________________________

[SetIRQ   | or  D$IRQLine #2 | copy D$#1Clock D$IRQClock | call CheckIRQLine]
[ClearIRQ | and D$IRQLine 0FF-#1 | call CheckIRQLine]
[IRQ            IRQ_FRAME+IRQ_DMC+IRQ_CARTRIDGE
 IRQ_RESET      080
 IRQ_NMI        040
 IRQ_FRAME      020
 IRQ_DMC        010
 IRQ_CARTRIDGE   08]

HandleInterrupts:

    ; Reset?
    ifFlag. D$IRQLine IRQ_RESET
        call DoReset
        mov D$IRQLine 0
        call CheckIRQLine
        ret
    endif

    ; 6502 pipelining! If next opcode has already been fetched, execute that one first
    mov eax D$CPUClock | sub eax D$IRQClock | js Q0>>
    if. eax <= D$CPUCycles
        ; Execute
        getNextByte
        copy D$InstructionTable+eax*4 D$Instruction
        call D$AddressingTable+eax*4
    endif
    mov D$IRQClock 0-100

    ; NMI?
    ifFlag. D$IRQLine IRQ_NMI
        call DoNMI
        ClearIRQ IRQ_NMI
        call CheckIRQLine
        ret
    endif

    ; IRQ?
    ifFlag D$IRQLine IRQ, call DoIRQ
    call CheckIRQLine
Q0: ret

CheckIRQLine:

    test D$IRQLine IRQ_RESET+IRQ_NMI | setnz B$CheckIRQStatus
    ifFlag B$IRQLine IRQ,
        if B$Flag.I = &FALSE,
            mov B$CheckIRQStatus &TRUE
    ret


InterruptSequence:

    ; Do interrupt
    cpuTick 7
    pushWord W$PC
    call PackFlags | pushByte al
    ret

DoIRQ:

    ; IRQs enabled?
    if. B$Flag.I = &FALSE
        ;ClearIRQ IRQ_CARTRIDGE
        call InterruptSequence
        mov B$Flag.I &TRUE
        mov B$Flag.D &FALSE
        ; Update PC
        mov eax IRQ_VECTOR
        getWrappedWordEax
        mov D$PC eax
        mov B$Flag.I &TRUE
        mov B$Flag.D &FALSE
    endif
    ret

DoNMI:

    call InterruptSequence

    ; Update PC
    mov B$Flag.I &TRUE
    mov B$Flag.D &FALSE
    mov eax NMI_VECTOR
    getWrappedWordEax
    mov D$PC eax
    ret

DoReset:

    ; Reset other
    call ClearAudio
    call FlushVideo | call ClearVideo
    ; Reset NES

    call ResetMemoryMapping
    call APUReset
    call PPUReset
    call CartridgeReset
    call ResetGameGenie
    call ZeroMemory Ports LENGTH_PORTS | mov B$Zapper 08

    ; Reset CPU
    mov eax RESET_VECTOR | getWrappedWordEax | mov D$PC eax
    mov B$S 0FF
    mov B$Flag.I &TRUE
    mov B$Flag.D &FALSE
    ret

____________________________________________________________________________________________
; Flags
____________________________________________________________________________________________

[BREAK_FLAG  010
 UNUSED_FLAG 020]
[saveSZCV | sets B$Flag.S | setz B$Flag.Z | setc B$Flag.C | seto B$Flag.V]
[saveSZC  | sets B$Flag.S | setz B$Flag.Z | setc B$Flag.C]
[saveSZ   | sets B$Flag.S | setz B$Flag.Z]

PackFlags:

    mov al B$Flag.S | shl al 1
    or  al B$Flag.V | shl al 3
    or  al B$Flag.D | shl al 1
    or  al B$Flag.I | shl al 1
    or  al B$Flag.Z | shl al 1
    or  al B$Flag.C
    or  al UNUSED_FLAG

    call CheckIRQLine
    ret

UnpackFlags:

    shr al 1 | setc B$Flag.C
    shr al 1 | setc B$Flag.Z
    shr al 1 | setc B$Flag.I
    shr al 1 | setc B$Flag.D
    shr al 3 | setc B$Flag.V
    shr al 1 | setc B$Flag.S
    ret

____________________________________________________________________________________________
; Memory access
____________________________________________________________________________________________

; Memory read
;[doRead             | mov D$Temp eax | call D$eax*4+CPUReadTable | mov B$CPUBus al | ifFlag eax 0FFFF_FF00, int 3]
[doRead             | call D$eax*4+CPUReadTable | mov B$CPUBus al]
[getWrappedWordEax  | mov edx eax | doRead | xchg eax edx | inc al | doRead | mov ah al | mov al dl] ; Both bytes from same page
[getNextByte        | movzx eax W$PC | doRead | inc W$PC]
[getNextWord        | getNextByte | mov dl al | getNextByte | mov ah al | mov al dl]
ReadVoid:    | outhex eax | movzx eax B$CPUBus     | ret
ReadRAM:       movzx eax B$eax+RAM    | ret
ReadMirrorRAM: and eax 07FF  | doRead | ret
ReadMirrorPPU: and eax 02007 | doRead | ret
ReadWRAM:      and eax 01FFF | movzx eax B$eax+WRAM | ret

ReadPROM:

    mov ebx eax | shr ebx 13 | and ebx 3
    and eax 01FFF
    add eax D$IndexPROM+ebx*4
    add eax D$pPROM
    movzx eax B$eax
    ret

; Memory write
[doWrite | call D$edx*4+CPUWriteTable]
WriteVoid:       outWrite | ret
WriteRAM:        mov B$edx+RAM al | ret
WriteMirrorRAM:  and edx 07FF  | doWrite | ret
WriteMirrorPPU:  and edx 02007 | doWrite | ret
WriteWRAM:       and edx 01FFF | mov B$edx+WRAM al | ret
____________________________________________________________________________________________
; Stack
____________________________________________________________________________________________
[pushByte | mov al #1 | movzx edx B$S | mov  B$edx+RAM+0100 al | dec B$S]
[pullByte | inc B$S   | movzx edx B$S | mov  al B$edx+RAM+0100 | mov #1 al]
[pushWord | mov bx #1 | pushByte bh | pushByte bl]
[pullWord | pullByte bl | pullByte bh | mov #1 bx]
____________________________________________________________________________________________
; Addressing modes
____________________________________________________________________________________________
[doReadCycle  |             doRead  | cpuTick 1]
[doWriteCycle | cpuTick 1 | doWrite] ; Must tick before or Dizzy the adventurer will shake
;      Save address, read data,     write back unmodified data,    do operation,        write data,    done
Read:                doReadCycle |                                 call D$Instruction                | ret
RMW:   mov edx eax | doReadCycle | pushad | doWriteCycle | popad | call D$Instruction | doWriteCycle | ret
Write: mov edx eax |                                               call D$Instruction | doWriteCycle | ret

; Zero page
ZP_RMW:   cpuTick 2 | getNextByte | jmp RMW
ZP_W:     cpuTick 2 | getNextByte | jmp Write
ZP_R:     cpuTick 2 | getNextByte | jmp Read
ZPX_RMW:  cpuTick 3 | getNextByte | add al B$X | jmp RMW
ZPX_W:    cpuTick 3 | getNextByte | add al B$X | jmp Write
ZPX_R:    cpuTick 3 | getNextByte | add al B$X | jmp Read
ZPY_RMW:  cpuTick 3 | getNextByte | add al B$Y | jmp RMW
ZPY_W:    cpuTick 3 | getNextByte | add al B$Y | jmp Write
ZPY_R:    cpuTick 3 | getNextByte | add al B$Y | jmp Read

; Absolute
Abs_RMW:  cpuTick 3 | getNextWord | jmp RMW
Abs_W:    cpuTick 3 | getNextWord | jmp Write
Abs_R:    cpuTick 3 | getNextWord | jmp Read
AbsX_RMW: cpuTick 4 | getNextWord | mov dx W$X |  add ax dx | jmp RMW
AbsX_W:   cpuTick 4 | getNextWord | mov dx W$X |  add ax dx | jmp Write
AbsX_R:   cpuTick 3 | getNextWord | mov dx W$X | xadd ax dx | if ah != dh, cpuTick 1 | jmp Read
AbsY_RMW: cpuTick 4 | getNextWord | mov dx W$Y |  add ax dx | jmp RMW
AbsY_W:   cpuTick 4 | getNextWord | mov dx W$Y |  add ax dx | jmp Write
AbsY_R:   cpuTick 3 | getNextWord | mov dx W$Y | xadd ax dx | if ah != dh, cpuTick 1 | jmp Read

; Indirect
IndX_RMW: cpuTick 5 | getNextByte | add al B$X | getWrappedWordEax | jmp RMW
IndX_W:   cpuTick 5 | getNextByte | add al B$X | getWrappedWordEax | jmp Write
IndX_R:   cpuTick 5 | getNextByte | add al B$X | getWrappedWordEax | jmp Read
IndY_RMW: cpuTick 5 | getNextByte | getWrappedWordEax | mov dx W$Y |  add ax dx | jmp RMW
IndY_W:   cpuTick 5 | getNextByte | getWrappedWordEax | mov dx W$Y |  add ax dx | jmp Write
IndY_R:   cpuTick 4 | getNextByte | getWrappedWordEax | mov dx W$Y | xadd ax dx | if ah != dh, cpuTick 1 | jmp Read

; Misc
Imm:      cpuTick 1 | movzx eax W$PC | inc W$PC | jmp Read
Acc:      cpuTick 2 | mov eax D$A | call D$Instruction | mov D$A eax | ret
Imp:      cpuTick 2 | call D$Instruction | ret
Spec:     call D$Instruction | ret
Rel:      cpuTick 2 | call D$Instruction
          ; Don't branch!
          je L0> | inc W$PC | ret | L0:
          ; Branch!
          getNextByte | cbw | add ax W$PC
          cpuTick 1
          if ah != B$PC+1, cpuTick 1
          mov W$PC ax
          ret
____________________________________________________________________________________________
; Instructions
____________________________________________________________________________________________

; Read
ADC: shr B$Flag.C 1 |       adc B$A al       | saveSZCV | ret
SBC: shr B$Flag.C 1 | cmc | sbb B$A al | cmc | saveSZCV | ret
AND: and B$A al | saveSZ | ret
EOR: xor B$A al | saveSZ | ret
ORA:  or B$A al | saveSZ | ret
LDA: mov B$A al | add al 0 | saveSZ | ret
LDX: mov B$X al | add al 0 | saveSZ | ret
LDY: mov B$Y al | add al 0 | saveSZ | ret
CMP: cmp B$A al | cmc | saveSZC | ret
CPX: cmp B$X al | cmc | saveSZC | ret
CPY: cmp B$Y al | cmc | saveSZC | ret
BIT: test al B$A | setz  B$Flag.Z
     test al 040 | setnz B$Flag.V
     test al 080 | setnz B$Flag.S
     ret

; Write
STA: mov eax D$A | ret
STX: mov eax D$X | ret
STY: mov eax D$Y | ret

; Read-Modify-Write
ASL: shl al 1 | saveSZC | ret
LSR: shr al 1 | saveSZC | ret
ROL: shr B$Flag.C 1 | rcl al, 1 | setc B$Flag.C | add al 0 | saveSZ | ret
ROR: shr B$Flag.C 1 | rcr al, 1 | setc B$Flag.C | add al 0 | saveSZ | ret
INC: inc al | saveSZ | ret
DEC: dec al | saveSZ | ret

; Special
NOP: ret
JMPi:cpuTick 5 | getNextWord | getWrappedWordEax | mov W$PC ax | ret
JMP: cpuTick 3 | getNextWord | mov W$PC ax | ret
JSR: cpuTick 6 | getNextWord | xchg ax W$PC | dec ax | pushWord ax | ret
RTI: cpuTick 6 | pullByte al | call UnpackFlags | pullWord W$PC | ret
RTS: cpuTick 6 | pullWord W$PC | inc W$PC | ret
PHA: cpuTick 3 | pushByte B$A | ret
PLA: cpuTick 4 | pullByte B$A | add B$A 0 | saveSZ | ret
PHP: cpuTick 3 | call PackFlags | or al BREAK_FLAG | pushByte al | ret
PLP: cpuTick 4 | pullByte al | call UnpackFlags | ret
BRK: cpuTick 7
     inc W$PC | pushWord W$PC
     call PackFlags | or al BREAK_FLAG | pushByte al
     mov eax BRK_VECTOR | getWrappedWordEax | mov D$PC eax
     mov B$Flag.I &TRUE
     ret

; Branch
BCS: cmp B$Flag.C &TRUE  | ret
BEQ: cmp B$Flag.Z &TRUE  | ret
BMI: cmp B$Flag.S &TRUE  | ret
BVS: cmp B$Flag.V &TRUE  | ret
BCC: cmp B$Flag.C &FALSE | ret
BNE: cmp B$Flag.Z &FALSE | ret
BPL: cmp B$Flag.S &FALSE | ret
BVC: cmp B$Flag.V &FALSE | ret

; Implied
SEC: mov B$Flag.C &TRUE  | ret
SED: mov B$Flag.D &TRUE  | ret
SEI: mov B$Flag.I &TRUE  | ret
CLC: mov B$Flag.C &FALSE | ret
CLD: mov B$Flag.D &FALSE | ret
CLI: mov B$Flag.I &FALSE | ret
CLV: mov B$Flag.V &FALSE | ret
DEX: dec B$X | saveSZ | ret
DEY: dec B$Y | saveSZ | ret
INX: inc B$X | saveSZ | ret
INY: inc B$Y | saveSZ | ret
TAX: copy D$A D$X | add B$X 0 | saveSZ | ret
TSX: copy D$S D$X | add B$X 0 | saveSZ | ret
TXA: copy D$X D$A | add B$A 0 | saveSZ | ret
TYA: copy D$Y D$A | add B$A 0 | saveSZ | ret
TAY: copy D$A D$Y | add B$Y 0 | saveSZ | ret
TXS: copy D$X D$S | ret

____________________________________________________________________________________________
; Illegal opcodes
____________________________________________________________________________________________
LAX:SHY:SHX:SBX:ANE:LXA:ARR: ; Read
KIL:ANC:ASR:LAS: ; ?
SHA:SAX:SHS: ; Write
SLO:SRE:ISB:RRA:DCP:RLA:;RMW
;;
    pushad
        outhex D$PC
        closeMessage 'Illegal opcode.'
        copy D$CyclesPerFrame D$CPUClock
    popad
;;
    ret

____________________________________________________________________________________________
; CPU data
____________________________________________________________________________________________
[cpuTick | mov ecx D$CPUCycles | imul ecx #1 | add D$CPUClock ecx]
[ppuTick | mov ecx D$PPUCycles | imul ecx #1 | add D$CPUClock ecx]
[Instruction: ?]

; Sizes
[LENGTH_CPU      <(SIZE_CPU shl 2)>]
[SIZE_CPU        SIZE_CPU_REGS+SIZE_CPU_MEMORY]
[SIZE_CPU_REGS   11+5]
[SIZE_CPU_MEMORY SIZE_RAM+SIZE_WRAM]
[SIZE_RAM        0200]
[SIZE_WRAM       02000]

; 6502 vectors
[NMI_VECTOR 0FFFA   RESET_VECTOR 0FFFC   IRQ_VECTOR 0FFFE   BRK_VECTOR 0FFFE]

; 6502 registers
[CPU:
 PC: ?
 X:  ?
 Y:  ?
 A:  ?
 S:  ?
 Flag.C: ?
 Flag.Z: ?
 Flag.I: ?
 Flag.D: ?
 Flag.V: ?
 Flag.S: ?

; Variables
 IRQClock:       ?
 IRQLine:        ?
 CheckIRQStatus: ?
 CPUClock:       ?
 CPUBus:         ?

; Memory
 CPUMemory:
 RAM:   ? #SIZE_RAM
 WRAM:  ? #SIZE_WRAM]

____________________________________________________________________________________________
; Tables
____________________________________________________________________________________________

; Instructions
[InstructionTable:
 BRK ORA KIL SLO NOP ORA ASL SLO _ PHP ORA ASL ANC NOP  ORA ASL SLO _ BPL ORA KIL SLO NOP ORA ASL SLO _ CLC ORA NOP SLO NOP ORA ASL SLO
 JSR AND KIL RLA BIT AND ROL RLA _ PLP AND ROL ANC BIT  AND ROL RLA _ BMI AND KIL RLA NOP AND ROL RLA _ SEC AND NOP RLA NOP AND ROL RLA
 RTI EOR KIL SRE NOP EOR LSR SRE _ PHA EOR LSR ASR JMP  EOR LSR SRE _ BVC EOR KIL SRE NOP EOR LSR SRE _ CLI EOR NOP SRE NOP EOR LSR SRE
 RTS ADC KIL RRA NOP ADC ROR RRA _ PLA ADC ROR ARR JMPi ADC ROR RRA _ BVS ADC KIL RRA NOP ADC ROR RRA _ SEI ADC NOP RRA NOP ADC ROR RRA
 NOP STA NOP SAX STY STA STX SAX _ DEY NOP TXA ANE STY  STA STX SAX _ BCC STA KIL SHA STY STA STX SAX _ TYA STA TXS SHS SHY STA SHX SHA
 LDY LDA LDX LAX LDY LDA LDX LAX _ TAY LDA TAX LXA LDY  LDA LDX LAX _ BCS LDA KIL LAX LDY LDA LDX LAX _ CLV LDA TSX LAS LDY LDA LDX LAX
 CPY CMP NOP DCP CPY CMP DEC DCP _ INY CMP DEX SBX CPY  CMP DEC DCP _ BNE CMP KIL DCP NOP CMP DEC DCP _ CLD CMP NOP DCP NOP CMP DEC DCP
 CPX SBC NOP ISB CPX SBC INC ISB _ INX SBC NOP SBC CPX  SBC INC ISB _ BEQ SBC KIL ISB NOP SBC INC ISB _ SED SBC NOP ISB NOP SBC INC ISB

; Addressing modes
 AddressingTable:
 Spec IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Spec Imm Acc Imm Abs_R Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Spec IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Spec Imm Acc Imm Abs_R Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Spec IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Spec Imm Acc Imm Spec  Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Spec IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Spec Imm Acc Imm Spec  Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Imm  IndX_W Imm IndX_W   ZP_W ZP_W ZP_W   ZP_W    Imp  Imm Imp Imm Abs_W Abs_W Abs_W   Abs_W    Rel IndY_W Imp IndY_W   ZPX_W ZPX_W ZPY_W   ZPY_W    Imp AbsY_W Imp AbsY_W   AbsX_W AbsX_W AbsY_W   AbsY_W
 Imm  IndX_R Imm IndX_R   ZP_R ZP_R ZP_R   ZP_R    Imp  Imm Imp Imm Abs_R Abs_R Abs_R   Abs_R    Rel IndY_R Imp IndY_R   ZPX_R ZPX_R ZPY_R   ZPY_R    Imp AbsY_R Imp AbsY_R   AbsX_R AbsX_R AbsY_R   AbsY_R
 Imm  IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Imp  Imm Imp Imm Abs_R Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Imm  IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Imp  Imm Imp Imm Abs_R Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW]
____________________________________________________________________________________________
TITLE PPU

; Emulates the Picture Processing Unit.
; Read200x/Write200x handle PPU register access.
; PPUSynchronize makes the PPU catch up in time with the CPU.
; PPUReset resets all PPU regs, but not any CiRAM or CRAM.

____________________________________________________________________________________________
; Memory
____________________________________________________________________________________________
[DoPPURead  | push ebx | call D$PPUReadTable+eax*4 | pop ebx]
[DoPPUWrite | push ebx | call D$PPUWriteTable+edx*4 | pop ebx]

ReadVRAM:

    on eax < 02000, ReadCROM
    on eax < 03000, ReadCiRAM
    on eax < 03F00, ReadHighCiRAM
    on eax < 04000, ReadPalette
    msgError 'Invalid PPU read address.'
    ret

ReadPlainCRAM:

    movzx eax B$CRAM+eax
    ret

ReadCRAM:

    movzx ebx ah | and ebx 01C
    and eax 03FF | add eax D$IndexCRAM+ebx
    movzx eax B$CRAM+eax
    ret

ReadCROM:

    on D$Cartridge@SizeCROM = 0, ReadCRAM

    movzx ebx ah | and ebx 01C
    and eax 03FF | add eax D$IndexCROM+ebx
    add eax D$pCROM
    movzx eax B$eax
    ret

ReadCiROM:

    movzx ebx ah | and ebx 0C
    and eax 03FF | add eax D$IndexCiROM+ebx
    add eax D$pCROM
    movzx eax B$eax
    ret

ReadCiRAM:

    movzx ebx ah | and ebx 0C
    and eax 03FF | add eax D$IndexCiRAM+ebx
    movzx eax B$NameTable+eax
    ret

ReadHighCiRAM:

    ; $3000-$3EFF
    and eax 02FFF
    DoPPURead
    ret

ReadPalette:

    and eax 01F
    movzx eax B$eax+Palette
    ret
____________________________________________________________________________________________

WriteVRAM:

    on edx < 02000, WriteCRAM
    on edx < 03000, WriteCiRAM
    on edx < 03F00, WriteHighCiRAM
    on edx < 04000, WritePalette
    msgError 'Invalid PPU write address.'
    ret

WritePlainCRAM:

    mov B$CRAM+edx al
    ret

WriteCRAM:

    if. D$Cartridge@SizeCRAM > 0
        movzx ebx dh | and ebx 01C
        and edx 03FF | add edx D$IndexCRAM+ebx
        mov B$CRAM+edx al
    endif
    ret

WriteCiRAM:

    movzx ebx dh | and ebx 0C
    and edx 03FF | add edx D$IndexCiRAM+ebx
    mov B$NameTable+edx al
    ret

WriteHighCiRAM:

    ; $3000-$3EFF
    and edx 02FFF
    DoPPUWrite
    ret

WritePalette:

    and edx 01F
    and eax 03F

    ; Color #0?
    ifNotFlag. edx 0F
        mov edx 0
    L0: mov B$Palette+edx*4 al | inc edx | on edx < 8, L0<
    endif

    ; Write to palette
    ifFlag edx 3, mov B$Palette+edx al
    ret

____________________________________________________________________________________________
; Reset
____________________________________________________________________________________________

PPUReset:

    ; Zero out all data
    call ZeroMemory PPU       (SIZE_PPU_REGS shl 2)
    call ZeroMemory Rendering (SIZE_RENDERING shl 2)
    mov D$HSyncHook &NULL

    ; Begin at VBlank
    call StartVBlank

    ; Set up memory mapping
    PPURead  00000, 01FFF, ReadCROM
    PPURead  02000, 02FFF, ReadCiRAM
    PPURead  03000, 03EFF, ReadHighCiRAM
    PPURead  03F00, 03FFF, ReadPalette
    PPUWrite 00000, 01FFF, WriteCRAM
    PPUWrite 02000, 02FFF, WriteCiRAM
    PPUWrite 03000, 03EFF, WriteHighCiRAM
    PPUWrite 03F00, 03FFF, WritePalette
    ret

____________________________________________________________________________________________
; Register reads
____________________________________________________________________________________________

Read200x:
    call PPUSynchronize

    ; Write-only registers
    movzx eax B$VRAMBuffer
    ret

Write2002:
    call PPUSynchronize

    ; Reset latch
    and B$Status 0FF-IN_VBLANK
    mov D$FlipFlop 0
    ret

Read2002:
    call PPUSynchronize

    ; Return PPU status
    movzx eax B$VRAMBuffer
    and al 01F
    or al B$Status

    ; Reset latch
    and B$Status 0FF-IN_VBLANK
    mov D$FlipFlop 0
    ret

Read2004:
    call PPUSynchronize

    ; Read from OAM
    movzx eax B$OAMAddress
    inc B$OAMAddress
    if B$OAMLatch < 8, movzx eax B$OAMLatch
    inc B$OAMLatch
    movzx eax B$eax+OAM
    ret

Read2007:
    call PPUSynchronize

    ; Read from VRAM
    mov eax D$VRAMAddress | and eax 03FFF
    DoPPURead
    xchg al B$VRAMBuffer

ifFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE,
    if D$PPUCycle < 341, ret

    ; Increase VRAM address
    ifFlag. B$Control VERTICAL_WRITE
        add D$VRAMAddress 020
    else
        inc D$VRAMAddress
    endif
    and D$VRAMAddress 07FFF
    ret

____________________________________________________________________________________________
; Register writes
____________________________________________________________________________________________

Write2000:
    call PPUSynchronize

    ; Save register
    mov B$Control al

    ; Update TempAddress
    and eax NAME_TABLE_SELECT
    shl eax, 10
    and D$TempAddress 07FFF-0C00
    or D$TempAddress eax
    ret

Write2001:
    call PPUSynchronize

    ; Save register
    mov B$Mask al
    ; Update masks
    mov D$PixelMask 0
    mov D$ClipMask 0
    ifFlag al BG_VISIBLE,      mov B$PixelMask 0FF
    ifFlag al SPRITES_VISIBLE, mov B$PixelMask+1 0FF
    ifFlag al BG_CLIPPING,     mov B$ClipMask 0FF
    ifFlag al SPRITE_CLIPPING, mov B$ClipMask+1 0FF
    ret

Write2003:
    call PPUSynchronize

    ; Save new OAM address
    mov B$OAMAddress al
    and al 7 | mov B$OAMLatch al
    ret

Write2004:
    call PPUSynchronize

    ; Write
    if. B$OAMLatch < 8
        movzx edx B$OAMLatch
        mov B$edx+OAM al
    else
        movzx edx B$OAMAddress
        if edx >= 8, mov B$edx+OAM al
    endif

    ; Increase address
    inc B$OAMAddress
    inc B$OAMLatch
    ret

Write2005:
    call PPUSynchronize

    ; Horizontal scroll
    if. D$FlipFlop = 0

        mov D$XOffset eax
        and D$XOffset 7

        shr eax, 3  | and eax 01F
        and D$TempAddress 07FFF-01F
        or  D$TempAddress eax

    ; Vertical scroll
    else
        push eax
            shl eax, 12 | and eax 07000
            and D$TempAddress 07FFF-07000
            or  D$TempAddress eax
        pop eax

        shl eax, 2  | and eax 03E0
        and D$TempAddress 07FFF-03E0
        or  D$TempAddress eax
    endif

    ; Toggle
    xor D$FlipFlop 1
    ret

Write2006:

    call PPUSynchronize

    ; Write MSB
    if. D$FlipFlop = 0
        and al 03F
        mov B$TempAddress+1 al

    ; Write LSB, update address
    else
        mov B$TempAddress al
        copy D$TempAddress D$VRAMAddress
    endif

    ; Toggle
    xor B$FlipFlop 1
    ret

Write2007:
    call PPUSynchronize

    ; Write to VRAM
    mov edx D$VRAMAddress | and edx 03FFF
    DoPPUWrite

ifFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE,
    if D$PPUCycle < 341, ret

    ; Increase VRAM address
    ifFlag. B$Control VERTICAL_WRITE
        add W$VRAMAddress 020
    else
        inc W$VRAMAddress
    endif
    and W$VRAMAddress 07FFF
    ret

Write4014:

    ; MSB of 16-bit address
    shl eax 8

    ; Do DMA
    and B$OAMAddress 0FC
    and B$OAMLatch 0FC
L0: push eax
        doRead
        call Write2004
        cpuTick 2
    pop eax
    inc al | jnz L0<
    ret

____________________________________________________________________________________________
; Mirroring
____________________________________________________________________________________________

[mirror | #=1 | call SetMirroring #1]
enumerate 0,
    ONE_SCREEN_2000,
    ONE_SCREEN_2400,
    HORIZONTAL,
    VERTICAL,
    FOUR_SCREEN,
    MIRRORING_0001

SetMirroring:

    call PPUSynchronize
    arguments @Mirroring

    if D@Mirroring = MIRRORING_0001,  setNameTables 0000, 0000, 0000, 0400
    if D@Mirroring = FOUR_SCREEN,     setNameTables 0000, 0400, 0800, 0C00
    if D@Mirroring = ONE_SCREEN_2000, setNameTables 0000, 0000, 0000, 0000
    if D@Mirroring = ONE_SCREEN_2400, setNameTables 0400, 0400, 0400, 0400

    if. D$Cartridge@Mirroring != FOUR_SCREEN_MIRRORING
        if D@Mirroring = HORIZONTAL,  setNameTables 0000, 0000, 0400, 0400
        if D@Mirroring = VERTICAL,    setNameTables 0000, 0400, 0000, 0400
    endif
    return

[SetNameTables | #=4
    mov D$IndexCiRAM+00 #1
    mov D$IndexCiRAM+04 #2
    mov D$IndexCiRAM+08 #3
    mov D$IndexCiRAM+0C #4]

____________________________________________________________________________________________
; Rendering
____________________________________________________________________________________________

PPUSynchronize:

    [@Busy: ?]
    if B$@Busy = &TRUE, ret ; For MMC2 CROM switch during rendering
    mov B$@Busy &TRUE

    pushad

        ; PPU really behind?
        test D$CPUClock 0FFFF_FFFF | js Q0>
        mov eax D$PPUClock | on eax >= D$CPUClock, Q0>

        ; Run PPU until synchronized
    L0: mov eax D$PPUCycle
        inc D$PPUCycle
        push eax
            call D$PPUJumpTable+eax*4
        pop eax
        call D$AddressUpdateTable+eax*4

        ; Loop
        mov eax D$PPUClock | add eax D$PPUCycles | mov D$PPUClock eax
        on eax < D$CPUClock, L0<

Q0: popad
    mov B$@Busy &FALSE
    ret

____________________________________________________________________________________________
; Empty scanline (#241), VBlank.
____________________________________________________________________________________________

IdleEmptyLine:

    ; Continue idle
    mov D$PPUCycle 341
    ; Count down
    dec D$EmptyLineCounter
    jz StartVBlank
    ret

IdleVBlank:

    ; Continue idle
    mov D$PPUCycle 342

    ; MMC5
    if. D$HSyncHook != &NULL
        mov eax D$VBlankCounter
        mov edx 0, ecx 341
        div ecx
        if edx = 0, call D$HSyncHook
    endif

    ; Count down
    dec D$VBlankCounter
    jz EndVBlank
    ret

StartVBlank:

    ; Start waiting in VBlank
    mov D$PPUCycle 342
    copy D$PPUCyclesInVBlank D$VBlankCounter

    ; Adjust regs
    or B$Status IN_VBLANK
    ifFlag B$Control NMI_ON_VBLANK, SetIRQ PPU, IRQ_NMI
    mov D$OAMAddress 0
    mov D$OAMLatch 0

    ; First VBlank cycle
    call IdleVBlank
    ret

EndVBlank:

    if D$HSyncHook != &NULL, call D$HSyncHook

;mov D$Divide42 32
;mov D$A13 0
    if D$VideoCursor != 0, call FlushVideo
    mov D$PPUCycle 0
    mov D$nScanline 0
    mov B$Status 0
    ret

____________________________________________________________________________________________
; Render one pixel
____________________________________________________________________________________________

RenderPixel:

    ; Fetch pixels
    mov esi D$nPixel  |              mov dh B$SpritePipeline+esi
    add esi D$XOffset | and esi 0F | mov dl B$BackgroundPipeline+esi

    ; Clipping
    if D$nPixel < 8, and edx D$ClipMask

    ; GFX visible?
    and edx D$PixelMask

    ; Sprite #0?
    ifFlag. dh 040
        ifFlag dh 3,
            ifFlag dl 3,
                or B$Status SPRITE_ZERO
;;
        if B$Emulate@SpriteZero = &FALSE, break
        mov dh B$APUClock
        or dh 3
;;
    endif

    ; Output sprite
    ifFlag. dh 3
;        on B$Emulate@Sprites = &FALSE, I8>
        movzx eax dh | or eax 010 | and eax 01F
        ifFlag dl 03,
            ifFlag dh 080,
                jmp S0>

    ; Output background
    else
S0:     ;if B$Emulate@Background = &FALSE, mov dl 0
        movzx eax dl
        and eax 0F
    endif

    ; Output pixel
    movzx eax B$eax+Palette
    ifFlag B$Mask MONOCHROME, and al 030
    call VideoOutput
    inc D$nPixel
    ret

____________________________________________________________________________________________
; Cycle 0 - 255
____________________________________________________________________________________________
GetNameTable:
    call ReadNameTable
    call UpdateBackgroundPipeline
    call FindSpritesInRange
    call RenderPixel
    ret

GetAttribute:
    call ReadAttribute
    call RenderPixel
    ret

GetPattern:

    ; MMC5 or normal?
    if. D$Cartridge@Mapper = 5
        call DoMMC5Graphics
    else
        call ReadPattern
    endif

    call FindSpritesInRange
    call RenderPixel
    ret

FindSpritesInRange:

    ; Determine sprite height
    ifFlag. B$Control DOUBLE_SPRITE_SIZE
        mov edx 15
    else
        mov edx 7
    endif

    ; Get OAM data
    ; YYYYYYYY, PATTERN#, VHPxxxAA, XXXXXXXX
    mov ebx D$nPixel | shr ebx 2
    mov eax D$ebx*4+OAM

    ; Out of screen?
    if al >= 240, ret

    ; Calculate new Y
    not al | add al B$nScanline

    ; Out of range?
    if al > dl, ret

    ; Sprite memory full?
    if. D$nSpritesFound < 8
        ; Modify Y to new result
        and al dl

        ; Invert vertical?
        ifFlag eax 080_0000, xor al dl

        ; Sprite #0?
        and eax 0FFFF_FFFF-010_0000
        if ebx = 0, or eax 010_0000

        ; Store in temporary memory
        ; xxxxYYYY, PATTERN#, xHPZxxAA, XXXXXXXX
        mov ebx D$nSpritesFound | inc D$nSpritesFound
        mov D$ebx*4+SpriteMemory eax

        if D$nSpritesFound = 8, or B$Status EIGHT_SPRITES
    endif
    ret
____________________________________________________________________________________________
; Cycle 255
____________________________________________________________________________________________
LineAddress:

    call ZeroMemory SpritePipeline, 0100

    ; GFX visible?
    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Scroll Y
    add W$VRAMAddress 01000 | jns L0>
    and W$VRAMAddress 0FFF
    mov ax W$VRAMAddress | and ax 03E0 | shr ax 5
           if. ax = 29 | and W$VRAMAddress 0FFFF-03E0 | xor W$VRAMAddress 0800
    else | if. ax = 31 | and W$VRAMAddress 0FFFF-03E0
    else | add W$VRAMAddress 020
    endif

L0: ; Update VRAM address(X)
    mov ax W$TempAddress | and ax 00_0000_0100_0001_1111
    and W$VRAMAddress      (07FFF-00_0000_0100_0001_1111)
    or W$VRAMAddress ax

    ; Update VRAM address(Y) at end of dummy scanline
    if. D$nScanline = 0
        copy W$TempAddress W$VRAMAddress
    endif
    ret

____________________________________________________________________________________________
; Cycle 256 - 319
____________________________________________________________________________________________
GetSprite:

    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Any sprites left to fetch?
    mov ebx D$nSprite
    if. ebx = D$nSpritesFound
        mov eax 0
        DoPPURead
        DoPPURead
        ret
    endif
    inc D$nSprite

    ; Get sprite info
    ; xxxxYYYY, PATTERN#, xHPZxxAA, XXXXXXXX
S0: mov ebx D$ebx*4+SpriteMemory

    ; Pattern address -> eax
    ; (Horrible code, but it works)
    mov eax ebx
    ifFlag. B$Control DOUBLE_SPRITE_SIZE
        ; 8x16 sprites
        mov ecx 0
        shr ah 1 | rcl ecx 13
        shl al 5
        rcl ah 1
        shr al 1
        shr ax 4
        and eax 0FFF | or eax ecx
    else
        ; 8x8 sprites
        and al 7
        shl al 4 | shr ax 4
        and eax 0FFF
        ifFlag B$Control SPRITE_ADDRESS_1000, or eax 01000
    endif

    ; Patterns --> dx
    push eax  | DoPPURead | mov dl al | pop eax
    or  eax 8 | DoPPURead | mov dh al

    ; xHPZxxAA --> bl
    ; XXXXXXXX --> bh
    shr ebx 16

    ; Get x coordinate
    movzx edi bh
    add edi SpritePipeline
    mov ecx 8

    ; PZxxAAxx --> bl
    shl bl 2 | jc S0>


    ; - No horizontal flip -
    ; Extract pixel-> al
L1: mov al 0
    shl dh, 1 | rcl al, 1
    shl dl, 1 | rcl al, 1
    ; Add attribute, store
    ifFlag B$edi 3, mov al 0
    if. al != 0
        or al bl
        stosb
    else
        inc edi
    endif
    dec ecx | jnz L1<
    ret

    ; - Horizontal flip -
S0:
    ; Extract pixel-> al
L1: mov al 0
    shr dh, 1 | rcl al, 1
    shr dl, 1 | rcl al, 1
    ; Add attribute, store
    ifFlag B$edi 3, mov al 0
    if. al != 0
        or al bl
        stosb
    else
        inc edi
    endif
    dec ecx | jnz L1<
    ret

____________________________________________________________________________________________
; Cycles 320 - 335
____________________________________________________________________________________________
PreNameTable:
    call ReadNameTable
    call UpdateBackgroundPipeline
    call SimRender
    ret

PreAttribute:
    call ReadAttribute
    call SimRender
    ret

PrePattern:

    ; MMC5 or normal?
    if. D$Cartridge@Mapper = 5
        call DoMMC5Graphics
    else
        call ReadPattern
    endif

    call SimRender
    ret

SimRender: inc D$nPixel | ret

____________________________________________________________________________________________
; Cycle 336 - 339
____________________________________________________________________________________________
DummyFetch: call ReadNameTable | ret
DoNothing:                       ret
____________________________________________________________________________________________
; Cycle 340
____________________________________________________________________________________________
HBlank:

    if D$HSyncHook != &NULL, call D$HSyncHook

    mov D$nPixel 0
    call ZeroMemory SpriteMemory 32
    mov D$nSpritesFound 0
    mov D$nSprite 0
    mov D$PPUCycle 0
    and B$Status 0FF-EIGHT_SPRITES

    inc D$nScanline
    if B$nScanline < 241, ret

    mov D$PPUCycle 341
    mov D$EmptyLineCounter NUM_CYCLES_LINE
    ret
____________________________________________________________________________________________
; Pipeline
____________________________________________________________________________________________
; Sets the pipelined pixels according to W$Patterns
UpdateBackgroundPipeline:

    ; Set up edi
    mov edi D$nPixel
    add edi 8+6 ; Last pixels of next tile
    and edi 0F | add edi BackgroundPipeline

    ; Set up cx/dx (even/odd pixels)
    mov dx W$Patterns | mov cx dx
    and dx 00_01010101_10101010
    and cx 00_10101010_01010101
    shr dl 1 | shl dh 1
    or dl ch | or dh cl

    ; Extract pixel data
    mov ecx D$Attribute
    std
        mov ax dx | and ax 0303 | jz L0> | ifFlag al 3, or al cl | ifFlag ah 3, or ah cl | L0: | stosw | shr dx 2
        mov ax dx | and ax 0303 | jz L0> | ifFlag al 3, or al cl | ifFlag ah 3, or ah cl | L0: | stosw | shr dx 2
        mov ax dx | and ax 0303 | jz L0> | ifFlag al 3, or al cl | ifFlag ah 3, or ah cl | L0: | stosw | shr dx 2
        mov ax dx | and ax 0303 | jz L0> | ifFlag al 3, or al cl | ifFlag ah 3, or ah cl | L0: | stosw | shr dx 2
    cld
    ret
____________________________________________________________________________________________
; Memory reads
____________________________________________________________________________________________

ReadNameTable:

    ; GFX visible?
    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Read name table
    mov eax D$VRAMAddress
    and eax 0FFF | or eax 02000
    DoPPURead
    mov B$NTByte al
    ret

ReadAttribute:

    ; GFX visible?
    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Read attribute
    mov eax 023C0
    mov edx D$VRAMAddress  | shr edx 4 | and dx 038  | or eax edx
    mov edx D$VRAMAddress  | shr edx 2 | and dx 7    | or eax edx
    mov edx D$VRAMAddress  |             and dx 0C00 | or eax edx
    DoPPURead

    ; Shift attribute
    ifFlag W$VRAMAddress 040, shr al 4
    ifFlag W$VRAMAddress 02,  shr al 2
    and al 3 | shl al 2
    mov B$Attribute al
    ret

ReadPattern:

    ; GFX visible?
    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Get pattern address
    movzx eax B$NTByte | shl eax 4
    ifFlag B$Control BG_ADDRESS_1000, or eax 01000
    mov dx W$VRAMAddress | shr dx 12 | or ax dx

    ; Get pattern #1
    push eax
        DoPPURead
        mov B$Patterns al
    pop eax

    ; Get pattern #2
    add eax 8
    DoPPURead
    mov B$Patterns+1 al

    ; No BG pixels
    ifNotFlag B$Mask BG_VISIBLE, mov D$Patterns 0
    ret

____________________________________________________________________________________________
; Address updates
____________________________________________________________________________________________

TileAddress:

    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret
    ; Next tile
    inc W$VRAMAddress
    ifNotFlag. W$VRAMAddress 01F
        sub W$VRAMAddress 020
        xor W$VRAMAddress 0400
    endif
    ret
____________________________________________________________________________________________
; PPU data
____________________________________________________________________________________________
[LENGTH_PPU      <(SIZE_PPU shl 2)>]
[SIZE_PPU        SIZE_PPU_REGS+SIZE_PPU_MEMORY
 SIZE_PPU_REGS   6+3+1
 SIZE_PPU_MEMORY SIZE_OAM+SIZE_NAMETABLE+SIZE_PALETTE+SIZE_CRAM
 SIZE_OAM        040
 SIZE_NAMETABLE  0400
 SIZE_PALETTE    08
 SIZE_CRAM       02000] ; Extra big for CRAM switching (mappers #13 and #96)
[PPU:
 ; Registers
 Control:      ?
 Mask:         ?
 Status:       ?
 OAMAddress:   ?
 OAMLatch:     ?
 VRAMAddress:  ?

 ; Latches
 TempAddress:  ?
 VRAMBuffer:   ?
 FlipFlop:     ?

 PPUClock:     ?

 ; Memory
 PPUMemory:
 OAM:          ? #SIZE_OAM
 NameTable:    ? #SIZE_NAMETABLE
 Palette:      ? #SIZE_PALETTE
 CRAM:         ? #SIZE_CRAM]

; Register flags
[NAME_TABLE_SELECT   03  ; $2000
 VERTICAL_WRITE      04
 SPRITE_ADDRESS_1000 08
 BG_ADDRESS_1000     010
 DOUBLE_SPRITE_SIZE  020
 NMI_ON_VBLANK       080

 MONOCHROME          01  ; $2001
 BG_CLIPPING         02
 SPRITE_CLIPPING     04
 BG_VISIBLE          08
 SPRITES_VISIBLE     010
 COLOR_MASK          0E0

 EIGHT_SPRITES       020 ; $2002
 SPRITE_ZERO         040
 IN_VBLANK           080]
____________________________________________________________________________________________
; Rendering data
____________________________________________________________________________________________
[LENGTH_RENDERING         <(SIZE_RENDERING shl 2)>]
[SIZE_RENDERING           5+2+04E+014]

[Rendering:
 nScanline:            ?
 nPixel:               ?
 PPUCycle:             ?
 VBlankCounter:        ?
 EmptyLineCounter:     ?

 PixelMask:            ?
 ClipMask:             ?

 nSprite:              ?
 nSpritesFound:        ?
 SpriteMemory:         ? #8
 SpritePipeline:       ? #044

 BackgroundRenderer:
 XOffset:              ?
 NTByte:               ?
 Attribute:            ?
 Patterns:             ?
 BackgroundPipeline:   ? #010]

[NUM_CYCLES_LINE   341]
____________________________________________________________________________________________
; PPU jump table
____________________________________________________________________________________________
[PPUJumpTable:
; Background (0 - 255)
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel

GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel

GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel

GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel

; Sprites (256 - 319)
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
; Background for next line (320 - 335)
PreNameTable   SimRender       PreAttribute    SimRender       PrePattern      SimRender       SimRender       SimRender
PreNameTable   SimRender       PreAttribute    SimRender       PrePattern      SimRender       SimRender       SimRender
; Get garbage, HBlank (336 - 340)
DummyFetch     DoNothing       DummyFetch      DoNothing       HBlank
; Do nothing (Empty scanline)
IdleEmptyLine
; Do nothing (VBlank)
IdleVBlank]

[AddressUpdateTable:
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       LineAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing]
_____________________________________________________________________________________________
TITLE APU

; Emulates the Audio Processing Unit.
; Read40xx/Write40xx handle APU register access.
; APUSynchronize makes the APU catch up in time with the CPU.
; APUReset resets the regs.

____________________________________________________________________________________________
; APU synchronization
____________________________________________________________________________________________
APUSynchronize:

    pushad

        ; Is APU behind?
        mov eax D$APUClock | on eax >= D$CPUClock, Q0>
        copy D$CPUClock D$APUClock

        ; How much?
        neg eax | add eax D$CPUClock
        mov edx 0 | div D$CPUCycles

        ; How many samples is that?
        mov edx 0
        add eax D$SampleCounter
        div D$CyclesPerSample
        mov D$SampleCounter edx

        ; Output samples
        test eax eax | jz Q0>
    L0: push eax
            call GetSample | call AudioOutput
        pop eax
        dec eax | jnz L0<

Q0: popad
    ret

____________________________________________________________________________________________
; Output
____________________________________________________________________________________________

GetSample:

    ; Clock all timers
    mov eax D$CyclesPerSample
    add D$Square1+Timer eax
    add D$Square2+Timer eax
    add D$Triangle+Timer eax
    add D$Noise+Timer eax
    if B$DMC+LengthEnabled = &TRUE, add D$DMC+Timer eax
    if D$TimerHook != &NULL, call D$TimerHook

    ; Frame counter
    sub D$FrameCounter eax | jns S0> | call FrameCounterZero | S0:

    ; Get channel amplitudes
    mov edi 0
    call SampleSquare Square1 | add edi D$Square1+Output
    call SampleSquare Square2 | add edi D$Square2+Output
    call SampleTriangle       | add edi D$Triangle+Output
    call SampleNoise          | add edi D$Noise+Output
    call SampleDMC            | mov eax D$DMC+Output | shr eax 2 | add edi eax
    if D$SampleHook != &NULL, call D$SampleHook

    mov eax edi
    ret

____________________________________________________________________________________________
; Signal generation
____________________________________________________________________________________________

SampleSquare:

    pop eax ebx | push eax

or D$ebx+TimerLatch 01

    ; Count steps
    mov eax D$ebx+Timer, edx 0
    div D$ebx+TimerLatch
    mov D$ebx+Timer edx

    if B$ebx+Active = &FALSE, ret

    ; Wave high or low?
    add eax D$ebx+DutyCycleCounter | and eax 0F
    mov D$ebx+DutyCycleCounter eax
    movzx edx B$ebx+DutyCycleType
    shr al 1 | and eax 07

; Send decay or volume?
if. B$ebx+DecayDisabled = &TRUE
    copy W$ebx+Volume W$ebx+Output
else
    copy W$ebx+DecayCounter W$ebx+Output
endif

    if B$DutyCycleTable+edx*8+eax = 0, mov B$ebx+Output 0
    ret

SampleTriangle:

or D$Triangle+TimerLatch 01

    ; Count steps
    mov eax D$Triangle+Timer, edx 0
    div D$Triangle+TimerLatch
    mov D$Triangle+Timer edx
    if B$Triangle+Active = &FALSE, ret

    ; Triangle step
    add al B$Triangle+Step
    mov B$Triangle+Step al
    ifFlag al 010, xor al 0F | and al 0F
    mov B$Triangle+Output al
    ret

SampleNoise:

or D$Noise+TimerLatch 01

    ; Count steps
    mov eax D$Noise+Timer, edx 0
    div D$Noise+TimerLatch
    mov D$Noise+Timer edx
    if B$Noise+Active = &FALSE, ret
    if eax = 0, ret

; Send decay or volume?
if. B$Noise+DecayDisabled = &TRUE
    copy W$Noise+Volume W$Noise+Output
else
    copy W$Noise+DecayCounter W$Noise+Output
endif
shr B$Noise+Output 1

; Shift reg
if B$Noise+RandomMode93 = &FALSE
    mov dx 02000
else
    mov dx 0100
endif

    mov ecx eax
L0: test W$Noise+ShiftRegister 04000 | setnz al
    ifFlag W$Noise+ShiftRegister dx, xor al 1
    shr al 1 | rcl W$Noise+ShiftRegister 1
    dec ecx | jnz L0<

    ifFlag W$Noise+ShiftRegister 04000, mov B$Noise+Output 0
    ret
____________________________________________________________________________________________
; Frame counter
____________________________________________________________________________________________
Write4017:

    call APUSynchronize
    ClearIRQ IRQ_FRAME

    mov B$Reg4017 al
    ; Short: 4, 0,1,2,3, 0,1...
    ; Long:  0,1,2,3,4, 0,1...
    mov B$Update 0
    ifFlag. B$Reg4017 LONG_FRAMES
        mov D$FrameCounter 0
        call FrameCounterZero
    else
        mov D$FrameCounter 7457
    endif

    ; Frame IRQ
    ifNotFlag. B$Reg4017 0C0
        mov D$FrameIRQCounter 29830
    else
        mov D$FrameIRQCounter 0
    endif
    ret

FrameCounterZero:

    ; Reset counter
    add D$FrameCounter 7457

    ; Which update?
    if B$Update = 0, call Clock240Hz
    if B$Update = 1, call Clock120Hz
    if B$Update = 2, call Clock240Hz
    if B$Update = 3, call Clock120Hz

    ; Final update?
    inc B$Update
    ifFlag. B$Reg4017 LONG_FRAMES
        if B$Update >= 5, mov B$Update 0
    else
        if B$Update >= 4, mov B$Update 0
    endif
Q0: ret

Clock120Hz:

    inc D$FrameCounter

    ; Sweep units
    call ClockSweepUnit SQUARE1
    call ClockSweepUnit SQUARE2

    ; Length counters
    call ClockLengthCounter SQUARE1
    call ClockLengthCounter SQUARE2
    call ClockLengthCounter Triangle
    call ClockLengthCounter Noise

    if D$FrameCounterHook120Hz != &NULL, call D$FrameCounterHook120Hz
    call Clock240Hz
    ret

Clock240Hz:

    ; Decay units, linear counter
    call ClockEnvelopeDecay SQUARE1
    call ClockEnvelopeDecay SQUARE2
    call ClockEnvelopeDecay Noise
    call ClockLinearCounter

    if D$FrameCounterHook240Hz != &NULL, call D$FrameCounterHook240Hz
    ret
____________________________________________________________________________________________

ClockEnvelopeDecay:

    pop eax ebx | push eax

    ; Decay clock
    if. B$ebx+DecayDivider != 0
        dec B$ebx+DecayDivider
        ret
    endif
    mov al B$ebx+DecayRate | mov B$ebx+DecayDivider al

    ; Decrease decay volume
    if. B$ebx+DecayCounter != 0
        dec B$ebx+DecayCounter
    else
        if B$ebx+DecayLooping = &TRUE, mov B$ebx+DecayCounter 0F
    endif
    ret

ClockSweepUnit:

    pop eax ebx | push eax

    ; Channel active?
    if B$ebx+Active = &FALSE, ret
    if B$ebx+SweepEnabled = &FALSE, ret

    ; Sweep clock
    if. B$ebx+SweepDivider > 0
        dec B$ebx+SweepDivider
        ret
    endif
    mov al B$ebx+SweepRate | mov B$ebx+SweepDivider al

    ; Shift
    mov eax D$ebx+TimerLatch
    mov cl B$ebx+ShiftAmount
    if cl = 0, ret
    shr eax cl

    ; Update wavelength
    if. B$ebx+SweepNegate = &TRUE
        ; Decrease wavelength
        if ebx = SQUARE1, not eax
        if ebx = SQUARE2, neg eax
    endif
    add eax D$ebx+TimerLatch
    if. eax >= D$ebx+MaxWavelength | mov B$ebx+Active &FALSE | ret | endif
;    if. eax >= 07FF | mov B$ebx+Active &FALSE | ret | endif
    if. eax <= MIN_SQUARE_WAVELENGTH | mov B$ebx+Active &FALSE | ret | endif
    mov D$ebx+TimerLatch eax
    ret

ClockLengthCounter:

    pop eax ebx | push eax

    ; Count down length counter
    if B$ebx+LengthCounter = 0, mov B$ebx+Active &FALSE
    if B$ebx+LengthHalted = &FALSE,
        if B$ebx+LengthCounter != 0,
            dec B$ebx+LengthCounter
    ret

ClockLinearCounter:

    if. B$Triangle+LinearCounter =  0
        mov B$Triangle+Active &FALSE
    else
        if B$Triangle+LinearCounterHalt = &FALSE, dec B$Triangle+LinearCounter
    endif

    if B$Triangle+LinearCounterControl = &FALSE, mov B$Triangle+LinearCounterHalt &FALSE
    ret
____________________________________________________________________________________________
; Length counters
____________________________________________________________________________________________

Write4015:

    call APUSynchronize

    ; Kill length counters?
    ifNotFlag al  SQUARE1_ENABLED, mov B$Square1+LengthCounter  0
    ifNotFlag al  SQUARE2_ENABLED, mov B$Square2+LengthCounter  0
    ifNotFlag al TRIANGLE_ENABLED, mov B$Triangle+LengthCounter 0
    ifNotFlag al    NOISE_ENABLED, mov B$Noise+LengthCounter    0

    ; DMC
    ifFlag. al DMC_ENABLED
        if B$DMC+LengthEnabled = &FALSE, call ResetDMC
    else
        mov B$DMC+LengthEnabled &FALSE
        mov D$DmcIrqCounter 0
    endif

    ; Enable channels?
    shr al 1 | setc B$Square1+LengthEnabled
    shr al 1 | setc B$Square2+LengthEnabled
    shr al 1 | setc B$Triangle+LengthEnabled
    shr al 1 | setc B$Noise+LengthEnabled
    call ChannelsActive

    ; Reset DMC IRQ status
    ClearIRQ IRQ_DMC
    ret

Read4015:

    call APUSynchronize

    ; Length counter non-zero?
    mov eax 0
    if B$Square1+LengthCounter  > 0, or al SQUARE1_ENABLED
    if B$Square2+LengthCounter  > 0, or al SQUARE2_ENABLED
    if B$Triangle+LengthCounter > 0, or al TRIANGLE_ENABLED
    if B$Noise+LengthCounter    > 0, or al NOISE_ENABLED
    if B$DMC+LengthEnabled      > 0, or al DMC_ENABLED

    ; IRQ status
    ifFlag D$IRQLine IRQ_FRAME, or al 040
    ifFlag D$IRQLine IRQ_DMC,   or al 080

    ; Reset frame IRQ status
    ClearIRQ IRQ_FRAME
    ret
____________________________________________________________________________________________

ChannelsActive:

    call SquareActive Square1
    call SquareActive Square2
    call TriangleActive
    call NoiseActive
    ret

SquareActive:

    pop eax, ebx | push eax

    mov edx D$ebx+MaxWavelength
    mov B$ebx+Active &FALSE

    if B$ebx+LengthEnabled = &TRUE,
        if B$ebx+LengthCounter != 0,
            if W$ebx+TimerLatch >= MIN_SQUARE_WAVELENGTH,
;                if W$ebx+TimerLatch <= MAX_SQUARE_WAVELENGTH,
                if D$ebx+TimerLatch < edx,
                    mov B$ebx+Active &TRUE
    ret

TriangleActive:

    mov B$Triangle+Active &FALSE
    if B$Triangle+LengthEnabled = &TRUE,
        if B$Triangle+LengthCounter != 0,
            if B$Triangle+LinearCounter != 0,
                    if W$Triangle+TimerLatch >= MIN_TRIANGLE_WAVELENGTH,
                        mov B$Triangle+Active &TRUE
    ret

NoiseActive:

    mov B$Noise+Active &FALSE
    if B$Noise+LengthEnabled = &TRUE,
        if B$Noise+LengthCounter != 0,
            mov B$Noise+Active &TRUE
    ret

____________________________________________________________________________________________
; Reset
____________________________________________________________________________________________

APUReset:

    [TimerHook: ?
     SampleHook: ?
     FrameCounterHook120Hz: ?
     FrameCounterHook240Hz: ?]
    call ZeroMemory TimerHook 010

    call ZeroMemory APU (SIZE_APU shl 2)
    mov B$Noise+ShiftRegister 1
    mov B$Reg4017 DISABLE_FRAME_IRQ
    mov W$Square1+MaxWavelength MAX_SQUARE_WAVELENGTH
    mov W$Square2+MaxWavelength MAX_SQUARE_WAVELENGTH
    ret
____________________________________________________________________________________________
; Register writes
____________________________________________________________________________________________

; Square
____________________________________________________________________________________________

Write4000: call WriteSquareReg0 Square1 | ret
Write4001: call WriteSquareReg1 Square1 | ret
Write4002: call WriteSquareReg2 Square1 | ret
Write4003: call WriteSquareReg3 Square1 | ret
Write4004: call WriteSquareReg0 Square2 | ret
Write4005: call WriteSquareReg1 Square2 | ret
Write4006: call WriteSquareReg2 Square2 | ret
Write4007: call WriteSquareReg3 Square2 | ret

; ddle vvvv
WriteSquareReg0:

    pop edx, ebx | push edx
    call APUSynchronize

    ; --le ----
    test al 020 | setnz B$ebx+LengthHalted
    test al 010 | setnz B$ebx+DecayDisabled
    ; ---- vvvv
    mov B$ebx+Volume al
    and B$ebx+Volume 0F
    ; dd-- ----
    mov B$ebx+DutyCycleType al
    shr B$ebx+DutyCycleType 6
    ret

; errr nsss
WriteSquareReg1:

    pop edx, ebx | push edx
    call APUSynchronize

    ; ---- -sss
    mov B$ebx+ShiftAmount al
    and B$ebx+ShiftAmount 07
    ; ---- n---
    test al 08  | setnz B$ebx+SweepNegate
    ; -rrr ----
    mov B$ebx+SweepRate al
    shr B$ebx+SweepRate 4
    and B$ebx+SweepRate 07
    ; e--- ----
    test al 080 | setnz B$ebx+SweepEnabled

    ; Reset sweep divider
    mov al B$ebx+SweepRate | mov B$ebx+SweepDivider al

    ; Max wavelength
;;
    [SquareMax: 0400, 0556, 0667, 071D, 0788, 07C2, 07E1, 07F1]
    movzx eax B$ebx+SweepRate | xor al 07 | mov ax W$SquareMax+eax*4
    if B$ebx+SweepEnabled = &FALSE, mov ax MAX_SQUARE_WAVELENGTH
    if B$ebx+SweepNegate  = &TRUE,  mov ax MAX_SQUARE_WAVELENGTH
    mov W$ebx+MaxWavelength ax
;;
    call SquareActive ebx
    ret

; wwww wwww
WriteSquareReg2:

    pop edx, ebx | push edx
    call APUSynchronize

    ; wwww wwww
    mov B$ebx+TimerLatch al

    call SquareActive ebx
    ret

; llll lwww
WriteSquareReg3:

    pop edx, ebx | push edx
    call APUSynchronize

    ; ---- -www
    mov B$ebx+TimerLatch+1 al
    and B$ebx+TimerLatch+1 07
    ; llll l---
    if. B$ebx+LengthEnabled = &TRUE
        shr eax 3 | and eax 01F
        mov al B$eax+LengthCounterTable | mov B$ebx+LengthCounter al
    endif

    ; Reset duty cycle counter and decay counter
    mov B$ebx+DutyCycleCounter 0
    mov B$ebx+DecayCounter 0F
    mov al B$ebx+DecayRate | mov B$ebx+DecayDivider al

    ; Channel active?
    call SquareActive ebx
    ret

; Triangle
____________________________________________________________________________________________

; hlll llll
Write4008:

    call APUSynchronize

    ; h--- ----
    test eax 080 | setnz B$Triangle+LengthHalted
    ; -lll llll
    and al 07F
    mov B$Triangle+LinearCounterLoad al
    if B$Triangle+LinearCounterHalt = &TRUE, mov B$Triangle+LinearCounter al

    call TriangleActive
    ret

; tttt tttt
Write400A:

    call APUSynchronize

    ; tttt tttt
    mov B$Triangle+TimerLatch al

    call TriangleActive
    ret

; llll lttt
Write400B:

    call APUSynchronize

    ; ---- -ttt
    mov B$Triangle+TimerLatch+1 al
    and B$Triangle+TimerLatch+1 07
    ; llll l---
    if. B$Triangle+LengthEnabled = &TRUE
        shr eax 3 | and eax 01F
        mov al B$eax+LengthCounterTable | mov B$Triangle+LengthCounter al
    endif

    ; Halt linear counter
    mov B$Triangle+LinearCounterHalt &TRUE
    mov al B$Triangle+LinearCounterLoad | mov B$Triangle+LinearCounter al

    call TriangleActive
    ret

; Noise
____________________________________________________________________________________________

; --ld vvvv
Write400C:

    call APUSynchronize

    ; ---- vvvv
    mov B$Noise+Volume al | and B$Noise+Volume 0F
    ; ---d ----
    test al 010 | setnz B$Noise+DecayDisabled
    ; --l- ----
    test al 020 | setnz B$Noise+LengthHalted

    call NoiseActive
    ret

; m--- wwww
Write400E:

    call APUSynchronize

    ; m--- ----
    test al 080 | setnz B$Noise+RandomMode93
    ; --- wwww
    and eax 0F
    copy W$NoiseWaveLengthTable+eax*2 W$Noise+TimerLatch

    call NoiseActive
    ret

; llll l---
Write400F:

    call APUSynchronize

    ; llll l---
    if. B$Noise+LengthEnabled = &TRUE
        shr eax 3 | and eax 01F
        mov al B$eax+LengthCounterTable | mov B$Noise+LengthCounter al
    endif

    ; Reset decay
    mov B$Noise+DecayCounter 0F
    mov al B$Noise+DecayRate | mov B$Noise+DecayDivider al

    call NoiseActive
    ret
____________________________________________________________________________________________
; APU data
____________________________________________________________________________________________

; Frame counter
[DISABLE_FRAME_IRQ  040
 FRAME_IRQ          040
 LONG_FRAMES        080]

; Length counter
[SQUARE1_ENABLED    01
 SQUARE2_ENABLED    02
 TRIANGLE_ENABLED   04
 NOISE_ENABLED      08
 DMC_ENABLED        010]
[LengthCounterTable: B$
  0A  0FE
 014   02
 028   04
 050   06
 0A0   08
 03C   0A
  0E   0C
 01A   0E
  0C   010
 018   012
 030   014
 060   016
 0C0   018
 048   01A
 010   01C
 020   01E]

; Square
[DutyCycleTable: B$
 000 0FF 000 000 000 000 000 000
 000 0FF 0FF 000 000 000 000 000
 000 0FF 0FF 0FF 0FF 000 000 000
 0FF 000 000 0FF 0FF 0FF 0FF 0FF]
[MIN_SQUARE_WAVELENGTH   8
 MAX_SQUARE_WAVELENGTH   07B7]

; Triangle
[MIN_TRIANGLE_WAVELENGTH 4]

; Noise
[NoiseWaveLengthTable: W$ 02 04 08 010   020 030 040 050   065 07F 0BE 0FE   017D 01FC 03F9 07F2]

; APU/channels
[LENGTH_APU     <(SIZE_APU shl 2)>]
[SIZE_APU       <7+(10 shl 5)>]
[APU:
 APUClock:              ?
 SampleCounter:         ?
 FrameCounter:          ?
 Update:                ?
 Reg4017:               ?
 FrameIRQCounter:       ?
 DmcIRQCounter:         ?

____________________________________________________________________________________________

 Channels:
 Square1:     ? #020
 Square2:     ? #020
 Triangle:    ? #020
 Noise:       ? #020
 DMC:         ? #020

 ExChannel:
 ExChannel0:  ? #020
 ExChannel1:  ? #020
 ExChannel2:  ? #020
 ExChannel3:  ? #020
 ExChannel4:  ? #020]

enumerate4 0,

    ; Common
    Timer,
    TimerLatch,
    Output, Active,
    LengthHalted, LengthEnabled, LengthCounter,

    ; Square/Noise
    DecayRate, DecayCounter, DecayDivider, DecayDisabled,

    ; Square
    DutyCycleType, DutyCycleCounter,
    SweepNegate, SweepEnabled, SweepDivider, SweepRate, ShiftAmount,
    MaxWaveLength, MaxWaveLengthHigh

    ; Triangle
enumerate4 030,
    LinearCounter, LinearCounterLoad, LinearCounterHalt,
    Step, ChangeMode

    ; Noise
enumerate4 030,
    RandomMode93, ShiftRegister, ShiftRegisterHigh

    ; DMC
enumerate4 030,
    LengthCounterLatch, DMCLoop, nBits,
    pSample, pSampleLatch, Sample, DMCEnableIRQ,
    ConvertedTimer, ConvertedTimerLatch

[Volume               DecayRate
 LinearCounterControl LengthHalted
 DecayLooping         LengthHalted]
____________________________________________________________________________________________
TITLE DMC

; I am still not satisfied with the Delta Modulation Channel,
; that's why I gave it a TITLE of its own.
; * Mig-29 doesn't look perfect when refueling
; * Blarggs Tetris music sounds awful

____________________________________________________________________________________________
; DMC
____________________________________________________________________________________________

[DMCWavelengthTable: B$ 0D6 0BE 0AA 0A0 08F 07F 071 06B
                        05F 050 047 040 035 02A 024 01B]
[DMC_IRQ  080]
____________________________________________________________________________________________
SetDmcIRQCounter:

    ; Calculate number of cycles left
    mov eax D$DMC+LengthCounterLatch | inc eax

    mul D$DMC+ConvertedTimerLatch

    if D$DMC+LengthEnabled = &FALSE, mov eax 0
    mov D$DmcIRQCounter eax
    ret

SampleDMC:
;mov edi 0
or D$DMC+TimerLatch 1
if D$DMC+TimerLatch = 0, ret

    ; Count steps
    mov eax D$DMC+Timer, edx 0
    div D$DMC+TimerLatch
    mov D$DMC+Timer edx

    ; Get sample and adjust output
    ifNotFlag eax eax, ret
L0: shr B$DMC+Sample 1 | jc S0>
    if B$DMC+Output > 001, sub B$DMC+Output 2 | jmp S1> | S0:
    if B$DMC+Output < 07E, add B$DMC+Output 2           | S1:

    ; 8 bits yet?
    inc B$DMC+nBits
    if. B$DMC+nBits = 8
        push eax
            call FetchSample
        pop eax
        mov B$DMC+nBits 0
    endif
    dec eax | jnz L0<
    ret

FetchSample:

    ; Fetch next sample byte
    mov eax D$DMC+pSample | or eax 08000
    inc W$DMC+pSample
    doRead
    mov D$DMC+Sample eax

    if. D$DMC+LengthCounter = 0
        call APUSynchronize
        mov D$DMC+LengthEnabled &FALSE
        ; Loop?
        if D$DMC+DMCLoop = &TRUE, call ResetDMC
    else
        dec D$DMC+LengthCounter
    endif
    ret

ResetDMC:

    mov B$DMC+LengthEnabled &TRUE
    mov D$DMC+nBits 0
    mov D$DMC+Timer 0
    copy D$DMC+LengthCounterLatch D$DMC+LengthCounter
    copy D$DMC+pSampleLatch       D$DMC+pSample
push eax
call SetDmcIRQCounter
pop eax
    ret

____________________________________________________________________________________________
; Register writes
____________________________________________________________________________________________

; il-- wwww
Write4010:

    call APUSynchronize

    ; i--- ----
    test eax 080 | setnz B$DMC+DMCEnableIRQ
    if D$DMC+DMCEnableIRQ = &FALSE, ClearIRQ IRQ_DMC

    ; -l-- ----
    test eax 040 | setnz B$DMC+DMCLoop

    ; ---- wwww
    mov ebx eax
    and ebx 0F

    ; Table value
    movzx eax B$DMCWavelengthTable+ebx
    if. B$CPUMode = PAL
        mov ecx  PAL_CPU_CYCLES_PER_SECOND | mul ecx
        mov ecx NTSC_CPU_CYCLES_PER_SECOND | div ecx
    endif

    ; Update shift timer
    mov D$DMC+TimerLatch eax
    shl D$DMC+TimerLatch 1

     ; Convert
    xor bl 0F | inc bl
    sub al bl
    shl eax 4
    mov D$DMC+ConvertedTimerLatch eax
call SetDmcIRQCounter
    ret

; oooo oooo
Write4011:

    call APUSynchronize

    ; oooo oooo
    and eax 07F | mov D$DMC+Output eax
    ret

; aaaa aaaa
Write4012:

    call APUSynchronize

    ; aaaa aaaa --> 11aa aaaa aa00_0000   (Data << 6) + $C000
    shl eax 6 | or eax 0C000
    mov D$DMC+pSampleLatch eax
    ret

; llll llll
Write4013:

    call APUSynchronize

    ; llll llll (number of samples / $10)
    shl eax 4
    mov D$DMC+LengthCounterLatch eax
call SetDmcIRQCounter
    ret
____________________________________________________________________________________________
TITLE Ports

; Emulates the NES I/O ports.
; As of now, only the two standard controllers are emulated, and the 4-player adapter.
; Through GetButtonStatus, the input engine is translated to the NES.

____________________________________________________________________________________________
; Register reads/writes
____________________________________________________________________________________________

Read4016:

    ; Transparent?
    if B$Strobe = 1, call GetButtonStatus
    mov eax 0

    if. B$Port1 = CONTROLLER_ZAPPER

        ; Zapper
        call CheckZapperHit
        if B$Zapper@Triggered = &TRUE, or al 010

    else

        ; Four-score
        if.. B$nRead4016 < 8  | shr B$JoypadStatus+0 1 | setc al | else.. ; Pad #1
        if.. B$nRead4016 < 16 | shr B$JoypadStatus+2 1 | setc al | else.. ; Pad #3
        if.. B$nRead4016 < 24 | test B$nRead4016 19    | sete al | else.. ; 4-player signature
        mov al 0, B$nRead4016 24 | endif..

    endif

    and B$CPUBus 040
    or al B$CPUBus
    inc B$nRead4016
    ret

Read4017:

    ; Transparent?
    if B$Strobe = 1, call GetButtonStatus
    mov eax 0

    if. B$Port2 = CONTROLLER_ZAPPER

        ; Zapper
        call CheckZapperHit
        if B$Zapper@Triggered = &TRUE, or al 010

    else

        ; Four-score
        if.. B$nRead4017 < 8  | shr B$JoypadStatus+1 1 | setc al | else.. ; Pad #2
        if.. B$nRead4017 < 16 | shr B$JoypadStatus+3 1 | setc al | else.. ; Pad #4
        if.. B$nRead4017 < 24 | test B$nRead4017 18    | sete al | else.. ; 4-player signature
        mov al 0, B$nRead4017 24 | endif..

    endif

    and B$CPUBus 040
    or al B$CPUBus
    inc B$nRead4017
    ret

Write4016:

    ; Register toggled 1->0?  -->  get button status
    and al 1
    xchg al B$Strobe
    if al = 1,
        if B$Strobe = 0,
            call GetButtonStatus
    ret

GetButtonStatus:

    mov B$nRead4016 0
    mov B$nRead4017 0
    mov ebx 0, ecx 0

    ; Pads
L0: mov al 0
    ifFlag  B$InputArray+ebx+PAD1_A      BUTTON_DOWN, or al 01
    ifFlag  B$InputArray+ebx+PAD1_B      BUTTON_DOWN, or al 02
    ifFlag  B$InputArray+ebx+PAD1_SELECT BUTTON_DOWN, or al 04
    ifFlag  B$InputArray+ebx+PAD1_START  BUTTON_DOWN, or al 08
    ifFlag  B$InputArray+ebx+PAD1_UP     BUTTON_DOWN, or al 010
    ifFlag  B$InputArray+ebx+PAD1_LEFT   BUTTON_DOWN, or al 040
    ifFlag. B$InputArray+ebx+PAD1_DOWN   BUTTON_DOWN | or al 020 | and al 0FF-010 | endif
    ifFlag. B$InputArray+ebx+PAD1_RIGHT  BUTTON_DOWN | or al 080 | and al 0FF-040 | endif
    mov B$JoypadStatus+ecx al
    add ebx 10
    inc ecx | on ecx < 4, L0<<
    ret

____________________________________________________________________________________________
; Zapper
____________________________________________________________________________________________

ZapperCoordinates:

    pop eax, W@X, W@Y | push eax

    [@Coordinates:
     @X: ?
     @Y: ?]
    call 'User32.ClientToScreen' D$hMainWindow, @Coordinates

    ; x
    mov eax D@X | sub eax D$ClientRect+00
    mov ecx D$SourceRect+08 | sub ecx D$SourceRect+00 | mul ecx
    mov ecx D$ClientRect+08 | sub ecx D$ClientRect+00 | div ecx
    mov D$Zapper@X eax

    ; y
    mov eax D@Y | sub eax D$ClientRect+04
    mov ecx D$SourceRect+0C | sub ecx D$SourceRect+04 | mul ecx
    mov ecx D$ClientRect+0C | sub ecx D$ClientRect+04 | div ecx
    mov D$Zapper@Y eax
    ret

CheckZapperHit:

    ; Find pixel
    mov ebx D$Zapper@Y
    add ebx D$SourceRect+04
    shl ebx 8
    mov bl B$Zapper@X

    mov al 08

    ; Maybe it's not rendered yet?
    if ebx > D$VideoCursor, ret

    ; Check pixel
    if B$VideoBuffer+ebx = 015, mov al 0 ; Chiller (U)
    if B$VideoBuffer+ebx = 020, mov al 0
    if B$VideoBuffer+ebx = 030, mov al 0
    ret

SetZapper:

    arguments @pPort, @Enabled
    mov ebx D@pPort
    if. B@Enabled = &TRUE
        mov D$ebx CONTROLLER_ZAPPER
    else
        mov D$ebx CONTROLLER_GAMEPAD
    endif

    call UpdateCursor
    return

____________________________________________________________________________________________
; Data
____________________________________________________________________________________________

[LENGTH_PORTS <(SIZE_PORTS shl 2)>]
[SIZE_PORTS 7]
[Ports:
 nRead4016:     ?
 nRead4017:     ?
 JoypadStatus:  ?
 Strobe:        ?

 Zapper:
 @Triggered:    ?
 @X:            ?
 @Y:            ?]
____________________________________________________________________________________________
TITLE Cartridge

; Sets up the cartridge hardware.
; Can use either CRC32 and the database resource, or the iNES header.

____________________________________________________________________________________________
; Synchronize the cartridge hardware (IRQ counters)
____________________________________________________________________________________________

[setIRQCounter | #=2
 call InstallIRQCounter #1 #2]
InstallIRQCounter:

    arguments @Type, @Callback
    copy D@Type D$IRQType
    copy D@Callback D$IRQHook

    if. D$IRQType = IRQHookPPU

        ; Monitor read line
        PPURead 00000, 01FFF, ReadA13Low
        PPURead 02000, 02FFF, ReadA13High

        ; Make sure the PPU is synchronized at all times
        mov D$IRQType PPUSynchronize

    else

        if D$IRQHook = &NULL, mov D$IRQHook TriggerIRQ

    endif
    return
____________________________________________________________________________________________

IRQHookPPU:ret

ReadA13Low:

    ; Memorize A13, read
    mov D$A13 0
    jmp ReadCROM

ReadA13High:

    ; A13?
    on D$A13 = 1, ReadCiRAM
    ; Divide-by-42
    dec B$Divide42 | jnz S0>
    mov B$Divide42 42

    ; IRQ counter
mov D$ResetIRQ 0
    on B$IRQEnabled = &FALSE, S0>
    dec D$IRQCounter | jns S0>

    ; IRQ!
    if. D$IRQHook = &NULL
mov B$IRQEnabled 0
        SetIRQ PPU, IRQ_CARTRIDGE
        copy D$IRQLatch D$IRQCounter
    else
        call D$IRQHook
    endif

    ; Memorize A13, read
S0: mov D$A13 1
    jmp ReadCiRAM
____________________________________________________________________________________________
[Mul113: 113+113+113+2
 Div113: 3]
IRQCountdown:

    ; Count down
    on B$IRQEnabled = &FALSE, Q0>
    sub D$IRQCounter eax | jns Q0>

    ; Adjust clock, call IRQ hook
    mov eax D$IRQCounter
    imul D$CPUCycles
    push D$CartridgeClock
        add D$CartridgeClock eax
        call D$IRQHook
    pop D$CartridgeClock
Q0: ret
____________________________________________________________________________________________

IRQCount:

    if B$IRQEnabled = &TRUE, jmp D$IRQHook
    ret
____________________________________________________________________________________________

TriggerIRQ:

    mov B$IRQEnabled &FALSE
    copy D$IRQLatch D$IRQCounter
    SetIRQ Cartridge, IRQ_CARTRIDGE
    ret
____________________________________________________________________________________________

CartridgeSynchronize:

    pushad

        ; Behind?
        mov eax D$CPUClock | on eax <= D$CartridgeClock, Q0>>

        sub eax D$CartridgeClock
        mov edx 0 | div D$CPUCycles
        test eax eax | jz Q0>>
        copy D$CPUClock D$CartridgeClock

        ; DMC DMA?
        if. D$DmcIRQCounter > 0
            push eax
                mov eax D$DmcIRQCounter, edx 0
                div D$DMC+ConvertedTimerLatch
                if edx = 0, cpuTick 2
            pop eax
        endif

        ; DNC IRQ?
        if. D$DmcIRQCounter > 0
            sub D$DmcIRQCounter eax | js F0> | jz F0> | break | F0:
            ; Stop
            mov D$DmcIRQCounter 0
            ; IRQ?
            if D$DMC+DMCEnableIRQ = &TRUE, SetIRQ Cartridge, IRQ_DMC
        endif

        ; Frame IRQ?
        if. D$FrameIRQCounter > 0
            sub D$FrameIRQCounter eax | js F0> | jz F0> | break | F0:
            mov D$FrameIRQCounter 29830
            SetIRQ Cartridge, IRQ_FRAME
        endif

        ; IRQ counter
        if D$IRQType != &NULL, call D$IRQType

Q0: popad
    ret

____________________________________________________________________________________________
; Memory mapping
____________________________________________________________________________________________

[CPURead  | call MapAddress CPUReadTable,  #1, #2, #3]
[CPUWrite | call MapAddress CPUWriteTable, #1, #2, #3]
[PPURead  | call MapAddress PPUReadTable,  #1, #2, #3]
[PPUWrite | call MapAddress PPUWriteTable, #1, #2, #3]

MapAddress:

    arguments @pTable, @Start, @End, @Label

    pushad

        mov edi D$@Start | shl edi 2 | add edi D$@pTable
        mov ecx D$@End | sub ecx D$@Start | inc ecx
        mov eax D$@Label
        rep stosd

    popad
    return

____________________________________________________________________________________________
; Reset
____________________________________________________________________________________________

[HSyncHook: ?]
[IRQHook: ?]
[IRQType: ?]

CartridgeReset:

    call ZeroMemory MapperData LENGTH_MAPPER_DATA-LENGTH_HARDRESET_DATA

    ; Got WRAM?
    if D$Cartridge@SizeWRAM = 0, mov D$Cartridge@SizeWRAM 8

    ; Nullify any hooks
    mov D$IRQType &NULL
    mov D$IRQHook &NULL
    mov D$HSyncHook &NULL

    ; Set PPU mirroring
    if D$Cartridge@Mirroring = HORIZONTAL_MIRRORING,  mirror HORIZONTAL
    if D$Cartridge@Mirroring = VERTICAL_MIRRORING,    mirror VERTICAL
    if D$Cartridge@Mirroring = FOUR_SCREEN_MIRRORING, mirror FOUR_SCREEN

    ; Default - swap in first PROM banks
    swap PROM, 16k, 08000, 0
    swap PROM, 16k, 0C000, 1

    ; Default - swap in first CROM bank
    swap CROM, 8k,  00000, 0

    ; Load trainer
    if D$Cartridge@Trainer = &TRUE, call CopyMemory D$pTrainer, WRAM+01000, 0200

    ; VsSystem?
    if B$Cartridge@VsSystem = &TRUE, call ResetVsSystem

    ; Reset mapper
    mov eax D$Cartridge@Mapper
    call D$eax*4+MapperTable
    ret

____________________________________________________________________________________________
; Load a NES ROM
____________________________________________________________________________________________

LoadNES:

    ; Unload previous cartridge
    call UnloadNES
    mov D$GameGenieCodes &INVALID_HANDLE_VALUE
    copy D$pFile D$pImage | mov D$pFile &NULL

    ; File ID?
    mov eax D$pImage
    if. D$eax != 01A53454E
        closeMessage 'Bad file format.'
        ret
    endif

    ; Insert cartridge
    mov B$ROMTitle 0 | call AddString ROMTitle FileTitle
    mov B$ROMDirectory 0
    call AddString ROMDirectory, FilesDirectory
    call AddString ROMDirectory, ROMTitle
    call AddString ROMDirectory, BackSlash
    call CartridgeGetInfo

    ; - Validate -
    ; Mo PROM?
    if. D$Cartridge@SizePROM < 16
        closeMessage 'No program data.'
        ret
    endif
    ; Mapper supported?
    mov eax D$Cartridge@Mapper
    if. D$MapperTable+eax*4 = &NULL
        closeMessage 'Mapper not supported yet.'
        ret
    endif
    mov B$WindowTitle 0
    call AddString WindowTitle D$Cartridge@pGamename
    ; Bad dump?
    if. B$Cartridge@BadDump = &TRUE
        call AddString WindowTitle BadDump
    endif
    ; Wrong size?
    mov eax D$FileSize | sub eax 010 | shr eax 10
    mov edx D$Cartridge@SizePROM | add edx D$Cartridge@SizeCROM
    if. eax != edx
        call AddString WindowTitle WrongSize
    endif
    ; Set window text
    call 'User32.SendMessageA' D$hMainWindow, &WM_SETTEXT, &NULL, WindowTitle

    ; - Preferences -
    ; Set mode (NTSC/PAL)
    if. B$Preferences@PreferNTSC = &TRUE
        call MenuNTSC
    else
        call MenuPAL
    endif
    if B$Cartridge@PAL = &TRUE,
        if B$Cartridge@NTSC = &FALSE,
            call MenuPAL
    if B$Cartridge@NTSC = &TRUE,
        if B$Cartridge@PAL = &FALSE,
            call MenuNTSC
    ; Power ON
    if B$Preferences@AutoPowerOn = &TRUE, call SetPower POWER_ON
    ; Remove menu?
    if. B$Preferences@AutoHideMenu = &TRUE
        call 'User32.SetMenu' D$hMainWindow, &NULL | winErrorCheck 'Could not set menu.'
    endif


    ; Palette
    [PALExtension: B$ '*.pal', 0]
    push D$pFile
        copy D$pPalette D$pFile | mov D$FileSize 0C0 | call LoadPAL
    pop D$pFile
    call LoadFirstFile NessieDirectory, PALExtension
    call LoadFirstFile ROMDirectory, PALExtension

    ; IPS patch
    [IPSExtension: B$ '*.ips', 0]
    call LoadFirstFile ROMDirectory, IPSExtension

    ; Configuration
    call LoadConfiguration

    ; Zapper?
    mov eax D$Cartridge@PROMCRC32 | mov D$EndZapperTable eax
    mov edi ZapperTable, ecx 0-1 | repne scasd
    [ZapperTable:
    ZapperPort1Port2:
    0BC1D_CE96 ; Chiller (Unl) (U)
    0639C_B8B8 ; Chiller (HES)
    0BE1B_F10C ; Chiller [o1]
    ZapperPort2:
    0FBFC_6A6C ; Adventures of Bayou Billy, The (E)
    0CB27_5051 ; Adventures of Bayou Billy, The (U)
    0FB69_C131 ; Baby Boomer (Unl) (U)
    0F264_1AD0 ; Barker Bill's Trick Shooting (U)
    090CA_616D ; Duck Hunt (JUE)
    0EEB7_A62B ; Duck Hunt (JUE) [p1]
    07521_7873 ; Duck Hunt (JUE) [p2]
    041CF_3616 ; Duck Hunt (JUE) [h1]
    0F428_6D0F ; Duck Hunt (JUE) [h2]
    0D947_6F2E ; Duck Hunt (JUE) [h3]
    0AD6C_DF29 ; Duck Hunt (JUE) [b1]
    059E3_343F ; Freedom Force (U)
    0242A_270C ; Gotcha! (U)
    0D83C_671A ; Gotcha! (U) [p1]
    07B5B_D2DE ; Gumshoe (UE)
    08D21_13BC ; Gumshoe (UE) (bad)
    0255B_129C ; Gun Sight (J)
    08963_AE6E ; Hogan's Alley (JU)
    0E5ED_CDE3 ; Hogan's Alley (JU) [p1]
    0DCCD_F578 ; Hogan's Alley (JU) [p2]
    0818F_DB2B ; Hogan's Alley (JU) [o1]
    0A9A1_07B5 ; Hogan's Alley (JU) [p2][o1]
    02716_BD4B ; Hogan's Alley (JU) [h1]
    0B092_DD8E ; Hogan's Alley (JU) [b1]
    051D2_112F ; Laser Invasion (U)
     0A86_6C94 ; Lone Ranger, The (U)
    0E4C0_4EEA ; Mad City (J)
    09EEF_47AA ; Mechanized Attack (U)
    084AB_73C1 ; Mechanized Attack (U) (bad)
               ; Mechanized attach [b2] and [b5] won't run anyway
    0C2DB_7551 ; Shooting Range (U) (bad)
    0885A_8A6D ; Shooting Range (U)
    08106_9812 ; Super Mario Bros / Duck Hunt (U)
    05D37_3F6A ; Super Mario Bros / Duck Hunt (U) [b1]
    0E8F8_F7A5 ; Super Mario Bros / Duck Hunt (E)
    0D4F0_18F5 ; Super Mario Bros / Duck Hunt / Track Meet
    0163E_86C0 ; To The Earth (U)
    03899_60DB ; Wild Gunman (U)
     0D3C_F705 ; Wild Gunman (U) [!]
    04A60_A644 ; Wild Gunman (U) [p1]
    0ED58_8f00 ; VS Duck Hunt
    017AE_56BE ; VS Freedom Force
    0FF51_35A3 ; VS Hogan's Alley
    ZapperNoPort:
    EndZapperTable: 0]
    sub edi 4
    if. edi >= ZapperNoPort
        call SetZapper Port1, &FALSE
        call SetZapper Port2, &FALSE
    else
    if. edi >= ZapperPort2
        call SetZapper Port1, &FALSE
        call SetZapper Port2, &TRUE
    else
    if. edi >= ZapperPort1Port2
        call SetZapper Port1, &TRUE
        call SetZapper Port2, &TRUE
    endif

    ; Enable NES menu and Close
    call 'User32.EnableMenuItem' D$hMenu, 1, &MF_BYPOSITION+&MF_ENABLED
    call 'User32.EnableMenuItem' D$hMenu, IDM_CLOSE, &MF_BYCOMMAND+&MF_ENABLED
    call 'User32.DrawMenuBar' D$hMainWindow
    ret

UnloadNES:


    if D$pImage = &NULL, ret
    call 'User32.SendMessageA' D$hMainWindow, &WM_SETTEXT, &NULL, Classname
    call SetPower POWER_OFF
    call 'Kernel32.GlobalFree' D$pImage | mov D$pImage &NULL
    call ZeroMemory Cartridge (SIZE_CARTRIDGE shl 2)

    ; Disable NES menu and Close
    call 'User32.EnableMenuItem' D$hMenu, 1, &MF_BYPOSITION+&MF_GRAYED
    call 'User32.EnableMenuItem' D$hMenu, IDM_CLOSE, &MF_BYCOMMAND+&MF_GRAYED
    call 'User32.DrawMenuBar' D$hMainWindow
    ret

____________________________________________________________________________________________
; Cartridge info
____________________________________________________________________________________________

CartridgeGetInfo:

    ; Read that crappy iNES header
    call CopyMemory D$pImage, INESHeader, LENGTH_INES_HEADER

    ; File CRC (no iNES interference)
    mov ebx D$pImage, ecx D$FileSize
    add ebx LENGTH_INES_HEADER                              ; Header
    and ecx 0FFFF_E200 | ifFlag ecx 0200, add ebx 0200      ; Trainer
    and ecx 0FFFF_E000                                      ; 8k aligned
  . D$Cartridge@CRC32 = CRC32 ebx, ecx

    ; PROM CRC (based on iNES data)
    mov eax D$pImage | add eax LENGTH_INES_HEADER           ; Skip header
    ifFlag B$INESHeader@ROMControl1 04, add eax 0200        ; Skip trainer
    movzx ecx B$INESHeader@nPROMBanks | shl ecx 14          ; Get PROM size from header
    if ecx < D$Filesize,                                    ; Do not allow size overflow
        . D$Cartridge@PROMCRC32 = CRC32 eax, ecx

    ; Query database
    call DatabaseFind D$Cartridge@CRC32 D$pCRC32
    if esi = &NULL, call DatabaseFind D$Cartridge@PROMCRC32 D$pPROMCRC32
    ; Get cartridge info (esi = data, edi = name)
    if. esi = &NULL
        call UseINES
    else
        call UseDatabase
    endif

    ; If battery but no WRAM, default to 8k WRAM
    if D$Cartridge@SizeWRAM = 0,
        if D$Cartridge@Battery = &TRUE,
            mov D$Cartridge@SizeWRAM 8

    ; pTrainer
    mov eax D$pImage
    add eax LENGTH_INES_HEADER
    if. B$Cartridge@Trainer = &TRUE
        mov D$pTrainer eax
        add eax 0200
    else
        mov D$pTrainer &NULL
    endif

    ; pPROM
    mov D$pPROM eax

    ; pCROM
    if. D$Cartridge@SizeCROM > 0
        mov edx D$Cartridge@SizePROM | shl edx 10 | add eax edx
        mov D$pCROM eax
    else ; Default to 8k of CRAM
        mov D$Cartridge@SizeCRAM 8
        mov D$pCROM &NULL
    endif
    ret

UseDatabase:  ; edi = name, esi = data

    ; Save game name
    mov D$Cartridge@pGamename edi

    ; Get cartridge info from database
    lodsb | movzx eax al | shl eax 4                 ; 16k PROM banks
            mov D$Cartridge@SizePROM eax
    lodsb | movzx eax al | shl eax 3                 ; 8k CROM banks
            mov D$Cartridge@SizeCROM eax
    lodsb | movzx eax al | shl eax 3                 ; 8k WRAM banks
            mov D$Cartridge@SizeWRAM eax
    lodsb | mov B$Cartridge@Mapper al                ; Mapper
    lodsw                                            ; Various stuff
    shr ax 1 | setc B$Cartridge@PAL
    shr ax 1 | setc B$Cartridge@NTSC
    shr ax 1 | setc B$Cartridge@VsSystem
    shr ax 1 | setc B$Cartridge@Playchoice
                mov B$Cartridge@Mirroring al
                and B$Cartridge@Mirroring 3
    shr ax 3 | setc B$Cartridge@Battery
    shr ax 1 | setc B$Cartridge@Trainer
    shr ax 1 | setc B$Cartridge@BadDump
    shr ax 1 | setc B$Cartridge@Hacked
    shr ax 1 | setc B$Cartridge@Translated
    shr ax 1 | setc B$Cartridge@Unlicensed
    shr ax 1 | setc B$Cartridge@Bootleg
    ret

UseINES:

    ; Get name (file title)
    mov D$Cartridge@pGamename FileTitle

    ; PROM banks
    movzx eax B$INESHeader@nPROMBanks
    shl eax 4 | mov D$Cartridge@SizePROM eax

    ; CROM banks
    movzx eax B$INESHeader@nCROMBanks
    shl eax 3 | mov D$Cartridge@SizeCROM eax

    ; Mapper
    mov ax W$INESHeader@ROMControl1
    shr ah, 4 | shr ax, 4 | and eax 0FF
    mov D$Cartridge@Mapper eax

    ; Mirroring
    mov B$Cartridge@Mirroring HORIZONTAL_MIRRORING
    ifFlag B$INESHeader@ROMControl1 01, mov B$Cartridge@Mirroring VERTICAL_MIRRORING
    ifFlag B$INESHeader@ROMControl1 08, mov B$Cartridge@Mirroring FOUR_SCREEN_MIRRORING
    test B$INESHeader@ROMControl1 04 | setnz B$Cartridge@Trainer
    test B$INESHeader@ROMControl1 02 | setnz B$Cartridge@Battery


    ; Default - activate 8k WRAM at $6000
    mov D$Cartridge@SizeWRAM 8
    ret
____________________________________________________________________________________________
; Database
____________________________________________________________________________________________

DatabaseFind:

    arguments @CRC, @pSearch

    ; Database in use?
    mov esi &NULL
    on B$Preferences@UseDatabase = &FALSE, Q0>>
    on D@CRC = &NULL, Q0>>

    ; Scan through
    mov edi D@pSearch, eax D@CRC, ecx D$nROMs
    repne scasd | jecxz Q0>
    sub ecx D$nROMs | not ecx

    ; Data pointer
    move eax ecx*8
    add eax D$pData | mov esi eax

    ; Name pointer
    movzx edx W$esi+6
    mov edi D$pNames, al 0, ecx 0-1
L0: dec edx | js Q0>
    repne scasb | jmp L0<

Q0: return
____________________________________________________________________________________________
; ROM name
____________________________________________________________________________________________

[ROMTitle:  B$ ? #&MAX_PATH]
[BadDump:   B$ ' (Bad dump)',   0
 WrongSize: B$ ' (Wrong size)', 0]
____________________________________________________________________________________________
; Cartridge data
____________________________________________________________________________________________

; iNES header
[LENGTH_INES_HEADER 010]
[INESHeader:
 @ID:           D$ ?
 @nPROMBanks:   B$ ?
 @nCROMBanks:   B$ ?
 @ROMControl1:  B$ ?
 @ROMControl2:  B$ ?
 @Unused:       B$ ? #8]

; Pointers to image
[pImage:        ?
 pTrainer:      ?
 pPROM:         ?
 pCROM:         ?]

; Cartridge data
[SIZE_CARTRIDGE 21]
[Cartridge:
 @CRC32:        ?
 @PROMCRC32:    ?

 @SizePROM:     ?
 @SizeCROM:     ?
 @SizeCRAM:     ?
 @SizeWRAM:     ?

 @Mapper:       ?

 @PAL:          ?
 @NTSC:         ?
 @VsSystem:     ?
 @Playchoice:   ?
 @Mirroring:    ?
 @Battery:      ?
 @Trainer:      ?
 @BadDump:      ?
 @Hacked:       ?
 @Translated:   ?
 @Unlicensed:   ?
 @Bootleg:      ?

 @pGamename:    ?]

[HORIZONTAL_MIRRORING  00
 VERTICAL_MIRRORING    01
 FOUR_SCREEN_MIRRORING 02]
____________________________________________________________________________________________
TITLE Mappers

; 0,1,2,3,4,5,6,7,9                             (8 - ???)
; 10,11,13,15,16,18                             (17 - mirroring)
; 21,22,23,24,25,26
; 32,33,34
; 40,41,44,45,46,49
; 50,51,52,57
; 60,61,62,65,66,67,68,69
; 70,71,72,73,75,76,77,78,79                    (74 - No other emu handles it either)
; 80,82,86,87,88,89                             (85 - vrc7 sound)
; 91,92,93,94,95,97                             (96 - CRAM switch, 99 - VS)
; 101
; 112,113,114,117,118,119
;
; 133
; 144
; 152,155                                       (151 - VS)
; 160
;
; 180,182,184,185,188,189
;                                               (198 - no ROMs)
;
;
; 225,226,227,228,229                           (222 - IRQ counter)
; 230,231,232,233,234,235
; 240,241,242,243,244,246,248                   (245 - supposed to be different)
; 250                                           (255)
[MapperTable:
 Mapper000 Mapper001 Mapper002 Mapper003 Mapper004 Mapper005 Mapper006 Mapper007 Mapper008 Mapper009
 Mapper010 Mapper011 &NULL     Mapper013 &NULL     Mapper015 Mapper016 Mapper017 Mapper018 Mapper019
 &NULL     Mapper021 Mapper022 Mapper023 Mapper024 Mapper025 Mapper026 &NULL     &NULL     &NULL
 &NULL     &NULL     Mapper032 Mapper033 Mapper034 &NULL     &NULL     &NULL     &NULL     &NULL
 Mapper040 Mapper041 Mapper042 &NULL     Mapper044 Mapper045 Mapper046 Mapper047 &NULL     Mapper049
 Mapper050 Mapper051 Mapper052 &NULL     &NULL     &NULL     &NULL     Mapper057 Mapper058 &NULL
 Mapper060 Mapper061 Mapper062 &NULL     &NULL     Mapper065 Mapper066 Mapper067 Mapper068 Mapper069
 Mapper070 Mapper071 Mapper072 Mapper073 Mapper074 Mapper075 Mapper076 Mapper077 Mapper078 Mapper079
 Mapper080 &NULL     Mapper082 Mapper083 &NULL     Mapper085 Mapper086 Mapper087 Mapper088 Mapper089
 &NULL     Mapper091 Mapper092 Mapper093 Mapper094 Mapper095 Mapper096 Mapper097 &NULL     Mapper099
 &NULL     Mapper101 &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL
 &NULL     &NULL     Mapper112 Mapper113 Mapper114 mapper115 &NULL     Mapper117 Mapper118 Mapper119
 &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL
 &NULL     &NULL     &NULL     Mapper133 &NULL     &NULL     &NULL     &NULL     &NULL     &NULL
 Mapper140 &NULL     &NULL     &NULL     Mapper144 &NULL     &NULL     &NULL     &NULL     &NULL
 &NULL     Mapper151 Mapper152 &NULL     &NULL     Mapper155 &NULL     &NULL     &NULL     &NULL
 Mapper160 &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL
 &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL
 Mapper180 &NULL     Mapper182 &NULL     Mapper184 Mapper185 &NULL     Mapper187 Mapper188 Mapper189
 &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     Mapper198 &NULL
 &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL
 &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL     &NULL
 &NULL     &NULL     Mapper222 &NULL     &NULL     Mapper225 Mapper226 Mapper227 Mapper228 Mapper229
 Mapper230 Mapper231 Mapper232 Mapper233 Mapper234 Mapper235 &NULL     &NULL     &NULL     &NULL
 Mapper240 Mapper241 Mapper242 Mapper243 Mapper244 Mapper245 Mapper246 &NULL     Mapper248 Mapper249
 Mapper250 &NULL     &NULL     &NULL     &NULL     Mapper255]

____________________________________________________________________________________________
; Mapper registers
____________________________________________________________________________________________
[LENGTH_MAPPER_DATA <(SIZE_MAPPER_DATA shl 2)>]
[LENGTH_HARDRESET_DATA 4]
[SIZE_MAPPER_DATA SIZE_INDEX+SIZE_REGISTERS+SIZE_MMC3+SIZE_IRQCOUNTER+1+SIZE_MMC5+1]
[SIZE_INDEX      020]
[SIZE_REGISTERS  020+020+5]
[SIZE_MMC3       6]
[SIZE_MMC5       23]
[SIZE_IRQCOUNTER 3]

[fnPROMUpdate:  ?]
[fnCROMUpdate:  ?]
[MapperData:

 ; Memory mapping
 IndexPROM:      ? #4
 IndexXRAM:      ? #4
 IndexCROM:      ? #8
 IndexCiROM:     ? #4
 IndexCRAM:      ? #8
 IndexCiRAM:     ? #4

 ; Mapper registers
 Register:      ? #020
 Bank:          ? #020
 Mode:          ?
 Offset:        ?
 Segment:       ?
 Latch:         ? ?

 ; MMC3 data
 WRAMEnabled:   ?
 Command:       ?
 A13:           ?
 Divide42:      ?
 Latched:       ?
 ResetIRQ:      ?

 ; IRQ counters
 IRQCounter:    ?
 IRQLatch:      ?
 IRQEnabled:    ?

 CartridgeClock:?

 ; MMC5
 IRQStop:       ?
 IRQStatus:     ?
 IRQClear:      ?

 PMode:         ?
 CMode:         ?
 GMode:         ?
 XRAMWrite:     ?
 XRAMEnabled:   ?

 FillCharacter: ?
 FillAttribute: ?

 SplitEnabled:  ?
 SplitFromRight:?
 SplitTile:     ?
 SplitScroll:   ?
 SplitOffset:   ?

 Factor1:       ?
 Factor2:       ?

 IndexBgROM:    ? #8

 HardResetData:
 Game:          ?]

[CROMBank Bank+000
 PROMBank Bank+040]
____________________________________________________________________________________________
; Bank switching
____________________________________________________________________________________________

[LAST_BANK 0FFFF]

; swap CROM, 4k, 01000, eax
[swap | #=4 | call Swap#1 LEN_#2, #3, #4]
[LEN_1K  0400
 LEN_2K  0800
 LEN_4K  01000
 LEN_8K  02000
 LEN_16K 04000
 LEN_32K 08000]

SwapCRAM:

    arguments @Length, @Address, @Bank
    call PPUSynchronize
    on D$Cartridge@SizeCRAM = 0, Q0>

    pushad

        ; Adjust bank number
        mov eax D@Bank
        mul D@Length
        mov ebx D$Cartridge@SizeCRAM | shl ebx 10 | mov edx 0 | div ebx | mov eax edx

        ; Update index
        mov ecx D@Length  | shr ecx 10
        mov ebx D@Address | shr ebx 10
    L0: mov D$IndexCRAM+ebx*4 eax | add eax 0400 | inc ebx | dec ecx | jnz L0<

    popad
Q0: return

SwapCiROM:

    arguments @Length, @Address, @Bank
    if D$Cartridge@SizeCROM = 0, return
    call PPUSynchronize

    pushad

        ; Adjust bank number
        mov eax D@Bank
        mul D@Length
        mov ebx D$Cartridge@SizeCROM | shl ebx 10 | mov edx 0 | div ebx | mov eax edx

        ; Update index
        mov ecx D@Length  | shr ecx 10
        mov ebx D@Address | shr ebx 10 | and ebx 03
    L0: mov D$IndexCiROM+ebx*4 eax | add eax 0400 | inc ebx | dec ecx | jnz L0<

    popad
    return

SwapCROM:

    on D$Cartridge@SizeCROM = 0, SwapCRAM

    arguments @Length, @Address, @Bank
    call PPUSynchronize

    pushad

        ; Adjust bank number
        mov eax D@Bank
        mul D@Length
        mov ebx D$Cartridge@SizeCROM | shl ebx 10 | mov edx 0 | div ebx | mov eax edx

        ; Update index
        mov ecx D@Length  | shr ecx 10
        mov ebx D@Address | shr ebx 10
    L0: mov D$IndexCROM+ebx*4 eax | add eax 0400 | inc ebx | dec ecx | jnz L0<

    popad
    return

SwapPROM:

    arguments @Length, @Address, @Bank
    on D$Cartridge@SizePROM = 0, Q0>
    pushad

        ; Adjust bank number
        mov eax D@Bank
        mul D@Length
        mov ebx D$Cartridge@SizePROM | shl ebx 10 | mov edx 0 | div ebx | mov eax edx

        ; Update index
        mov ecx D@Length  | shr ecx 13
        mov ebx D@Address | and ebx 07FFF | shr ebx 13
    L0: mov D$IndexPROM+ebx*4 eax | add eax 02000 | inc ebx | dec ecx | jnz L0<
    popad
Q0: return

____________________________________________________________________________________________
; Memory mapping
____________________________________________________________________________________________

ResetMemoryMapping:

    ; - Reads -

    CPURead  00000, 0FFFF, ReadVoid

    ; RAM
    CPURead  00000,  07FF, ReadRAM
    CPURead   0800, 01FFF, ReadMirrorRAM

    ; PPU registers
    CPURead  02000, 02007, Read200x
    CPURead  02002, 02002, Read2002
    CPURead  02004, 02004, Read2004
    CPURead  02007, 02007, Read2007
    CPURead  02008, 03FFF, ReadMirrorPPU

    ; APU registers
    CPURead  04015, 04015, Read4015
    CPURead  04016, 04016, Read4016
    CPURead  04017, 04017, Read4017

    ; Misc
    CPURead  06000, 07FFF, ReadWRAM
    CPURead  08000, 0FFFF, ReadPROM


    ; - Writes -

    CPUWrite 00000, 0FFFF, WriteVoid

    ; RAM
    CPUWrite 00000,  07FF, WriteRAM
    CPUWrite  0800, 01FFF, WriteMirrorRAM

    ; PPU registers
    CPUWrite 02000, 02000, Write2000
    CPUWrite 02001, 02001, Write2001
    CPUWrite 02002, 02002, Write2002
    CPUWrite 02003, 02003, Write2003
    CPUWrite 02004, 02004, Write2004
    CPUWrite 02005, 02005, Write2005
    CPUWrite 02006, 02006, Write2006
    CPUWrite 02007, 02007, Write2007
    CPUWrite 02008, 03FFF, WriteMirrorPPU

    ; APU registers
    CPUWrite 04000, 04000, Write4000
    CPUWrite 04001, 04001, Write4001
    CPUWrite 04002, 04002, Write4002
    CPUWrite 04003, 04003, Write4003
    CPUWrite 04004, 04004, Write4004
    CPUWrite 04005, 04005, Write4005
    CPUWrite 04006, 04006, Write4006
    CPUWrite 04007, 04007, Write4007
    CPUWrite 04008, 04008, Write4008
CPUWrite 04009, 04009, DoNothing
    CPUWrite 0400A, 0400A, Write400A
    CPUWrite 0400B, 0400B, Write400B
    CPUWrite 0400C, 0400C, Write400C
CPUWrite 0400D, 0400D, DoNothing
    CPUWrite 0400E, 0400E, Write400E
    CPUWrite 0400F, 0400F, Write400F
    CPUWrite 04010, 04010, Write4010
    CPUWrite 04011, 04011, Write4011
    CPUWrite 04012, 04012, Write4012
    CPUWrite 04013, 04013, Write4013

    ; Sprite DMA
    CPUWrite 04014, 04014, Write4014

    ; Channel enable
    CPUWrite 04015, 04015, Write4015

    ; I/O
    CPUWrite 04016, 04016, Write4016

    ; Frame counter
    CPUWrite 04017, 04017, Write4017

    ; Misc
    CPUWrite 06000, 07FFF, WriteWRAM
    ret
____________________________________________________________________________________________
TITLE Simple
____________________________________________________________________________________________
; Mapper #000, NROM
____________________________________________________________________________________________

Mapper000: | ret

; ____________________________________________________________________________________________
;
;                            PROM switching mappers
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #188, Bandai Karaoke Studio

; $8000-$FFFF: xxxSxPPP - swap 16k PROM at $8000
;                       - Use first/second 128k PROM segment (1/0)
____________________________________________________________________________________________

Mapper188:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    CPURead  06000, 07FFF, @ReadWROM

    ; Load last bank of first segment
    swap PROM, 16k, 0C000, 7
    ret

@8000_FFFF:

    ; xxxxxxxx = 0?
    if. eax = 0
        swap PROM, 16k, 08000, 7
        if D$Cartridge@SizePROM > 128, swap PROM, 16k, 08000, 8
        ret
    endif

    ; xxxSxPPP
    ifNotFlag. eax 010
        and eax 07
        or eax  08
    else
        and eax 07
    endif

    ; Swap bank
    swap PROM, 16k, 08000, eax
    ret

@ReadWROM: mov eax 3 | ret

____________________________________________________________________________________________
; Mapper #241, Fon Serm Bon

; $8000: - PPPPPPPP - swap 32k PROM at $8000
____________________________________________________________________________________________

Mapper241:

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 05FF0, 05FF0, @5FF0
    ret

; ????????
@5FF0:ret

@8000:

    ; PPPPPPPP
    swap PROM, 32k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #002, UNROM

; $8000-$FFFF: xxxxPPPP - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper002:

    ; Set ports
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Swap last 16k PROM bank to $C000
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_FFFF:

    ; xxxxPPPP
    swap PROM, 16k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #093, Sunsoft (Fantasy Zone)

; $6000: PPPPPPPP - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper093:

    ; Memory mapping
    CPUWrite 06000, 06000, @6000

    ; Swap last PROM bank to $C000
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@6000:

    ; PPPPPPPP
    swap PROM, 16k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #094, Capcom 74HC161/32

; $8000-$FFFF: xxxPPPxx - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper094:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Swap in last PROM bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_FFFF:

    ; xxxPPPxx
    shr eax 2
    swap PROM, 16k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #180, Nichibutsu

; $8000-$FFFF: xxxxxPPP - swap 16k PROM at $C000
____________________________________________________________________________________________

Mapper180:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxxxxPPP
    and eax 7
    swap PROM, 16k, 0C000, eax
    ret

____________________________________________________________________________________________
; Mapper #232, Camerica/Codemasters Quattro Games

; $6000-$9FFF: xxxPPxxx - select 64k PROM segment
; $A000-$FFFF: xxxxxxPP - select 16k PROM (in selected segment) at $8000

; - swap selected bank of 64k segment at $8000
; - swap last     bank of 64k segment at $C000
____________________________________________________________________________________________

Mapper232:

    ; Memory mapping
    CPUWrite 08000, 0BFFF, @8000_BFFF
    CPUWrite 0C000, 0FFFF, @C000_FFFF

    ; Select last 64k segment
    mov D$Bank 0C
    call @SetBanks
    ret

; Two MSB of PROM select
@8000_BFFF:

    ; Hack: Swap bits on (Aladdin) games
    on D$Cartridge@PROMCRC32 = 0A045_FE1D, F0> ; Super Sports Challenge (Aladdin) (E)
    on D$Cartridge@PROMCRC32 = 06C04_0686, F0> ; Quattro Adventure (Aladdin) (U)
    on D$Cartridge@PROMCRC32 = 062EF_6C79, F0> ; Quattro Sports (Aladdin) (U)
    jmp S0>
F0: if. eax = 08  | mov eax 010 | else
    if. eax = 010 | mov eax 08  | endif

    ; xxxPPxxx
S0: shr eax 1
    and eax 0C
    and D$Bank 03
    or  D$Bank eax
    jmp @SetBanks

; Two LSB of PROM select
@C000_FFFF:
    ; xxxxxxPP
    and eax 03
    and D$Bank 0C
    or  D$Bank eax
    jmp @SetBanks

@SetBanks:

    ; Selected bank --> $8000
    ; Last     bank --> $C000
    mov eax D$Bank
    swap PROM, 16k, 08000, eax | or eax 3
    swap PROM, 16k, 0C000, eax
    ret

; ____________________________________________________________________________________________
;
;                            CROM switching mappers
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #101, Jaleco 74HC161/32

; $8000-$FFFF: xxxxxxCC - swap 8k CROM at $0000
____________________________________________________________________________________________

Mapper101:

    ; Memory mapping
    CPUWrite 06000, 0FFFF, @6000_FFFF
    ret

@6000_FFFF:

    ; xxxxxxCC
    swap CROM, 8k,  00000, eax
    ret

____________________________________________________________________________________________
; Mapper #003, CNROM

; $8000-$FFFF: xxxxCCCC - swap 8k CROM at $0000
____________________________________________________________________________________________

Mapper003:

    ; Set ports
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxxxCCCC
    swap CROM, 8k, 00000, eax
    ret

____________________________________________________________________________________________
; Mapper #087, Konami 74HC161/32

; $6000-$FFFF: CCCCCCCx - swap 8k CROM at $0000
____________________________________________________________________________________________

Mapper087:

    ; Memory mapping
    CPUWrite 06000, 0FFFF, @6000_FFFF
    ret

@6000_FFFF:

    ; CCCCCCCx
    shr eax 1
    swap CROM, 8k, 00000, eax
    ret

____________________________________________________________________________________________
; Mapper #184

; $6000-$FFFF: ccccCCCC - swap 4k CROM at $0000
;                       - swap 4k cROM at $1000
____________________________________________________________________________________________

Mapper184:

    ; Memory mapping
    CPUWrite 06000, 0FFFF, @6000_FFFF
    ret

@6000_FFFF:

    ; ccccCCCC
    swap CROM, 4k, 00000, eax | shr eax 4
    swap CROM, 4k, 01000, eax
    ret

; ____________________________________________________________________________________________
;
;                        PROM/mirror switching mappers
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #235, 150-in-1

; $8000-$FFFF: (address) 1-MpSmPP ---PPPPP - (m = 1) mirror one screen
;                                          - mirror VERTICAL/HORIZONTAL (M = 0/1)
;                                          - (S = 1) swap 16k PROM at $8000 and $C000 (PPPPPPPp)
;                                          - (S = 0) swap 32k PROM at $8000 (PPPPPPP)
____________________________________________________________________________________________

Mapper235:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    CPURead  08000, 0FFFF, @ReadPROM
    ret

@8000_FFFF:

    ; -M---m-- --------
    ifFlag. edx 0400  | mirror ONE_SCREEN_2000 | else
    ifFlag. edx 02000 | mirror HORIZONTAL      | else
                        mirror VERTICAL        | endif

    ; ------PP ---PPPPP
    mov eax edx | and eax 01F
    mov ecx edx | and ecx 0300

    if. D$Cartridge@SizePROM = 2048
        test ecx 0100 | setnz B$Register
        if ecx = 0200, shr ecx 1
    endif
    shr ecx 3 | or eax ecx

    ; ---pS--- --------
    ifFlag. edx 0800
        shl eax 1
        ifFlag edx 01000, or eax 01
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    else
        swap PROM, 32k, 08000, eax
    endif
    ret

@ReadPROM:

    on B$Register = &FALSE, ReadPROM
    jmp ReadVoid
____________________________________________________________________________________________
; Mapper #051, Ball Games 11-in-1

; $6000-$7FFF: xxxBxxAx - mode = BA
;                       - mirror VERTICAL/HORIZONTAL (mode = 0,1,2 / mode = 3)
; $8000-$FFFF: xxxxPpPP - (mode = 1,3) swap 32k PROM at $8000 (PpPP)
;                       - (mode = 1,3) swap 8k  PROM at $6000 (1pPP11)
;                       - (mode = 0,2) swap 16k PROM at $8000 (PpPPB)
;                       - (mode = 0,2) swap 16k PROM at $C000 (Pp111)
;                       - (mode = 0,2) swap 8k  PROM at $6000 (1p1111)
; $C000-$DFFF: xxxBxxxx - mode = BA
;                       - mirror VERTICAL/HORIZONTAL (mode = 0,1,2 / mode = 3)
____________________________________________________________________________________________

Mapper051:

    ; Memory mapping
    CPUWrite 06000, 07FFF, @6000_7FFF
    CPURead  06000, 07FFF, @ReadExPROM
    CPUWrite 08000, 0FFFF, @8000_FFFF
    CPUWrite 0C000, 0DFFF, @C000_DFFF

    ; Force CRAM
    mov D$Cartridge@SizeCROM 0
    mov D$Cartridge@SizeCRAM 8
    swap CRAM 8k, 0000, 0

    ; Registers
    mov B$Mode 01
    call @UpdateBanks
    ret

@ReadExPROM:

    sub eax 06000
    add eax D$Offset
    add eax D$pPROM
    movzx eax B$eax
    ret

@6000_7FFF:

    ; xxxBxxAx
    mov B$Mode 0
    ifFlag eax 02, or B$Mode 01
    ifFlag eax 010, or B$Mode 02
    call @UpdateBanks
    ret

@C000_DFFF:

    ; xxxBxxxx
    and B$Mode 01
    ifFlag eax 010, or D$Mode 02

    ; xxxxPPPP
    and eax 0F
    mov D$Bank eax
    call @UpdateBanks
    ret

@8000_FFFF:

    ; xxxxPPPP
    and eax 0F
    mov D$Bank eax
    call @UpdateBanks
    ret

@UpdateBanks:

    ; Mirroring
    if. D$Mode = 3
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif

    ifFlag. D$Mode 01

        ; Swap PROM
        swap PROM, 32k, 08000, D$Bank

        ; Find bank at $6000-$7FFF
        mov eax D$Bank
        shl eax 2 | or eax 023

        ; mul 8k
        shl eax 13
        mov D$Offset eax

    else

        ; Swap PROM
        mov eax D$Bank
        shl eax 1
        ifFlag D$Mode 02, or eax 01
        swap PROM, 16k, 08000, eax | or eax 7
        swap PROM, 16k, 0C000, eax

        ; Find bank at $6000-$7FFF
        mov eax D$Bank
        shl eax 2 | or eax 02F

        ; mul 8k
        shl eax 13
        mov D$Offset eax

    endif
    ret

____________________________________________________________________________________________
; Mapper #230, 20-in-1 (Contra)

; $8000-$FFFF: xMSPPPPp - mirror HORIZONTAL/VERTICAL (0/1)
;                       - (S = 0) swap 32k PROM at $8000
;                       - (S = 1) swap 16k PROM at $8000 and $C000
____________________________________________________________________________________________

[Contra Game]

Mapper230:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Toggle ROM switch
    xor B$Contra &TRUE

    ; Load PROM
    if. B$Contra = &TRUE
        swap PROM, 16k, 0C000, 7
    else
        swap PROM, 16k, 08000, 8
        swap PROM, 16k, 0C000, 8+LAST_BANK
    endif
    ret

@8000_FFFF:

    ; Contra?
    if. B$Contra = &TRUE

        ; xxxxxPPp
        and eax 7
        swap PROM, 16k, 08000, eax
        ret

    endif

    ; xMxxxxxx
    ifFlag. eax 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxSPPPPp
    ifNotFlag. eax 020
        and eax 01F
        add eax 8 ; Get past first 128k
        shr eax 1
        swap PROM, 32k, 08000, eax
    else
        and eax 01F
        add eax 8 ; Get past first 128k
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    endif
    ret

____________________________________________________________________________________________
; Mapper #227, 1200-in-1

; $8000-$FFFF: (address): xxxxxxAH_OPPPPpMS - mirror VERTICAL/HORIZONTAL (0/1)
;                                           - (S = 0) swap 16k PROM at $8000 and $C000 (HPPPPp)
;                                           - (S = 1) swap 32k PROM at $8000 (HPPPP)
;                                           - (O = 0) swap 16k PROM at $C000 (HPPAAA)
____________________________________________________________________________________________

Mapper227:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Load first bank
    swap PROM, 16k, 0C000, 0
    ret

@8000_FFFF:

    ; xxxxxxxx_xxxxxxxx
    ifNotFlag. edx 02
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxxxxxH_xPPPPpxx
    mov eax edx
    shr eax 2
    and eax 01F
    ifFlag edx 0100, or eax 020

    ; xxxxxxxx_xxxxxxxS
    ifFlag. edx 01
        shr eax 1
        swap PROM, 32k, 08000, eax
    else
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
        shr eax 1
    endif

    ; xxxxxxAx_Oxxxxxxx
    ifNotFlag. edx 080
        and eax 01C
        shl eax 1
        ifFlag edx 0200, or eax 7
        swap PROM, 16k, 0C000, eax
    endif
    ret

____________________________________________________________________________________________
; Mapper #061, 20-in-1

; $8000-$FFFF: (LSB of address): MxABPPPP - mirror VERTICAL/HORIZONTAL (0/1)
;                                         - (A =  B) swap 32k PROM at $8000 (PPPP)
;                                         - (A != B) swap 16k PROM at $8000 and $C000 (PPPPA)
____________________________________________________________________________________________

Mapper061:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_FFFF:

    ; Mxxxxxxx
    ifNotFlag. edx 080
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxABxxxx
    mov eax edx
    and edx 030

    ; xxABPPPP
    if edx = 000, swap PROM, 32k, 08000, eax
    if edx = 030, swap PROM, 32k, 08000, eax

    shl eax 1
    if. edx = 010
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    endif

    or eax 2
    if. edx = 020
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    endif

    ret

____________________________________________________________________________________________
; Mapper #226, 76-in-1

; $8000-$FFFF: (LSB of address): xxxxxxxR
;     (R = 0) (data): PMSPPPPP (5xx43210) - mirror HORIZONTAL/VERTICAL (0/1)
;     (R = 1) (data): xxxxxxxP (xxxxxxx6) - (S = 0) Use bits 654321  to swap 32k PROM at $8000
;                                         - (S = 1) Use bits 6543210 to swap 16k PROM at $8000 and $C000
____________________________________________________________________________________________

Mapper226:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxxxxxxR
    and edx 1 | mov B$Register+edx al

    ; xMxxxxxx
    ifNotFlag. B$Register+0 040
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif

    ; Assemble bits
    mov eax D$Register | mov edx eax
    and eax 01F
    and edx 0180 | shr edx 2
    or eax edx
    ifFlag. B$Register+0 020
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    else
        shr eax 1
        swap PROM, 32k, 08000, eax
    endif
    ret

____________________________________________________________________________________________
; Mapper #233, Super 42-in-1

; $8000-$FFFF: MMSPPPPP: - (S = 1) swap 16k PROM at $8000 and $C000
;                        - (S = 0) Use 4 MSB of PROM select and swap 32k PROM at $8000
;                        - mirror 0-0-0-1/VERTICAL/HORIZONTAL/$2400 (0/1/2/3)
____________________________________________________________________________________________

Mapper233:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; MMxxxxxx
    mov edx eax | shr edx 6 | and edx 3
    if dl = 0, mirror MIRRORING_0001
    if dl = 1, mirror VERTICAL
    if dl = 2, mirror HORIZONTAL
    if dl = 3, mirror ONE_SCREEN_2400

    ; xxSPPPPP
    ifFlag. eax 020
        and eax 01F
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    else
        and eax 01F
        shr eax 1
        swap PROM, 32k, 08000, eax
    endif
    ret

____________________________________________________________________________________________
; Mapper #231, 20-in-1

; $8000-$FFFF:
; (LSB of address) MxSPPPPx - swap 32k PROM at $8000
;                           - (S = 0) mirror $C000-$FFFF from $8000-$BFFF (16k games)
;                           - mirror VERTICAL/HORIZONTAL (0/1)
____________________________________________________________________________________________

Mapper231:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; Mxxxxxxx
    ifNotFlag. edx 080
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxSPPPPx
    and edx 03E
    swap PROM, 16k, 08000, edx
    ifFlag edx 020, inc edx
    swap PROM, 16k, 0C000, edx
    ret

____________________________________________________________________________________________
; Mapper #7, AOROM

; $8000-$FFFF: xxxMPPPP - swap 32k PROM at $8000
;                       - mirror $2000/$2400 (0/1)
____________________________________________________________________________________________

Mapper007:

    ; Set ports
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxxxPPPP
    swap PROM, 32k, 08000, eax

    ; xxxMxxxx
    ifNotFlag. al 010
        mirror ONE_SCREEN_2000
    else
        mirror ONE_SCREEN_2400
    endif

    ; Mirroring hack
    on D$Cartridge@PROMCRC32 = 04C7C_1AF3, F0> ; Caesar's Palace (U) [o1]
    on D$Cartridge@PROMCRC32 = 03C9F_E649, F0> ; WWF Wrestlemania Challenge (U) [hM07]
    ret

    ; Two-screen mirroring
F0: ifNotFlag. al 010
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

____________________________________________________________________________________________
; Mapper #071, Camerica

; $9000-$9FFF: xxxMxxxx - mirror $2000/$2400 (0/1)
; $C000-$FFFF: PPPPPPPP - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper071:

    ; Set ports
    CPUWrite 09000, 09FFF, @9000_9FFF
    CPUWrite 0C000, 0FFFF, @C000_FFFF

    ; Swap last 16k PROM bank at $C000
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@9000_9FFF:

    ; xxxMxxxx
    ifFlag. al 010
        mirror ONE_SCREEN_2400
    else
        mirror ONE_SCREEN_2000
    endif
    ret

@C000_FFFF:

    ; PPPPPPPP
    swap PROM, 16k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #097, Irem 74161

; $8000-$BFFF: MxxxPPPP - swap 16k PROM at $C000
;                       - mirror HORIZONTAL/VERTICAL (0/1)
;;
;;;;
FCE Ultra 0.96 code:
switch(V>>6)
{
    case 0:break;
    case 1:MIRROR_SET2(0);break;
    case 2:MIRROR_SET2(1);break;
    case 3:break;
}
;;
____________________________________________________________________________________________

Mapper097:

    ; Memory mapping
    CPUWrite 08000, 0BFFF, @8000_BFFF

    ; Last bank --> $8000, First bank --> $C000
    swap PROM, 16k, 08000, LAST_BANK
    swap PROM, 16k, 0C000, 0
    ret

@8000_BFFF:

    ; MxxxPPPP
    swap PROM, 16k, 0C000, eax
    ifFlag. al 080
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

____________________________________________________________________________________________
; Mapper #242, Wai Xing Zhan Shi

; $8000-$FFFF: xPPPPxMM - swap 32k PROM at $8000
;                       - mirror: VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
____________________________________________________________________________________________

Mapper242:

    ; Set ports
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; PROM switch
    shr edx 3
    swap PROM, 32k, 08000, edx

    ; Mirror switch
    and al 3
    if al = 0, mirror VERTICAL
    if al = 1, mirror HORIZONTAL
    if al = 2, mirror ONE_SCREEN_2000
    if al = 3, mirror ONE_SCREEN_2400
    ret

; ____________________________________________________________________________________________
;
;                       PROM/CROM switching mappers
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #160, PC-Aladdin

; $8000: PPPPPPPP - swap 8k PROM at $8000
; $8001: PPPPPPPP - swap 8k PROM at $A000
; $8002: PPPPPPPP - swap 8k PROM at $C000
; $9000: CCCCCCCC - (mode = 1) swap 1k CROM at $0000
; $9001: CCCCCCCC - (mode = 1) swap 1k CROM at $0400
;                 - (mode = 4) swap 4k CROM at $0000 and $1000 (Use CCCCCCCx)
; $9002: CCCCCCCC - (mode = 1) swap 1k CROM at $0800
; $9003: CCCCCCCC - (mode = 1) swap 1k CROM at $0C00
; $9004: CCCCCCCC - (mode = 1) swap 1k CROM at $1000
; $9005: CCCCCCCC - (mode = 1) swap 1k CROM at $1400
; $9006: CCCCCCCC - (mode = 1) swap 1k CROM at $1800
; $9007: CCCCCCCC - (mode = 1) swap 1k CROM at $1C00
; $D000-$FFFF: MMMMMMMM - mode
____________________________________________________________________________________________

;$C00x???
Mapper160:

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 08001, 08001, @8001
    CPUWrite 08002, 08002, @8002
    CPUWrite 09000, 09000, @9000
    CPUWrite 09001, 09001, @9001
    CPUWrite 09002, 09002, @9002
    CPUWrite 09003, 09003, @9003
    CPUWrite 09004, 09004, @9004
    CPUWrite 09005, 09005, @9005
    CPUWrite 09006, 09006, @9006
    CPUWrite 09007, 09007, @9007
    CPUWrite 0D000, 0FFFF, @D000_FFFF

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK

    mov D$Mode 0
    ret

; PPPPPPPP
@8000: swap PROM, 8k, 08000, eax | ret
@8001: swap PROM, 8k, 0A000, eax | ret
@8002: swap PROM, 8k, 0C000, eax | ret

; CCCCCCCC
@9000: if D$Mode = 1, swap CROM, 1k,  0000, eax | ret
@9002: if D$Mode = 1, swap CROM, 1k,  0800, eax | ret
@9003: if D$Mode = 1, swap CROM, 1k,  0C00, eax | ret
@9004: if D$Mode = 1, swap CROM, 1k, 01000, eax | ret
@9005: if D$Mode = 1, swap CROM, 1k, 01400, eax | ret
@9006: if D$Mode = 1, swap CROM, 1k, 01800, eax | ret
@9007: if D$Mode = 1, swap CROM, 1k, 01C00, eax | ret

@9001:

    ; CCCCCCCC
    if D$Mode = 1, swap CROM, 1k,  0400, eax

    ; CCCCCCCx
    if. D$Mode = 4
        shr eax 1
        swap CROM, 4k, 00000, eax
        swap CROM, 4k, 01000, eax
    endif
    ret

@D000_FFFF:

    ; Mode
    mov D$Mode eax
    ret

____________________________________________________________________________________________
; Mapper #046, Rumble Station 15-in-1

; $6000-$7FFF: CCCCPPPP (6543xxxx/xxxx4321)
; $8000-$FFFF: xCCCxxxP (x210xxxx/xxxxxxx0)
;                       - swap 8k  CROM at $0000 (6543210)
;                       - swap 32k PROM at $8000 (43210)
____________________________________________________________________________________________

Mapper046:

    ; Set ports
    CPUWrite 06000, 07FFF, @6000_7FFF
    CPUWrite 08000, 0FFFF, @8000_FFFF

    mirror VERTICAL
    ret

@6000_7FFF:

    ; xxxxPPPP
    shr D$PROMBank 1
    mov D$PROMBank eax
    rcl D$PROMBank 1

    ; CCCCxxxx
    and eax 0F0 | shr eax 1
    and D$CROMBank 07
    or  D$CROMBank eax

    ; Swap banks
    swap PROM, 32k, 08000, D$PROMBank
    swap CROM, 8k,  00000, D$CROMBank
    ret

@8000_FFFF:

    ; xxxxxxxP
    ifFlag. eax 01
        or D$PROMBank 01
    else
        and D$PROMBank 01E
    endif

    ; xCCCxxxx
    shr eax 4 | and eax 7
    and D$CROMBank 078
    or D$CROMBank eax

    ; Swap banks
    swap PROM, 32k, 08000, D$PROMBank
    swap CROM, 8k,  00000, D$CROMBank
    ret

____________________________________________________________________________________________
; Mapper #246, Fong Shen Bang - Zhu Lu Zhi Zhan

; $6000: PPPPPPPP - swap 8k PROM at $8000
; $6001: PPPPPPPP - swap 8k PROM at $A000
; $6002: PPPPPPPP - swap 8k PROM at $C000
; $6003: PPPPPPPP - swap 8k PROM at $E000
; $6004: CCCCCCCC - swap 2k CROM at $0000
; $6005: CCCCCCCC - swap 2k CROM at $0800
; $6006: CCCCCCCC - swap 2k CROM at $1000
; $6007: CCCCCCCC - swap 2k CROM at $1800

; $6800-$6FFF: ???????? - (???)
; $8AD9:       ???????? - writes $01 here on startup (???)
____________________________________________________________________________________________

Mapper246:

    ; Set ports
    mov ecx 06000
L0: push ecx
        mov eax WriteVoid
        and ecx 0F007
        if ecx = 06000, mov eax @6000
        if ecx = 06001, mov eax @6001
        if ecx = 06002, mov eax @6002
        if ecx = 06003, mov eax @6003
        if ecx = 06004, mov eax @6004
        if ecx = 06005, mov eax @6005
        if ecx = 06006, mov eax @6006
        if ecx = 06007, mov eax @6007
    pop ecx

    CPUWrite ecx, ecx, eax
    inc cx
    on ecx < 06800, L0<<
    ret

@6000: swap PROM, 8k, 08000, eax | ret
@6001: swap PROM, 8k, 0A000, eax | ret
@6002: swap PROM, 8k, 0C000, eax | ret
@6003: swap PROM, 8k, 0E000, eax | ret
@6004: swap CROM, 2k, 00000, eax | ret
@6005: swap CROM, 2k,  0800, eax | ret
@6006: swap CROM, 2k, 01000, eax | ret
@6007: swap CROM, 2k, 01800, eax | ret

____________________________________________________________________________________________
; Mapper #086, Jaleco early mapper #2

; $6000: xCPPxxCC (x2xxxx10) - swap 32k PROM at $8000
;                            - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper086:

    ; Memory mapping
    CPUWrite 06000, 06000, @6000
    ret

@6000:

    ; xCxxxxCC
    mov edx eax | and edx 3
    ifFlag eax 040, or edx 04
    swap CROM, 8k, 00000, edx

    ; xxPPxxxx
    shr eax 4
    swap PROM, 32k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #066, GNROM

; $6000-$FFFF: PPPPCCCC - swap 32k PROM at $8000
;                       - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper066:

    ; Set ports
    CPUWrite 06000, 0FFFF, @6000_FFFF
    ret

@6000_FFFF:

    ; PPPPCCCC
    swap CROM, 8k,  00000, eax | shr eax 4
    swap PROM, 32k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #011, Color Dreams

; $8000-$FFFF: CCCCPPPP - swap 32k PROM at $8000
;                       - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper011:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; CCCCPPPP
    swap PROM, 32k, 08000, eax | shr eax 4
    swap CROM, 8k,  00000, eax
    ret

____________________________________________________________________________________________
; Mapper #144

; $8001-$FFFF: CCCCPPPP - swap 32k PROM at $8000
;                       - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper144:

    ; Almost exactly like Color Dreams...
    call Mapper011

    ; ...except for $8000
    CPUWrite 08000, 08000, WriteVoid
    ret

____________________________________________________________________________________________
; Mapper #008, FFE F8xxx

; $8000-$FFFF: PPPPPCCC - swap 8k  CROM at $0000
;                       - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper008:

    ; Set ports
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; PPPPPCCC
    swap CROM, 8k,  00000, eax | shr eax 3
    swap PROM, 16k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #034, Nina-1

; $7FFD:       PPPPPPPP - swap 32k PROM at $8000
; $7FFE:       CCCCCCCC - swap 4k  CROM at $0000
; $7FFF:       CCCCCCCC - swap 4k  CROM at $1000
; $8000-$FFFF: PPPPPPPP - swap 32k PROM at $8000
____________________________________________________________________________________________

Mapper034:

    ; Memory mapping
    CPUWrite 07FFD, 07FFD, @7FFD
    CPUWrite 07FFE, 07FFE, @7FFE
    CPUWrite 07FFF, 07FFF, @7FFF
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Swap last 16k PROM bank to $C000
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@7FFD:
@8000_FFFF:

    ; PPPPPPPP
    swap PROM, 32k, 08000, eax
    ret

@7FFE:

    ; CCCCCCCC
    swap CROM, 4k,  00000, eax
    ret

@7FFF:

    ; CCCCCCCC
    swap CROM, 4k,  01000, eax
    ret

____________________________________________________________________________________________
; Mapper #072, Jaleco type 1 - low bankswitch

; $8000-$FFFF: PCxxBBBB - (P = 1) swap 16k PROM at $8000 from bank BBBB
;                       - (C = 1) swap 8k  CROM at $0000 from bank BBBB
____________________________________________________________________________________________

Mapper072:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Swap last bank at $C000
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_FFFF:

    ; PCxxBBBB
    ifFlag eax 080, swap PROM, 16k, 08000, eax
    ifFlag eax 040, swap CROM, 8k,  00000, eax
    ret

____________________________________________________________________________________________
; Mapper #092, Jaleco type 1 - high bankswitch

; $8x70-$8x7F: (LSB of address): 0111CCCC - swap 8k  CROM at $0000
; $8xB0-$8xBF: (LSB of address): 1011PPPP - swap 16k PROM at $C000
; $9xD0-$FxDF: (LSB of address): 1101PPPP - swap 16k PROM at $C000
; $9xE0-$FxEF: (LSB of address): 1110CCCC - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper092:

    ; Set ports
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 09000, 0FFFF, @9000_FFFF

    ; Swap in last 16k PROM bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_8FFF:

    ; XXXXxxxx
    mov eax edx | and eax 0F0
    on eax = 00_0111_0000, @SwitchCROM
    on eax = 00_1011_0000, @SwitchPROM
    ret

@9000_FFFF:

    ; XXXXxxxx
    mov eax edx | and eax 0F0
    on eax = 00_1101_0000, @SwitchPROM
    on eax = 00_1110_0000, @SwitchCROM
    ret

@SwitchPROM:

    ; xxxxPPPP
    swap PROM, 16k, 0C000, edx
    ret

@SwitchCROM:

    ; xxxxCCCC
    swap CROM, 8k,  00000, edx
    ret

____________________________________________________________________________________________
; Mapper #240, Jing Ke Xin Zhuan

; $4020-$5FFF: PPPPCCCC - swap 32k PROM at $8000
;                       - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper240:

    ; Set ports
    CPUWrite 04020, 05FFF, @4020_5FFF

    ; Swap last bank into $C000
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@4020_5FFF:

    ; PPPPCCCC
    swap CROM, 8k,  00000, eax | shr eax 4
    swap PROM, 32k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #140, Jaleco

; $6000-$7000: PPPPCCCC - swap 32k PROM at $8000
;                       - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper140:

    ; Memory mapping
    CPUWrite 04080, 04080, @4080
    CPUWrite 06000, 07FFF, @6000_7FFF
    ret

@4080:

    ; ????????
    ret

@6000_7FFF:

    ; PPPPCCCC
    swap CROM, 8k,  00000, eax | shr eax 4
    swap PROM, 32k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #079, Nina-3

; $4020-$5FFF, $8000-$FFFF: xxxxxCCC - swap 8k  CROM at $0000
; $4100:                    xxxxPCCC - swap 32k PROM at $8000
;                                    - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper079:

    ; Memory mapping
    CPUWrite 04020, 05FFF, @4020_5FFF
    CPUWrite 08000, 0FFFF, @8000_FFFF
    CPUWrite 04100, 04100, @4100

    ; Hacks: force one-screen
    if D$Cartridge@PROMCRC32 =  08EB_DE64, mirror ONE_SCREEN_2000 ; Trolls on Treasure Island (U)
    if D$Cartridge@PROMCRC32 = 0627A_D380, mirror ONE_SCREEN_2000 ; Dudes With Attitude (U)
    ret

@4100:

    ; xxxxPCCC
    ifNotFlag. eax 08
        swap PROM, 32k, 08000, 0
    else
        swap PROM, 32k, 08000, 1
    endif
    call @8000_FFFF
    ret

@4020_5FFF:
@8000_FFFF:

    ; xxxxxCCC
    swap CROM, 8k, 00000, eax
    ret
____________________________________________________________________________________________
; Mapper #244, Decathlon

; $8065-$80A4: - swap 32k PROM at $8000 from bank (address - $8065) & 3
; $80A5-$80E4: - swap 8k  CROM at $0000 from bank (address - $80A5) & 7
____________________________________________________________________________________________

Mapper244:

    ; Set ports
    CPUWrite 08065, 080A4, @SwitchPROM
    CPUWrite 080A5, 080E4, @SwitchCROM

    ; NTSC hack
    if D$Cartridge@PROMCRC32 = 089A7_35D8, call MenuNTSC ; Decathlon
    ret

@SwitchPROM:

    sub edx 08065
    swap PROM, 32k, 08000, edx
    ret

@SwitchCROM:

    sub edx 080A5
    swap CROM, 8k,  00000, edx
    ret

____________________________________________________________________________________________
; Mapper #133, Sachen Chen

; $4120: xxxxxPCC - swap 32k PROM at $8000
;                 - swap 8k  CROM at $0000
____________________________________________________________________________________________

; $8000/$8001???
Mapper133:

    ; Set ports
    CPUWrite 04120, 04120, @4120
    ret

@4120:

    ; xxxxxPCC
    swap CROM, 8k,  00000, eax | shr eax 2
    swap PROM, 32k, 08000, eax
    ret
____________________________________________________________________________________________
TITLE PRom_CRom_Mirror

; ____________________________________________________________________________________________
;
;                      PROM/CROM/mirror switching mappers
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #058, 68-in-1 + 32-in-1 jammed together (sigh)

; $8000-$FFFF: ------M- - mirror HORIZONTAL/VERTICAL (0/1)
;    (address) -SCCCPPp - (S = 0) swap 32k PROM at $8000 (PP)
;                       - (S = 1) swap 16k PROM at $8000 and $C000 (PPp)
;                       - swap 8k CROM at $0000
____________________________________________________________________________________________

Mapper058:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; Hack for Study and Game 32-in-1
    if. D$Cartridge@PROMCRC32 = 0ABB2_F974 ; Study and Game 32-in-1 [!]
        swap PROM, 32k, 08000, eax
    endif

    ; ------M-
    ifNotFlag. al 02
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif

    ; -S---PPp
    mov eax edx | and eax 07
    ifNotFlag. edx 040
        shr eax 1
        swap PROM, 32k, 08000, eax
    else
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    endif

    ; --CCC
    shr edx 3
    swap CROM, 8k, 00000, edx
    ret
____________________________________________________________________________________________
; Mapper #068, Sunsoft #4

; $8000-$8FFF: CCCCCCCC - swap 2k CROM at $0000
; $9000-$9FFF: CCCCCCCC - swap 2k CROM at $0800
; $A000-$AFFF: CCCCCCCC - swap 2k CROM at $1000
; $B000-$BFFF: CCCCCCCC - swap 2k CROM at $1800
; $C000-$CFFF: xCCCCCCC - swap 1k CiROM at $2000 (from last 128k segment)
; $D000-$DFFF: xCCCCCCC - swap 1k CiROM at $2400 (from last 128k segment)
; $E000-$EFFF: xxxExxMM - enable CiROM at $2000-$2FFF
;                       - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; $F000-$FFFF: PPPPPPPP - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper068:

    ; Memory mapping
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 09000, 09FFF, @9000_9FFF
    CPUWrite 0A000, 0AFFF, @A000_AFFF
    CPUWrite 0B000, 0BFFF, @B000_BFFF
    CPUWrite 0C000, 0CFFF, @C000_CFFF
    CPUWrite 0D000, 0DFFF, @D000_DFFF
    CPUWrite 0E000, 0EFFF, @E000_EFFF
    CPUWrite 0F000, 0FFFF, @F000_FFFF

    ; CiRAM/CiROM
    PPURead  02000, 02FFF, @ReadCiRAM
    PPUWrite 02000, 02FFF, @WriteCiRAM

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK

    call @UpdateMirroring
    ret

; CCCCCCCC
@8000_8FFF: swap CROM, 2k,  0000, eax | ret
@9000_9FFF: swap CROM, 2k,  0800, eax | ret
@A000_AFFF: swap CROM, 2k, 01000, eax | ret
@B000_BFFF: swap CROM, 2k, 01800, eax | ret
; xCCCCCCC
@C000_CFFF: mov D$Bank+00 eax | call @UpdateMirroring | ret
@D000_DFFF: mov D$Bank+04 eax | call @UpdateMirroring | ret

@E000_EFFF:

    ; xxxExxxx
    test eax 010 | setnz B$Register

    ; xxxxxxMM
    and eax 03 | mov D$Register+04 eax
    call @UpdateMirroring
    ret

@UpdateMirroring:

    mov eax D$Register+04


    if.. B$Register = &FALSE

        ; xxxxxxMM
        if al = 0, mirror VERTICAL
        if al = 1, mirror HORIZONTAL
        if al = 2, mirror ONE_SCREEN_2000
        if al = 3, mirror ONE_SCREEN_2400

    else..

        or D$Bank+00 080
        or D$Bank+04 080

        ; xxxxxxMM
        if. al = 0
            swap CiROM, 1k, 02000, D$Bank+00
            swap CiROM, 1k, 02400, D$Bank+04
            swap CiROM, 1k, 02800, D$Bank+00
            swap CiROM, 1k, 02C00, D$Bank+04
        endif
        if. al = 1
            swap CiROM, 1k, 02000, D$Bank+00
            swap CiROM, 1k, 02400, D$Bank+00
            swap CiROM, 1k, 02800, D$Bank+04
            swap CiROM, 1k, 02C00, D$Bank+04
        endif
        if. al = 2
            swap CiROM, 1k, 02000, D$Bank+00
            swap CiROM, 1k, 02400, D$Bank+00
            swap CiROM, 1k, 02800, D$Bank+00
            swap CiROM, 1k, 02C00, D$Bank+00
        endif
        if. al = 3
            swap CiROM, 1k, 02000, D$Bank+04
            swap CiROM, 1k, 02400, D$Bank+04
            swap CiROM, 1k, 02800, D$Bank+04
            swap CiROM, 1k, 02C00, D$Bank+04
        endif

    endif..
    ret

; PPPPPPPP
@F000_FFFF: swap PROM, 16k, 08000, eax | ret

@ReadCiRAM:

    on B$Register = &FALSE, ReadCiRAM
    jmp ReadCiROM

@WriteCiRAM:

    on B$Register = &FALSE, WriteCiRAM
    ret
____________________________________________________________________________________________
; Mapper #082, Taito

; $7EF0: CCCCCCCx - swap 2k CROM at $0000
; $7EF1: CCCCCCCx - swap 2k CROM at $0800
; $7EF2: CCCCCCCC - swap 1k CROM at $1000
; $7EF3: CCCCCCCC - swap 1k CROM at $1400
; $7EF4: CCCCCCCC - swap 1k CROM at $1800
; $7EF5: CCCCCCCC - swap 1k CROM at $1C00
; $7EF6: xxxxxxSM - mirror HORIZONTAL/VERTICAL (0/1)
;                 - (S = 0) xor CROM swap address with $1000
; $7EFA: PPPPPPxx - swap 8k PROM at $8000
; $7EFA: PPPPPPxx - swap 8k PROM at $A000
; $7EFA: PPPPPPxx - swap 8k PROM at $C000
____________________________________________________________________________________________

Mapper082:

    ; Memory mapping
    CPUWrite 07EF0, 07EF0, @7EF0
    CPUWrite 07EF1, 07EF1, @7EF1
    CPUWrite 07EF2, 07EF2, @7EF2
    CPUWrite 07EF3, 07EF3, @7EF3
    CPUWrite 07EF4, 07EF4, @7EF4
    CPUWrite 07EF5, 07EF5, @7EF5
    CPUWrite 07EF6, 07EF6, @7EF6
    CPUWrite 07EFA, 07EFA, @7EFA
    CPUWrite 07EFB, 07EFB, @7EFB
    CPUWrite 07EFC, 07EFC, @7EFC

    mirror VERTICAL
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; CCCCCCCx
@7EF0: mov edx  0000 | xor edx D$Register | shr eax 1 | swap CROM, 2k, edx, eax | ret
@7EF1: mov edx  0800 | xor edx D$Register | shr eax 1 | swap CROM, 2k, edx, eax | ret
; CCCCCCCC
@7EF2: mov edx 01000 | xor edx D$Register | swap CROM, 1k, edx, eax | ret
@7EF3: mov edx 01400 | xor edx D$Register | swap CROM, 1k, edx, eax | ret
@7EF4: mov edx 01800 | xor edx D$Register | swap CROM, 1k, edx, eax | ret
@7EF5: mov edx 01C00 | xor edx D$Register | swap CROM, 1k, edx, eax | ret

@7EF6:

    ; xxxxxxxM
    ifNotFlag. eax 01
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif

    ; xxxxxxSx
    ifNotFlag. eax 02
        mov D$Register 00000
    else
        mov D$Register 01000
    endif
    ret

@7EFA: shr eax 2 | swap PROM, 8k, 08000, eax | ret
@7EFB: shr eax 2 | swap PROM, 8k, 0A000, eax | ret
@7EFC: shr eax 2 | swap PROM, 8k, 0C000, eax | ret

____________________________________________________________________________________________
; Mapper #080, Taito X-005

; $7EF0: xCCCCCCx - swap 2k CROM at $0000
; $7EF1: xCCCCCCx - swap 2k CROM at $0800
; $7EF2: xCCCCCCC - swap 1k CROM at $1000
; $7EF3: xCCCCCCC - swap 1k CROM at $1400
; $7EF4: xCCCCCCC - swap 1k CROM at $1800
; $7EF5: xCCCCCCC - swap 1k CROM at $1C00
; $7EF6: xxxxxxxM - mirror HORIZONTAL/VERTICAL (0/1)
; $7EFA:
; $7EFB: PPPPPPPP - swap 8k PROM at $8000
; $7EFC:
; $7EFD: PPPPPPPP - swap 8k PROM at $A000
; $7EFE:
; $7EFF: PPPPPPPP - swap 8k PROM at $C000
____________________________________________________________________________________________

Mapper080:

    ; Memory mapping
    CPUWrite 07EF0, 07EF0, @7EF0
    CPUWrite 07EF1, 07EF1, @7EF1
    CPUWrite 07EF2, 07EF2, @7EF2
    CPUWrite 07EF3, 07EF3, @7EF3
    CPUWrite 07EF4, 07EF4, @7EF4
    CPUWrite 07EF5, 07EF5, @7EF5
    CPUWrite 07EF6, 07EF6, @7EF6
    CPUWrite 07EFA, 07EFB, @7EFA_7EFB
    CPUWrite 07EFC, 07EFD, @7EFC_7EFD
    CPUWrite 07EFE, 07EFF, @7EFE_7EFF

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK

    ; Mirroring hack
    if. D$Cartridge@PROMCRC32 = 09832_D15A ; Fudou Myouou Den (J)
        mov D$Bank+00 0000
        mov D$Bank+04 0000
        mov D$Bank+08 0400
        mov D$Bank+0C 0400
        CPUWrite 07EF0, 07EF0, @Hack7EF0
        CPUWrite 07EF1, 07EF1, @Hack7EF1
        CPUWrite 07EF6, 07EF6, @Hack7EF6
    endif
    ret

; xCCCCCCx
@7EF0: shr eax 1 | swap CROM, 2k, 0000, eax | ret
@7EF1: shr eax 1 | swap CROM, 2k, 0800, eax | ret
; xCCCCCCC
@7EF2: swap CROM, 1k, 01000, eax | ret
@7EF3: swap CROM, 1k, 01400, eax | ret
@7EF4: swap CROM, 1k, 01800, eax | ret
@7EF5: swap CROM, 1k, 01C00, eax | ret

@7EF6:

    ; xxxxxxxM
    ifNotFlag. eax 01
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif
    ret

; PPPPPPPP
@7EFA_7EFB: swap PROM, 8k, 08000, eax | ret
@7EFC_7EFD: swap PROM, 8k, 0A000, eax | ret
@7EFE_7EFF: swap PROM, 8k, 0C000, eax | ret
____________________________________________________________________________________________

@Hack7EF0:

    ifNotFlag. eax 080
        mov D$Bank+00 0000
        mov D$Bank+04 0000
    else
        mov D$Bank+00 0400
        mov D$Bank+04 0400
    endif
    mov esi Bank, edi IndexCiRAM, ecx 4 | rep movsd
    jmp @7EF0

@Hack7EF1:

    ifNotFlag. eax 080
        mov D$Bank+08 0000
        mov D$Bank+0C 0000
    else
        mov D$Bank+08 0400
        mov D$Bank+0C 0400
    endif
    mov esi Bank, edi IndexCiRAM, ecx 4 | rep movsd
    jmp @7EF1

@Hack7EF6:

    mov D$Bank+00 0000
    mov D$Bank+0C 0400
    ifNotFlag. eax 01
        mov D$Bank+04 0000
        mov D$Bank+08 0400
    else
        mov D$Bank+04 0400
        mov D$Bank+08 0000
    endif
    jmp @7EF6
____________________________________________________________________________________________
; Mapper #032, Irem G-101

; $8000-$8FFF: PPPPPPPP - swap 8k PROM at $8000/$C000 (S = 0/1)
; $9000-$9FFF: xxxxxxSM - mirror VERTICAL/HORIZONTAL (0/1)
; $A000-$AFFF: PPPPPPPP - swap 8k PROM ar $A000
; $B000:       CCCCCCCC - swap 1k CROM at $0000
; $B001:       CCCCCCCC - swap 1k CROM at $0400
; $B002:       CCCCCCCC - swap 1k CROM at $0800
; $B003:       CCCCCCCC - swap 1k CROM at $0C00
; $B004:       CCCCCCCC - swap 1k CROM at $1000
; $B005:       CCCCCCCC - swap 1k CROM at $1400
; $B006:       CCCCCCCC - swap 1k CROM at $1800
; $B007:       CCCCCCCC - swap 1k CROM at $1C00
____________________________________________________________________________________________

Mapper032:

    ; Memory mapping
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 09000, 09FFF, @9000_9FFF
    CPUWrite 0A000, 0AFFF, @A000_AFFF
    mov ecx 0B000
L0: push ecx
        mov eax WriteVoid
        and ecx 0B007
        if ecx = 0B000, mov eax @B000
        if ecx = 0B001, mov eax @B001
        if ecx = 0B002, mov eax @B002
        if ecx = 0B003, mov eax @B003
        if ecx = 0B004, mov eax @B004
        if ecx = 0B005, mov eax @B005
        if ecx = 0B006, mov eax @B006
        if ecx = 0B007, mov eax @B007
    pop ecx
    CPUWrite ecx, ecx, eax
    inc ecx | on ecx < 0C000, L0<<

    ; Last PROM bank
    swap PROM, 16k, 0C000, LAST_BANK

    ; Ai Sensei no Oshiete - Watashi no Hoshi
    if. D$Cartridge@PROMCRC32 = 0FD3F_C292
        swap PROM, 16k, 08000, 0F
        swap PROM, 16k, 0C000, 0F
    endif

    ; Major League
    if D$Cartridge@PROMCRC32 = 0C0FE_D437
    mirror ONE_SCREEN_2000
    ret

@8000_8FFF:

    ; PPPPPPPP
    if. B$Register = 0
        swap PROM, 8k, 08000, eax
    else
        swap PROM, 8k, 0C000, eax
    endif
    ret

@9000_9FFF:

    ; xxxxxxxM
    ifNotFlag. eax 01
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxxxxSx
    test eax 02 | setnz B$Register
    ret

@A000_AFFF: swap PROM, 8k, 0A000, eax | ret
@B000: swap CROM, 1k,  0000, eax | ret
@B001: swap CROM, 1k,  0400, eax | ret
@B002: swap CROM, 1k,  0800, eax | ret
@B003: swap CROM, 1k,  0C00, eax | ret
@B004: swap CROM, 1k, 01000, eax | ret
@B005: swap CROM, 1k, 01400, eax | ret
@B006: swap CROM, 1k, 01800, eax | if D$Cartridge@PROMCRC32 = 0C0FE_D437, ifFlag eax 040, mirror MIRRORING_0001  | ret ; Major League
@B007: swap CROM, 1k, 01C00, eax | if D$Cartridge@PROMCRC32 = 0C0FE_D437, ifFlag eax 040, mirror ONE_SCREEN_2000 | ret ; Major League
____________________________________________________________________________________________
; Mapper #234, Maxi-15

; (Read/write)
; $FF80-$FF9F: MSxxABCD - high bits of bank select
; $FFE8-$FFF7: xEFGxxxH - low  bits of bank select
;                       - (S = 0) swap 32k PROM at $8000 (ABCD)
;                       - (S = 0) swap 8k  CROM at $0000 (ABCDFG)
;                       - (S = 1) swap 32k PROM at $8000 (ABCH)
;                       - (S = 1) swap 8k  CROM at $0000 (ABCEFG)
;                       - mirror VERTICAL/HORIZONTAL (0/1)
____________________________________________________________________________________________

Mapper234:

    ; Memory mapping
    CPUWrite 0FF80, 0FF9F, @FF80_FF9F
    CPUWrite 0FFE8, 0FFF7, @FFE8_FFF7
    CPURead  0FF80, 0FF9F, @ReadFF80_FF9F
    CPURead  0FFE8, 0FFF7, @ReadFFE8_FFF7
    CPURead 06000, 07FFF, ReadVoid

    mirror VERTICAL
    ret

@FF80_FF9F:

    if. D$Bank+00 = 0

        ; Mxxxxxxx
        ifNotFlag.. eax 080
            mirror VERTICAL
        else..
            mirror HORIZONTAL
        endif..

        mov D$Bank+00 eax
        call @UpdateBanks

    endif
    ret

@FFE8_FFF7:

    mov D$Bank+04 eax
    call @UpdateBanks
    ret

@ReadFF80_FF9F:

    call ReadPROM
    jmp @FF80_FF9F

@ReadFFE8_FFF7:

    call ReadPROM
    jmp @FFE8_FFF7
____________________________________________________________________________________________

@UpdateBanks:

    ; PROM
    mov eax D$Bank+00
    mov edx D$Bank+04
    ifFlag. eax 040
        and eax 0E
        and edx 01
        or eax edx
    endif
    and eax 0F
    swap PROM, 32k, 08000, eax

    ; CROM
    mov eax D$Bank+00
    mov edx D$Bank+04 | shr edx 4
    ifFlag. eax 040
        and eax 0E
        and edx 07
    else
        and eax 0F
        and edx 03
    endif
    shl eax 2
    or eax edx
    swap CROM, 8k, 00000, eax
    ret

____________________________________________________________________________________________
; Mapper #096, Bandai 74HC161/32

; $8000-$FFFF: xxxxxSPP - swap 32k PROM at $8000
;                       - Select 16k CRAM segment
; Select 4k CRAM at $0000 from VRAM address
; Keep last 4k CRAM of segment in $1000

____________________________________________________________________________________________

Mapper096:

    ; Set 32k of CRAM
    mov D$Cartridge@SizeCRAM 32

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    PPURead  02000, 023BF, @NameTable
    PPURead  02400, 027BF, @NameTable
    PPURead  02800, 02BBF, @NameTable
    PPURead  02C00, 02FBF, @NameTable

    swap CRAM, 4k, 00000, 0
    swap CRAM, 4k, 01000, 3
    ret

@8000_FFFF:

    ; xxxxxxPP
    swap PROM, 32k, 08000, eax

    ; xxxxxSxx
    and eax 4 | mov D$Register eax
    or eax D$Register+04 | swap CRAM, 4k, 00000, eax
    or eax 3             | swap CRAM, 4k, 01000, eax
    ret

@NameTable:

    push eax

        and ah 03 | movzx eax ah
        if. eax != D$Register+04
            mov D$Register+04 eax
            or eax D$Register
            swap CRAM, 4k, 00000, eax
        endif

    pop eax
    call ReadCiRAM
    ret

____________________________________________________________________________________________
; Mapper #112, ASDER

; $8000: xxxxxNNN - command number
; $A000: BBBBBBBb - (NNN = 0) swap 8k PROM at $8000 (BBBBBBBb)
;                 - (NNN = 1) swap 8k PROM at $A000 (BBBBBBBb)
;                 - (NNN = 2) swap 2k PROM at $0000 (BBBBBBB)
;                 - (NNN = 2) swap 2k PROM at $0800 (BBBBBBB)
;                 - (NNN = 4) swap 1k PROM at $1000 (BBBBBBBb)
;                 - (NNN = 5) swap 1k PROM at $1400 (BBBBBBBb)
;                 - (NNN = 6) swap 1k PROM at $1800 (BBBBBBBb)
;                 - (NNN = 7) swap 1k PROM at $1C00 (BBBBBBBb)
; $E000: xxxxxxxM - mirror VERTICAL/HORIZONTAL (0/1)
____________________________________________________________________________________________

Mapper112:

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 0A000, 0A000, @A000
    CPUWrite 0C000, 0C000, @C000
    CPUWrite 0E000, 0E000, @E000

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000: mov D$Command eax | ret

@A000:

    mov edx D$Command | and edx 07
    mov ecx eax | shr ecx 1

    if edx = 0, swap PROM, 8k, 08000, eax
    if edx = 1, swap PROM, 8k, 0A000, eax
    if edx = 2, swap CROM, 2k,  0000, ecx
    if edx = 3, swap CROM, 2k,  0800, ecx
    if edx = 4, swap CROM, 1k, 01000, eax
    if edx = 5, swap CROM, 1k, 01400, eax
    if edx = 6, swap CROM, 1k, 01800, eax
    if edx = 7, swap CROM, 1k, 01C00, eax
    ret

; ????????
@C000: ret

@E000:

    ; xxxxxxxM
    ifNotFlag. eax 01
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

____________________________________________________________________________________________
; Mapper #243, Sachen 74LS374N

; $4100: xxxxxNNN - command number
; $4101: xxxxxxBB - (NNN = 000) swap first 32k PROM at $8000, and 4th 8k CROM at $0000
;                 - (NNN = 100) xB = LSB of 8k CROM select
;                 - (NNN = 110) BB = Two MSB of 8k CROM select
;                 - (NNN = 101) xB = swap 32k PROM at $8000
;                 - (NNN = 111) xB = mirror VERTICAL/HORIZONTAL (1/0)
____________________________________________________________________________________________

Mapper243:

    ; Memory mapping
    mov ecx 04100
L0: push ecx
        mov eax WriteVoid
        and ecx 04101
        if ecx = 04100, mov eax @4100
        if ecx = 04101, mov eax @4101
    pop ecx
    CPUWrite ecx, ecx, eax
    inc ecx | on ecx < 08000, L0<

    ; Reset regs
    call @4101
    ret

@4100: mov D$Command eax | ret

@4101:

    mov edx D$Command | and edx 7

    if. edx = 0
        swap PROM, 32k, 08000, 0
        mov D$Register 3
    endif

    if. edx = 4
        and D$Register 06
        and eax 01
        or  D$Register eax
    endif

    if. edx = 5
        and eax 01
        swap PROM, 32k, 08000, eax
    endif

    if. edx = 6
        and D$Register 01
        and eax 03
        shl eax 1
        or D$Register eax
    endif

    if. edx = 7
        mirror HORIZONTAL
        ifFlag eax 01, mirror VERTICAL
    endif

    swap CROM, 8k, 00000, D$Register
    ret

____________________________________________________________________________________________
; Mapper #057, Game star GK-54

; $x000-$x7FF: xIxxxxCC (xxxxxx10) - (I = 1) swap 8k CROM at $0000 (3210)
; $x800-$xFFF: CPpSMCCC (3xxxx210) - (S = 0) swap 16k PROM at $8000 and $C000 (Pp)
;                                  - (S = 1) swap 32k PROM at $8000 (1P)
;                                  - mirror VERTICAL/HORIZONTAL (0/1)
;                                  - swap 8k CROM at $0000 (3210)
____________________________________________________________________________________________

Mapper057:

    ; Memory mapping
    mov ecx 08000
L0: CPUWrite ecx, ecx, @8000
    ifFlag ecx 0800, CPUWrite ecx, ecx, @8800
    inc cx | jnz L0<

    ; Load first bank
    swap PROM, 16k, 0C000, 0
    ret

@8000:

    ; xIxxxxxx
    ifFlag. eax 040

        ; Swap CROM
        and eax 03
        or  eax D$Register
        swap CROM, 8k, 00000, eax

    endif
    ret

@8800:

    ; xxxxMxxx
    ifNotFlag. eax 08
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xPpSxxxx
    mov edx eax
    shr edx 5
    ifFlag. eax 010
        shr edx 1
        swap PROM, 32k, 08000, edx
    else
        swap PROM, 16k, 08000, edx
        swap PROM, 16k, 0C000, edx
    endif

    ; CxxxxCCC
    ifFlag. eax 080
        and eax 07
        or  eax 08
    else
        and eax 07
    endif
    mov D$Register eax
    swap CROM, 8k, 00000, eax
    ret

____________________________________________________________________________________________
; Mapper #255, 110-in-1

; $5800-$5FFF: access the four registers
; $8000-$FFFF: (address) 1HMSPPPP_PpCCCCCC - mirror VERTICAL/HORIZONTAL (0/1)
;                                          - (S = 0) swap 32k PROM at $8000 (PPPPP)
;                                          - (S = 1) swap 16k PROM at $8000 and $C000 (PPPPPp)
;                                          - swap 8k CROM at $0000
;                                          - H = high bit of PROM/CROM bank select
____________________________________________________________________________________________

Mapper255:

    ; Memory mapping
    CPURead  05800, 05FFF, @ReadRegister
    CPUWrite 05800, 05FFF, @WriteRegister
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Reset regs
    mov D$Register 0
    ret

@ReadRegister:
    and edx 03
    movzx eax B$Register+edx
    ret

@WriteRegister:

    and edx 03
    and eax 0F
    mov B$Register+edx al
    ret

@8000_FFFF:

    ; xxMxxxxx_xxxxxxxx
    ifNotFlag. edx 02000
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xHxxxxxx_xxCCCCCC
    mov eax edx | and eax 03F
    ifFlag edx 04000, or eax 040
    swap CROM, 8k, 00000, eax

    ; xHxSPPPP_Ppxxxxxx
    mov eax edx
    shr eax 6 | and eax 03F
    ifFlag edx 04000, or eax 040
    ifNotFlag. edx 01000
        and eax 03E
        swap PROM, 16k, 08000, eax | inc eax
        swap PROM, 16k, 0C000, eax
    else
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    endif
    ret

____________________________________________________________________________________________
; Mapper #225, 72-in-1

; $8000-$FFFF: (address) 1xMSPPPP_PpCCCCCC - mirror VERTICAL/HORIZONTAL (0/1)
;                                          - (S = 0) swap 32k PROM at $8000 (PPPPP)
;                                          - (S = 1) swap 16k PROM at $8000 and $C000 (PPPPPp)
;                                          - swap 8k CROM at $0000
____________________________________________________________________________________________

Mapper225:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxMxxxxx_xxxxxxxx
    ifNotFlag. edx 02000
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxxxxxx_xxCCCCCC
    swap CROM, 8k, 00000, edx

    ; xxxSPPPP_Ppxxxxxx
    ifNotFlag. edx 01000
        shr edx 7
        swap PROM, 32k, 08000, edx
    else
        shr edx 6
        swap PROM, 16k, 08000, edx
        swap PROM, 16k, 0C000, edx
    endif
    ret
____________________________________________________________________________________________
; Mapper #075, Konami VRC1 / Jaleco D65005

; $8000-$8FFF: PPPPPPPP - swap 8k PROM at $8000
; $9000-$9FFF: xxxxxCcM - mirror VERTICAL/HORIZONTAL (0/1)
;                       - c = bit 4 of CROM select at $0000
;                       - C = bit 4 of CROM select at $1000
; $A000-$AFFF: PPPPPPPP - swap 8k PROM at $A000
; $C000-$CFFF: PPPPPPPP - swap 8k PROM at $C000
; $E000-$EFFF: xxxxCCCC - swap 4k CROM at $0000
; $F000-$FFFF: xxxxCCCC - swap 4k CROM at $1000
____________________________________________________________________________________________

[CROMLow  Register+00
 CROMHigh Register+04]

Mapper075:

    ; Memory mapping
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 09000, 09FFF, @9000_9FFF
    CPUWrite 0A000, 0AFFF, @A000_AFFF
    CPUWrite 0C000, 0CFFF, @C000_CFFF
    CPUWrite 0E000, 0EFFF, @E000_EFFF
    CPUWrite 0F000, 0FFFF, @F000_FFFF

    ; Load last PROM bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@8000_8FFF: swap PROM, 8k, 08000, eax | ret
@A000_AFFF: swap PROM, 8k, 0A000, eax | ret
@C000_CFFF: swap PROM, 8k, 0C000, eax | ret

; xxxxCCCC
@E000_EFFF: and eax 0F | and D$CROMLow  010 | or D$CROMLow  eax | swap CROM, 4k, 00000, D$CROMLow  | ret
@F000_FFFF: and eax 0F | and D$CROMHigh 010 | or D$CROMHigh eax | swap CROM, 4k, 01000, D$CROMHigh | ret

@9000_9FFF:

    ; xxxxxxxM
    ifNotFlag. eax 01
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxxxxcx
    ifFlag. eax 02
        or D$CROMLow 010
    else
        and D$CROMLow 0F
    endif

    ; xxxxxCxx
    ifFlag. eax 04
        or D$CROMHigh 010
    else
        and D$CROMHigh 0F
    endif

    ; Swap new banks
    swap CROM, 4k, 00000, D$CROMLow
    swap CROM, 4k, 01000, D$CROMHigh
    ret

____________________________________________________________________________________________
; Mapper #022, Konami VRC4 1B

; $8000: PPPPPPPP - swap 8k PROM at $8000
; $9000: xxxxxxMM - mirror VERTICAL/HORIZONTAL/$2400/$2000 (0/1/2/3)
; $A000: PPPPPPPP - swap 8k PROM at $A000
; $B000: CCCCCCCx - swap 1k CROM at $0000
; $B001: CCCCCCCx - swap 1k CROM at $0400
; $C000: CCCCCCCx - swap 1k CROM at $0800
; $C001: CCCCCCCx - swap 1k CROM at $0C00
; $D000: CCCCCCCx - swap 1k CROM at $1000
; $D001: CCCCCCCx - swap 1k CROM at $1400
; $E000: CCCCCCCx - swap 1k CROM at $1800
; $E001: CCCCCCCx - swap 1k CROM at $1C00
____________________________________________________________________________________________

Mapper022:

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 09000, 09000, @9000
    CPUWrite 0A000, 0A000, @A000
    CPUWrite 0B000, 0B000, @B000
    CPUWrite 0B001, 0B001, @B001
    CPUWrite 0C000, 0C000, @C000
    CPUWrite 0C001, 0C001, @C001
    CPUWrite 0D000, 0D000, @D000
    CPUWrite 0D001, 0D001, @D001
    CPUWrite 0E000, 0E000, @E000
    CPUWrite 0E001, 0E001, @E001

    ; Load first 1k CROM
    swap CROM, 1k,  0000, 0
    swap CROM, 1k,  0400, 0
    swap CROM, 1k,  0800, 0
    swap CROM, 1k,  0C00, 0
    swap CROM, 1k, 01000, 0
    swap CROM, 1k, 01400, 0
    swap CROM, 1k, 01800, 0
    swap CROM, 1k, 01C00, 0

    ; Load last 16k PROM
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@8000: swap PROM, 8k, 08000, eax | ret
@A000: swap PROM, 8k, 0A000, eax | ret

; CCCCCCCx
@B000: shr eax 1 | swap CROM, 1k,  0000, eax | ret
@B001: shr eax 1 | swap CROM, 1k,  0400, eax | ret
@C000: shr eax 1 | swap CROM, 1k,  0800, eax | ret
@C001: shr eax 1 | swap CROM, 1k,  0C00, eax | ret
@D000: shr eax 1 | swap CROM, 1k, 01000, eax | ret
@D001: shr eax 1 | swap CROM, 1k, 01400, eax | ret
@E000: shr eax 1 | swap CROM, 1k, 01800, eax | ret
@E001: shr eax 1 | swap CROM, 1k, 01C00, eax | ret

@9000:

    ; xxxxxxMM
    and eax 3
    if eax = 0, mirror VERTICAL
    if eax = 1, mirror HORIZONTAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2400
    ret

____________________________________________________________________________________________
; Mapper #152

; $6000-$FFFF: MPPPCCCC - mirror $2000/$2400 (0/1)
;                       - swap 16k PROM at $8000
;                       - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper152:

    ; Memory mapping
    CPUWrite 06000, 0FFFF, @6000_FFFF

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@6000_FFFF:

    ; Mxxxxxxx
    ifNotFlag. eax 080
        mirror ONE_SCREEN_2000
    else
        mirror ONE_SCREEN_2400
    endif

    ; xxxxCCCC
    swap CROM, 8k,  00000, eax

    ; xPPPxxxx
    shr eax 4
    swap PROM, 16k, 00000, eax
    ret

____________________________________________________________________________________________
; Mapper #113, MB-91

; 4020-$7FFF: $8008-$8009: xHPPPCCC - swap 32k PROM at $8000 (HPPP)
;                                    - swap 8k  CROM at $0000 (HCCC)
; $8E66-$8E67:              xxxxxCCC - swap 8k  CROM at $0000 from bank 0/1 (CCC != 0 / CCC = 0)
; $E00A:                    xxxxxxxx - mirror $2000
____________________________________________________________________________________________

Mapper113:

    ; Memory mapping
    CPUWrite 04020, 07FFF, @4020_7FFF
    CPUWrite 08008, 08009, @8008_8009
    CPUWrite 08E66, 08E67, @8E66_8E67
    CPUWrite 0E00A, 0E00A, @E00A
    ret

@4020_7FFF:

    ; Mirroring hack
    if. D$Cartridge@PROMCRC32 = 0A75A_EDE5 ; HES 6-in-1
        mirror HORIZONTAL
        ifFlag eax 080, mirror VERTICAL
    endif

@8008_8009:

    ; xHxxxCCC
    mov edx eax | and edx 7
    ifFlag eax 040, or edx 08
    swap CROM, 8k, 00000, edx

    ; xHPPPxxx
    shr eax 3
    swap PROM, 32k, 08000, eax
    ret

@8E66_8E67:

    ; xxxxxCCC
    ifFlag. eax 7
        swap CROM, 8k, 00000, 0
    else
        swap CROM, 8k, 00000, 1
    endif
    ret

@E00A:

    mirror ONE_SCREEN_2000
    ret
____________________________________________________________________________________________
; Mapper #228. Action 52

; $8000-$FFFF: (address): 1xMPPPPP_PiSxCCCC - select 64k CROM segment
;                                           - mirror VERTICAL/HORIZONTAL (0/1)

;                 (data):          xxxxxxCC - swap 8k CROM at $0000 from selected segment
____________________________________________________________________________________________

Mapper228:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxMxxxxx_xxxxxxxx
    ifNotFlag. edx 02000
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxxxxxx_xxxxCCCC / xxxxxxCC
    push edx
        shl edx 2 | or eax edx
        swap CROM, 8k, 00000, eax
    pop edx

    ; xxxPPPPP_PiSxxxxx
    mov eax edx | shr edx 7
    and edx 03F
    ifFlag edx 020, and edx 02F ; if bit 5 is set, clear bit 4 (?)
    ; 16k game (S = 1)
    ifFlag. eax 020
        shl edx 2
        ifFlag eax 040, or edx 02
        swap PROM, 16k, 08000, edx
        swap PROM, 16k, 0C000, edx
    ; 32k game (S = 0)
    else
        swap PROM, 32k, 08000, edx
    endif
    ret

____________________________________________________________________________________________
; Mapper #041, Caltron 6-in-1

; $6000-$67FF: (LSB of address): 0xMSSPPP - swap 32k PROM at $8000
;                                         - mirror VERTICAL/HORIZONTAL (0/1)
;                                         - select 32k CROM segment
;                                         - memorize MSB of PROM select
; $8000-$FFFF: xxxxxxCC - (if MSB of PROM select = 1) swap 8k CROM from selected segment at $0000
____________________________________________________________________________________________

Mapper041:

    ; Set ports
    CPUWrite 06000, 067FF, @6000_67FF

    ; Enable CROM switching
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@6000_67FF:

    ; xxMxxxxx
    ifNotFlag. edx 020
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxxxPPP
    swap PROM, 32k, 08000, edx

    ; xxxxx!xx
    ifFlag. edx 4
        CPUWrite 08000, 0FFFF, @8000_FFFF
    else
        CPUWrite 08000, 0FFFF, WriteVoid
    endif

    ; xxxSSxxx
    shr edx 1 | and edx 0C
    mov D$Segment edx
    ret

@8000_FFFF:

    ; xxxxxxCC
    and eax 3
    or eax D$Segment
    swap CROM, 8k,  00000, eax
    ret

____________________________________________________________________________________________
; Mapper #062, 700-in-1

; $8000-$FFFF: (address): 1xPPPPPP_MPSCCCCC  - select 32k CROM segment
;                        (xx543210_x6xxxxxx) - swap 16k PROM at $8000, swap next 16k PROM at $C000
;                                            - (S = 1) 16k game, mirror $C000 from $8000
;                                            - mirror VERTICAL/HORIZONTAL (0/1)

;              (data):              xxxxxxCC - swap 8k bank from selected segment at $0000
____________________________________________________________________________________________

Mapper062:

    ; Set ports
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxxxxxxx_Mxxxxxxx
    ifNotFlag. edx 080
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxxxxxx_xxxCCCCC
    ; xxxxxxCC
    shl al 6 | mov ah dl
    shr ax 6
    swap CROM, 8k,  00000, eax

    ; xxPPPPPP_xPSxxxxx
    mov al dh | and eax 03F
    ifFlag edx 040, or eax 040
    ifNotFlag. edx 020
        shr eax 1
        swap PROM, 32k, 08000, eax
    else
        swap PROM, 16k, 08000, eax
        swap PROM, 16k, 0C000, eax
    endif
    ret

____________________________________________________________________________________________
; Mapper #229, 31-in-1

; $8000-$FFFF: (LSB of address): xxMBBBBB - swap 16k PROM at $8000 and $C000
;                                         - swap 8k  CROM at $0000
;                                         - mirror VERTICAL/HORIZONTAL (0/1)
____________________________________________________________________________________________

Mapper229:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxMxxxxx
    ifNotFlag. edx 020
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxPPPPP
    swap PROM, 16k, 08000, edx
    swap PROM, 16k, 0C000, edx
    ifNotFlag edx 01E, swap PROM, 32k, 08000, 0

    ; xxxCCCCC
    swap CROM, 8k,  00000, edx
    ret

____________________________________________________________________________________________
; Mapper #060, Reset-triggered 4-in-1

; $8000-$FFFF: (LSB of address): SPPpMCCC - (S = 1) swap 16k PROM at $8000 and $C000 from bank PPp
;                                         - (S = 0) swap 32k PROM at $8000 from bank PP
;                                         - swap 8k CROM at $0000
;                                         - mirror VERTICAL/HORIZONTAL (0/1)
____________________________________________________________________________________________

Mapper060:

    ; Increase bank number after each reset. (Stolen from Nestopia 1.08)
    ; Shouldn't this game have its own mapper number?
    if. D$Cartridge@PROMCRC32 = 0F9C484A0 ; Reset-triggered 4-in-1
        swap CROM, 8k,  00000, D$Game
        swap PROM, 16k, 08000, D$Game
        swap PROM, 16k, 0C000, D$Game
        inc D$Game | and D$Game 3
    else
        ; Set ports
        CPUWrite 08000, 0FFFF, @8000_FFFF
    endif
    ret

@8000_FFFF:

    ; xxxxMxxx
    ifNotFlag. edx 8
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; xxxxxCCC
    swap CROM, 8k, 00000, edx

    ; SPPpxxxx
    ifFlag. edx 080
        shr edx 4
        swap PROM, 16k, 08000, edx
        swap PROM, 16k, 0C000, edx
    else
        shr edx 5
        swap PROM, 32k, 08000, edx
    endif
    ret

____________________________________________________________________________________________
; Mapper #078, Jaleco (Cosmo Carrier)

; $8000-$FDFF: CCCCMPPP - swap 16k PROM at $8000
;                       - swap 8k  CROM at $0000
;                       - mirror $2000/$2400 (0/1)
;
; $FE00-$FFFF: CCCCxPPP - swap 16k PROM at $8000
;                       - swap 8k  CROM at $0000
____________________________________________________________________________________________

Mapper078:

    ; Memory mapping
    CPUWrite 08000, 0FDFF, @8000_FDFF
    CPUWrite 0FE00, 0FFFF, @FE00_FFFF

    ; Swap in last PROM bank at $C000
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_FDFF:

    ; xxxxMxxx
    ifFlag. eax 8
        mirror ONE_SCREEN_2400
    else
        mirror ONE_SCREEN_2000
    endif

@FE00_FFFF:

    ; CCCCPPPP
    mov edx eax | and eax 0F | shr edx 4
    swap PROM, 16k, 08000, eax
    swap CROM, 8k,  00000, edx
    ret

____________________________________________________________________________________________
; Mapper #015, 100-in-1

; $8000: RMPPPPPP - swap 16k PROM $8000
;                 - swap 16k PROM (+1) at $C000
;                 - (R = 1) --> swap ($8000 <--> $A000) and ($C000 <--> $E000)
;                 - mirror VERTICAL/HORIZONTAL (0/1)
; $8001: PPPPPPPP - swap 16k PROM at $8000
;                 - swap last 16k PROM bank at $C000
;                 - mirror VERTICAL
; $8002: PxPPPPPP - swap 8k PROM at $8000, $A000, $C000 and $E000 (Rotate value left first)
; $8003: RMPPPPPP - swap 16k PROM $C000
;                 - (R = 1) --> swap ($8000 <--> $C000)
;                 - mirror VERTICAL/HORIZONTAL (0/1)
____________________________________________________________________________________________

Mapper015:

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 08001, 08001, @8001
    CPUWrite 08002, 08002, @8002
    CPUWrite 08003, 08003, @8003
    ret

@8000:

    ifNotFlag. al 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ; Hack for SMB1...
    if al = 048, mirror VERTICAL

    rol al 1
    swap PROM, 8k, 08000, eax | xor eax 1
    swap PROM, 8k, 0A000, eax | xor eax 1 | add eax 2
    swap PROM, 8k, 0C000, eax | xor eax 1
    swap PROM, 8k, 0E000, eax | xor eax 1
    ret

@8001:

    ifNotFlag. al 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    rol al 1
    swap PROM, 8k, 08000, eax | xor eax 1
    swap PROM, 8k, 0A000, eax | xor eax 1 | or eax 0FE
    swap PROM, 8k, 0C000, eax | xor eax 1
    swap PROM, 8k, 0E000, eax | xor eax 1 | or eax 0FE
    ret

@8002:

    ifNotFlag. al 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    rol al 1
    swap PROM, 8k, 08000, eax
    swap PROM, 8k, 0A000, eax
    swap PROM, 8k, 0C000, eax
    swap PROM, 8k, 0E000, eax
    ret

@8003:

    ifNotFlag. al 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    rol al 1
    swap PROM, 8k, 08000, eax | xor eax 1
    swap PROM, 8k, 0A000, eax | xor eax 1
    swap PROM, 8k, 0C000, eax | xor eax 1
    swap PROM, 8k, 0E000, eax | xor eax 1
    ret
____________________________________________________________________________________________
; Mapper #089, Sunsoft

; $8000-$FFFF: CPPPMCCC - swap 16k PROM at $8000
;                       - swap 8k  CROM at $0000
;                       - mirror $2000/$2400 (0/1)
____________________________________________________________________________________________

Mapper089:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_FFFF:

    ; xxxxMxxx
    ifFlag. al 8
        mirror ONE_SCREEN_2400
    else
        mirror ONE_SCREEN_2000
    endif

    ; xPPPxxxx
    mov edx eax | shr edx 4
    swap PROM, 16k, 08000, edx

    ; CxxxxCCC
    ifFlag. eax 080
        and eax 7
        or  eax 8
    else
        and eax 7
    endif
    swap CROM, 8k, 00000, eax
    ret

____________________________________________________________________________________________
; Mapper #070

; $8000-$FFFF: MPPPCCCC - swap 8k  CROM at $0000
;                       - swap 16k PROM at $8000
;                       - mirror $2000/$2400 (0/1)
____________________________________________________________________________________________

Mapper070:

    ; Set ports
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Swap in last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_FFFF:

    ; Mxxxxxxx
    ifNotFlag. eax 080
        mirror ONE_SCREEN_2000
        if D$Cartridge@PROMCRC32 = 0A59C_A2EF, ; Kamen Rider Kurabu (J)
            mirror VERTICAL
        if D$Cartridge@PROMCRC32 = 010BB_8F9A, ; Family Trainer - Manhattan Police (J)
            mirror VERTICAL
    else
        mirror ONE_SCREEN_2400
        if D$Cartridge@PROMCRC32 = 0A59C_A2EF, ; Kamen Rider Kurabu (J)
            mirror HORIZONTAL
        if D$Cartridge@PROMCRC32 = 010BB_8F9A, ; Family Trainer - Manhattan Police (J)
            mirror HORIZONTAL
    endif

    ; xxxxCCCC
    swap CROM, 8k,  00000, eax

    ; xPPPxxxx
    shr eax 4
    swap PROM, 16k, 08000, eax
    ret

; ____________________________________________________________________________________________
;
;                                 Special mappers
;
;____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #185, CROM disable protect

; $8000-$FFFF: xxxxxxEE - CROM LengthEnabled/disabled (EE != 0 / EE = 0)
; [Hack for Spy vs Spy] - CROM LengthEnabled/disabled (xxExxxxE = $21 / xxExxxxE != $21
____________________________________________________________________________________________
[CROMDisabled Register]
Mapper185:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    PPURead 00000, 01FFF, @ReadDisabledCROM
    ret

@8000_FFFF:

    if. D$Cartridge@PROMCRC32 = 0B364_57C7 ; Spy vs Spy (J)
        ; xxExxxxE
        cmp eax 021 | setne B$CROMDisabled
    else
        ; xxxxxxEE
        test eax 3 | setz B$CROMDisabled
    endif
    ret

@ReadDisabledCROM:

    on B$CROMDisabled = &FALSE, ReadCROM
    mov eax 0FF
    ret

____________________________________________________________________________________________
; Mapper #013, CPROM

; $8000-$FFFF: xxPPxxCC - swap 4k  CRAM at $1000
;                       - swap 32k PROM at $8000
____________________________________________________________________________________________

Mapper013:

    ; Resize CRAM to 16k
    mov D$Cartridge@SizeCRAM 16

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; xxPPxxCC
    swap CRAM, 4k,  01000, eax | shr eax 4
    swap PROM, 32k, 08000, eax
    ret

____________________________________________________________________________________________
; Mapper #077, Irem

; $8000-$FFFF: CCCCxPPP - swap 32k PROM at $8000
;                       - swap 2k  CROM at $0000
____________________________________________________________________________________________

Mapper077:

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Enable CRAM for $0800-$1FFF
    mov D$Cartridge@SizeCRAM 8
    swap CRAM, 8k, 00000, 0
    PPURead  0800, 01FFF, ReadCRAM
    PPUWrite 0800, 01FFF, WriteCRAM

    ; Enable CROM for $0000-$07FF
    swap CROM, 2k, 00000, 0
    ret

@8000_FFFF:

    ; CCCCxPPP
    swap PROM, 32k, 08000, eax | shr eax 4
    swap CROM, 2k,  00000, eax
    ret

____________________________________________________________________________________________
TITLE IRQCounter
; ____________________________________________________________________________________________
;
;                               CPU cycle counters
;
; ____________________________________________________________________________________________
____________________________________________________________________________________________
; Mapper #083, Dragon Ball Z 4-in-1
; (Probably a bunch of different mappers jammed togehter)

; $5000-$5FFF: RRRRRRRR - Register
; $8000,$B000: --SSPPPP - select ROM segment
;                       - swap 16k PROM at $8000 (SSPPPP)
;                       - swap 16k PROM at $C000 (SS1111)
; $8100: i-----MM - IRQ mode
;                 - mirror VERTICAL/HORIZONTAL/$2000/$2400
; $8200: IIIIIIII - LSB of IRQ counter
; $8201: IIIIIIII - MSB of IRQ counter
;                 - IRQ disable/enable (i = 0/1)
; $8300: PPPPPPPP - swap 8k PROM at $8000
; $8301: PPPPPPPP - swap 8k PROM at $A000
; $8302: PPPPPPPP - swap 8k PROM at $C000
; $8310: CCCCCCCC - swap 1k CROM at $0000 (SSCCCCCCCC)
; $8311: CCCCCCCC - swap 1k CROM at $0400 (SSCCCCCCCC)
; $8312: CCCCCCCC - swap 1k CROM at $0800 (SSCCCCCCCC)
; $8313: CCCCCCCC - swap 1k CROM at $0C00 (SSCCCCCCCC)
; $8314: CCCCCCCC - swap 1k CROM at $1000 (SSCCCCCCCC)
; $8315: CCCCCCCC - swap 1k CROM at $1400 (SSCCCCCCCC)
; $8316: CCCCCCCC - swap 1k CROM at $1800 (SSCCCCCCCC)
; $8317: CCCCCCCC - swap 1k CROM at $1C00 (SSCCCCCCCC)
____________________________________________________________________________________________

Mapper083:

    setIRQCounter IRQCountDown, &NULL

    ; Memory mapping
    CPUWrite 05000, 05FFF, @5000_5FFF
    CPURead  05000, 05FFF, @Read5000_5FFF
    ; PROM segment
    CPUWrite 08000, 08000, @8000
    CPUWrite 0B000, 0B000, @8000
    ; Mirroring
    CPUWrite 08100, 08100, @8100
    ; IRQ counter
    CPUWrite 08200, 08200, @8200
    CPUWrite 08201, 08201, @8201
    ; 8k PROM
    CPUWrite 08300, 08300, @8300
    CPUWrite 08301, 08301, @8301
    CPUWrite 08302, 08302, @8302
    ; 1k CROM
    CPUWrite 08310, 08310, @8310
    CPUWrite 08311, 08311, @8311
    CPUWrite 08312, 08312, @8312
    CPUWrite 08313, 08313, @8313
    CPUWrite 08314, 08314, @8314
    CPUWrite 08315, 08315, @8315
    CPUWrite 08316, 08316, @8316
    CPUWrite 08317, 08317, @8317
    ; 2k CROM
    if. D$Cartridge@PROMCRC32 = 01461_D1F8 ; World heroes 2
        CPUWrite 08310, 08310, @8310_2k
        CPUWrite 08311, 08311, @8311_2k
        CPUWrite 08316, 08316, @8316_2k
        CPUWrite 08317, 08317, @8317_2k
    endif

    ; Last bank
    swap PROM, 16k, 0C000, 0F
    ret

; RRRRRRRR
@5000_5FFF:     mov D$Register+08 eax | ret
@Read5000_5FFF: mov eax D$Register+08 | ret

@8000:

    ; --SSPPPP
    swap PROM, 16k, 08000, eax
    and eax 030 | or eax 0F
    swap PROM, 16k, 0C000, eax

    ; --SS----
    and eax 030 | shl eax 4
    mov D$Register+00 eax
    ret

@8100:

    ; I-------
    test al 080 | setnz B$Register+04

    ; ------MM
    and al 03
    if al = 0, mirror VERTICAL
    if al = 1, mirror HORIZONTAL
    if al = 2, mirror ONE_SCREEN_2000
    if al = 3, mirror ONE_SCREEN_2400
    ret
____________________________________________________________________________________________

; PPPPPPPP
@8300: swap PROM, 8k, 08000, eax | ret
@8301: swap PROM, 8k, 0A000, eax | ret
@8302: swap PROM, 8k, 0C000, eax | ret
; CCCCCCCC
@8310:    or eax D$Register+00 | swap CROM, 1k,  0000, eax | ret
@8311:    or eax D$Register+00 | swap CROM, 1k,  0400, eax | ret
@8312:    or eax D$Register+00 | swap CROM, 1k,  0800, eax | ret
@8313:    or eax D$Register+00 | swap CROM, 1k,  0C00, eax | ret
@8314:    or eax D$Register+00 | swap CROM, 1k, 01000, eax | ret
@8315:    or eax D$Register+00 | swap CROM, 1k, 01400, eax | ret
@8316:    or eax D$Register+00 | swap CROM, 1k, 01800, eax | ret
@8317:    or eax D$Register+00 | swap CROM, 1k, 01C00, eax | ret
; CCCCCCCC
@8310_2k: or eax D$Register+00 | swap CROM, 2k,  0000, eax | ret
@8311_2k: or eax D$Register+00 | swap CROM, 2k,  0800, eax | ret
@8316_2k: or eax D$Register+00 | swap CROM, 2k, 01000, eax | ret
@8317_2k: or eax D$Register+00 | swap CROM, 2k, 01800, eax | ret
____________________________________________________________________________________________

@8200:

    call CartridgeSynchronize

    ; IIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    mov B$IRQCounter+00 al
    ret

@8201:

    call CartridgeSynchronize

    ; IIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    mov B$IRQCounter+01 al
    copy D$Register+04 D$IRQEnabled
    ret
____________________________________________________________________________________________
; Mapper #042, Mario Baby

; $6000-$7FFF: swappable PROM
; (mask $E003)
; $E000: ----PPPP - swap 8k ExROM at $6000
; $E001: ----M--- - mirror VERTICAL/HORIZONTAL (0/1)
; $E002: ------I- - disable/enable IRQ (0/1)

; $40xx: ????????
____________________________________________________________________________________________

Mapper042:

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL
    mov D$IRQLatch 06000

    ; Memory mapping
    CPURead  06000, 07FFF, @ReadExROM
    mov ecx 0E000
L0: push ecx

        and ecx 0E003
        mov eax WriteVoid

        if ecx = 0E000, mov eax @E000
        if ecx = 0E001, mov eax @E001
        if ecx = 0E002, mov eax @E002

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<

    ; Last bank
    swap PROM, 32k, 08000, LAST_BANK
    ret

@E000:

    ; ----PPPP
    and eax 0F | shl eax 3
    mov edx 0 | div D$Cartridge@SizePROM
    shl edx 10 | mov D$Register edx
    ret

@E001:

    ; ----M---
    ifNotFlag. al 08
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

@E002:

    call CartridgeSynchronize

    ; ------I-
    ClearIRQ IRQ_CARTRIDGE
    test al 02 | setnz B$IRQEnabled
    if B$IRQEnabled = &FALSE, copy D$IRQLatch D$IRQCounter
    ret
____________________________________________________________________________________________

@ReadExROM:

    and eax 01FFF
    add eax D$Register
    add eax D$pPROM
    movzx eax B$eax
    ret

____________________________________________________________________________________________
; Mapper #019, Namcot 106

; (mask $F800)
; $5000: IIIIIIII - 8 LSB of IRQ counter
; $5800: iIIIIIII - 7 MSB of IRQ counter, IRQ enable
; $8000: CCCCCCCC - swap 1k CROM at $0000
; $8800: CCCCCCCC - swap 1k CROM at $0400
; $9000: CCCCCCCC - swap 1k CROM at $0800
; $9800: CCCCCCCC - swap 1k CROM at $0C00
; $A000: CCCCCCCC - swap 1k CROM at $1000
; $A800: CCCCCCCC - swap 1k CROM at $1400
; $B000: CCCCCCCC - swap 1k CROM at $1800
; $B800: CCCCCCCC - swap 1k CROM at $1C00
; (CROM bank > $E0 --> use non-swappable CRAM)
; $C000: CCCCCCCC - swap 1k CiROM at $1000
; $C800: CCCCCCCC - swap 1k CiROM at $1400
; $D000: CCCCCCCC - swap 1k CiROM at $1800
; $D800: CCCCCCCC - swap 1k CiROM at $1C00
; (CiROM bank > $E0 --> use CiRAM based on bit 0)
; $E000: --PPPPPP - swap 8k PROM at $8000
; $E800: CcPPPPPP - swap 8k PROM at $A000
;                 - (c = 1) Force CROM at $0000
;                 - (C = 1) Force CROM at $1000
; $F000: --PPPPPP - swap 8k PROM at $C000
____________________________________________________________________________________________

Mapper019:

    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    ; Extra memory/extra sound
    CPUWrite 04800, 04FFF, @4800
    CPURead  04800, 04FFF, @Read4800
    CPUWrite 0F800, 0FFFF, @F800
    call Namco106Sound
    ; IRQ
    CPUWrite 05000, 057FF, @5000
    CPUWrite 05800, 05FFF, @5800
    CPURead  05000, 057FF, @Read5000
    CPURead  05800, 05FFF, @Read5800
    ; CROM
    CPUWrite 08000, 087FF, @8000
    CPUWrite 08800, 08FFF, @8800
    CPUWrite 09000, 097FF, @9000
    CPUWrite 09800, 09FFF, @9800
    CPUWrite 0A000, 0A7FF, @A000
    CPUWrite 0A800, 0AFFF, @A800
    CPUWrite 0B000, 0B7FF, @B000
    CPUWrite 0B800, 0BFFF, @B800
    ; CiROM
    on D$Cartridge@PROMCRC32 = 0B62A_7B71, S0> ; Family Circuit '91
    on D$Cartridge@PROMCRC32 = 01494_2C06, S0> ; Wagan Land 3
        CPUWrite 0C000, 0C7FF, @C000
        CPUWrite 0C800, 0CFFF, @C800
        CPUWrite 0D000, 0D7FF, @D000
        CPUWrite 0D800, 0DFFF, @D800
    S0:
    ; PROM
    CPUWrite 0E000, 0E7FF, @E000
    CPUWrite 0E800, 0EFFF, @E800
    CPUWrite 0F000, 0F7FF, @F000

    ; PPU settings
    if. D$Cartridge@SizeCROM > 0

        ; Override address line
        PPUWrite 00000, 01FFF, @WriteCROM
        PPURead  00000, 01FFF, @ReadCROM

        ; CiROM, but not for these two games
        on D$Cartridge@PROMCRC32 = 0B62A_7B71, S0> ; Family Circuit '91
        on D$Cartridge@PROMCRC32 = 01494_2C06, S0> ; Wagan Land 3
        PPUWrite 02000, 02FFF, @WriteCiROM
        PPURead  02000, 02FFF, @ReadCiROM

        ; Enable 8k CRAM
    S0: mov D$Cartridge@SizeCRAM 8
        swap CRAM, 8k, 00000, 0

        ; Set all banks to CROM
        mov D$Bank+00 01010101
        mov D$Bank+04 01010101

    endif

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; CCCCCCCC
@8000: swap CROM,  1k,  0000, eax | cmp al 0E0 | setb B$Bank+00 | ifFlag B$Register 040, mov B$Bank+00 &TRUE | ret
@8800: swap CROM,  1k,  0400, eax | cmp al 0E0 | setb B$Bank+01 | ifFlag B$Register 040, mov B$Bank+01 &TRUE | ret
@9000: swap CROM,  1k,  0800, eax | cmp al 0E0 | setb B$Bank+02 | ifFlag B$Register 040, mov B$Bank+02 &TRUE | ret
@9800: swap CROM,  1k,  0C00, eax | cmp al 0E0 | setb B$Bank+03 | ifFlag B$Register 040, mov B$Bank+03 &TRUE | ret
@A000: swap CROM,  1k, 01000, eax | cmp al 0E0 | setb B$Bank+04 | ifFlag B$Register 080, mov B$Bank+04 &TRUE | ret
@A800: swap CROM,  1k, 01400, eax | cmp al 0E0 | setb B$Bank+05 | ifFlag B$Register 080, mov B$Bank+05 &TRUE | ret
@B000: swap CROM,  1k, 01800, eax | cmp al 0E0 | setb B$Bank+06 | ifFlag B$Register 080, mov B$Bank+06 &TRUE | ret
@B800: swap CROM,  1k, 01C00, eax | cmp al 0E0 | setb B$Bank+07 | ifFlag B$Register 080, mov B$Bank+07 &TRUE | ret
; CCCCCCCC
@C000: swap CiROM, 1k, 02000, eax | cmp al 0E0 | setb B$Bank+08 | and al 01 | shl eax 10 | mov D$IndexCiRAM+00 eax | ret
@C800: swap CiROM, 1k, 02400, eax | cmp al 0E0 | setb B$Bank+09 | and al 01 | shl eax 10 | mov D$IndexCiRAM+04 eax | ret
@D000: swap CiROM, 1k, 02800, eax | cmp al 0E0 | setb B$Bank+0A | and al 01 | shl eax 10 | mov D$IndexCiRAM+08 eax | ret
@D800: swap CiROM, 1k, 02C00, eax | cmp al 0E0 | setb B$Bank+0B | and al 01 | shl eax 10 | mov D$IndexCiRAM+0C eax | ret
; PPPPPPPP
@E000: swap PROM,  8k, 08000, eax | ret; if (pRomCrc == 0x14942C06UL) // Wagan Land 3  ppu.SetMirroring( (data & 0x40) ? MIRROR_VERTICAL : MIRROR_ZERO );
@E800: mov B$Register al | swap PROM,  8k, 0A000, eax | ret
@F000: swap PROM,  8k, 0C000, eax | ret
____________________________________________________________________________________________

@WriteCROM:

    mov ebx edx | shr ebx 10
    on B$Bank+ebx = &FALSE, WritePlainCRAM
    ret

@WriteCiROM:

    mov ebx edx | shr ebx 10
    on B$Bank+ebx = &FALSE, WriteCiRAM
    ret

@ReadCROM:

    mov ebx eax | shr ebx 10
    on B$Bank+ebx = &FALSE, ReadPlainCRAM
    jmp ReadCROM

@ReadCiROM:

    mov ebx eax | shr ebx 10
    on B$Bank+ebx = &FALSE, ReadCiRAM
    jmp ReadCiROM
____________________________________________________________________________________________

[ExRAM CRAM+02000]
@F800: mov D$Register+04 eax | ret

@4800:

    mov edx D$Register+04
    and edx 07F | mov B$ExRAM+edx al
    ifFlag. B$Register+04 080
        inc B$Register+04
        or  B$Register+04 080
    endif

    on eax = 0329F_FF7A, S0> ; Battle Fleet
    on eax = 09653_3999, S0> ; Dokuganryuu Masamune
    on eax = 0429F_D177, S0> ; Famista '90
    on eax = 0DD45_4208, S0> ; Hydlide 3 - Yami Kara no Houmonsha
    on eax = 0B1B9_E187, S0> ; Kaijuu Monogatari
    on eax = 0AF15_338F, S0> ; Mindseeker
    call Namco106Sound@4800
S0: ret

@Read4800:

    mov eax D$Register+04
    and eax 07F | mov al B$ExRAM+eax
    ifFlag. B$Register+04 080
        inc B$Register+04
        or  B$Register+04 080
    endif
    ret

____________________________________________________________________________________________

@5000:

    call CartridgeSynchronize

    ; IIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    not al
    mov B$IRQCounter+00 al
    ret

@5800:

    call CartridgeSynchronize

    ; iIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    test al 080 | setnz B$IRQEnabled
    not al | and al 07F
    mov B$IRQCounter+01 al
    ret

@Read5000:

    call CartridgeSynchronize

    movzx eax B$IRQCounter+00 | not al
    ret

@Read5800:

    call CartridgeSynchronize

    movzx eax B$IRQCounter+01 | not al
    ret
____________________________________________________________________________________________
; Mapper #018, Jaleco SS8806

; $8000: xxxxPPPP (xxxx3210) - swap 8k PROM at $8000
; $8001: xxxxPPPP (xxxx7654) - swap 8k PROM at $8000
; $8002: xxxxPPPP (xxxx3210) - swap 8k PROM at $A000
; $8003: xxxxPPPP (xxxx7654) - swap 8k PROM at $A000
; $9000: xxxxPPPP (xxxx3210) - swap 8k PROM at $C000
; $9001: xxxxPPPP (xxxx7654) - swap 8k PROM at $C000

; $A000: xxxxCCCC (xxxx3210) - swap 1k CROM at $0000
; $A001: xxxxCCCC (xxxx7654) - swap 1k CROM at $0000
; $A002: xxxxCCCC (xxxx3210) - swap 1k CROM at $0400
; $A003: xxxxCCCC (xxxx7654) - swap 1k CROM at $0400
; $B000: xxxxCCCC (xxxx3210) - swap 1k CROM at $0800
; $B001: xxxxCCCC (xxxx7654) - swap 1k CROM at $0800
; $B002: xxxxCCCC (xxxx3210) - swap 1k CROM at $0C00
; $B003: xxxxCCCC (xxxx7654) - swap 1k CROM at $0C00
; $C000: xxxxCCCC (xxxx3210) - swap 1k CROM at $1000
; $C001: xxxxCCCC (xxxx7654) - swap 1k CROM at $1000
; $C002: xxxxCCCC (xxxx3210) - swap 1k CROM at $1400
; $C003: xxxxCCCC (xxxx7654) - swap 1k CROM at $1400
; $D000: xxxxCCCC (xxxx3210) - swap 1k CROM at $1800
; $D001: xxxxCCCC (xxxx7654) - swap 1k CROM at $1800
; $D002: xxxxCCCC (xxxx3210) - swap 1k CROM at $1C00
; $D003: xxxxCCCC (xxxx7654) - swap 1k CROM at $1C00

; $E000: xxxxIIII (xxxx3210) - 1st nibble of IRQ counter latch
; $E001: xxxxIIII (xxxx7654) - 2nd nibble of IRQ counter latch
; $E002: xxxxIIII (xxxxBA98) - 3rd nibble of IRQ counter latch
; $E003: xxxxIIII (xxxxFEDC) - 4th nibble of IRQ counter latch

; $F000: xxxxxxxx - Reset IRQ counter
; $F001: xxxxxxxI - disable/enable IRQ (0/1)
; $F002: xxxxxxMM - mirror HORIZONTAL/VERTICAL/$2000/$2000 (0/1/2/3)
____________________________________________________________________________________________

Mapper018:

    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        and ecx 0F003
        mov eax WriteVoid

        ; PROM
        if ecx = 08000, mov eax @8000
        if ecx = 08001, mov eax @8001
        if ecx = 08002, mov eax @8002
        if ecx = 08003, mov eax @8003
        if ecx = 09000, mov eax @9000
        if ecx = 09001, mov eax @9001
        ; CROM
        if ecx = 0A000, mov eax @A000
        if ecx = 0A001, mov eax @A001
        if ecx = 0A002, mov eax @A002
        if ecx = 0A003, mov eax @A003
        if ecx = 0B000, mov eax @B000
        if ecx = 0B001, mov eax @B001
        if ecx = 0B002, mov eax @B002
        if ecx = 0B003, mov eax @B003
        if ecx = 0C000, mov eax @C000
        if ecx = 0C001, mov eax @C001
        if ecx = 0C002, mov eax @C002
        if ecx = 0C003, mov eax @C003
        if ecx = 0D000, mov eax @D000
        if ecx = 0D001, mov eax @D001
        if ecx = 0D002, mov eax @D002
        if ecx = 0D003, mov eax @D003
        ; IRQ
        if ecx = 0E000, mov eax @E000
        if ecx = 0E001, mov eax @E001
        if ecx = 0E002, mov eax @E002
        if ecx = 0E003, mov eax @E003
        if ecx = 0F000, mov eax @F000
        if ecx = 0F001, mov eax @F001
        ; Mirroring
        if ecx = 0F002, mov eax @F002

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Set up banks
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; xxxxPPPP
@8000: and eax 0F |           | and D$Bank+00  0F0 | or D$Bank+00  eax | swap PROM, 8k, 08000, D$Bank+00 | ret
@8001: and eax 0F | shl eax 4 | and D$Bank+00   0F | or D$Bank+00  eax | swap PROM, 8k, 08000, D$Bank+00 | ret
@8002: and eax 0F |           | and D$Bank+04  0F0 | or D$Bank+04  eax | swap PROM, 8k, 0A000, D$Bank+04 | ret
@8003: and eax 0F | shl eax 4 | and D$Bank+04   0F | or D$Bank+04  eax | swap PROM, 8k, 0A000, D$Bank+04 | ret
@9000: and eax 0F |           | and D$Bank+08  0F0 | or D$Bank+08  eax | swap PROM, 8k, 0C000, D$Bank+08 | ret
@9001: and eax 0F | shl eax 4 | and D$Bank+08   0F | or D$Bank+08  eax | swap PROM, 8k, 0C000, D$Bank+08 | ret
;9002: and eax 0F |           | and D$Bank+0C  0F0 | or D$Bank+0C  eax | swap PROM, 8k, 0E000, D$Bank+0C | ret
;9003: and eax 0F | shl eax 4 | and D$Bank+0C   0F | or D$Bank+0C  eax | swap PROM, 8k, 0E000, D$Bank+0C | ret


; xxxxCCCC
@A000: and eax 0F |           | and D$Bank+010 0F0 | or D$Bank+010 eax | swap CROM, 1k,  0000, D$Bank+010 | ret
@A001: and eax 0F | shl eax 4 | and D$Bank+010  0F | or D$Bank+010 eax | swap CROM, 1k,  0000, D$Bank+010 | ret
@A002: and eax 0F |           | and D$Bank+014 0F0 | or D$Bank+014 eax | swap CROM, 1k,  0400, D$Bank+014 | ret
@A003: and eax 0F | shl eax 4 | and D$Bank+014  0F | or D$Bank+014 eax | swap CROM, 1k,  0400, D$Bank+014 | ret
@B000: and eax 0F |           | and D$Bank+018 0F0 | or D$Bank+018 eax | swap CROM, 1k,  0800, D$Bank+018 | ret
@B001: and eax 0F | shl eax 4 | and D$Bank+018  0F | or D$Bank+018 eax | swap CROM, 1k,  0800, D$Bank+018 | ret
@B002: and eax 0F |           | and D$Bank+01C 0F0 | or D$Bank+01C eax | swap CROM, 1k,  0C00, D$Bank+01C | ret
@B003: and eax 0F | shl eax 4 | and D$Bank+01C  0F | or D$Bank+01C eax | swap CROM, 1k,  0C00, D$Bank+01C | ret
@C000: and eax 0F |           | and D$Bank+020 0F0 | or D$Bank+020 eax | swap CROM, 1k, 01000, D$Bank+020 | ret
@C001: and eax 0F | shl eax 4 | and D$Bank+020  0F | or D$Bank+020 eax | swap CROM, 1k, 01000, D$Bank+020 | ret
@C002: and eax 0F |           | and D$Bank+024 0F0 | or D$Bank+024 eax | swap CROM, 1k, 01400, D$Bank+024 | ret
@C003: and eax 0F | shl eax 4 | and D$Bank+024  0F | or D$Bank+024 eax | swap CROM, 1k, 01400, D$Bank+024 | ret
@D000: and eax 0F |           | and D$Bank+028 0F0 | or D$Bank+028 eax | swap CROM, 1k, 01800, D$Bank+028 | ret
@D001: and eax 0F | shl eax 4 | and D$Bank+028  0F | or D$Bank+028 eax | swap CROM, 1k, 01800, D$Bank+028 | ret
@D002: and eax 0F |           | and D$Bank+02C 0F0 | or D$Bank+02C eax | swap CROM, 1k, 01C00, D$Bank+02C | ret
@D003: and eax 0F | shl eax 4 | and D$Bank+02C  0F | or D$Bank+02C eax | swap CROM, 1k, 01C00, D$Bank+02C | ret

@F002:

    ; xxxxxxMM
    and eax 3
    if eax = 0, mirror HORIZONTAL
    if eax = 1, mirror VERTICAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2000
    ret
____________________________________________________________________________________________

; xxxxIIII
@E000: call CartridgeSynchronize | and eax 0F | shl eax 00 | and D$IRQLatch 0FFF0 | or D$IRQLatch eax | ret
@E001: call CartridgeSynchronize | and eax 0F | shl eax 04 | and D$IRQLatch 0FF0F | or D$IRQLatch eax | ret
@E002: call CartridgeSynchronize | and eax 0F | shl eax 08 | and D$IRQLatch 0F0FF | or D$IRQLatch eax | ret
@E003: call CartridgeSynchronize | and eax 0F | shl eax 0C | and D$IRQLatch  0FFF | or D$IRQLatch eax | ret

@F000:

    call CartridgeSynchronize

    ; xxxxxxxI
    copy D$IRQLatch D$IRQCounter
    ret

@F001:

    call CartridgeSynchronize

    ; xxxxxxxI
    ClearIRQ IRQ_CARTRIDGE
    test al 01 | setnz B$IRQEnabled
    ret
____________________________________________________________________________________________
; Mapper #017, FFE F8xxx

; $42FE: xxxMxxxx - mirror $2000/$2400 (0/1)
; $42FF: xxxMxxxx - mirror VERTICAL/HORIZONTAL (0/1)
; $4501: xxxxxxxx - disable IRQ
; $4502: IIIIIIII - LSB of IRQ counter
; $4503: IIIIIIII - MSB of IRQ counter
; $4504: PPPPPPPP - swap 8k PROM at $8000
; $4505: PPPPPPPP - swap 8k PROM at $A000
; $4506: PPPPPPPP - swap 8k PROM at $C000
; $4507: PPPPPPPP - swap 8k PROM at $E000
; $4510: CCCCCCCC - swap 1k CROM at $0000
; $4511: CCCCCCCC - swap 1k CROM at $0400
; $4512: CCCCCCCC - swap 1k CROM at $0800
; $4513: CCCCCCCC - swap 1k CROM at $0C00
; $4514: CCCCCCCC - swap 1k CROM at $1000
; $4515: CCCCCCCC - swap 1k CROM at $1400
; $4516: CCCCCCCC - swap 1k CROM at $1800
; $4517: CCCCCCCC - swap 1k CROM at $1C00
____________________________________________________________________________________________

Mapper017:

    setIRQCounter IRQCountdown @IRQ

    ; Set ports
    CPUWrite 042FE, 042FE, @42FE
    CPUWrite 042FF, 042FF, @42FF
    CPUWrite 04500, 04500, @4500
    CPUWrite 04501, 04501, @4501
    CPUWrite 04502, 04502, @4502
    CPUWrite 04503, 04503, @4503
    CPUWrite 04504, 04504, @4504
    CPUWrite 04505, 04505, @4505
    CPUWrite 04506, 04506, @4506
    CPUWrite 04507, 04507, @4507
    CPUWrite 04510, 04510, @4510
    CPUWrite 04511, 04511, @4511
    CPUWrite 04512, 04512, @4512
    CPUWrite 04513, 04513, @4513
    CPUWrite 04514, 04514, @4514
    CPUWrite 04515, 04515, @4515
    CPUWrite 04516, 04516, @4516
    CPUWrite 04517, 04517, @4517

    ; Set IRQ counter length
    mov D@IRQLength 010000
    mov eax D$Cartridge@PROMCRC32
    if eax != 057BA_F095, ; Doki! Doki! Yuuenchi
    if eax != 0E641_38EC, ; Dragon Ball Z 2 - Gekishin Freeza!!
    if eax != 0C7A4_583E, ; Dragon Ball Z 3 - Ressen Jinzou Ningen
    if eax != 0FF2C_8EE4, ; Dynamite Batman
    if eax != 0CB7E_529D, ; SD Gundam Gaiden - Knight Gundam Monogatari 2 - Hikari no Kishi
    if eax != 08F3F_8B1F, ; Spartan X2
    if eax != 0A304_7263, ; Spartan X2
    if eax != 0C529_C604, ; Spartan X2
        mov D@IRQLength 0D000

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@4504: swap PROM, 8k, 08000, eax | ret
@4505: swap PROM, 8k, 0A000, eax | ret
@4506: swap PROM, 8k, 0C000, eax | ret
@4507: swap PROM, 8k, 0E000, eax | ret

; CCCCCCCC
@4510: swap CROM, 1k,  0000, eax | ret
@4511: swap CROM, 1k,  0400, eax | ret
@4512: swap CROM, 1k,  0800, eax | ret
@4513: swap CROM, 1k,  0C00, eax | ret
@4514: swap CROM, 1k, 01000, eax | ret
@4515: swap CROM, 1k, 01400, eax | ret
@4516: swap CROM, 1k, 01800, eax | ret
@4517: swap CROM, 1k, 01C00, eax | ret

@4500:
;outwrite
test al al | setnz B$IRQEnabled
    ret

@42FE:
;outwrite

    ; xxxMxxxx
    ifNotFlag. eax 010
        mirror ONE_SCREEN_2000
    else
        mirror ONE_SCREEN_2400
    endif
    ret

@42FF:
;outwrite

    ; xxxMxxxx
shr eax 4
and eax 3
if al = 2, mirror VERTICAL
if al = 3, mirror HORIZONTAL
if al = 0, mirror ONE_SCREEN_2000
if al = 1, mirror ONE_SCREEN_2400
ret

    ifNotFlag. eax 020
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

@4501:

    call CartridgeSynchronize

    ; Disable IRQs
    ClearIRQ IRQ_CARTRIDGE
    mov B$IRQEnabled &FALSE
    ret

@4502:

    call CartridgeSynchronize

    ; IIIIIIII
    mov B$IRQLatch al
    ret

@4503:

    call CartridgeSynchronize

    ; IIIIIIII
    mov B$IRQLatch+1 al

    ; Enable IRQs
    ClearIRQ IRQ_CARTRIDGE
    mov B$IRQEnabled &TRUE
    mov eax D@IRQLength | sub eax D$IRQLatch
    mov D$IRQCounter eax
    ret

@IRQ:

    ; IRQ!
    copy D@IRQLength D$IRQCounter
    mov D$IRQEnabled &FALSE
    SetIRQ Cartridge, IRQ_CARTRIDGE
    ret

[@IRQLength: ?]
____________________________________________________________________________________________
; Mapper #016, Bandai

; xxx0: CCCCCCCC - swap 1k  CROM at $0000
; xxx1: CCCCCCCC - swap 1k  CROM at $0400
; xxx2: CCCCCCCC - swap 1k  CROM at $0800
; xxx3: CCCCCCCC - swap 1k  CROM at $0C00
; xxx4: CCCCCCCC - swap 1k  CROM at $1000
; xxx5: CCCCCCCC - swap 1k  CROM at $1400
; xxx6: CCCCCCCC - swap 1k  CROM at $1800
; xxx7: CCCCCCCC - swap 1k  CROM at $1C00
; xxx8: PPPPPPPP - swap 16k PROM at $8000
; xxx9: xxxxxxMM - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; xxxA: xxxxxxxI - IRQ enable/disable (1/0)
; xxxB: IIIIIIII - LSB of IRQ counter latch
; xxxC: IIIIIIII - MSB of IRQ counter latch
; xxxD: DDDDDDDD - EEPROM (?)
____________________________________________________________________________________________

Mapper016:

    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    mov ecx 06000
L0: push ecx

        mov eax WriteVoid
        and ecx 0F

        [@WriteTable: @xxx0 @xxx1 @xxx2 @xxx3 @xxx4 @xxx5 @xxx6     @xxx7
                      @xxx8 @xxx9 @xxxA @xxxB @xxxC @xxxD WriteVoid WriteVoid]

        mov eax D@WriteTable+ecx*4

        ; Don't swap CRAM
        if ecx < 08,
            if D$Cartridge@SizeCROM = 0,
                mov eax WriteVoid

    pop ecx
    CPUWrite ecx, ecx, eax
    inc ecx | on ecx <= 0FFFF, L0<<

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; CCCCCCCC
@xxx0: swap CROM, 1k,   0000, eax | ret
@xxx1: swap CROM, 1k,   0400, eax | ret
@xxx2: swap CROM, 1k,   0800, eax | ret
@xxx3: swap CROM, 1k,   0C00, eax | ret
@xxx4: swap CROM, 1k,  01000, eax | ret
@xxx5: swap CROM, 1k,  01400, eax | ret
@xxx6: swap CROM, 1k,  01800, eax | ret
@xxx7: swap CROM, 1k,  01C00, eax | ret

; PPPPPPPP
@xxx8: swap PROM, 16k, 08000, eax | ret

@xxx9:

    ; xxxxxxMM
    and eax 3
    if eax = 0, mirror VERTICAL
    if eax = 1, mirror HORIZONTAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2400
    ret

@xxxA:

    call CartridgeSynchronize

    ; xxxxxxxI
    ClearIRQ IRQ_CARTRIDGE
    test al 01 | setnz B$IRQEnabled
    if B$IRQEnabled = &TRUE, copy D$IRQLatch D$IRQCounter
    ret

; IIIIIIII
@xxxB: call CartridgeSynchronize | mov B$IRQLatch+00 al | ret
@xxxC: call CartridgeSynchronize | mov B$IRQLatch+01 al | ret

; EEPROM (?)
@xxxD: ret

____________________________________________________________________________________________
; Mapper #050, SMB2j rev A

; $4020: xxxxxxxI - IRQ disabled/enabled (0/1)
; $4120: xxxxPPPP - swap 8k PROM at $C000
____________________________________________________________________________________________

Mapper050:

    setIRQCounter IRQCountdown, &NULL
    mov D$IRQLatch 01000

    ; Set ports
    CPURead  06000, 07FFF, @ReadWROM
    mov ecx 04020
L0: push ecx

        mov eax WriteVoid
        and ecx 0E160
        if ecx = 04020, mov eax @4020
        if ecx = 04120, mov eax @4120

    pop ecx
    CPUWrite ecx, ecx, eax
    inc ecx | on ecx < 06000, L0<

    ; Swap in PROM banks
    swap PROM, 32k, 08000, 2
    ret

@ReadWROM:

    ; Read from 8k PROM bank #F
    and eax 01FFF
    add eax (0F shl 13)
    add eax D$pPROM
    movzx eax B$eax
    ret

@4020:

    ; xxxxPPPP (xxxx3102)
    mov edx eax
    and edx 8                          ; 3xxx
    ifFlag eax 01, or edx 4            ; x2xx
    and eax 7 | shr eax 1 | or edx eax ; xx10
    swap PROM, 8k, 0C000, edx
    ret

@4120:

    call CartridgeSynchronize
    ClearIRQ IRQ_CARTRIDGE

    ; xxxxxxxI
    test al 01 | setnz B$IRQEnabled
    if B$IRQEnabled = &TRUE, mov D$IRQCounter 01000
    ret
____________________________________________________________________________________________
; Mapper #40, SMB2j

; $8000-$9FFF: xxxxxxxx - disable IRQ counter
; $A000-$BFFF: xxxxxxxx - enable IRQ counter
; $E000-$FFFF: xxxxxPPP - swap 8k PROM at $C000
____________________________________________________________________________________________

Mapper040:

    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    CPURead  06000, 07FFF, @ReadWROM
    CPUWrite 08000, 09FFF, @8000_9FFF
    CPUWrite 0A000, 0BFFF, @A000_BFFF
    CPUWrite 0E000, 0FFFF, @E000_FFFF
CPUWrite 04020, 04021, @4020_4021

    ; Swap PROM banks
    swap PROM, 32k, 08000, 1
    ret

; ????????
@4020_4021:ret

@ReadWROM:

    ; Read from 8k PROM bank #6
    add eax D$pPROM
    sub eax 06000
    movzx eax B$eax+0C000
    ret

@E000_FFFF:

    ; xxxxxPPP
    swap PROM, 8k, 0C000, eax
    ret

@8000_9FFF:

    call CartridgeSynchronize

    ; Disable IRQ
    ClearIRQ IRQ_CARTRIDGE
    mov D$IRQEnabled &FALSE
    ret

@A000_BFFF:

    call CartridgeSynchronize

    ; Enable IRQ
    ClearIRQ IRQ_CARTRIDGE
    mov D$IRQEnabled &TRUE
    mov D$IRQCounter 4096
    ret
____________________________________________________________________________________________
; Mapper #065, Irem H3001

; $8000: PPPPPPPP - swap 8k PROM at $8000
; $A000: PPPPPPPP - swap 8k PROM at $A000
; $C000: PPPPPPPP - swap 8k PROM at $C000
; $9000: xMxxxxxx - mirror HORIZONTAL/VERTICAL (0/1)
; $9003: Ixxxxxxx - disable/enable IRQ (0/1)
; $9004: xxxxxxxx - update IRQ counter
; $9005: IIIIIIII - high byte of IRQ latch
; $9006: IIIIIIII - low  byte of IRQ latch
; $B000: CCCCCCCC - swap 1k CROM at $0000
; $B001: CCCCCCCC - swap 1k CROM at $0400
; $B002: CCCCCCCC - swap 1k CROM at $0800
; $B003: CCCCCCCC - swap 1k CROM at $0C00
; $B004: CCCCCCCC - swap 1k CROM at $1000
; $B005: CCCCCCCC - swap 1k CROM at $1400
; $B006: CCCCCCCC - swap 1k CROM at $1800
; $B007: CCCCCCCC - swap 1k CROM at $1C00
____________________________________________________________________________________________

Mapper065:

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 0A000, 0A000, @A000
    CPUWrite 0C000, 0C000, @C000
    CPUWrite 09000, 09000, @9000
    CPUWrite 09003, 09003, @9003
    CPUWrite 09004, 09004, @9004
    CPUWrite 09005, 09005, @9005
    CPUWrite 09006, 09006, @9006
    CPUWrite 0B000, 0B000, @B000
    CPUWrite 0B001, 0B001, @B001
    CPUWrite 0B002, 0B002, @B002
    CPUWrite 0B003, 0B003, @B003
    CPUWrite 0B004, 0B004, @B004
    CPUWrite 0B005, 0B005, @B005
    CPUWrite 0B006, 0B006, @B006
    CPUWrite 0B007, 0B007, @B007

    ; IRQ counter and mirroring hack
    if. D$Cartridge@PROMCRC32 = 0E30B_7F64 ; Kaiketsu Yanchamaru 3
        CPUWrite 09000, 09004, WriteVoid
        CPUWrite 09001, 09001, @9001
        CPUWrite 09005, 09005, @Hack9005
        CPUWrite 09006, 09006, @Hack9006
    endif

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@8000: swap PROM, 8k, 08000, eax | ret
@A000: swap PROM, 8k, 0A000, eax | ret
@C000: swap PROM, 8k, 0C000, eax | ret

; CCCCCCCC
@B000: swap CROM, 1k,  0000, eax | ret
@B001: swap CROM, 1k,  0400, eax | ret
@B002: swap CROM, 1k,  0800, eax | ret
@B003: swap CROM, 1k,  0C00, eax | ret
@B004: swap CROM, 1k, 01000, eax | ret
@B005: swap CROM, 1k, 01400, eax | ret
@B006: swap CROM, 1k, 01800, eax | ret
@B007: swap CROM, 1k, 01C00, eax | ret

@9000:

    ; xMxxxxxx
    ifNotFlag. eax 040
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif
    ret

@9001:

    ; Mxxxxxxx
    ifNotFlag. eax 080
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

@9003:

    call CartridgeSynchronize

    ; Ixxxxxxx
    ClearIRQ IRQ_CARTRIDGE
    test eax 080
    setnz B$IRQEnabled
    ret

@9004:

    call CartridgeSynchronize

    ; xxxxxxxx
    copy D$IRQLatch D$IRQCounter
    ret

; IIIIIIII
@9005: call CartridgeSynchronize | mov B$IRQLatch+01 al | ret
@9006: call CartridgeSynchronize | mov B$IRQLatch+00 al | ret

@Hack9005:

    call CartridgeSynchronize

    ClearIRQ IRQ_CARTRIDGE
    test eax eax | setnz B$IRQEnabled
    shl al 1 | mov D$IRQCounter eax
    ret

@Hack9006:

    call CartridgeSynchronize

    mov B$IRQEnabled &TRUE
    ret
____________________________________________________________________________________________
; Mapper #069, Sunsoft FME-7

; $8000-$9FFF: xxxxNNNN - command number
; $A000-$BFFF: ABCDEFGH - (command = 0) swap 1k CROM at $0000
;                       - (command = 1) swap 1k CROM at $0400
;                       - (command = 2) swap 1k CROM at $0800
;                       - (command = 3) swap 1k CROM at $0C00
;                       - (command = 4) swap 1k CROM at $1000
;                       - (command = 5) swap 1k CROM at $1400
;                       - (command = 6) swap 1k CROM at $1800
;                       - (command = 7) swap 1k CROM at $1C00
;                       - (command = 8) swap 8k PROM at $6000
;                       - (command = 9) swap 8k PROM at $8000
;                       - (command = A) swap 8k PROM at $A000
;                       - (command = B) swap 8k PROM at $C000
;                       - (command = C) mirror VERTICAL/HORIZONTAL/$2000/$2400 (GH = 0/1/2/3)
;                       - (command = D) IRQ disabled/enabled (= 0 / != 0)
;                       - (command = E) low  byte of IRQ counter
;                       - (command = F) high byte of IRQ counter
; $C000-$DFFF: External sound
; $E000-$FFFF: External sound
____________________________________________________________________________________________

Mapper069:

    ; Sound
    call FME7Sound

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL
    mov D$IRQLatch 0FFFF

    ; Memory mapping
    CPUWrite 08000, 09FFF, @8000_9FFF
    CPUWrite 0A000, 0BFFF, @A000_BFFF
    CPUWrite 06000, 07FFF, @WriteWRAM
    CPURead  06000, 07FFF, @ReadWRAM

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_9FFF:

    ; xxxxNNNN
    and eax 0F
    mov D$Command eax
    ret

@A000_BFFF:

    mov edx D$Command

    ; CCCCCCCC
    if. dl = 00 | swap CROM, 1k,  0000, eax | else
    if. dl = 01 | swap CROM, 1k,  0400, eax | else
    if. dl = 02 | swap CROM, 1k,  0800, eax | else
    if. dl = 03 | swap CROM, 1k,  0C00, eax | else
    if. dl = 04 | swap CROM, 1k, 01000, eax | else
    if. dl = 05 | swap CROM, 1k, 01400, eax | else
    if. dl = 06 | swap CROM, 1k, 01800, eax | else
    if. dl = 07 | swap CROM, 1k, 01C00, eax | else

    ; PPPPPPPP
    if. dl = 08

        ifFlag.. eax 040

            ; Use WRAM
            ifFlag eax 080, mov B$WRAMEnabled &TRUE

        else..

            ; Use PROM
            mov B$WRAMEnabled &FALSE
            shl eax 3
            mov edx 0 | div D$Cartridge@SizePROM
            shl edx 10
            mov D$Bank edx

        endif..
    else
    if. dl = 09 | swap PROM, 8k, 08000, eax | else
    if. dl = 0A | swap PROM, 8k, 0A000, eax | else
    if. dl = 0B | swap PROM, 8k, 0C000, eax | else

    ; ------MM
    if. dl = 0C | and al 03
        if al = 0, mirror VERTICAL
        if al = 1, mirror HORIZONTAL
        if al = 2, mirror ONE_SCREEN_2000
        if al = 3, mirror ONE_SCREEN_2400
    else

    ; IIIIIIII
    if. dl = 0D | call CartridgeSynchronize | ClearIRQ IRQ_CARTRIDGE | test al al | setnz B$IRQEnabled | else
    if. dl = 0E | call CartridgeSynchronize | mov B$IRQCounter+0 al | else
    if. dl = 0F | call CartridgeSynchronize | mov B$IRQCounter+1 al | endif
    ret

@ReadWRAM:

    on B$WRAMEnabled = &TRUE, ReadWRAM
    sub eax 06000
    add eax D$Bank
    add eax D$pPROM
    movzx eax B$eax
    ret

@WriteWRAM:

    on B$WRAMEnabled = &TRUE, WriteWRAM
    ret
____________________________________________________________________________________________
; Mapper #067, Sunsoft #3

; $8800-$8FFF: CCCCCCCC - swap 2k CROM at $0000
; $9800-$9FFF: CCCCCCCC - swap 2k CROM at $0800
; $A800-$AFFF: CCCCCCCC - swap 2k CROM at $1000
; $B800-$BFFF: CCCCCCCC - swap 2k CROM at $1800
; $C000-$CFFF: IIIIIIII - high/low byte of IRQ counter (first/second write)
; $D800-$DFFF: xxxIxxxx - disable/enable IRQ (0/1)
; $E800-$EFFF: xxxxxxMM - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; $F800-$FFFF: PPPPPPPP - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper067:

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL
    mov D$IRQLatch 0FFFF

    ; Memory mapping
    CPUWrite 08800, 08FFF, @8800_8FFF
    CPUWrite 09800, 09FFF, @9800_9FFF
    CPUWrite 0A800, 0AFFF, @A800_AFFF
    CPUWrite 0B800, 0BFFF, @B800_BFFF
    CPUWrite 0C000, 0CFFF, @C000_CFFF
    CPUWrite 0D800, 0DFFF, @D800_DFFF
    CPUWrite 0E800, 0EFFF, @E800_EFFF
    CPUWrite 0F800, 0FFFF, @F800_FFFF

    ; Last banks
    swap PROM, 16k, 0C000, LAST_BANK
    swap CROM, 4k,  01000, LAST_BANK
    ret

; CCCCCCCC
@8800_8FFF: swap CROM, 2k,   0000, eax | ret
@9800_9FFF: swap CROM, 2k,   0800, eax | ret
@A800_AFFF: swap CROM, 2k,  01000, eax | ret
@B800_BFFF: swap CROM, 2k,  01800, eax | ret

@E800_EFFF:

    ; xxxxxxMM
    and eax 03
    if eax = 0, mirror VERTICAL
    if eax = 1, mirror HORIZONTAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2400
    ret

; PPPPPPPP
@F800_FFFF: swap PROM, 16k, 08000, eax | ret

@C000_CFFF:

    call CartridgeSynchronize

    ; Set up mask
    mov edx 0FF00
    xor B$Register &TRUE | jz F0>

    ; 1st write - high byte
    xchg al ah
    xchg dl dh

    ; IIIIIIII
F0: and D$IRQCounter edx
    or  D$IRQCounter eax
    ret

@D800_DFFF:

    call CartridgeSynchronize

    ; xxxIxxxx
    ClearIRQ IRQ_CARTRIDGE
    test al 010 | setnz B$IRQEnabled
    mov B$Register 0
    ret
____________________________________________________________________________________________
; Mapper #073, Konami VRC3

; $8000-$8FFF: xxxxIIII - bits 3210 of IRQ counter
; $9000-$9FFF: xxxxIIII - bits 7654 of IRQ counter
; $A000-$AFFF: xxxxIIII - bits BA98 of IRQ counter
; $B000-$BFFF: xxxxIIII - bits FEDC of IRQ counter
; $C000-$CFFF: xxxxxxIx - disable/enable IRQ (0/1)
; $F000-$FFFF: PPPPPPPP - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper073:

    setIRQCounter IRQCount, @IRQ
    mov D$IRQLatch 0

    ; Memory mapping
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 09000, 09FFF, @9000_9FFF
    CPUWrite 0A000, 0AFFF, @A000_AFFF
    CPUWrite 0B000, 0BFFF, @B000_BFFF
    CPUWrite 0C000, 0CFFF, @C000_CFFF
    CPUWrite 0D000, 0DFFF, @D000_DFFF
    CPUWrite 0F000, 0FFFF, @F000_FFFF

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; ????????
@D000_DFFF: ret

; PPPPPPPP
@F000_FFFF: swap PROM, 16k, 08000, eax | ret

; xxxxIIII
@8000_8FFF: and eax 0F |            | and D$IRQCounter 0FFF0 | or D$IRQCounter eax | ret
@9000_9FFF: and eax 0F | shl eax 4  | and D$IRQCounter 0FF0F | or D$IRQCounter eax | ret
@A000_AFFF: and eax 0F | shl eax 8  | and D$IRQCounter 0F0FF | or D$IRQCounter eax | ret
@B000_BFFF: and eax 0F | shl eax 12 | and D$IRQCounter  0FFF | or D$IRQCounter eax | ret

@C000_CFFF:

    call CartridgeSynchronize

    ; xxxxxxIx
    ClearIRQ IRQ_CARTRIDGE
    test al 02 | setnz B$IRQEnabled
    ret

@IRQ:

    ; Increase IRQ counter
    add D$IRQCounter eax
    on D$IRQCounter <= 0FFFF, Q0>

    ; Trigger IRQ
    mov D$IRQEnabled &FALSE
    copy D$IRQLatch D$IRQCounter
    SetIRQ Cartridge, IRQ_CARTRIDGE
Q0: ret

; ____________________________________________________________________________________________
;
;                             PPU read line counters
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #117, San Guo Zhi 4 - Chi Bi Feng Yun (Sangokushi 4)

; $8000: PPPPPPPP - swap 8k PROM at $8000
; $8001: PPPPPPPP - swap 8k PROM at $A000
; $8002: PPPPPPPP - swap 8k PROM at $C000
; $8003: PPPPPPPP - swap 8k PROM at $E000
; $A000: CCCCCCCC - swap 1k CROM at $0000
; $A001: CCCCCCCC - swap 1k CROM at $0400
; $A002: CCCCCCCC - swap 1k CROM at $0800
; $A003: CCCCCCCC - swap 1k CROM at $0C00
; $A004: CCCCCCCC - swap 1k CROM at $1000
; $A005: CCCCCCCC - swap 1k CROM at $1400
; $A006: CCCCCCCC - swap 1k CROM at $1800
; $A007: CCCCCCCC - swap 1k CROM at $1C00
; $C001: IIIIIIII - IRQ counter latch
; $C002: -------- - Clear IRQ line
; $C003: -------- - Reset IRQ
; $E000: -------I - disable/enable IRQ (0/1)
____________________________________________________________________________________________

Mapper117:

    setIRQCounter IRQHookPPU, @IRQ

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 08001, 08001, @8001
    CPUWrite 08002, 08002, @8002
    CPUWrite 08003, 08003, @8003
    CPUWrite 0A000, 0A000, @A000
    CPUWrite 0A001, 0A001, @A001
    CPUWrite 0A002, 0A002, @A002
    CPUWrite 0A003, 0A003, @A003
    CPUWrite 0A004, 0A004, @A004
    CPUWrite 0A005, 0A005, @A005
    CPUWrite 0A006, 0A006, @A006
    CPUWrite 0A007, 0A007, @A007
    CPUWrite 0C001, 0C001, @C001
    CPUWrite 0C002, 0C002, @C002
    CPUWrite 0C003, 0C003, @C003
    CPUWrite 0E000, 0E000, @E000

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@8000: swap PROM, 8k, 08000, eax | ret
@8001: swap PROM, 8k, 0A000, eax | ret
@8002: swap PROM, 8k, 0C000, eax | ret
@8003: swap PROM, 8k, 0E000, eax | ret
; CCCCCCCC
@A000: swap CROM, 1k,  0000, eax | ret
@A001: swap CROM, 1k,  0400, eax | ret
@A002: swap CROM, 1k,  0800, eax | ret
@A003: swap CROM, 1k,  0C00, eax | ret
@A004: swap CROM, 1k, 01000, eax | ret
@A005: swap CROM, 1k, 01400, eax | ret
@A006: swap CROM, 1k, 01800, eax | ret
@A007: swap CROM, 1k, 01C00, eax | ret


@C001:

    call CartridgeSynchronize

    ; IIIIIIII
    mov D$IRQLatch eax
    ret

@C002:

    call CartridgeSynchronize

    ; --------
    ClearIRQ IRQ_CARTRIDGE
    ret

@C003:

    call CartridgeSynchronize

    ; --------
    copy D$IRQLatch D$IRQCounter
    mov B$Register 01
    cmp D$Register 0101 | sete B$IRQEnabled
    ret

@E000:

    call CartridgeSynchronize

    ; -------I
    ClearIRQ IRQ_CARTRIDGE
    test al 01 | setnz B$Register+1
    cmp D$Register 0101 | sete B$IRQEnabled
    ret

@IRQ:

    mov B$Register 00
    mov B$IRQEnabled &FALSE
    copy D$IRQLatch D$IRQCounter
    SetIRQ PPU, IRQ_CARTRIDGE
    ret

____________________________________________________________________________________________
; Mapper #033, Taito TC0190

; $8000: xxPPPPPP - swap 8k PROM at $8000
; $8001: xxPPPPPP - swap 8k PROM at $A000
; $8002: CCCCCCCC - swap 2k CROM at $0000
; $8003: CCCCCCCC - swap 2k CROM at $0800
; $A000: CCCCCCCC - swap 1k CROM at $1000
; $A001: CCCCCCCC - swap 1k CROM at $1400
; $A002: CCCCCCCC - swap 1k CROM at $1800
; $A003: CCCCCCCC - swap 1k CROM at $1C00
; $C000: IIIIIIII - IRQ latch
; $C001: xxxxxxxx - Update IRQ counter
; $C002: xxxxxxxx - Disable IRQ
; $C003: xxxxxxxx - ENable IRQ
; $E000: xMxxxxxx - mirror VERTICAL/HORIZONTAL (0/1) (some games use $8000 for this)
____________________________________________________________________________________________

Mapper033:

    setIRQCounter IRQHookPPU, &NULL

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        mov eax WriteVoid
        and ecx 0F003
        if cx = 08000, mov eax @8000
        if cx = 08001, mov eax @8001
        if cx = 08002, mov eax @8002
        if cx = 08003, mov eax @8003
        if cx = 0A000, mov eax @A000
        if cx = 0A001, mov eax @A001
        if cx = 0A002, mov eax @A002
        if cx = 0A003, mov eax @A003
        if cx = 0C000, mov eax @C000
        if cx = 0C001, mov eax @C001
        if cx = 0C002, mov eax @C002
        if cx = 0C003, mov eax @C003
        if cx = 0E000, mov eax @E000

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000:

    ; xxPPPPPP
    swap PROM, 8k, 08000, eax
    if B$Register = &TRUE, ret
    ifNotFlag. eax 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

; xxPPPPPP
@8001: swap PROM, 8k, 0A000, eax | ret

; CCCCCCCC
@8002: swap CROM, 2k,  0000, eax | ret
@8003: swap CROM, 2k,  0800, eax | ret
@A000: swap CROM, 1k, 01000, eax | ret
@A001: swap CROM, 1k, 01400, eax | ret
@A002: swap CROM, 1k, 01800, eax | ret
@A003: swap CROM, 1k, 01C00, eax | ret

; IIIIIIII
@C000: neg al | jmp MMC3@IRQLatch
; xxxxxxxx
@C001: call MMC3@ResetIRQ | mov D$Divide42 35 | ret
@C002: jmp MMC3@EnableIRQ
@C003: jmp MMC3@DisableIRQ

@E000:

    ; xMxxxxxx
    mov B$Register &TRUE
    ifNotFlag. eax 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

____________________________________________________________________________________________
; Mapper #091, PC-HK-SF3

; $6000: CCCCCCCC - swap 2k CROM at $0000
; $6001: CCCCCCCC - swap 2k CROM at $0800
; $6002: CCCCCCCC - swap 2k CROM at $1000
; $6003: CCCCCCCC - swap 2k CROM at $1800

; $7000: PPPPPPPP - swap 8k PROM at $8000
; $7001: PPPPPPPP - swap 8k CROM at $A000
; $7002: IIIIIIII - IRQ counter
; $7003: IIIIIIII - Disable/Enable IRQ (0/1)
____________________________________________________________________________________________

Mapper091:

    setIRQCounter IRQHookPPU, &NULL

    ; Memory mapping
    mov ecx 06000
L0: push ecx

        mov eax WriteVoid
        and ecx 0F003

        if ecx = 06000, mov eax @6000
        if ecx = 06001, mov eax @6001
        if ecx = 06002, mov eax @6002
        if ecx = 06003, mov eax @6003
        if ecx = 07000, mov eax @7000
        if ecx = 07001, mov eax @7001
        if ecx = 07002, mov eax @7002
        if ecx = 07003, mov eax @7003

    pop ecx
    CPUWrite ecx, ecx, eax
    inc ecx | on ecx < 08000, L0<<

    ; Load last bank
    swap PROM, 16k, 08000, LAST_BANK
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; CCCCCCCC
@6000: swap CROM, 2k,  0000, eax | ret
@6001: swap CROM, 2k,  0800, eax | ret
@6002: swap CROM, 2k, 01000, eax | ret
@6003: swap CROM, 2k, 01800, eax | ret

; PPPPPPPP
@7000: swap PROM, 8k, 08000, eax | ret
@7001: swap PROM, 8k, 0A000, eax | ret

; IIIIIIII
@7002: mov D$IRQEnabled &FALSE | mov D$IRQCounter 7 | ClearIRQ IRQ_CARTRIDGE | ret
@7003: mov B$IRQEnabled &TRUE  | ret

; ____________________________________________________________________________________________
;
;                               Scanline counters
;
; ____________________________________________________________________________________________
____________________________________________________________________________________________
; Mappers #024 and #026, VRC6

; (mask $F003)
; $8000: PPPPPPPP - swap 16k PROM at $8000
; $A000: \
; $A001:  \
; $A002:   \ - - > VRC6 sound channels
; $B000:   /
; $B001:  /
; $B002: /
; $B003: xxxxMMxx - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; $C000: PPPPPPPP - swap 9k  PROM at $C000
; $D000: CCCCCCCC - swap 1k CROM at $0000
; $D001: CCCCCCCC - swap 1k CROM at $0400
; $D002: CCCCCCCC - swap 1k CROM at $0800
; $D003: CCCCCCCC - swap 1k CROM at $0C00
; $E000: CCCCCCCC - swap 1k CROM at $1000
; $E001: CCCCCCCC - swap 1k CROM at $1400
; $E002: CCCCCCCC - swap 1k CROM at $1800
; $E003: CCCCCCCC - swap 1k CROM at $1C00
; $F000: IIIIIIII - IRQ counter latch
; $F001: xxxxxxIi - disable/enable IRQ (I = 0/1)
; $F002: xxxxxxxx - disable/enable IRQ (i = 0/1)
____________________________________________________________________________________________

Mapper024:
Mapper026:

    call VRC6
    setIRQCounter IRQCountdown, @IRQ

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        ; Mapper #026 has A0 and A1 reversed
        if. D$Cartridge@Mapper = 26
            mov eax ecx   | and eax 01 | shl eax 1
            mov edx ecx   | and edx 02 | shr edx 1
            and ecx 0FFFC | or ecx eax | or ecx edx
        endif

        and ecx 0F003
        mov eax WriteVoid

        if ecx = 08000, mov eax @8000
        if ecx = 09000, mov eax VRC6@9000
        if ecx = 09001, mov eax VRC6@9001
        if ecx = 09002, mov eax VRC6@9002
        if ecx = 0A000, mov eax VRC6@A000
        if ecx = 0A001, mov eax VRC6@A001
        if ecx = 0A002, mov eax VRC6@A002
        if ecx = 0B000, mov eax VRC6@B000
        if ecx = 0B001, mov eax VRC6@B001
        if ecx = 0B002, mov eax VRC6@B002
        if ecx = 0B003, mov eax @B003
        if ecx = 0C000, mov eax @C000
        if ecx = 0D000, mov eax @D000
        if ecx = 0D001, mov eax @D001
        if ecx = 0D002, mov eax @D002
        if ecx = 0D003, mov eax @D003
        if ecx = 0E000, mov eax @E000
        if ecx = 0E001, mov eax @E001
        if ecx = 0E002, mov eax @E002
        if ecx = 0E003, mov eax @E003
        if ecx = 0F000, mov eax @F000
        if ecx = 0F001, mov eax @F001
        if ecx = 0F002, mov eax @F002

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@8000: swap PROM, 16k, 08000, eax | ret
@C000: swap PROM, 8k,  0C000, eax | ret

; CCCCCCCC
@D000: swap CROM, 1k,  0000, eax | ret
@D001: swap CROM, 1k,  0400, eax | ret
@D002: swap CROM, 1k,  0800, eax | ret
@D003: swap CROM, 1k,  0C00, eax | ret
@E000: swap CROM, 1k, 01000, eax | ret
@E001: swap CROM, 1k, 01400, eax | ret
@E002: swap CROM, 1k, 01800, eax | ret
@E003: swap CROM, 1k, 01C00, eax | ret

@B003:

    ; xxxxMMxx
    shr al 2 | and al 03
    if al = 0, mirror VERTICAL
    if al = 1, mirror HORIZONTAL
    if al = 2, mirror ONE_SCREEN_2000
    if al = 3, mirror ONE_SCREEN_2400
    ret

@F000:

    call CartridgeSynchronize

    ; IIIIIIII
    not al | inc eax
    mul D$Mul113 | div D$Div113
    mov D$IRQLatch eax
    ret

@F001:

    call CartridgeSynchronize

    ; xxxxxxIi
    ClearIRQ IRQ_CARTRIDGE
    test al 02 | setnz B$IRQEnabled
    test al 01 | setnz B$Register
    copy D$IRQLatch D$IRQCounter
    ret

@F002:

    call CartridgeSynchronize

    ; xxxxxxxx
    ClearIRQ IRQ_CARTRIDGE
    copy D$Register D$IRQEnabled
    ret

@IRQ:

;    mov B$IRQEnabled &FALSE
    mov eax D$IRQLatch
    add D$IRQCounter eax
    SetIRQ Cartridge, IRQ_CARTRIDGE
    ret
____________________________________________________________________________________________
; Mapper #085, VRC7
____________________________________________________________________________________________

Mapper085:

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        mov eax WriteVoid
        and ecx 0F038
        if cx = 08000, mov eax @8000
        if cx = 08008, mov eax @8010 | if cx = 08010, mov eax @8010
        if cx = 09000, mov eax @9000
    if cx = 09010, mov eax @9010
    if cx = 09030, mov eax @9030
        if cx = 0A000, mov eax @A000
        if cx = 0A008, mov eax @A010 | if cx = 0A010, mov eax @A010
        if cx = 0B000, mov eax @B000
        if cx = 0B008, mov eax @B010 | if cx = 0B010, mov eax @B010
        if cx = 0C000, mov eax @C000
        if cx = 0C008, mov eax @C010 | if cx = 0C010, mov eax @C010
        if cx = 0D000, mov eax @D000
        if cx = 0D008, mov eax @D010 | if cx = 0D010, mov eax @D010
        if cx = 0E000, mov eax @E000
        if cx = 0E008, mov eax @E010 | if cx = 0E010, mov eax @E010
        if cx = 0F000, mov eax @F000
        if cx = 0F008, mov eax @F010 | if cx = 0F010, mov eax @F010

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; ????????
@9010:
@9030:ret

; PPPPPPPP
@8000: swap PROM, 8k, 08000, eax | ret
@8010: swap PROM, 8k, 0A000, eax | ret
@9000: swap PROM, 8k, 0C000, eax | ret

; CCCCCCCC
@A000: swap CROM, 1k,  0000, eax | ret
@A010: swap CROM, 1k,  0400, eax | ret
@B000: swap CROM, 1k,  0800, eax | ret
@B010: swap CROM, 1k,  0C00, eax | ret
@C000: swap CROM, 1k, 01000, eax | ret
@C010: swap CROM, 1k, 01400, eax | ret
@D000: swap CROM, 1k, 01800, eax | ret
@D010: swap CROM, 1k, 01C00, eax | ret

@E000:

    ; xxxxxxMM
    and al 03
    if al = 0, mirror VERTICAL
    if al = 1, mirror HORIZONTAL
    if al = 2, mirror ONE_SCREEN_2000
    if al = 3, mirror ONE_SCREEN_2400
    ret

@E010:

    call CartridgeSynchronize

    ; IIIIIIII
    not al | inc eax
    mul D$Mul113 | div D$Div113
    mov D$IRQLatch eax
    ret

@F000:

    call CartridgeSynchronize

    ; xxxxxxEE
    ClearIRQ IRQ_CARTRIDGE
    and eax 03 | mov D$Register eax
    test eax 02 | setnz B$IRQEnabled
    if B$IRQEnabled = &TRUE, copy D$IRQLatch D$IRQCounter
    ret

@F010:

    call CartridgeSynchronize

    ; xxxxxxxx
    ClearIRQ IRQ_CARTRIDGE
    copy D$Register D$IRQEnabled
    ret

____________________________________________________________________________________________
; Mapper #025, Konami VRC4

; --> Mapper #021, with a different mask and slightly different PROM swapping
____________________________________________________________________________________________

Mapper025:

    ; Almost mapper #021
    call Mapper021

    ; Memory mapping
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 0A000, 0AFFF, Mapper021@A000

    mov edx 0F00F
    mov ecx 08000
L0: push ecx

        mov eax &NULL
        and ecx 0F00F
        if cx = 09000, mov eax Mapper021@9000
        if cx = 0B000, mov eax Mapper021@B000
        if cx = 0C000, mov eax Mapper021@C000
        if cx = 0D000, mov eax Mapper021@D000
        if cx = 0E000, mov eax Mapper021@E000
        if cx = 0F000, mov eax Mapper021@F000

        if cx = 09001, mov eax Mapper025@9004 | if cx = 09004, mov eax Mapper025@9004
        if cx = 0B001, mov eax Mapper021@B004 | if cx = 0B004, mov eax Mapper021@B004
        if cx = 0B002, mov eax Mapper021@B002 | if cx = 0B008, mov eax Mapper021@B002
        if cx = 0B003, mov eax Mapper021@B006 | if cx = 0B00C, mov eax Mapper021@B006
        if cx = 0C001, mov eax Mapper021@C004 | if cx = 0C004, mov eax Mapper021@C004
        if cx = 0C002, mov eax Mapper021@C002 | if cx = 0C008, mov eax Mapper021@C002
        if cx = 0C003, mov eax Mapper021@C006 | if cx = 0C00C, mov eax Mapper021@C006
        if cx = 0D001, mov eax Mapper021@D004 | if cx = 0D004, mov eax Mapper021@D004
        if cx = 0D002, mov eax Mapper021@D002 | if cx = 0D008, mov eax Mapper021@D002
        if cx = 0D003, mov eax Mapper021@D006 | if cx = 0D00C, mov eax Mapper021@D006
        if cx = 0E001, mov eax Mapper021@E004 | if cx = 0E004, mov eax Mapper021@E004
        if cx = 0E002, mov eax Mapper021@E002 | if cx = 0E008, mov eax Mapper021@E002
        if cx = 0E003, mov eax Mapper021@E006 | if cx = 0E00C, mov eax Mapper021@E006
        if cx = 0F001, mov eax Mapper021@F004 | if cx = 0F004, mov eax Mapper021@F004
        if cx = 0F002, mov eax Mapper021@F002 | if cx = 0F008, mov eax Mapper021@F002
        if cx = 0F003, mov eax Mapper021@F006 | if cx = 0F00C, mov eax Mapper021@F006

    pop ecx
    if eax != &NULL, CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    mov D$Bank+020 0
    mov D$Bank+024 LAST_BANK-1
    ret

@8000_8FFF:

    if. B$Register+04 = &FALSE
        mov D$Bank+020 eax
        swap PROM, 8k, 08000, eax
    else
        mov D$Bank+024 eax
        swap PROM, 8k, 0C000, eax
    endif
    ret

@9004:

    test eax 02 | setnz al
    xchg al B$Register+04
    if. al != B$Register+04
        push D$Bank+020 | push D$Bank+024
        pop  D$Bank+020 | pop  D$Bank+024
        swap PROM, 8k, 08000, D$Bank+020
        swap PROM, 8k, 0C000, D$Bank+024
    endif
    ret
____________________________________________________________________________________________
; Mapper #023, Konami VRC2 type B

; --> Mapper #021, with a different mask
____________________________________________________________________________________________

Mapper023:

    ; Almost mapper #021
    call Mapper021

    ; Address mask
    mov edx 0FFFF
    if D$Cartridge@PROMCRC32 = 09379_4634, mov edx 0F00C ; Akumajou Special - Boku Dracula Kun
    if D$Cartridge@PROMCRC32 = 0C782_9DAE, mov edx 0F00C ; Akumajou Special - Boku Dracula Kun (t.eng)
    if D$Cartridge@PROMCRC32 = 06A50_B553, mov edx 0F00C ; Akumajou Special - Boku Dracula Kun (b)
    if D$Cartridge@PROMCRC32 = 08C62_37FD, mov edx 0F00C ; Kaiketsu Yanchamaru 2 - Karakuri Land

    if D$Cartridge@PROMCRC32 = 0E4CE_EAD1, mov edx 0F00C ; Parodius

    ; Set ports
    mov ecx 08000
L0: push ecx

        mov eax &NULL
        and ecx edx

        if cx = 08000, mov eax Mapper021@8000 | if cx = 08004, mov eax Mapper021@8000
        if cx = 08008, mov eax Mapper021@8000 | if cx = 0800C, mov eax Mapper021@8000
;        if cx = 09008, mov eax Mapper021@9002
        if cx = 0A000, mov eax Mapper021@A000 | if cx = 0A004, mov eax Mapper021@A000
        if cx = 0A008, mov eax Mapper021@A000 | if cx = 0A00C, mov eax Mapper021@A000

        if cx = 0B001, mov eax Mapper021@B002 | if cx = 0B004, mov eax Mapper021@B002
        if cx = 0B002, mov eax Mapper021@B004 | if cx = 0B008, mov eax Mapper021@B004
        if cx = 0B003, mov eax Mapper021@B006 | if cx = 0B00C, mov eax Mapper021@B006
        if cx = 0C001, mov eax Mapper021@C002 | if cx = 0C004, mov eax Mapper021@C002
        if cx = 0C002, mov eax Mapper021@C004 | if cx = 0C008, mov eax Mapper021@C004
        if cx = 0C003, mov eax Mapper021@C006 | if cx = 0C00C, mov eax Mapper021@C006
        if cx = 0D001, mov eax Mapper021@D002 | if cx = 0D004, mov eax Mapper021@D002
        if cx = 0D002, mov eax Mapper021@D004 | if cx = 0D008, mov eax Mapper021@D004
        if cx = 0D003, mov eax Mapper021@D006 | if cx = 0D00C, mov eax Mapper021@D006
        if cx = 0E001, mov eax Mapper021@E002 | if cx = 0E004, mov eax Mapper021@E002
        if cx = 0E002, mov eax Mapper021@E004 | if cx = 0E008, mov eax Mapper021@E004
        if cx = 0E003, mov eax Mapper021@E006 | if cx = 0E00C, mov eax Mapper021@E006

        if cx = 09000, mov eax Mapper021@9000
        if cx = 09008, mov eax Mapper021@9002
        if cx = 0B000, mov eax Mapper021@B000
        if cx = 0C000, mov eax Mapper021@C000
        if cx = 0D000, mov eax Mapper021@D000
        if cx = 0E000, mov eax Mapper021@E000
        if cx = 0F000, mov eax Mapper021@F000
        if cx = 0F004, mov eax Mapper021@F002
        if cx = 0F008, mov eax Mapper021@F004
        if cx = 0F00C, mov eax Mapper021@F006

    pop ecx
    if eax != &NULL, CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<
    ret
____________________________________________________________________________________________
; Mapper #021, Konami VRC4 2B

; $8000:               PPPPPPPP - swap 8k PROM at $8000 or $C000
; $9000:               xxxxxxMM - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; $9002: $9080:        xxxxxxSx - ensable swapping for $8000/$C000 (0/1)
; $A000:               PPPPPPPP - swap 8k PROM at $C000
;        $B000:        xxxxCCCC (xxxx3210) - swap 1k CROM at $0000
;        $B002: $B040: xxxxCCCC (xxxx7654) - swap 1k CROM at $0000
; $B001: $B004: $B080: xxxxCCCC (xxxx3210) - swap 1k CROM at $0400
; $B003: $B006: $B0C0: xxxxCCCC (xxxx7654) - swap 1k CROM at $0400
;        $C000:        xxxxCCCC (xxxx3210) - swap 1k CROM at $0800
;        $C002: $C040: xxxxCCCC (xxxx7654) - swap 1k CROM at $0800
; $C001: $C004: $C080: xxxxCCCC (xxxx3210) - swap 1k CROM at $0C00
; $C003: $C006: $C0C0: xxxxCCCC (xxxx7654) - swap 1k CROM at $0C00
;        $D000:        xxxxCCCC (xxxx3210) - swap 1k CROM at $1000
;        $D002: $D040: xxxxCCCC (xxxx7654) - swap 1k CROM at $1000
; $D001: $D004: $D080: xxxxCCCC (xxxx3210) - swap 1k CROM at $1400
; $D003: $D006: $D0C0: xxxxCCCC (xxxx7654) - swap 1k CROM at $1400
;        $E000:        xxxxCCCC (xxxx3210) - swap 1k CROM at $1800
;        $E002: $E040: xxxxCCCC (xxxx7654) - swap 1k CROM at $1800
; $E001: $E004: $E080: xxxxCCCC (xxxx3210) - swap 1k CROM at $1C00
; $E003: $E006: $E0C0: xxxxCCCC (xxxx7654) - swap 1k CROM at $1C00

;        $F000:        xxxxIIII (xxxx3210) - IRQ counter latch
;        $F002: $F040: xxxxIIII (xxxx7654) - IRQ counter latch
; $F001: $F004: $F080: xxxxxxIi - i = IRQ enable latch
;                               - I = IRQ enable
; $F003: $F006: $F0C0: xxxxxxxx - disable/enable IRQ depending on IRQ enable latch
____________________________________________________________________________________________

Mapper021:

    setIRQCounter IRQCount, @IRQ

    ; Set ports
    mov ecx 08000
L0: push ecx

        and ecx 0F0CF
        mov eax WriteVoid

        if cx = 08000, mov eax @8000
        if cx = 09000, mov eax @9000
        if cx = 09002, mov eax @9002 | if ecx = 09080, mov eax @9002
        if cx = 0A000, mov eax @A000

        if cl = 040, mov cl 02
        if cl = 01,  mov cl 04
        if cl = 080, mov cl 04
        if cl = 03,  mov cl 06
        if cl = 0C0, mov cl 06

        if cx = 0B000, mov eax @B000
        if cx = 0B002, mov eax @B002
        if cx = 0B004, mov eax @B004
        if cx = 0B006, mov eax @B006
        if cx = 0C000, mov eax @C000
        if cx = 0C002, mov eax @C002
        if cx = 0C004, mov eax @C004
        if cx = 0C006, mov eax @C006
        if cx = 0D000, mov eax @D000
        if cx = 0D002, mov eax @D002
        if cx = 0D004, mov eax @D004
        if cx = 0D006, mov eax @D006
        if cx = 0E000, mov eax @E000
        if cx = 0E002, mov eax @E002
        if cx = 0E004, mov eax @E004
        if cx = 0E006, mov eax @E006

        if cx = 0F000, mov eax @F000
        if cx = 0F002, mov eax @F002
        if cx = 0F004, mov eax @F004
        if cx = 0F006, mov eax @F006

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Load last PROM bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000:

    ; PPPPPPPP
    if. B$Register+04 = &FALSE
        swap PROM, 8k, 08000, eax
    else
        swap PROM, 8k, 0C000, eax
    endif
    ret

; PPPPPPPP
@A000: swap PROM, 8k, 0A000, eax | ret

@9000:

    ; xxxxxxMM
    and eax 03
    if eax = 0, mirror VERTICAL
    if eax = 1, mirror HORIZONTAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2400
    ret

@9002:

    ; xxxxxxSx
    test eax 02 | setnz B$Register+04
    ret

; xxxxCCCC
@B000: and eax 0F |           | and D$Bank+00  0F0 | or D$Bank+00  eax | swap CROM, 1k,  0000, D$Bank+00  | ret
@B002: and eax 0F | shl eax 4 | and D$Bank+00  0F  | or D$Bank+00  eax | swap CROM, 1k,  0000, D$Bank+00  | ret
@B004: and eax 0F |           | and D$Bank+04  0F0 | or D$Bank+04  eax | swap CROM, 1k,  0400, D$Bank+04  | ret
@B006: and eax 0F | shl eax 4 | and D$Bank+04  0F  | or D$Bank+04  eax | swap CROM, 1k,  0400, D$Bank+04  | ret
@C000: and eax 0F |           | and D$Bank+08  0F0 | or D$Bank+08  eax | swap CROM, 1k,  0800, D$Bank+08  | ret
@C002: and eax 0F | shl eax 4 | and D$Bank+08  0F  | or D$Bank+08  eax | swap CROM, 1k,  0800, D$Bank+08  | ret
@C004: and eax 0F |           | and D$Bank+0C  0F0 | or D$Bank+0C  eax | swap CROM, 1k,  0C00, D$Bank+0C  | ret
@C006: and eax 0F | shl eax 4 | and D$Bank+0C  0F  | or D$Bank+0C  eax | swap CROM, 1k,  0C00, D$Bank+0C  | ret
@D000: and eax 0F |           | and D$Bank+010 0F0 | or D$Bank+010 eax | swap CROM, 1k, 01000, D$Bank+010 | ret
@D002: and eax 0F | shl eax 4 | and D$Bank+010 0F  | or D$Bank+010 eax | swap CROM, 1k, 01000, D$Bank+010 | ret
@D004: and eax 0F |           | and D$Bank+014 0F0 | or D$Bank+014 eax | swap CROM, 1k, 01400, D$Bank+014 | ret
@D006: and eax 0F | shl eax 4 | and D$Bank+014 0F  | or D$Bank+014 eax | swap CROM, 1k, 01400, D$Bank+014 | ret
@E000: and eax 0F |           | and D$Bank+018 0F0 | or D$Bank+018 eax | swap CROM, 1k, 01800, D$Bank+018 | ret
@E002: and eax 0F | shl eax 4 | and D$Bank+018 0F  | or D$Bank+018 eax | swap CROM, 1k, 01800, D$Bank+018 | ret
@E004: and eax 0F |           | and D$Bank+01C 0F0 | or D$Bank+01C eax | swap CROM, 1k, 01C00, D$Bank+01C | ret
@E006: and eax 0F | shl eax 4 | and D$Bank+01C 0F  | or D$Bank+01C eax | swap CROM, 1k, 01C00, D$Bank+01C | ret

; xxxxIIII
@F000: and eax 0F |           | and D$IRQLatch 0F0 | or D$IRQLatch eax | ret
@F002: and eax 0F | shl eax 4 | and D$IRQLatch  0F | or D$IRQLatch eax | ret

@F004:

    call CartridgeSynchronize

    ; xxxxxxIi
    ClearIRQ IRQ_CARTRIDGE
    test eax 01 | setnz B$Register
    test eax 02 | setnz B$IRQEnabled
    if. B$IRQEnabled = &TRUE
        mov eax D$IRQLatch
        not al
        mul D$Mul113 | div D$Div113
        mov D$IRQCounter eax
    endif
    ret

@F006:

    call CartridgeSynchronize

    ; xxxxxxxx
    ClearIRQ IRQ_CARTRIDGE
    copy D$Register D$IRQEnabled
    ret

@IRQ:

    ; Count
    sub D$IRQCounter eax | jns Q0>
    ; IRQ latch
    mov eax D$IRQLatch
    not al
    mul D$Mul113 | div D$Div113
    mov D$IRQCounter eax
    ; IRQ
    SetIRQ Cartridge, IRQ_CARTRIDGE
Q0: ret

; ____________________________________________________________________________________________
;
;                               Other IRQ counters
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #222, Dragon Ninja

; $8000: PPPPPPPP - swap 8k PROM at $8000
; $A000: PPPPPPPP - swap 8k PROM at $A000
; $B000: CCCCCCCC - swap 1k CROM at $0000
; $B002: CCCCCCCC - swap 1k CROM at $0400
; $C000: CCCCCCCC - swap 1k CROM at $0800
; $C002: CCCCCCCC - swap 1k CROM at $0C00
; $D000: CCCCCCCC - swap 1k CROM at $1000
; $D002: CCCCCCCC - swap 1k CROM at $1400
; $E000: CCCCCCCC - swap 1k CROM at $1800
; $E002: CCCCCCCC - swap 1k CROM at $1C00
; $F00x: Some IRQ thing
____________________________________________________________________________________________

Mapper222:

    setIRQCounter IRQCountdown, &NULL

    ; Set ports
CPUWrite 05000, 05000, @5000
    mov ecx 08000
L0: push ecx

        and ecx 0F003
        mov eax WriteVoid
        if ecx = 08000, mov eax @8000
        if ecx = 09000, mov eax @9000
        if ecx = 09002, mov eax @9002
        if ecx = 0A000, mov eax @A000
        if ecx = 0B000, mov eax @B000
        if ecx = 0B002, mov eax @B002
        if ecx = 0C000, mov eax @C000
        if ecx = 0C002, mov eax @C002
        if ecx = 0D000, mov eax @D000
        if ecx = 0D002, mov eax @D002
        if ecx = 0E000, mov eax @E000
        if ecx = 0E002, mov eax @E002
        if ecx = 0F000, mov eax @F000
        if ecx = 0F002, mov eax @F002

ifFlag ecx 1, mov eax @1

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Swap in last bank
    swap PROM, 16k, 0C000, LAST_BANK

    ; Vertical mirroring
    mirror VERTICAL
@1:
    ret

@8000: swap PROM, 8k, 08000, eax | ret
@A000: swap PROM, 8k, 0A000, eax | ret
@B000: swap CROM, 1k,  0000, eax | ret
@B002: swap CROM, 1k,  0400, eax | ret
@C000: swap CROM, 1k,  0800, eax | ret
@C002: swap CROM, 1k,  0C00, eax | ret
@D000: swap CROM, 1k, 01000, eax | ret
@D002: swap CROM, 1k, 01400, eax | ret
@E000: swap CROM, 1k, 01800, eax | ret
@E002: swap CROM, 1k, 01C00, eax | ret

@9002:

@9000:

    ret

@F000:
@F002:

    call CartridgeSynchronize

    ; IIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    not al
    mul D$Mul113 | div D$Div113
    mov D$IRQCounter eax
    ret

@5000:

    call CartridgeSynchronize

    test al al | setnz B$IRQEnabled
    ret

____________________________________________________________________________________________
; Mapper #006, FFE F4xxx
____________________________________________________________________________________________

Mapper006:

    ; 32k of CRAM
    if D$Cartridge@SizeCROM = 0, mov D$Cartridge@SizeCRAM 32

    ; Memory mapping
    CPUWrite 042FE, 042FE, @42FE
    CPUWrite 042FF, 042FF, @42FF
    CPUWrite 04501, 04501, @4501
    CPUWrite 04502, 04502, @4502
    CPUWrite 04503, 04503, @4503
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Set PROM bank
    swap PROM, 16k, 0C000, 7
    ret

@42FE:
    ifFlag. eax 010
        mirror ONE_SCREEN_2400
    else
        mirror ONE_SCREEN_2000
    endif
    ret

@42FF:
    ifFlag. eax 010
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif
    ret

@8000_FFFF:

    mov edx eax | and eax 03 | shr edx 2 | and edx 0F
    swap PROM, 16k, 08000, edx
    swap CROM, 8k,  00000, eax
    ret

@4501:
@4502:
@4503:

    ; Some ROM tries to use the IRQ counter (haven't found one that does yet)
    ret
____________________________________________________________________________________________
TITLE MMC
;____________________________________________________________________________________________
;
;                                   Nintendo MMC1-MMC6
;
;____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #001, Nintendo MMC1
____________________________________________________________________________________________

Mapper001:

    ; Memory mapping
    CPURead  06000, 07FFF, @ReadWRAM
    CPUWrite 06000, 07FFF, @WriteWRAM
    CPUWrite 08000, 0FFFF, @WriteRegs
mov B$WRAMEnabled &TRUE

    ; Set up registers
    mov B$BitCount 0
    mov D$Register 0C ; Reset all regs and set #1 to $C ($8000/16k)

    ; Set up banks
    call @UpdateMirroring
    call @UpdateCROM
    call @UpdatePROM
    ret

@WriteRegs:

    ; Reg number changed since last write?
    shr edx 13 | and edx 3
    if. dl != B$LastRegister
        mov B$BitCount 0
        mov B$LastRegister dl
    endif

    ; Reset?
    ifFlag. al 080
        mov B$BitCount 0
        or  B$Register+0 0C
        call @UpdateMirroring
        call @UpdateCROM
        call @UpdatePROM
        ret
    endif

    ; Save written bit
    shr al 1 | rcr B$Latch 1

    ; Five bits yet?
    inc B$BitCount
    if. B$BitCount >= 5

        mov B$BitCount 0

        ; Save register
        mov al B$Latch | shr al 3
        mov B$Register+edx al

        ; Change MMC setup
        if dl = 0, call @UpdateMirroring
        call @UpdateCROM
        call @UpdatePROM

    endif
    ret

@UpdateMirroring:

    movzx eax B$Register+0
    and eax MMC1_MIRRORING
    if eax = 0, mirror ONE_SCREEN_2000
    if eax = 1, mirror ONE_SCREEN_2400
    if eax = 2, mirror VERTICAL
    if eax = 3, mirror HORIZONTAL
    ret

@UpdateCROM:

    movzx eax B$Register+1
    movzx edx B$Register+2

    ; Swap it!
    ifFlag. B$Register+0 MMC1_SWITCH_4K
        swap CROM, 4k, 00000, eax
        swap CROM, 4k, 01000, edx
    else
        shr eax 1
        swap CROM, 8k, 00000, eax
    endif
    ret

@UpdatePROM:

    ; $8000 --> ecx, $C000 --> edx
    ifNotFlag. B$Register+0 MMC1_SWITCH_16K

        ; Switch 32k
        movzx ecx B$Register+3 | and ecx 0E
        move  edx ecx+1

    else

        ; Switch $8000
        movzx ecx B$Register+3     ; Selected bank --> $8000
        mov   edx LAST_BANK         ; Last     bank --> $C000

        ; Switch $C000
        ifNotFlag B$Register+0 MMC1_SWITCH_8000, mov   ecx 0               ; First    bank --> $8000
        ifNotFlag B$Register+0 MMC1_SWITCH_8000, movzx edx B$Register+3    ; Selected bank --> $C000

    endif

    ; Adjust bank numbers
    and ecx 0F
    and edx 0F
    ifFlag. B$Register+1 010
        or ecx 010
        or edx 010
    endif

    ; Swap
    swap PROM, 16k, 08000, ecx
    swap PROM, 16k, 0C000, edx

    ; WRAM enable hack
    if D$Cartridge@PROMCRC32 = 081DF_3D70, ret ; Tatakae!! Rahmen Man - Sakuretsu Choujin 102 Gei (J) [b2]

    ; WRAM Enabled?
    test B$Register+3 MMC1_WRAM_DISABLE | setz B$WRAMEnabled
    ret

@ReadWRAM:

    on B$WRAMEnabled = &TRUE, ReadWRAM
    jmp ReadVoid

@WriteWRAM:

    on B$WRAMEnabled = &TRUE, WriteWRAM
    jmp WriteVoid

[MMC1_MIRRORING     3
 MMC1_SWITCH_8000   4
 MMC1_SWITCH_C000   0
 MMC1_SWITCH_16K    8
 MMC1_SWITCH_32K    0
 MMC1_SWITCH_4K     010
 MMC1_WRAM_DISABLE  010]
[BitCount      Register+04
 LastRegister  Register+08]
____________________________________________________________________________________________
; Mapper #155, MMC1 without WRAM toggle (The Money Game (J))
____________________________________________________________________________________________

Mapper155:

    call Mapper001
    CPURead  06000, 07FFF, ReadWRAM
    CPUWrite 06000, 07FFF, WriteWRAM
    ret

____________________________________________________________________________________________
; Mapper #009, Nintendo MMC2 (Punch-out!)
____________________________________________________________________________________________

Mapper009:

    ; Monitor read line
    PPURead 0FD8,  0FDF,  @0FD0
    PPURead 0FE8,  0FEF,  @0FE0
    PPURead 01FD8, 01FDF, @1FD0
    PPURead 01FE8, 01FEF, @1FE0

    ; Set ports
    CPUWrite 0A000, 0AFFF, @A000_AFFF
    CPUWrite 0B000, 0BFFF, @B000_BFFF
    CPUWrite 0C000, 0CFFF, @C000_CFFF
    CPUWrite 0D000, 0DFFF, @D000_DFFF
    CPUWrite 0E000, 0EFFF, @E000_EFFF
    CPUWrite 0F000, 0FFFF, @F000_FFFF

    ; Load PROM
    swap PROM, 8k,  08000, 0
    swap PROM, 8k,  0A000, LAST_BANK-2
    swap PROM, 8k,  0C000, LAST_BANK-1
    swap PROM, 8k,  0E000, LAST_BANK

    ; Registers
    mov D$Register+00 0
    mov D$Register+04 0
    mov D$Register+08 1
    mov D$Register+0C 1
    mov D$Latch+00 0FD
    mov D$Latch+04 0FD
    ret

@A000_AFFF:

    ; PPPPPPPP
    swap PROM, 8k, 08000, eax
    ret

; CCCCCCCC
@B000_BFFF: mov D$Register+00 eax | if D$Latch+00 = 0FD, swap CROM, 4k, 00000, D$Register+00 | ret
@C000_CFFF: mov D$Register+04 eax | if D$Latch+00 = 0FE, swap CROM, 4k, 00000, D$Register+04 | ret
@D000_DFFF: mov D$Register+08 eax | if D$Latch+04 = 0FD, swap CROM, 4k, 01000, D$Register+08 | ret
@E000_EFFF: mov D$Register+0C eax | if D$Latch+04 = 0FE, swap CROM, 4k, 01000, D$Register+0C | ret

@F000_FFFF:

    ; xxxxxxxM
    ifNotFlag. eax 1
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

@0FD0: call ReadCROM | if D$Latch+00 != 0FD, swap CROM, 4k, 00000, D$Register+00 | mov D$Latch+00 0FD | ret
@0FE0: call ReadCROM | if D$Latch+00 != 0FE, swap CROM, 4k, 00000, D$Register+04 | mov D$Latch+00 0FE | ret
@1FD0: call ReadCROM | if D$Latch+04 != 0FD, swap CROM, 4k, 01000, D$Register+08 | mov D$Latch+04 0FD | ret
@1FE0: call ReadCROM | if D$Latch+04 != 0FE, swap CROM, 4k, 01000, D$Register+0C | mov D$Latch+04 0FE | ret
____________________________________________________________________________________________
; Mapper #004, Nintendo MMC3
____________________________________________________________________________________________

Mapper004: call MMC3 | ret

____________________________________________________________________________________________
; Mapper #010, Nintendo MMC4 - almost identical to MMC2
____________________________________________________________________________________________

Mapper010:

    ; Almost identical to MMC2
    call Mapper009

    ; Set ports
    CPUWrite 0A000, 0AFFF, @A000_AFFF

    ; Load PROM
    swap PROM, 16k, 08000, 0
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@A000_AFFF:
    swap PROM, 16k, 08000, eax
    ret
____________________________________________________________________________________________
TITLE MMC3

[MMC3_SWAP_PROM 040
 MMC3_SWAP_CROM 080]

MMC3:

    setIRQCounter IRQHookPPU, &NULL

    ; Hooks
    mov D$fnCROMUpdate @UpdatePROM
    mov D$fnPROMUpdate @UpdateCROM
    if D$Cartridge@SizeCROM = 0, mov D$fnCROMUpdate @UpdateCRAM

    ; - Memory mapping -

    ; MMC6B games don't toggle WRAM in the same way. (???)
    on D$Cartridge@PROMCRC32 = 0BEB8_8304, S0> ; Startropics (U)
    on D$Cartridge@PROMCRC32 = 0AC74_DD5C, S0> ; Startropics (E)
    on D$Cartridge@PROMCRC32 = 080FB_117E, S0> ; Startropics 2 - Zoda's Revenge (U)

    ; WRAM toggle
    CPUWrite 06000, 07FFF, @WriteWRAM
    CPURead  06000, 07FFF, @ReadWRAM

S0: mov ecx 08000
L0: push ecx

        and ecx 0E001
        if ecx = 08000, mov eax @Command
        if ecx = 08001, mov eax @Bankswitch
        if ecx = 0A000, mov eax @Mirroring
        if ecx = 0A001, mov eax @WRAMEnable
        if ecx = 0C000, mov eax @IRQLatch
        if ecx = 0C001, mov eax @ResetIRQ
        if ecx = 0E000, mov eax @DisableIRQ
        if ecx = 0E001, mov eax @EnableIRQ

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Swap last 16k PROM bank to $C000
    swap PROM, 16k, 0C000, LAST_BANK

    ; Reset banks
    mov D$Bank+00  0
    mov D$Bank+04  1
    mov D$Bank+08  4
    mov D$Bank+0C  5
    mov D$Bank+010 6
    mov D$Bank+014 7
    mov D$Bank+018 0
    mov D$Bank+01C 1

    mov B$WRAMEnabled &TRUE
    mov D$IRQCounter 0FF
    mov D$IRQLatch 0FF
    ret

____________________________________________________________________________________________

@Command:

    ; Save command
    mov D$Command eax
    call D$fnPROMUpdate
    call D$fnCROMUpdate
    ret

@Bankswitch:

    ; Set bank
    mov edx D$Command | and edx 7
    if edx <= 1, shr eax 1
    mov D$Bank+edx*4 eax

    ; Update banks
    call D$fnPROMUpdate
    call D$fnCROMUpdate
    ret

@Mirroring:

    ifFlag. eax 1
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif
    ret

@WRAMEnable:

    test al 080 | setnz B$WRAMEnabled
    ret
____________________________________________________________________________________________

@IRQLatch:

    call PPUSynchronize

    ; Hack for some games that lock up
    if eax = 240, inc eax

    mov D$IRQLatch eax

copy D$ResetIRQ D$Latched
xor D$Latched &TRUE
if D$ResetIRQ = &TRUE,
    mov D$IRQCounter eax
    ret

@ResetIRQ:

    call PPUSynchronize

    copy D$IRQLatch D$IRQCounter
    mov D$Divide42 33
    mov D$A13 1
    ret

@DisableIRQ:

    call PPUSynchronize
    ClearIRQ IRQ_CARTRIDGE

    mov B$IRQEnabled &FALSE
mov B$ResetIRQ &TRUE
    ret

@EnableIRQ:

    call PPUSynchronize

 mov B$IRQEnabled &TRUE

if D$Latched = &TRUE
    copy D$IRQLatch D$IRQCounter
    ret
____________________________________________________________________________________________

@UpdatePROM:

    ; $8000 <--> $C000 swap?
    ifNotFlag. D$Command MMC3_SWAP_PROM
        swap PROM, 8k, 08000, D$Bank+018
        swap PROM, 8k, 0C000, LAST_BANK-1
    else
        swap PROM, 8k, 0C000, D$Bank+018
        swap PROM, 8k, 08000, LAST_BANK-1
    endif

    ; $A000 and $E000
    swap PROM, 8k, 0A000, D$Bank+01C
    swap PROM, 8k, 0E000, LAST_BANK
    ret

@UpdateCROM:

    ; $0000 <--> $1000 swap?
    mov edx 0
    ifFlag D$Command MMC3_SWAP_CROM, mov edx 01000

    ; Set banks
    swap CROM, 2k, edx, D$Bank+00  | add edx 0800
    swap CROM, 2k, edx, D$Bank+04  | sub edx 0800
    xor edx 01000
    swap CROM, 1k, edx, D$Bank+08  | add edx 0400
    swap CROM, 1k, edx, D$Bank+0C  | add edx 0400
    swap CROM, 1k, edx, D$Bank+010 | add edx 0400
    swap CROM, 1k, edx, D$Bank+014
    ret

@UpdateCRAM:

    ; No bank switch, just acknowledge $0000 <--> $1000 swap
    mov edx 0
    ifFlag D$Command MMC3_SWAP_CROM, mov edx 01000
    swap CRAM, 4k, edx, 0 | xor edx 01000
    swap CRAM, 4k, edx, 1
    ret

____________________________________________________________________________________________

@ReadWRAM:

    on B$WRAMEnabled = &TRUE, ReadWRAM
    jmp ReadVoid

@WriteWRAM:

    on B$WRAMEnabled = &TRUE, WriteWRAM
    jmp WriteVoid
____________________________________________________________________________________________
; Mapper #115, Yuu Yuu Hakusho Final

; $6000: override PROM settings
____________________________________________________________________________________________

Mapper115:

    ; MMC3
    call MMC3

    ; Memory mapping
    CpuWrite 06000, 06000, @6000
    CPUWrite 06001, 07FFF, @6001_7FFF
    CPURead  06000, 07FFF, ReadVoid
    ret

@6000:

    mov D$Register eax

@6001_7FFF:

    ; Refresh PROM
    call MMC3@UpdatePROM
    ifFlag. B$Register 080
        mov eax D$Register | and eax 07
        swap PROM, 16k, 08000, eax
    endif
    ret
____________________________________________________________________________________________
; Mapper #249, ????????

; Encrypted MMC3
; $5000: ------S- - (S = 1) Convert bank numbers
____________________________________________________________________________________________

Mapper249:

    ; MMC3
    call MMC3

    ; Memory mapping
    CPUWrite 05000, 05000, @5000
S0: mov ecx 08000
L0: push ecx

        and ecx 0F701
        if ecx = 08000, mov eax @8000
        if ecx = 08001, mov eax @8001
        if ecx = 0A000, mov eax MMC3@Mirroring
        if ecx = 0A001, mov eax MMC3@WRAMEnable
        if ecx = 0C000, mov eax MMC3@IRQLatch
        if ecx = 0C001, mov eax MMC3@ResetIRQ
        if ecx = 0E000, mov eax MMC3@DisableIRQ
        if ecx = 0E001, mov eax MMC3@EnableIRQ

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<
    ret

@5000: test al 02 | setnz B$Register | ret
@8000: and eax 07 | jmp MMC3@Command

@8001:

    on B$Register = &FALSE, MMC3@Bankswitch

    if. B$Command >= 6
        if.. eax >= 020
            sub eax 020
            jmp S0>>
        endif..
    endif

    ; Convert #1
    mov edx 0
    ifFlag al 01,  or dl (1 shl 0)
    ifFlag al 02,  or dl (1 shl 1)
    ifFlag al 04,  or dl (1 shl 5)
    ifFlag al 08,  or dl (1 shl 2)
    ifFlag al 010, or dl (1 shl 6)
    ifFlag al 020, or dl (1 shl 7)
    ifFlag al 040, or dl (1 shl 4)
    ifFlag al 080, or dl (1 shl 3)
    mov eax edx
    jmp MMC3@BankSwitch

    ; Convert #2
S0: mov edx 0
    ifFlag eax 01,  or edx (1 shl 0)
    ifFlag eax 02,  or edx (1 shl 3)
    ifFlag eax 04,  or edx (1 shl 4)
    ifFlag eax 08,  or edx (1 shl 2)
    ifFlag eax 010, or edx (1 shl 1)
    mov eax edx
    jmp MMC3@Bankswitch
____________________________________________________________________________________________
; Mapper #198, ????????

; Has extra RAM at $4019-$7FFF
; No IRQ counter
____________________________________________________________________________________________

Mapper198:

    ; MMC3
    call MMC3

    ; Memory mapping
    CPUWrite 04019, 05FFF, @4019_5FFF
    CPUWrite 06000, 07FFF, @6000_7FFF
    CPURead  04019, 05FFF, @Read4019_5FFF
    CPURead  06000, 07FFF, @Read6000_7FFF
    mov ecx 08001
L0: CPUWrite ecx, ecx, @8001
    add ecx 02 | on ecx < 0A000, L0<
    CPUWrite 0C000, 0FFFF, WriteVoid
    ret

[@ExRAM WRAM+01000]

@4019_5FFF:

    sub edx 04019
    mov B@ExRAM+edx al
    ret

@Read4019_5FFF:

    sub eax 04019
    movzx eax B@ExRAM+eax
    ret

@6000_7FFF:

    and edx 0FFF
    mov B$WRAM+edx al
    ret

@Read6000_7FFF:

    and eax 0FFF
    movzx eax B$WRAM+eax
    ret

@8001:

    call MMC3@BankSwitch
    if D$Bank+018 >= 050, mov D$Bank+018 04F
    jmp D$fnPROMUpdate

____________________________________________________________________________________________
; Mapper #248, Bao Qing Tian

; $6000-$7FFF: override PROM settings
____________________________________________________________________________________________

Mapper248:

    ; MMC3
    call MMC3
    mov D$fnPROMUpdate @UpdatePROM

    ; Memory mapping
    CPUWrite 06000, 07FFF, @6000_7FFF
    CPURead  06000, 07FFF, ReadVoid
    mov ecx 08000
L0: CPUWrite ecx, ecx, @8000
    add ecx 2
    on ecx < 0A000, L0<
    ret

@6000_7FFF:

    mov D$Register eax
    jmp @UpdatePROM

@8000: and eax 07 | jmp MMC3@Command
____________________________________________________________________________________________

@UpdatePROM:

    ifFlag. B$Register 080
        mov eax D$Register | and eax 0F
        swap PROM, 16k, 08000, eax
    else
        and D$Bank+018 01F | swap PROM, 8k, 08000, D$Bank+018
        and D$Bank+01C 01F | swap PROM, 8k, 0A000, D$Bank+01C
    endif
    ret
____________________________________________________________________________________________
; Mapper #114, Lion King

; $6000-$7FFF: override PROM settings
; Uses a little different registers
____________________________________________________________________________________________

Mapper114:

    ; IRQ
    call MMC3
    mov D$fnPROMUpdate @UpdatePROM

    ; Memory mapping
    CPUWrite 06000, 07FFF, @6000_7FFF
    CPURead  06000, 07FFF, ReadVoid
    CPUWrite 08000, 09FFF, @8000_9FFF
    CPUWrite 0A000, 0BFFF, @A000_BFFF
    CPUWrite 0C000, 0DFFF, @C000_DFFF
    CPUWrite 0E000, 0FFFF, WriteVoid
    CPUwrite 0E002, 0E002, @E002
    CPUwrite 0E003, 0E003, @E003

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    mov D$Bank+018 0
    mov D$Bank+01C 1
    ret

@6000_7FFF:

    mov D$Register eax
    jmp @UpdatePROM

@8000_9FFF:

    jmp MMC3@Mirroring

@A000_BFFF:

    [@Table: B$ 0 3 1 5 6 7 2 4]
    mov B$Latch &TRUE
    and eax 07
    movzx eax B@Table+eax
    jmp MMC3@Command

@C000_DFFF:

    if B$Latch = &FALSE, ret
    mov B$Latch &FALSE
    jmp MMC3@Bankswitch

@E002:

    call PPUSynchronize

    ClearIRQ IRQ_CARTRIDGE
    ret

@E003:

    call PPUSynchronize

    test al al | setnz B$IRQEnabled
    mov D$IRQCounter eax
    ret
____________________________________________________________________________________________

@UpdatePROM:

    ifFlag. B$Register 080
        mov eax D$Register
        and eax 01F
        swap PROM, 16k, 08000, eax
    else
        swap PROM, 8k,  08000, D$Bank+018
        swap PROM, 8k,  0A000, D$Bank+01C
    endif
    ret
____________________________________________________________________________________________
; Mapper #47, NES-QJ

; $6000-$7FFF: Set PROM/CROM masks
____________________________________________________________________________________________

Mapper047:

    ; MMC3
    call MMC3
    mov D$fnPROMUpdate @UpdatePROM
    mov D$fnCROMUpdate @UpdateCROM

    ; Memory mapping
    CPUWrite 06000, 07FFF, @6000_7FFF
    CPURead  06000, 07FFF, ReadVoid
    call @UpdatePROM
    call @UpdateCROM
    ret

@6000_7FFF:

    if. D$Cartridge@PROMCRC32 = 07EEF_434C ; 3-in-1 (E) [!]
        shr al 1
    else
        shl al 1
    endif
    and eax 03
    mov D$Register eax
    call @UpdatePROM
    call @UpdateCROM
    ret

@UpdatePROM:

    mov ecx D$Register | shl ecx 3
    mov edx 0F
    if D$Cartridge@PROMCRC32 = 07EEF_434C, ; 3-in-1 (E) [!]
        if D$Register != 2,
            mov edx 7

    ; $8000 and $C000
    ifNotFlag. B$Command MMC3_SWAP_PROM
        mov eax D$Bank+018  |               or eax ecx | swap PROM, 8k, 08000, eax
        mov eax LAST_BANK-1 | and eax edx | or eax ecx | swap PROM, 8k, 0C000, eax
    else
        mov eax D$Bank+018  |               or eax ecx | swap PROM, 8k, 0C000, eax
        mov eax LAST_BANK-1 | and eax edx | or eax ecx | swap PROM, 8k, 08000, eax
    endif

    ; $A000 and $E000
    mov eax D$Bank+01C |             | or eax ecx | swap PROM, 8k, 0A000, eax
    mov eax LAST_BANK  | and eax edx | or eax ecx | swap PROM, 8k, 0E000, eax
    ret

@UpdateCROM:

    mov ecx D$Register | shr ecx 1 | shl ecx 7

    mov ebx 0
    ifFlag B$Command MMC3_SWAP_CROM, mov ebx 01000
    mov eax D$Bank+00  | shl eax 1 |         | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+00  | shl eax 1 | inc eax | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+04  | shl eax 1 |         | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+04  | shl eax 1 | inc eax | or eax ecx | swap CROM, 1k, ebx, eax | sub ebx 0C00 | xor ebx 01000
    mov eax D$Bank+08  |                     | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+0C  |                     | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+010 |                     | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+014 |                     | or eax ecx | swap CROM, 1k, ebx, eax
    ret

____________________________________________________________________________________________
; Mapper #187, Street Fighter Zero
____________________________________________________________________________________________

Mapper187:

    ; MMC3
    call MMC3
    mov D$fnPROMUpdate @UpdatePROM

    ; Memory mapping
    CPUWrite 05000, 07FFF, @5000_7FFF
    CPURead  05000, 07FFF, @Read5000_7FFF

    CPUWrite 08004, 09FFF, WriteVoid
    CPUWrite 0A004, 0BFFF, WriteVoid
    CPUWrite 0C004, 0DFFF, WriteVoid
    CPUWrite 0E004, 0FFFF, WriteVoid
    CPUWrite 08000, 08000, @8000
    CPUWrite 08001, 08001, @8001
    CPUWrite 08003, 08003, @8003

    ; Last 32k PROM
    mov D@Bank8000 LAST_BANK-3
    mov D@BankA000 LAST_BANK-2
    mov D@BankC000 LAST_BANK-1
    mov D@BankE000 LAST_BANK-0
    call @UpdatePROM
    ret

[@Bank8000 Bank+018
 @BankA000 Bank+01C
 @BankC000 Bank+020
 @BankE000 Bank+024

 @ExBank   Bank+028]

@5000_7FFF:
;outwrite
    mov B$Latch+04 al
    if edx > 05000, ret
    mov B$Mode al

    ifFlag. al 080

        ifFlag.. al 020

            and eax 01E | shl eax 1
            mov D@Bank8000 eax | inc eax
            mov D@BankA000 eax | inc eax
            mov D@BankC000 eax | inc eax
            mov D@BankE000 eax

        else..

            and eax 01F | shl eax 1
            mov D@BankC000 eax | inc eax
            mov D@BankE000 eax

        endif..

    else

        copy D@ExBank+00 D@Bank8000
        copy D@ExBank+04 D@BankA000
        mov D@BankC000 LAST_BANK-1
        mov D@BankE000 LAST_BANK

    endif

    call @UpdatePROM
    ret

@Read5000_7FFF:
;outhex eax

    mov dl B$Latch+04 | and dl 03
    if dl = 0, mov eax 083
    if dl = 1, mov eax 083
    if dl = 2, mov eax 042
    if dl = 3, mov eax 000
    ret
____________________________________________________________________________________________

@8000:
;outwrite

    mov D$Latch &FALSE
    mov B$Command al
    ret

@8001:
;outwrite

    mov edx D$Command | and edx 07

    if edx = 6, mov D@ExBank+00 eax
    if edx = 7, mov D@ExBank+04 eax

    if.. B$Latch = &TRUE

        if B$Command = 02A, mov D@BankA000  0F
        if B$Command = 028, mov D@BankC000 017
        jmp @UpdatePROM

    else..

        ; CROM
        if. edx < 2
            or eax 0100
            shr eax 1
        endif
        if. edx < 6
            mov D$Bank+edx*4 eax
            jmp D$fnPROMUpdate
        endif

        ; PROM
        mov ebx D$Mode
        and ebx 0A0
        if. ebx != 0A0
            if edx = 6, mov D@Bank8000 eax
            if edx = 7, mov D@BankA000 eax
            jmp @UpdatePROM
        endif

    endif..
    ret

@8003:
;outwrite

    mov B$Latch &TRUE
    mov B$Command al
    ifNotFlag. al 0F0
        mov D@BankC000 LAST_BANK-1
        call @UpdatePROM
    endif
    ret

@UpdatePROM:

    swap PROM, 8k, 08000, D@Bank8000
    swap PROM, 8k, 0A000, D@BankA000
    swap PROM, 8k, 0C000, D@BankC000
    swap PROM, 8k, 0E000, D@BankE000
    ret
____________________________________________________________________________________________
; Mapper #052, Mario 7-in-1 (MMC3)

; $6000-$7FFF: Set PROM/CROM masks
____________________________________________________________________________________________

Mapper052:

    ; MMC3
    call MMC3
    mov D$fnPROMUpdate @UpdatePROM
    mov D$fnCROMUpdate @UpdateCROM

    ; Memory mapping
    CPUWrite 06000, 07FFF, @6000_7FFF

    call @UpdatePROM
    call @UpdateCROM
    ret

@6000_7FFF:

    call MMC3@WriteWRAM

    ; Not written yet?
    if. D$Register+04 = &FALSE
        mov D$Register+04 &TRUE
        mov D$Register eax
        call @UpdatePROM
        call @UpdateCROM
    endif
    ret

@UpdatePROM:

    ; Masks
    ifNotFlag. B$Register 08
        mov edx 01F
        mov ecx D$Register | and ecx 06
    else
        mov edx 0F
        mov ecx D$Register | and ecx 07
    endif
    shl ecx 4

    ; $8000 and $C000
    ifNotFlag. B$Command MMC3_SWAP_PROM
        mov eax D$Bank+018  | and eax edx | or eax ecx | swap PROM, 8k, 08000, eax
        mov eax LAST_BANK-1 | and eax edx | or eax ecx | swap PROM, 8k, 0C000, eax
    else
        mov eax D$Bank+018  | and eax edx | or eax ecx | swap PROM, 8k, 0C000, eax
        mov eax LAST_BANK-1 | and eax edx | or eax ecx | swap PROM, 8k, 08000, eax
    endif

    ; $A000 and $E000
    mov eax D$Bank+01C | and eax edx | or eax ecx | swap PROM, 8k, 0A000, eax
    mov eax LAST_BANK  | and eax edx | or eax ecx | swap PROM, 8k, 0E000, eax
    ret

@UpdateCROM:

    ; ecx = 128k bank
    mov ecx 0
    ifFlag B$Register 020, or ecx 04
    ifFlag B$Register  04, or ecx 02
    ifFlag B$Register 010,
    ifFlag B$Register 040, or ecx 01
    shl ecx 7
    mov edx 0FF
    ifFlag B$Register 040, mov edx 07F

    ; CROM swapped?
    mov ebx 0
    ifFlag B$Command MMC3_SWAP_CROM, mov ebx 01000

    ; Set banks
    mov eax D$Bank+00  | shl eax 1 |         | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+00  | shl eax 1 | inc eax | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+04  | shl eax 1 |         | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+04  | shl eax 1 | inc eax | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | sub ebx 0C00 | xor ebx 01000
    mov eax D$Bank+08  |                     | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+0C  |                     | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+010 |                     | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+014 |                     | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax
    ret

____________________________________________________________________________________________
; Mapper #045, a bunch of pirate multicarts (MMC3)

; $6000-$7FFF: Set PROM/CROM masks
____________________________________________________________________________________________

Mapper045:

    ; MMC3
    call MMC3
    mov D$fnPROMUpdate @UpdatePROM
    mov D$fnCROMUpdate @UpdateCROM

    ; Memory mapping
    CPUWrite 06000, 07FFF, @6000_7FFF

    ; Banks
    call @UpdatePROM
    call @UpdateCROM
    ret

@6000_7FFF:

    ifNotFlag. D$Register+0C 040
        mov ebx D$Register+010
        inc D$Register+010
        and D$Register+010 03
        mov B$Register+ebx*4 al

        call @UpdatePROM
        call @UpdateCROM
    else
        call MMC3@WriteWRAM
    endif
    ret

@UpdatePROM:

    mov edx D$Register+0C
    not edx | and edx 03F

    ; $8000 and $C000
    ifNotFlag. B$Command MMC3_SWAP_PROM
        mov eax D$Bank+018  | and eax edx | or eax D$Register+04 | swap PROM, 8k, 08000, eax
        mov eax LAST_BANK-1 | and eax edx | or eax D$Register+04 | swap PROM, 8k, 0C000, eax
    else
        mov eax D$Bank+018  | and eax edx | or eax D$Register+04 | swap PROM, 8k, 0C000, eax
        mov eax LAST_BANK-1 | and eax edx | or eax D$Register+04 | swap PROM, 8k, 08000, eax
    endif

    ; $A000 and $E000
    mov eax D$Bank+01C | and eax edx | or eax D$Register+04 | swap PROM, 8k, 0A000, eax
    mov eax LAST_BANK  | and eax edx | or eax D$Register+04 | swap PROM, 8k, 0E000, eax
    ret

@UpdateCROM:

    ; and with
    mov edx 0FF
    mov ecx D$Register+08
    not ecx | and ecx 0F
    shr edx cl

    ; or with
    mov ecx D$Register+08 | and ecx 0F0 | shl ecx 4
    or ecx D$Register+00

    ; CROM swapped?
    mov ebx 0
    ifFlag B$Command MMC3_SWAP_CROM, mov ebx 01000

    ; Set banks
    mov eax D$Bank+00  | shl eax 1 |         | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+00  | shl eax 1 | inc eax | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+04  | shl eax 1 |         | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+04  | shl eax 1 | inc eax | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | sub ebx 0C00 | xor ebx 01000
    mov eax D$Bank+08  |                     | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+0C  |                     | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+010 |                     | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax | add ebx 0400
    mov eax D$Bank+014 |                     | and eax edx | or eax ecx | swap CROM, 1k, ebx, eax
    ret
____________________________________________________________________________________________
; Mapper #119, TQROM

; Use CRAM if CROM bank number is > $40
____________________________________________________________________________________________
[CRAM_ENABLED &TRUE]
Mapper119:

    ; Almost MMC3
    call MMC3
    mov D$fnCROMUpdate @UpdateCROM
    mov D$Cartridge@SizeCRAM 8

    ; Override PPU line to allow CRAM<-->CROM switch
    PPUWrite 00000, 01FFF, @WriteCRAM
    PPURead  00000, 01FFF, @ReadCRAM
    ret

@UpdateCROM:

    ; CRAM enabled?
    ifNotFlag.. B$Command MMC3_SWAP_CROM

        test B$Bank+00  020 | setnz B$Register+00 | setnz B$Register+01
        test B$Bank+04  020 | setnz B$Register+02 | setnz B$Register+03
        test B$Bank+08  040 | setnz B$Register+04
        test B$Bank+0C  040 | setnz B$Register+05
        test B$Bank+010 040 | setnz B$Register+06
        test B$Bank+014 040 | setnz B$Register+07

        if. B$Register+00 = CRAM_ENABLED | swap CRAM, 2k,  0000, D$Bank+00  | else | swap CROM, 2k,  0000, D$Bank+00  | endif
        if. B$Register+02 = CRAM_ENABLED | swap CRAM, 2k,  0800, D$Bank+04  | else | swap CROM, 2k,  0800, D$Bank+04  | endif
        if. B$Register+04 = CRAM_ENABLED | swap CRAM, 1k, 01000, D$Bank+08  | else | swap CROM, 1k, 01000, D$Bank+08  | endif
        if. B$Register+05 = CRAM_ENABLED | swap CRAM, 1k, 01400, D$Bank+0C  | else | swap CROM, 1k, 01400, D$Bank+0C  | endif
        if. B$Register+06 = CRAM_ENABLED | swap CRAM, 1k, 01800, D$Bank+010 | else | swap CROM, 1k, 01800, D$Bank+010 | endif
        if. B$Register+07 = CRAM_ENABLED | swap CRAM, 1k, 01C00, D$Bank+014 | else | swap CROM, 1k, 01C00, D$Bank+014 | endif

    else..

        test B$Bank+00  020 | setnz B$Register+04 | setnz B$Register+05
        test B$Bank+04  020 | setnz B$Register+06 | setnz B$Register+07
        test B$Bank+08  040 | setnz B$Register+00
        test B$Bank+0C  040 | setnz B$Register+01
        test B$Bank+010 040 | setnz B$Register+02
        test B$Bank+014 040 | setnz B$Register+03

        if. B$Register+00 = CRAM_ENABLED | swap CRAM, 1k,  0000, D$Bank+08  | else | swap CROM, 1k,  0000, D$Bank+08  | endif
        if. B$Register+01 = CRAM_ENABLED | swap CRAM, 1k,  0400, D$Bank+0C  | else | swap CROM, 1k,  0400, D$Bank+0C  | endif
        if. B$Register+02 = CRAM_ENABLED | swap CRAM, 1k,  0800, D$Bank+010 | else | swap CROM, 1k,  0800, D$Bank+010 | endif
        if. B$Register+03 = CRAM_ENABLED | swap CRAM, 1k,  0C00, D$Bank+014 | else | swap CROM, 1k,  0C00, D$Bank+014 | endif
        if. B$Register+04 = CRAM_ENABLED | swap CRAM, 2k, 01000, D$Bank+00  | else | swap CROM, 2k, 01000, D$Bank+00  | endif
        if. B$Register+06 = CRAM_ENABLED | swap CRAM, 2k, 01800, D$Bank+04  | else | swap CROM, 2k, 01800, D$Bank+04  | endif

    endif..
    ret

@WriteCRAM:

    mov ebx edx | shr ebx 10
    on B$Register+ebx = CRAM_ENABLED, WriteCRAM
    ret

@ReadCRAM:

    mov ebx eax | shr ebx 10
    mov D$A13 0
    on B$Register+ebx = CRAM_ENABLED, ReadCRAM
    jmp ReadCROM
____________________________________________________________________________________________
; Mapper #044, Super HIK 7-in-1

; $A001: XXXXXXXX: - Overrides bankswitching in a quite complicated way...

____________________________________________________________________________________________

Mapper044:

    ; Almost MMC3
    call MMC3
    mov D$fnPROMUpdate @UpdatePROM
    mov D$fnCROMUpdate @UpdateCROM

    ; Memory mapping
    CPUWrite 06000, 07FFF, WriteVoid
    CPURead  06000, 07FFF, ReadVoid
    mov ecx 0A001
L0: CPUWrite ecx, ecx, @A001
    add ecx 2
    on ecx < 0C000, L0<

    ; Reset banks
    call @UpdatePROM
    call @UpdateCROM
    ret

@A001:

    and eax 07
    if eax = 7, mov eax 6
    mov D$Register eax
    call @UpdatePROM
    call @UpdateCROM
    ret

@UpdatePROM:

    ; Set up masks
    mov ebx D$Register | shl ebx 4
    mov edx 0F | if B$Register = 6, mov edx 01F

    ; $8000 and $C000
    ifNotFlag. B$Command MMC3_SWAP_PROM
        mov eax D$Bank+018  | and eax edx | or eax ebx | swap PROM, 8k, 08000, eax
        mov eax LAST_BANK-1 | and eax edx | or eax ebx | swap PROM, 8k, 0C000, eax
    else
        mov eax D$Bank+018  | and eax edx | or eax ebx | swap PROM, 8k, 0C000, eax
        mov eax LAST_BANK-1 | and eax edx | or eax ebx | swap PROM, 8k, 08000, eax
    endif

    ; $A000 and $E000
    mov eax D$Bank+01C | and eax edx | or eax ebx | swap PROM, 8k, 0A000, eax
    mov eax LAST_BANK  | and eax edx | or eax ebx | swap PROM, 8k, 0E000, eax
    ret

@UpdateCROM:

    ; Set up mask
    mov ebx D$Register | shl ebx 7
    mov edx 07F | if B$Register = 6, mov edx 0FF

    ; $0000 <--> $1000?
    mov ecx 01000
    ifFlag B$Command MMC3_SWAP_CROM, mov ecx 0

    ; Swap banks
    mov eax D$Bank+08  | and eax edx | or eax ebx | swap CROM, 1k, ecx, eax | add ecx 0400
    mov eax D$Bank+0C  | and eax edx | or eax ebx | swap CROM, 1k, ecx, eax | add ecx 0400
    mov eax D$Bank+010 | and eax edx | or eax ebx | swap CROM, 1k, ecx, eax | add ecx 0400
    mov eax D$Bank+014 | and eax edx | or eax ebx | swap CROM, 1k, ecx, eax | sub ecx 0C00
    xor ecx 01000
    shr edx 1
    shr ebx 1
    mov eax D$Bank+00  | and eax edx | or eax ebx | swap CROM, 2k, ecx, eax | add ecx 0800
    mov eax D$Bank+04  | and eax edx | or eax ebx | swap CROM, 2k, ecx, eax
    ret
____________________________________________________________________________________________
; Mapper #049, Super HIK 4-in-1

; $6000-$7FFF: BBPPxxxM - (M = 0) swap 32k PROM at $8000 (PP)
;                       - (M = 1) use BB as the two MSB when swapping PROM
;                       - use BB as the two MSB when swapping CROM
____________________________________________________________________________________________

Mapper049:

    ; Almost like MMC3
    call MMC3
    mov D$fnPROMUpdate @UpdatePROM
    mov D$fnCROMUpdate @UpdateCROM

    ; Memory mapping
    CPUWrite 06000, 07FFF, @6000_7FFF
    CPURead  06000, 07FFF, ReadVoid

    call @UpdatePROM
    call @UpdateCROM
    ret

@6000_7FFF:

    if. B$WRAMEnabled = &TRUE
        mov D$Register eax
        call @UpdatePROM
        call @UpdateCROM
    endif
    ret

@UpdatePROM:

    mov edx D$Register
    ifFlag. edx 01

        and edx 0C0 | shr edx 2

        ; $8000 <--> $C000 swap?
        ifNotFlag.. D$Command MMC3_SWAP_PROM
            mov eax D$Bank+018  | and eax 0F | or eax edx | swap PROM, 8k, 08000, eax
            mov eax LAST_BANK-1 | and eax 0F | or eax edx | swap PROM, 8k, 0C000, eax
        else..
            mov eax D$Bank+018  | and eax 0F | or eax edx | swap PROM, 8k, 0C000, eax
            mov eax LAST_BANK-1 | and eax 0F | or eax edx | swap PROM, 8k, 08000, eax
        endif..

        ; $A000 and $E000
        mov eax D$Bank+01C | and eax 0F | or eax edx | swap PROM, 8k, 0A000, eax
        mov eax LAST_BANK  | and eax 0F | or eax edx | swap PROM, 8k, 0E000, eax

    else
        shr edx 4 | and edx 03
        swap PROM, 32k, 08000, edx
    endif
    ret

@UpdateCROM:

    mov edx D$Register | and edx 0C0
    mov ecx 0
    ifFlag B$Command MMC3_SWAP_CROM, mov ecx 01000

    mov eax D$Bank+00  | and eax 03F | or eax edx | swap CROM, 2k, ecx, eax | add ecx 0800
    mov eax D$Bank+04  | and eax 03F | or eax edx | swap CROM, 2k, ecx, eax | sub ecx 0800
    xor ecx 01000
    shl edx 1
    mov eax D$Bank+08  | and eax 07F | or eax edx | swap CROM, 1k, ecx, eax | add ecx 0400
    mov eax D$Bank+0C  | and eax 07F | or eax edx | swap CROM, 1k, ecx, eax | add ecx 0400
    mov eax D$Bank+010 | and eax 07F | or eax edx | swap CROM, 1k, ecx, eax | add ecx 0400
    mov eax D$Bank+014 | and eax 07F | or eax edx | swap CROM, 1k, ecx, eax
    ret

____________________________________________________________________________________________
; Mapper #182, Super Donkey Kong

; $8001: xxxxxxxM - mirror VERTICAL/HORIZONTAL (0/1)
; $A000: xxxxxCCC - command number
; $C000: BBBBBBBB - bank number for command
; $E003: IIIIIIII - IRQ counter
____________________________________________________________________________________________

Mapper182:

    call MMC3

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        and ecx 0F003
        mov eax WriteVoid
        if ecx = 08000, mov eax @8000
        if ecx = 08001, mov eax @8001
        if ecx = 0A000, mov eax @A000
        if ecx = 0A001, mov eax @A001
        if ecx = 0A002, mov eax @A002
        if ecx = 0C000, mov eax @C000
        if ecx = 0C001, mov eax @C001
        if ecx = 0C002, mov eax @C002
        if ecx = 0E002, mov eax @E002
        if ecx = 0E003, mov eax @E003

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; ????????
@8000:
@A001:
@A002:
@C001:
@C002:
@E002:ret

@8001:

    ; xxxxxxxM
    ifNotFlag. eax 01
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

@A000:
    [@Table: B$ 0 3 1 5 6 7 2 4]
    ; xxxxxCCC
    and eax 07
    movzx eax B$@Table+eax
    call MMC3@Command
    ret

@C000:

    call MMC3@BankSwitch
    ret

@E003:

    ClearIRQ IRQ_CARTRIDGE
    mov D$IRQLatch eax
    mov D$IRQCounter eax
    test eax eax | setnz B$IRQEnabled
    ret
____________________________________________________________________________________________
; Mapper #189, Street Fighter 2 (Yoko)

; MMC3 with different PROM swapping

; $41xx: xxPPxxxx - swap 32k PROM at $8000
; $61xx: xxxxxxPP - swap 32k PROM at $8000
; $8001: Do not swap PROM
____________________________________________________________________________________________

Mapper189:

    ; Like the MMC3
    call MMC3
    mov D$fnPROMUpdate DoNothing

    ; Memory mapping
    CPUWrite 04100, 041FF, @4100_41FF
    CPUWrite 04800, 04FFF, @4800_4FFF
    CPUWrite 05000, 057FF, @5000_57FF
    CPUWrite 05800, 05FFF, @5800_5FFF
    CPURead  05800, 05FFF, @Read5800_5FFF
    CPUWrite 06100, 061FF, @6100_61FF
CPUWrite 0C001, 0C001, @C001

    ; Load first 32k
    swap PROM, 32k, 08000, 0
    ret

; Hack for SF4, where the timing is off
@C001:

    if al > 0, dec al
    jmp MMC3@IRQLatch

@4100_41FF:

    ; xxPPxxxx
    shr eax 4
    swap PROM, 32k, 08000, eax
    ret

@4800_4FFF:

    ; --M-----
    ifNotFlag. eax 020
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif

    ; ---P---P
    ifFlag. eax 010
        and eax 01
        or eax 02
    else
        and eax 01
    endif
    swap PROM, 32k, 08000, eax
    ret

@Read5800_5FFF:

    and eax 03
    movzx eax B$Register+eax
    ret

@5000_57FF:

    mov D$Register+010 eax
    ret

@5800_5FFF:

    mov ebx D$Register+010
    xor al B@Table+ebx
    and edx 03
    mov B$Register+edx al
    ret

@6100_61FF:

    ; xxxxxxPP
    swap PROM, 32k, 08000, eax
    ret
____________________________________________________________________________________________

[@Table: B$
 059, 059, 059, 059, 059, 059, 059, 059, 059, 049, 019,  09, 059, 049, 019,  09,
 059, 059, 059, 059, 059, 059, 059, 059, 051, 041, 011,  01, 051, 041, 011,  01,
 059, 059, 059, 059, 059, 059, 059, 059, 059, 049, 019,  09, 059, 049, 019,  09,
 059, 059, 059, 059, 059, 059, 059, 059, 051, 041, 011,  01, 051, 041, 011,  01,
  00, 010, 040, 050,  00, 010, 040, 050,  00,  00,  00,  00,  00,  00,  00,  00,
  08, 018, 048, 058,  08, 018, 048, 058,  00,  00,  00,  00,  00,  00,  00,  00,
  00, 010, 040, 050,  00, 010, 040, 050,  00,  00,  00,  00,  00,  00,  00,  00,
  08, 018, 048, 058,  08, 018, 048, 058,  00,  00,  00,  00,  00,  00,  00,  00,
 059, 059, 059, 059, 059, 059, 059, 059, 058, 048, 018,  08, 058, 048, 018,  08,
 059, 059, 059, 059, 059, 059, 059, 059, 050, 040, 010,  00, 050, 040, 010,  00,
 059, 059, 059, 059, 059, 059, 059, 059, 058, 048, 018,  08, 058, 048, 018,  08,
 059, 059, 059, 059, 059, 059, 059, 059, 050, 040, 010,  00, 050, 040, 010,  00,
  01, 011, 041, 051,  01, 011, 041, 051,  00,  00,  00,  00,  00,  00,  00,  00,
  09, 019, 049, 059,  09, 019, 049, 059,  00,  00,  00,  00,  00,  00,  00,  00,
  01, 011, 041, 051,  01, 011, 041, 051,  00,  00,  00,  00,  00,  00,  00,  00,
  09, 019, 049, 059,  09, 019, 049, 059,  00,  00,  00,  00,  00,  00,  00,  00]

____________________________________________________________________________________________
; Mapper #245, Yong Zhe Dou E Long
____________________________________________________________________________________________

Mapper245: call MMC3 | ret
____________________________________________________________________________________________
; Mapper #074, Taiwanese MMC3
____________________________________________________________________________________________

Mapper074: call MMC3 | ret

____________________________________________________________________________________________
; Mapper #250, Time Diver Avenger (MMC3)

; Almost identical to MMC3
____________________________________________________________________________________________

Mapper250:

    ; Almost identical to MMC3
    call MMC3

    ; Memory mapping
    CPUWrite 08000, 0FFFF, @8000_FFFF
    ret

@8000_FFFF:

    ; Data = LSB of address
    movzx eax dl

    ; Write to a MMC3 reg
    and edx 0E400
    ifFlag edx 0400, or edx 01
    and edx 0E001
    on edx = 08000, MMC3@Command
    on edx = 08001, MMC3@Bankswitch
    on edx = 0A000, MMC3@Mirroring
    on edx = 0A001, MMC3@WRAMEnable
    on edx = 0C000, MMC3@IRQLatch
    on edx = 0C001, MMC3@ResetIRQ
    on edx = 0E000, MMC3@DisableIRQ
    on edx = 0E001, MMC3@EnableIRQ
    ret

____________________________________________________________________________________________
; Mapper #118, TLSROM/TKSROM (MMC3)

; MMC3
; When swapping 1k CROM, use the MSB for mirroring: $2400/$2000 (0/1)
____________________________________________________________________________________________

Mapper118:

    ; MMC3
    call Mapper004

    ; Set ports
    mov ecx 08001
L0: CPUWrite ecx, ecx, @8001
    add ecx 2
    on ecx <= 09FFF, L0<
    ret

@8001:

    mov edx D$Command
    and edx 7

    ; Skip if not 1k CROM switch
    on edx >= 6, S0>
    on edx <= 1, S0>

    ; Mxxxxxxx
    ifNotFlag. eax 080
        mirror ONE_SCREEN_2400
    else
        mirror ONE_SCREEN_2000
    endif

S0: call MMC3@Bankswitch
    ret
____________________________________________________________________________________________
; Mapper #088, Namcot 118

; $8000: xxxxxCCC - command number
; $8001: BBBBBBBB - swap banks almost like MMC3
; $C000: xMxxxxxx - mirror $2000/$2400 (0/1)
____________________________________________________________________________________________

Mapper088:

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        mov eax WriteVoid
        and ecx 08001
        if ecx = 08000, mov eax @8000
        if ecx = 08001, mov eax @8001

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000:

    ; Mirroring hack
    if. D$Cartridge@PROMCRC32 = 0C1B6_B2A6 ; Devil Man (J)
        mirror ONE_SCREEN_2000
        ifFlag eax 040, mirror ONE_SCREEN_2400
    endif

    ; xxxxxCCC
    and eax 7
    call MMC3@Command
    ret

@8001:

    ; 1k CROM? --> Add $40
    if D$Command >= 2,
        if D$Command <= 5,
            or eax 040

    call MMC3@Bankswitch
    ret

@C000:

    ; xMxxxxxx
    ifNotFlag. eax 040
        mirror ONE_SCREEN_2000
    else
        mirror ONE_SCREEN_2400
    endif
    ret

____________________________________________________________________________________________
; Mapper #095, Namcot 106M (Dragon Buster)

; $8000: xxxxxCCC - Command number
; $8001: xxMCCCCC - PROM swap like MMC3
;                 - CROM swap like MMC3
;                 - mirror $2400/$2000 (0/1)
____________________________________________________________________________________________

Mapper095:

    ; Memory mapping
    mov ecx 08000
L0: CPUWrite ecx, ecx, @8000 | inc ecx
    CPUWrite ecx, ecx, @8001 | inc ecx
    on ecx < 0A000, L0<

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000:

    and eax 07
    call MMC3@Command
    ret

@8001:

    on D$Command >= 6, S0>

    ; xxMxxxxx
    ifFlag. eax 020
        mirror ONE_SCREEN_2000
    else
        mirror ONE_SCREEN_2400
    endif
    and eax 01F

S0: call MMC3@Bankswitch
    ret

____________________________________________________________________________________________
; Mapper #076, Namco 109

; Quite similar to MMC3
____________________________________________________________________________________________

Mapper076:

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        and ecx 0E001
        mov eax WriteVoid
        if ecx = 08000, mov eax MMC3@Command
        if ecx = 08001, mov eax @8001
        if ecx = 0A000, mov eax @A000

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Swap in last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8001:

    ; 0-7: Switch banks according to command in $8000
    mov edx D$Command | and edx 7

    ; Load 2k CROM
    if dl = 2, swap CROM, 2k, 00000, eax
    if dl = 3, swap CROM, 2k,  0800, eax
    if dl = 4, swap CROM, 2k, 01000, eax
    if dl = 5, swap CROM, 2k, 01800, eax

    ; Load 8k PROM
    if dl = 6, swap PROM, 8k, 08000, eax
    if dl = 7, swap PROM, 8k, 0A000, eax
    ret

@A000:

    ifFlag. eax 1
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif
    ret
____________________________________________________________________________________________
TITLE MMC5

____________________________________________________________________________________________
; Mapper #005, Nintendo MMC5

; $5130: ????????
; $5800: ????????
____________________________________________________________________________________________

Mapper005:

    if D$Cartridge@SizeWRAM < 16, mov D$Cartridge@SizeWRAM 16

    ; Add MMC5 sound
    call MMC5Sound

    ; IRQ counter
    mov D$HSyncHook @HSync

    ; CROM reading
    PPURead 00000, 01FFF, @ReadCROM

    ; - Memory mapping -
    ; MMC settings
    CPUWrite 05100, 05100, @5100
    CPUWrite 05101, 05101, @5101
    CPUWrite 05102, 05102, @5102
    CPUWrite 05103, 05103, @5103
    CPUWrite 05104, 05104, @5104
    CPUWrite 05105, 05105, @5105
    CPUWrite 05106, 05106, @5106
    CPUWrite 05107, 05107, @5107
    ; PROM/XRAM switch
    CPUWrite 05113, 05113, @5113
    CPUWrite 05114, 05114, @5114
    CPUWrite 05115, 05115, @5115
    CPUWrite 05116, 05116, @5116
    CPUWrite 05117, 05117, @5117
    ; CROM switch
    CPUWrite 05120, 05120, @5120
    CPUWrite 05121, 05121, @5121
    CPUWrite 05122, 05122, @5122
    CPUWrite 05123, 05123, @5123
    CPUWrite 05124, 05124, @5124
    CPUWrite 05125, 05125, @5125
    CPUWrite 05126, 05126, @5126
    CPUWrite 05127, 05127, @5127
    CPUWrite 05128, 05128, @5128
    CPUWrite 05129, 05129, @5129
    CPUWrite 0512A, 0512A, @512A
    CPUWrite 0512B, 0512B, @512B
    ; Split
    CPUWrite 05200, 05200, @5200
    CPUWrite 05201, 05201, @5201
    CPUWrite 05202, 05202, @5202
    ; IRQ counter
    CPUWrite 05203, 05203, @5203
    CPUWrite 05204, 05204, @5204
    CPURead  05204, 05204, @Read5204
    ; Multiplier
    CPUWrite 05205, 05205, @5205
    CPUWrite 05206, 05206, @5206
    CPURead  05205, 05205, @Read5205
    CPURead  05206, 05206, @Read5206
    ; ExCiRAM
    CPUWrite 05C00, 05FFF, @5C00_5FFF
    CPURead  05C00, 05FFF, @Read5C00_5FFF
    ; PROM/XRAM
    CPUWrite 06000, 07FFF, @6000_7FFF
    CPUWrite 08000, 09FFF, @8000_9FFF
    CPUWrite 0A000, 0BFFF, @A000_BFFF
    CPUWrite 0C000, 0DFFF, @C000_DFFF
    CPURead  06000, 07FFF, @ReadXRAM
    CPURead  08000, 09FFF, @Read8000_9FFF
    CPURead  0A000, 0BFFF, @ReadA000_BFFF
    CPURead  0C000, 0DFFF, @ReadC000_DFFF

    ; Set modes
    mov B$PMode PMODE_8K
    mov B$CMode CMODE_1K

    ; Set banks
    setXRAM 06000, 0
    setPROM 08000, LAST_BANK
    setPROM 0A000, LAST_BANK
    setPROM 0C000, LAST_BANK
    setPROM 0E000, LAST_BANK
    mov eax 0
L0: mov D$CROMBank+eax*4 eax
    inc eax | on eax < 8, L0<
    mov eax 4
L0: mov D$BgROMBank+eax*4-010 eax
    mov D$BgROMBank+eax*4     eax
    inc eax | on eax < 8, L0<
    call @UpdateCROM
    call @UpdateBgROM

CPUWrite 05800, 05800, DoNothing
    ret

; Graphic modes
@5101: and al 3 | mov B$CMode al | ret
@5104: and al 3 | mov B$GMode al | ret

; Mirroring/fill/ExCiRAM
@5105:

    mov edx eax | and edx 03 | shl edx 10 | mov D$IndexCiRAM+00 edx | shr eax 2
    mov edx eax | and edx 03 | shl edx 10 | mov D$IndexCiRAM+04 edx | shr eax 2
    mov edx eax | and edx 03 | shl edx 10 | mov D$IndexCiRAM+08 edx | shr eax 2
    mov edx eax | and edx 03 | shl edx 10 | mov D$IndexCiRAM+0C edx | shr eax 2
    ret

; Fill settings
@5106: mov D$FillCharacter eax | ret
@5107: and al 03 | shl eax 2 | mov D$FillAttribute eax | ret

; CROM banks
@5120: mov D$CROMbank+00  eax | jmp @UpdateCROM
@5121: mov D$CROMbank+04  eax | jmp @UpdateCROM
@5122: mov D$CROMbank+08  eax | jmp @UpdateCROM
@5123: mov D$CROMbank+0C  eax | jmp @UpdateCROM
@5124: mov D$CROMbank+010 eax | jmp @UpdateCROM
@5125: mov D$CROMbank+014 eax | jmp @UpdateCROM
@5126: mov D$CROMbank+018 eax | jmp @UpdateCROM
@5127: mov D$CROMbank+01C eax | jmp @UpdateCROM

@UpdateCROM:

    if. B$CMode = CMODE_8K
        swap CROM, 8k,  0000, D$CROMBank+01C
    else
    if. B$CMode = CMODE_4K
        swap CROM, 4k,  0000, D$CROMBank+0C
        swap CROM, 4k, 01000, D$CROMBank+01C
    else
    if. B$CMode = CMODE_2K
        swap CROM, 2k,  0000, D$CROMBank+04
        swap CROM, 2k,  0800, D$CROMBank+0C
        swap CROM, 2k, 01000, D$CROMBank+014
        swap CROM, 2k, 01800, D$CROMBank+01C
    else
    if. B$CMode = CMODE_1K
        swap CROM, 1k,  0000, D$CROMBank+00
        swap CROM, 1k,  0400, D$CROMBank+04
        swap CROM, 1k,  0800, D$CROMBank+08
        swap CROM, 1k,  0C00, D$CROMBank+0C
        swap CROM, 1k, 01000, D$CROMBank+010
        swap CROM, 1k, 01400, D$CROMBank+014
        swap CROM, 1k, 01800, D$CROMBank+018
        swap CROM, 1k, 01C00, D$CROMBank+01C
    endif
    ret

; BgROM banks
[BgROMBank CROMBank+020]
@5128: mov D$BgROMBank+00 eax | mov D$BgROMBank+010 eax | jmp @UpdateBgROM
@5129: mov D$BgROMBank+04 eax | mov D$BgROMBank+014 eax | jmp @UpdateBgROM
@512A: mov D$BgROMBank+08 eax | mov D$BgROMBank+018 eax | jmp @UpdateBgROM
@512B: mov D$BgROMBank+0C eax | mov D$BgROMBank+01C eax | jmp @UpdateBgROM

@UpdateBgROM:

    if. B$CMode = CMODE_8K
        swap BgROM, 8k,  0000, D$BgROMBank+01C
    else
    if. B$CMode = CMODE_4K
        swap BgROM, 4k,  0000, D$BgROMBank+0C
        swap BgROM, 4k, 01000, D$BgROMBank+01C
    else
    if. B$CMode = CMODE_2K
        swap BgROM, 2k,  0000, D$BgROMBank+04
        swap BgROM, 2k,  0800, D$BgROMBank+0C
        swap BgROM, 2k, 01000, D$BgROMBank+014
        swap BgROM, 2k, 01800, D$BgROMBank+01C
    else
    if. B$CMode = CMODE_1K
        swap BgROM, 1k,  0000, D$BgROMBank+00
        swap BgROM, 1k,  0400, D$BgROMBank+04
        swap BgROM, 1k,  0800, D$BgROMBank+08
        swap BgROM, 1k,  0C00, D$BgROMBank+0C
        swap BgROM, 1k, 01000, D$BgROMBank+010
        swap BgROM, 1k, 01400, D$BgROMBank+014
        swap BgROM, 1k, 01800, D$BgROMBank+018
        swap BgROM, 1k, 01C00, D$BgROMBank+01C
    endif
    ret

@ReadCROM:

    ifNotFlag B$Mask SPRITES_VISIBLE+BG_VISIBLE, jmp ReadCROM
    on D$PPUCycle < 256, @ReadBgROM
    on D$PPUCycle < 320, ReadCROM
    on D$PPUCycle < 341, @ReadBgROM
    jmp ReadCROM

@ReadBgROM:

    movzx ebx ah | and ebx 01C
    and eax 03FF | add eax D$IndexBgROM+ebx
    add eax D$pCROM
    movzx eax B$eax
    ret
____________________________________________________________________________________________
; Split settings
____________________________________________________________________________________________

@5200:

    call PPUSynchronize

    ; ESxTTTTT
    test al 080 | setnz B$SplitEnabled
    test al 040 | setnz B$SplitFromRight
    and  al 01F | mov   B$SplitTile al
    ret

@5201:

    call PPUSynchronize

    mov B$SplitScroll al
    ret

@5202:

    call PPUSynchronize
    and al 03F | shl eax 12
    mov edx D$Cartridge@SizeCROM | shl edx 10
    dec edx | and eax edx
    mov D$SplitOffset eax
    ret

____________________________________________________________________________________________
; IRQ counter
____________________________________________________________________________________________

@5203:

    ClearIRQ IRQ_CARTRIDGE
    mov D$IRQStop eax
    ret

@5204:

    ClearIRQ IRQ_CARTRIDGE
    test al 080 | setnz B$IRQEnabled
    ret

@Read5204:

    mov eax D$IRQStatus
    and B$IRQStatus 0FF-080
    ret

@HSync:

    ; GFX on?
    ifFlag. B$Mask BG_VISIBLE+SPRITES_VISIBLE
        on D$PPUCycle = 342, end
        or B$IRQStatus 040
        inc D$IRQCounter
        mov B$IRQClear 0
    endif

    inc B$IRQClear
    if. B$IRQClear > 1
        ClearIRQ IRQ_CARTRIDGE
        mov D$IRQCounter 0-1
        and B$IRQStatus 0FF-0C0
    else
        ; Scanline hit?
        mov eax D$IRQStop
        if eax = D$IRQCounter, or B$IRQStatus 080
    endif

    ; Time for IRQ?
    if B$IRQEnabled = &TRUE,
        ifFlag B$IRQStatus 080,
            ifFlag B$IRQStatus 040,
                SetIRQ PPU, IRQ_CARTRIDGE
    ret

____________________________________________________________________________________________
; Multiplier
____________________________________________________________________________________________

@5205:     mov D$Factor1 eax | ret
@5206:     mov D$Factor2 eax | ret
@Read5205: mov eax D$Factor1 | mul D$Factor2 | movzx eax al | ret
@Read5206: mov eax D$Factor1 | mul D$Factor2 | movzx eax ah | ret

____________________________________________________________________________________________
; ExCiRAM
____________________________________________________________________________________________

[ExCiRAM NameTable+0800]
@5C00_5FFF:

    and edx 03FF
    if. B$GMode = GMODE_EXRAM
        mov B$ExCiRAM+edx al
    else
    if. B$GMode != GMODE_EXRAM_WP
        ;ifFlag B$IRQStatus 040, mov al 0
        mov B$ExCiRAM+edx al
    endif
    ret

@Read5C00_5FFF:

    on B$GMode < GMODE_EXRAM, ReadVoid
    and eax 03FF
    movzx eax B$ExCiRAM+eax
    ret

____________________________________________________________________________________________
; PROM/XRAM
____________________________________________________________________________________________

@5100: and al 3 | mov B$PMode        al | ret
@5102: and al 3 | mov B$XRAMWrite+00 al | ret
@5103: and al 3 | mov B$XRAMWrite+01 al | ret

@5113:

    ; $6000-$7FFF
    setXRAM 06000, eax
    ret

@5114:

    ; $8000-$9FFF
    if. B$PMode = PMODE_8K
        ifFlag.. al MMC5_SWITCH_PROM
            setPROM 08000, eax
        else..
            setXRAM 08000, eax
        endif..
    endif
    ret

@5115:

    ; $A000-$BFFF
    if. B$PMode = PMODE_8K
        ifFlag.. al MMC5_SWITCH_PROM
            setPROM 0A000, eax
        else..
            setXRAM 0A000, eax
        endif..
    endif

    ; $8000-$BFFF
    on  B$PMode = PMODE_16K_8K, F0>
    if. B$PMode = PMODE_16K
    F0: ifFlag.. al MMC5_SWITCH_PROM
            and al 0FE
            setPROM 08000, eax | inc eax
            setPROM 0A000, eax
        else..
            and al 0FE
            setXRAM 08000, eax | inc eax
            setXRAM 0A000, eax
        endif..
    endif
    ret

@5116:

    ; $C000-$DFFF
    on  B$PMode = PMODE_16K_8K, F0>
    if. B$PMode = PMODE_8K
    F0: ifFlag.. al MMC5_SWITCH_PROM
            setPROM 0C000, eax
        else..
            setXRAM 0C000, eax
        endif..
    endif
    ret

@5117:

    ifNotFlag eax MMC5_SWITCH_PROM, ret

    ; $8000-$FFFF
    if. B$Pmode = PMODE_32K
        and al 0FC
        setPROM 08000, eax | inc eax
        setPROM 0A000, eax | inc eax
        setPROM 0C000, eax | inc eax
        setPROM 0E000, eax
    endif

    ; $C000-$FFFF
    if. B$PMode = PMODE_16K
        and al 0FE
        setPROM 0C000, eax | inc eax
        setPROM 0E000, eax
    endif

    ; $E000-$FFFF
    on  B$PMode = PMODE_16K_8K, F0>
    if. B$PMode = PMODE_8K
    F0: setPROM 0E000, eax
    endif
    ret

; PROM/XRAM read/write
@6000_7FFF: if. W$XRAMWrite = 0102 | test B$XRAMEnabled 01 | jnz @WriteXRAM | endif | ret
@8000_9FFF: if. W$XRAMWrite = 0102 | test B$XRAMEnabled 02 | jnz @WriteXRAM | endif | ret
@A000_BFFF: if. W$XRAMWrite = 0102 | test B$XRAMEnabled 04 | jnz @WriteXRAM | endif | ret
@C000_DFFF: if. W$XRAMWrite = 0102 | test B$XRAMEnabled 08 | jnz @WriteXRAM | endif | ret
@Read8000_9FFF: test B$XRAMEnabled 02 | jnz @ReadXRAM | endif | jmp ReadPROM
@ReadA000_BFFF: test B$XRAMEnabled 04 | jnz @ReadXRAM | endif | jmp ReadPROM
@ReadC000_DFFF: test B$XRAMEnabled 08 | jnz @ReadXRAM | endif | jmp ReadPROM


@ReadXRAM:

    sub eax 06000
    mov ebx eax | shr ebx 13 | and ebx 3
    and eax 01FFF
    add eax D$IndexXRAM+ebx*4
    movzx eax B$WRAM+eax
    ret

@WriteXRAM:

    sub edx 06000
    mov ebx edx | shr ebx 13 | and ebx 3
    and edx 01FFF
    add edx D$IndexXRAM+ebx*4
    mov B$WRAM+edx al
    ret

[setPROM | #=2 | and B$XRAMEnabled (not (01 shl ((#1 - 06000) shr 13))) | swap PROM, 8k, #1, #2]
[setXRAM | #=2 | or  B$XRAMEnabled (    (01 shl ((#1 - 06000) shr 13))) | swap XRAM, 8k, #1, #2]
[MMC5_SWITCH_PROM 080]

____________________________________________________________________________________________
; MMC5 graphics
____________________________________________________________________________________________

DoMMC5Graphics:

    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Time to split?
    on B$SplitEnabled = &FALSE, Q0>
    mov eax D$PPUCycle
    if. eax >= 320 | sub eax 320
    else | add eax 010 | endif
    shr eax 3
    if. B$SplitFromRight != &FALSE
        on eax <  D$SplitTile, Q0>
    else
        on eax >= D$SplitTile, Q0>
    endif

    ; Split
    mov edx D$nScanline | shr edx 3 | shl edx 5
    movzx eax B$ExCiRAM+edx+eax | shl eax 4
    mov edx D$nScanline | and edx 07
    add eax edx
    add eax D$pCROM
    add eax D$SplitOffset
    mov dl B$eax, dh B$eax+8
    mov W$Patterns dx
    ret


Q0: mov eax D$VRAMAddress
    shr eax 10 | and eax 3
    if. D$IndexCiRAM+eax*4 = 0C00
        on B$GMode = GMODE_EXGFX, @FillExGfx
        jmp @FillNormal
    else
        on B$GMode = GMODE_EXGFX, @ExGfx
        jmp @Normal
    endif
    ret
____________________________________________________________________________________________

@Normal: jmp ReadPattern

@FillNormal:

    copy D$FillCharacter D$NTByte
    copy D$FillAttribute D$Attribute
    jmp ReadPattern
____________________________________________________________________________________________

@ExGfx:

    ; Tile
    mov edx D$NTByte | shl edx 4
    mov eax D$VRAMAddress | shr eax 12
    or edx eax

    ; Attribute
    mov eax D$VRAMAddress | and eax 03FF
    movzx eax B$ExCiRAM+eax
    push eax
        and eax 0C0 | shr eax 4
        mov D$Attribute eax
    pop eax

    ; Bank offset
    and eax 03F | shl eax 12
    or edx eax

    ; Pattern
    mov ecx D$Cartridge@SizeCROM | shl ecx 10 | dec ecx
    and edx ecx
    add edx D$pCROM
    mov al B$edx
    mov ah B$edx+8
    mov W$Patterns ax
    ret

@FillExGfx:

    ; Tile
    mov edx D$FillCharacter | shl edx 4
    mov eax D$VRAMAddress | shr eax 12
    or edx eax

    ; Attribute
    copy D$FillAttribute D$Attribute

    ; Bank offset
    mov eax D$VRAMAddress | and eax 03FF
    movzx eax B$ExCiRAM+eax
    and eax 03F | shl eax 12
    or edx eax

    ; Pattern
    mov ecx D$Cartridge@SizeCROM
    shl ecx 10 | dec ecx
    and edx ecx
    add edx D$pCROM
    mov ah B$edx
    mov al B$edx+8
    mov W$Patterns ax
    ret
____________________________________________________________________________________________
; Bankswitch
____________________________________________________________________________________________

SwapXRAM:

    arguments @Length, @Address, @Bank
    on D$Cartridge@SizeWRAM = 0, Q0>>
    pushad

        ; Adjust bank number
        mov eax D@Bank

        if. eax >= 4

            mov ecx D@Address | sub ecx 06000 | shr ecx 13
            mov edx 01 | shl edx cl | not edx

            if B$Cartridge@SizeWRAM <= 16, and D$XRAMEnabled edx
            if B$Cartridge@SizeWRAM =  32, mov eax 1
            if B$Cartridge@SizeWRAM =  64, and D$XRAMEnabled edx

        else

            if B$Cartridge@SizeWRAM <= 32, mov eax 0

        endif

        mul D@Length
        mov ebx D$Cartridge@SizeWRAM | shl ebx 10 | mov edx 0 | div ebx | mov eax edx

        ; Update index
        mov ecx D@Length  | shr ecx 13
        mov ebx D@Address | sub ebx 06000 | shr ebx 13
    L0: mov D$IndexXRAM+ebx*4 eax | add eax 02000 | inc ebx | dec ecx | jnz L0<
    popad
Q0: return

SwapBgROM:

    arguments @Length, @Address, @Bank
    call PPUSynchronize

    pushad

        ; Adjust bank number
        mov eax D@Bank
        mul D@Length
        mov ebx D$Cartridge@SizeCROM | shl ebx 10 | mov edx 0 | div ebx | mov eax edx

        ; Update index
        mov ecx D@Length  | shr ecx 10
        mov ebx D@Address | shr ebx 10
    L0: mov D$IndexBgROM+ebx*4 eax | add eax 0400 | inc ebx | dec ecx | jnz L0<

    popad
    return
____________________________________________________________________________________________

enumerate 0,
    PMODE_32K,
    PMODE_16K,
    PMODE_16K_8K,
    PMODE_8K
enumerate 0,
    GMODE_SPLIT,
    GMODE_EXGFX,
    GMODE_EXRAM,
    GMODE_EXRAM_WP
enumerate 0,
    CMODE_8K,
    CMODE_4K,
    CMODE_2K,
    CMODE_1K
____________________________________________________________________________________________
TITLE ExSound

; Various sound chips
____________________________________________________________________________________________
; Namco106 sound
____________________________________________________________________________________________

Namco106Sound:

    mov D$SampleHook @SampleHook
    mov D$TimerHook  @TimerHook
    mov D@NumChannels 1
    mov ebx ExChannel, esi 8
L0: mov D$ebx+WaveLength (1 shl 22)
    add ebx 040
    dec esi | jnz L0<
    mov D$Register+04 080
    ret

[@NumChannels Register+08]

enumerate4 02C,
    Frequency, Address, dummy, Speed, WaveLength

@4800:

    call APUSynchronize
    if edx < 040, ret

    if. edx = 07F
        mov ecx eax | shr ecx 4
        and ecx 7 | inc ecx
        mov D@NumChannels ecx
    endif

    mov ebx edx | shr ebx 3
    and ebx 07 | shl ebx 6 ; mul 040, half a channel
    add ebx ExChannel
    and edx 07
    on edx = 0, @WriteReg0
    on edx = 2, @WriteReg1
    on edx = 4, @WriteReg2
    on edx = 6, @WriteReg3
    on edx = 7, @WriteReg4
    ret

@WriteReg0: mov B$ebx+Frequency+00 al | ret
@WriteReg1: mov B$ebx+Frequency+01 al | ret
@WriteReg2:

    ; Frequency
    mov B$ebx+Frequency+02 al
    and B$ebx+Frequency+02 03
    copy D$ebx+Frequency D$ebx+Speed
    ; Wavelength
    shr eax 2 | not eax | and eax 07 | inc eax | shl eax 20
    xchg eax D$ebx+WaveLength
    if eax != D$ebx+WaveLength, mov D$ebx+Phase 0
    jmp @ChannelActive

@WriteReg3: mov B$ebx+Address al | ret
@WriteReg4: and al 0F | mov D$ebx+Volume eax | jmp @ChannelActive
____________________________________________________________________________________________

@TimerHook:
push eax
mul D$CPUCycles
mov ecx (1 shl 20) | mul ecx
mov ecx 45 | div ecx

    mov ebx ExChannel+(7 shl 6)
    mov esi D@NumChannels
L0: if B$ebx+Active = &TRUE, add D$ebx+Timer eax
    sub ebx 040
    dec esi | jnz L0<
pop eax
    ret

@SampleHook:
;mov edi 0
    mov ebx ExChannel+(7 shl 6)
    mov esi D@NumChannels
L0: call @Sample
    add edi eax
    sub ebx 040
    dec esi | jnz L0<
    ret

@Sample:

    mov eax 0
    if B$ebx+Active = &FALSE, ret

    ; Count steps
    mov eax D$ebx+Timer
    mov ecx D@NumChannels | shl ecx 20
    mov eax D$ebx+Speed
L0: if. D$ebx+Timer >= ecx
        sub D$ebx+Timer ecx
        add D$ebx+Phase eax
        jmp L0<
    endif

    mov ecx D$ebx+WaveLength
L0: if. D$ebx+Phase >= ecx
        sub D$ebx+Phase ecx
        jmp L0<
    endif

    mov edx D$ebx+Phase | shr edx 18
    add edx D$ebx+Address
    shr edx 1 | setc cl
    movzx eax B$ExRAM+edx
    ifFlag cl 01, shr eax 4
    and eax 0F

    ; Next channel
    [@VolumeDivider: 12]
    mul D$ebx+Volume
    div D@VolumeDivider
    ret

@ChannelActive:

    mov B$ebx+Active &FALSE
    if D$ebx+Speed > 0,
        if D$ebx+Wavelength > 0,
            if D$ebx+Volume > 0,
                mov B$ebx+Active &TRUE
    ret
____________________________________________________________________________________________
; VRC6 sound
____________________________________________________________________________________________

VRC6:

    ; Hooks
    mov D$TimerHook @TimerHook
    mov D$SampleHook @SampleHook
    ret

[@Square0  ExChannel0
 @Square1  ExChannel1
 @Sawtooth ExChannel2]

@9000: call @SquareReg0 @Square0 | ret
@9001: call @SquareReg1 @Square0 | ret
@9002: call @SquareReg2 @Square0 | ret

@A000: call @SquareReg0 @Square1 | ret
@A001: call @SquareReg1 @Square1 | ret
@A002: call @SquareReg2 @Square1 | ret

; --pp pppp
@B000:

    call APUSynchronize
    and al 03F
    mov B@Sawtooth+Phase al
    ret

@B001: call @SquareReg1 @Sawtooth | ret
@B002: call @SquareReg2 @Sawtooth | ret
____________________________________________________________________________________________

; Dddd vvvv
@SquareReg0:

    call APUSynchronize
    pop edx ebx | push edx

    ; ---- vvvv
    mov B$ebx+Volume al
    and B$ebx+Volume 0F

    ; Dddd
    shr al 4
    ifFlag al 08, mov al 0F
    mov B$ebx+DutyCycleType al
    ret

; wwww wwww
@SquareReg1:

    call APUSynchronize
    pop edx, ebx | push edx

    ; wwww wwww
    mov B$ebx+TimerLatch al
    ret


; e--- wwww
@SquareReg2:

    call APUSynchronize
    pop edx ebx | push edx

    ; e--- ----
    test al 080 | setnz B$ebx+LengthEnabled

    ; ---- wwww
    and al 0F
    mov b$ebx+TimerLatch+1 al
    ret
____________________________________________________________________________________________

@TimerHook:

    add D@Square0+Timer  eax
    add D@Square1+Timer  eax
    add D@Sawtooth+Timer eax
    ret

@SampleHook:

    call @SampleSquare @Square0 | add edi D@Square0+Output
    call @SampleSquare @Square1 | add edi D@Square1+Output
    call @SampleSawtooth        | mov eax D@Sawtooth+Output | shr eax 3 | add edi eax
    ret

@SampleSquare:

    pop eax, ebx | push eax

or D$ebx+TimerLatch 01

    ; Count steps
    mov eax D$ebx+Timer, edx 0
    div D$ebx+TimerLatch
    mov D$ebx+Timer edx

    if B$ebx+LengthEnabled = &FALSE, ret

    ; Wave high or low?
    add eax D$ebx+DutyCycleCounter | and eax 0F
    mov D$ebx+DutyCycleCounter eax

    if. eax > D$ebx+DutyCycleType
        mov B$ebx+Output 0
    else
        copy D$ebx+Volume D$ebx+Output
    endif
    ret

@SampleSawtooth:

or D@SawTooth+TimerLatch 01

    ; Count steps
    mov eax D@Sawtooth+Timer, edx 0
    mov ecx D@SawTooth+TimerLatch | shl ecx 1
    div ecx
    mov D@Sawtooth+Timer edx

    if B@Sawtooth+LengthEnabled = &FALSE, mov eax 0
    mov ecx eax | jecxz Q0>
    mov eax D@Sawtooth+Phase

    ; Add to accumulator
L0: add B@Sawtooth+Output al
    inc D@Sawtooth+Counter7
    if. D@Sawtooth+Counter7 >= 7
        mov D@SawTooth+Counter7 0
        mov D@SawTooth+Output 0
    endif
    dec ecx | jnz L0<
Q0: ret

enumerate4 030, Counter7, Phase

____________________________________________________________________________________________
; FME-7 sound

; !!! Noise channel not implemented yet (Gimmick doesn't seem to use it)

; $C000-$DFFF: xxxxCCCx - command number
; $E000-$FFFF: (CCC = 00)
____________________________________________________________________________________________

FME7Sound:

    ; Channel registers
    CPUWrite 0C000, 0DFFF, @C000_DFFF
    CPUWrite 0E000, 0FFFF, @E000_FFFF

    ; Sample hook
    mov D$TimerHook  @TimerHook
    mov D$SampleHook @SampleHook

    ; Noise
    mov D@Noise+ShiftRegister 01
    ret

____________________________________________________________________________________________

[@Square1       ExChannel0
 @Square2       ExChannel1
 @Square3       ExChannel2
 @Envelope      ExChannel3
 @Noise         ExChannel4]

[Flags          LinearCounter
 NoVolume       LinearCounterHalt
 SquareToggle   LinearCounterLoad
 WaveForm       DutyCycleType]

____________________________________________________________________________________________

@C000_DFFF:

    mov edx 0, ecx 0E
    div ecx | mov D$Command edx
    ret

@E000_FFFF:

    call APUSynchronize
    mov edx D$Command

    ; Timers
    if. dl <= 01
        and edx 1
        shr W@Square1+TimerLatch 3 | mov B@Square1+TimerLatch+edx al
        shl W@Square1+TimerLatch 3
    else
    if. dl <= 03
        and edx 1
        shr W@Square2+TimerLatch 3 | mov B@Square2+TimerLatch+edx al
        shl W@Square2+TimerLatch 3
    else
    if. dl <= 05
        and edx 1
        shr W@Square3+TimerLatch 3 | mov B@Square3+TimerLatch+edx al
        shl W@Square3+TimerLatch 3
    else

    if. dl = 06
        and al 01F | shl al 3
        mov B@Noise+TimerLatch al
    else

    ; Channel select
    if. dl = 07
        not al
ifFlag al 038, outwrite
        mov dl al | and dl 08+01 | mov B@Square1+Flags dl | shr al 1
        mov dl al | and dl 08+01 | mov B@Square2+Flags dl | shr al 1
        mov dl al | and dl 08+01 | mov B@Square3+Flags dl
    else

    ; Volume
    if. dl = 8
        test al 010 | setnz B@Square1+NoVolume | jnz end
        and al 0F | mov B@Square1+Volume al
    else
    if. dl = 09
        test al 010 | setnz B@Square2+NoVolume | jnz end
        and al 0F | mov B@Square2+Volume al
    else
    if. dl = 0A
        test al 010 | setnz B@Square3+NoVolume | jnz end
        and al 0F | mov B@Square3+Volume al
    else

    ; Envelope frequency
    if. dl = 0B
        shr W@Envelope+TimerLatch 3 | mov B@Envelope+TimerLatch+Timer+00 al
        shl W@Envelope+TimerLatch 3
    else
    if. dl = 0C
        shr W@Envelope+TimerLatch 3 | mov B@Envelope+TimerLatch+Timer+01 al
        shl W@Envelope+TimerLatch 3
    else

    ; Envelope waveform
    if. dl = 0D
        and al 0F
        mov B@Envelope+WaveForm al
        mov B@Envelope+Step 0
    endif

    call @SquareActive @Square1
    call @SquareActive @Square2
    call @SquareActive @Square3
    ret

____________________________________________________________________________________________

@TimerHook:

    add D@Envelope+Timer eax
    add D@Noise+Timer eax
    add D@Square1+Timer eax
    add D@Square2+Timer eax
    add D@Square3+Timer eax
    ret

@SampleHook:

    call @SampleEnvelope
    call @SampleNoise
    call @SampleSquare @Square1 | mov eax D@Square1+Output | shr eax 1 | add edi eax
    call @SampleSquare @Square2 | mov eax D@Square2+Output | shr eax 1 | add edi eax
    call @SampleSquare @Square3 | mov eax D@Square3+Output | shr eax 1 | add edi eax
    ret

____________________________________________________________________________________________

@SquareActive:

    pop eax, ebx | push eax
    mov B$ebx+Active &FALSE
    if B$ebx+Volume > 0,
        if W$ebx+TimerLatch > 0,
            mov B$ebx+Active &TRUE
    ret

@SampleSquare:

    pop eax, ebx | push eax
    on B$ebx+Active = &FALSE, Q0>

    ; Count steps
    mov eax D$ebx+Timer, edx 0
    div D$ebx+TimerLatch
    mov D$ebx+Timer edx

    ; Toggle
    add eax D$ebx+SquareToggle
    mov D$ebx+SquareToggle eax

    if. B$ebx+Flags != 0

        ; Envelope instead of volume?
        if B$ebx+NoVolume = &TRUE,
            copy W@Envelope+Output W$ebx+Volume

        ; Square output
        ifFlag.. B$ebx+Flags 01
            mov al B$ebx+Volume
            ifNotFlag B$ebx+SquareToggle 01, mov al 0
        endif..

        ; Noise output
        ifFlag.. B$ebx+Flags 08
            mov al B$ebx+Volume
            if B@Noise+Output = 0, mov al 0
        endif..

        mov B$ebx+Output al

    endif
Q0: ret
____________________________________________________________________________________________

@SampleEnvelope:

    on W@Envelope+TimerLatch = 0, Q0>

    ; Count steps
    mov eax D@Envelope+Timer, edx 0
    div D@Envelope+TimerLatch
    mov D@Envelope+Timer edx
    mov ecx eax | jecxz Q0>

    ; Step
    mov ebx D@Envelope+WaveForm
L0: call D@WaveForms+ebx*4
    dec ecx | jnz L0<

    ; Update output
    ifFlag B@Envelope+WaveForm 04, not al
    and al 01F
    mov B@Envelope+Output al
Q0: ret

[@WaveForms:
 @Once,  @Once,   @Once,     @Once,
 @Once,  @Once,   @Once,     @Once,
 @Loop,  @Once,   @Triangle, @Hold,
 @Loop,  @Once,   @Triangle, @Hold]

@Once:

    mov al B@Envelope+Step
    if B@Envelope+Step < 01F, inc B@Envelope+Step
    ret

@Loop:

    mov al B@Envelope+Step | inc B@Envelope+Step
    ret

@Triangle:

    mov al B@Envelope+Step | inc B@Envelope+Step
    ifFlag al 020, xor al 01F
    ret

@Hold:

    if. B@Envelope+Step < 020
        mov al B@Envelope+Step
        inc B@Envelope+Step
    else
        mov al 0
    endif
    ret

@SampleNoise:

    on W@Noise+TimerLatch = 0, Q0>

    ; Count steps
    mov eax D@Noise+Timer, edx 0
    div D@Noise+TimerLatch
    mov D@Noise+Timer edx
    mov ecx eax | jecxz Q0>

    mov eax D@Noise+ShiftRegister
L0: ifFlag. eax 01
        ifFlag eax 02, xor B@Noise+Output 01
        xor eax 028000
    endif
    shr eax 1
    dec ecx | jnz L0<
    mov D@Noise+ShiftRegister eax
Q0: ret
____________________________________________________________________________________________
; MMC5 sound
____________________________________________________________________________________________

MMC5Sound:

    ; Channel registers
    CPUWrite 05000, 05000, @5000
    CPUWrite 05002, 05002, @5002
    CPUWrite 05003, 05003, @5003
    CPUWrite 05004, 05004, @5004
    CPUWrite 05006, 05006, @5006
    CPUWrite 05007, 05007, @5007
    CPUWrite 05010, 05010, @5010
    CPUWrite 05011, 05011, @5011
    CPUWrite 05015, 05015, @5015

    ; Reset squares
    mov W@Square1+MaxWavelength MAX_SQUARE_WAVELENGTH
    mov W@Square2+MaxWavelength MAX_SQUARE_WAVELENGTH

    ; APU hooks
    mov D$TimerHook @TimerHook
    mov D$SampleHook @SampleHook
    mov D$FrameCounterHook120Hz @120Hz
    mov D$FrameCounterHook240Hz @240Hz
    ret

[@Square1 ExChannel0
 @Square2 ExChannel1
 @PCM     ExChannel2]

@5000: call WriteSquareReg0 @Square1 | ret
@5002: call WriteSquareReg2 @Square1 | ret
@5003: call WriteSquareReg3 @Square1 | ret
@5004: call WriteSquareReg0 @Square2 | ret
@5006: call WriteSquareReg2 @Square2 | ret
@5007: call WriteSquareReg3 @Square2 | ret
@5010: call APUSynchronize | test al 01 | setnz B@PCM+LengthEnabled | ret  ; Discarded (Shin 4 Nin Uchi Mahjong - Yakuman Tengoku)
@5011: call APUSynchronize | mov B@PCM+Output al | ret
@5015:

    call APUSynchronize
    test al 01 | setnz B@Square1+LengthEnabled
    test al 02 | setnz B@Square2+LengthEnabled
    call SquareActive @Square1
    call SquareActive @Square2
    ret

____________________________________________________________________________________________

@TimerHook:

    add D@Square1+Timer eax
    add D@Square2+Timer eax
    ret

@SampleHook:

    call SampleSquare @Square1 | add edi D@Square1+Output
    call SampleSquare @Square2 | add edi D@Square2+Output
    mov eax D@PCM+Output | shr eax 3 | add edi eax
    ret

@120Hz:

    call ClockSweepUnit @Square1
    call ClockSweepUnit @Square2
    call ClockLengthCounter @Square1
    call ClockLengthCounter @Square2
    ret

@240Hz:

    call ClockEnvelopeDecay @Square1
    call ClockEnvelopeDecay @Square2
    ret

____________________________________________________________________________________________
TITLE VsSystem

ResetVsSystem:
;;

  // VS. Dual-System Games are not supported

      case 0xB90497AAUL: // Tennis
  case 0x008A9C16UL: // Wrecking Crew 
  case 0xAD407F52UL: // Balloon Fight
  case 0x18A93B7BUL: // Mahjong (J)
  case 0x13A91937UL: // Baseball 
  case 0xF5DEBF88UL: // Baseball 
  case 0xF64D7252UL: // Baseball 
  case 0x968A6E9DUL: // Baseball
  case 0xF42DAB14UL: // Ice Climber
  case 0x7D6B764FUL: // Ice Climber
;;
    ; Extra bits in $4016/$4017
    copy D$CPUReadTable+(04016 shl 2) D@fn4016
    copy D$CPUReadTable+(04017 shl 2) D@fn4017
    CPURead 04016, 04016, @Read4016
    CPURead 04017, 04017, @Read4017

    ; Coin
    CPUWrite 04020, 04020, @4020
    CPURead  04020, 04020, @Read4020

    ; Select palette
    PPUWrite 03F00, 03FFF, @WritePalette
    mov eax D$Cartridge@PROMCRC32
    mov D@PaletteIndex 0
    on eax = 017AE_56BE, Q0>> ; Freedom Force
    on eax = 0D99A_2087, Q0>> ; Gradius
    on eax = 0F063_2142, Q0>> ; Gradius [b1]
    on eax = 0FF51_35A3, Q0>> ; Hogan's Alley
    on eax = 0EC46_1DB9, Q0>> ; Pinball
    on eax = 0E2C0_A2BE, Q0>> ; Platoon, [b1], [a1]
    inc D@PaletteIndex
    on eax = 0FFBE_F374, Q0>> ; Castlevania
    on eax = 0CC2C_4B5D, Q0>> ; Golf, [b1], [b2]
    on eax = 08616_7220, Q0>> ; Ladies Golf
    on eax =  0B65_A917, Q0>> ; Mach Rider
    on eax = 08A6A_9848, Q0>> ; Mach Rider [b1]
    on eax = 07090_1B25, Q0>> ; Slalom
    inc D@PaletteIndex
    on eax = 07043_3F2C, Q0>> ; Battle City -
    on eax = 08D15_A6E6, Q0>> ; bad .nes    -
    on eax = 0D5D7_EAC4, Q0>> ; Dr. Mario
    on eax = 0CBE8_5490, Q0>> ; Excitebike
    on eax = 01E43_8D52, Q0>> ; Goonies     -
    on eax = 0135A_DF7C, Q0>> ; RBI Baseball -
    on eax = 04691_4E3E, Q0>> ; Soccer
    inc D@PaletteIndex
    on eax =  0713_8C06, Q0>> ; Clu Clu Land
    on eax = 02915_5E0C, Q0>> ; Excitebike [a1]
    on eax = 043A3_57EF, Q0>> ; Ice Climber, [o1]
    on eax = 08192_C804, Q0>> ; VS Super Mario Bros, [b2], [o1]
    on eax = 08B60_CC58, Q0>> ; VS Super Mario Bros [a1]
    on eax = 0737D_D1BF, Q0>> ; VS Super Mario Bros [b1]
    on eax = 04BF3_972D, Q0>> ; VS Super Mario Bros [b3], [b3][o1]
    on eax = 0E528_F651, Q0>> ; Pinball [a1]

    ; Palette not recognized
    PPUWrite 03F00, 03FFF, WritePalette
    outtext 'Unknown palette!'

Q0: ret

[@fn4016:       ?
 @fn4017:       ?
 @PaletteIndex: ?]
____________________________________________________________________________________________

@Read4016:

    call D@fn4016

    ; Add dip switch bits
    nop
    ret

@Read4017:

    call D@fn4017

    ; Add dip switch bits
    nop
    ret
____________________________________________________________________________________________

@4020:     ret
@Read4020: ret
____________________________________________________________________________________________

@WritePalette:

    and eax 03F
    mov ebx D@PaletteIndex | shl ebx 6
    if B$VsPalette+ebx+eax != 0FF, mov al B$VsPalette+ebx+eax
    jmp WritePalette

____________________________________________________________________________________________
; Mapper #099, NROM (VS)

; $4016: xxxxxCxx - swap 8k CROM at $0000
____________________________________________________________________________________________

Mapper099:

    ; Monitor $4016 writes
    copy D$CPUReadTable+(04016 shl 2) D@fn4016
    CPUWrite 04016, 04016, @4016
    ret

[@fn4016: ?]

@4016:

    mov edx eax | shr edx 2
    swap CROM, 8k, 00000, edx
    jmp D@fn4016

____________________________________________________________________________________________
; Mapper #151, CNROM (VS)

; $8000-$8FFF: PPPPPPPP - swap 8k PROM at $8000
; $A000-$AFFF: PPPPPPPP - swap 8k PROM at $8000
; $C000-$CFFF: PPPPPPPP - swap 8k PROM at $8000
; $E000-$EFFF: CCCCCCCC - swap 4k CROM at $0000
; $F000-$FFFF: CCCCCCCC - swap 4k CROM at $1000
____________________________________________________________________________________________

Mapper151:

    ; Memory mapping
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 0A000, 0AFFF, @A000_AFFF
    CPUWrite 0C000, 0CFFF, @C000_CFFF
    CPUWrite 0E000, 0EFFF, @E000_EFFF
    CPUWrite 0F000, 0FFFF, @F000_FFFF
    ret

; PPPPPPPP
@8000_8FFF: swap PROM, 8k, 08000, eax | ret
@A000_AFFF: swap PROM, 8k, 0A000, eax | ret
@C000_CFFF: swap PROM, 8k, 0C000, eax | ret
; CCCCCCCC
@E000_EFFF: swap CROM, 4k, 00000, eax | ret
@F000_FFFF: swap CROM, 4k, 01000, eax | ret
____________________________________________________________________________________________
; VS palettes
____________________________________________________________________________________________

[VsPalette: B$

  035, 0FF, 016, 022, 01C, 0FF, 0FF, 015, 0FF,  00, 027,  05,  04, 027,  08, 030,
  021, 0FF, 0FF, 029, 03C, 0FF, 036, 012, 0FF, 02B, 0FF, 0FF, 0FF, 0FF, 0FF,  01,
  0FF, 031, 0FF, 02A, 02C,  0C, 0FF, 0FF, 0FF,  07, 034,  06, 013, 0FF, 026,  0F,
  0FF, 019, 010,  0A, 0FF, 0FF, 0FF, 017, 0FF, 011,  09, 0FF, 0FF, 025, 018, 0FF

  0FF, 027, 018, 0FF, 03A, 025, 0FF, 031, 016, 013, 038, 034, 020, 023, 0FF,  0B,
  0FF, 021,  06, 0FF, 01B, 029, 0FF, 022, 0FF, 024, 0FF, 0FF, 0FF,  08, 0FF,  03,
  0FF, 036, 026, 033, 011, 0FF, 010,  02, 014, 0FF,  00,  09, 012,  0F, 0FF, 030,
  0FF, 0FF, 02A, 017,  0C,  01, 015, 019, 0FF, 02C,  07, 037, 0FF,  05, 0FF, 0FF

  0FF, 0FF, 0FF, 010, 01A, 030, 031,  09,  01,  0F, 036,  08, 015, 0FF, 0FF, 0F0,
  022, 01C, 0FF, 012, 019, 018, 017, 0FF,  00, 0FF, 0FF,  02, 016,  06, 0FF, 035,
  023, 0FF, 08B, 0F7, 0FF, 027, 026, 020, 029, 0FF, 021, 024, 011, 0FF, 0EF, 0FF,
  02C, 0FF, 0FF, 0FF,  07, 0F9, 028, 0FF,  0A, 0FF, 032, 037, 013, 0FF, 0FF,  0C

  018, 0FF, 01C, 089, 0FF, 0FF,  01, 017, 010,  0F, 02A, 0FF, 036, 037, 01A, 0FF,
  025, 0FF, 012, 0FF,  0F, 0FF, 0FF, 026, 0FF, 0FF, 022, 0FF, 0FF,  0F, 03A, 021,
   05,  0A,  07, 0C2, 013, 0FF,  00, 015,  0C, 0FF, 011, 0FF, 0FF, 038, 0FF, 0FF,
  0FF, 0FF,  08, 045, 0FF, 0FF, 030, 03C,  0F, 027, 0FF, 060, 029, 0FF, 030,  09]
____________________________________________________________________________________________
TITLE Debug
; Ports
[CPUReadTable:  ? #010000]
[CPUWriteTable: ? #010000]
[PPUReadTable:  ? #04000]
[PPUWriteTable: ? #04000]
[closeMessage
    msgError #1
    call ClearVideo
    call ClearAudio
    call UnloadNES]
;
[ASSERT | NOPE]
[dxErrorCheck | NOPE]
[winErrorCheck | NOPE]
[outWrite  | NOPE]
[outDec    | NOPE]
[outHex    | NOPE]
[outString | NOPE]
[outText   | NOPE]
[outError  | NOPE]
;;
____________________________________________________________________________________________
; Debug macros
____________________________________________________________________________________________

[outWrite  | call DebugOutputWrite]
[outDec    | &1=#1 | { &0: B$ '&1', 0 } | call DebugOutputDec    &0 #1]
[outHex    | &1=#1 | { &0: B$ '&1', 0 } | call DebugOutputHex    &0 #1]
[outString | &1=#1 | { &0: B$ '&1', 0 } | call DebugOutputString &0 #1]
[outText   | &1=#1 | { &0: B$  &1,  0 } | call DebugOutputText   &0 ]
[outError  | call DebugOutputError]

[ASSERT | #=3 | on #1 #2 #3, M0> | M0:]
[winErrorCheck | test eax eax | jnz M0> | outError | outText #1 | M0:]
[dxErrorCheck  | test eax eax | jns M0> | outhex eax | outText #1 |  M0:]
____________________________________________________________________________________________
; Debug window
____________________________________________________________________________________________

[NewLine:         B$ 13, 10,         0
 Equal:              ' = ',          0
 DebugWindowName:    'Debug window', 0
 EditClass:          'edit',         0]
[hDebugWindow: ?]

LoadDebugWindow:

  . D$hDebugWindow = 'User32.FindWindowA' EditClass, DebugWindowName
    if D$hDebugWindow != &NULL, ret
  . D$hDebugWindow = 'User32.CreateWindowExA' &NULL,
                                              EditClass,
                                              DebugWindowName,
                                              &WS_VISIBLE+&WS_OVERLAPPEDWINDOW+&ES_MULTILINE+&ES_LEFT+&ES_AUTOVSCROLL+&WS_VSCROLL+&WS_HSCROLL,
                                              10, 550,
                                              800, 180,
                                              D$hMainWindow,
                                              &NULL,
                                              D$hInstance,
                                              &NULL
    winErrorCheck 'Could not create window.'
    call 'User32.SetWindowTextA' D$hDebugWindow, &NULL
    call 'User32.SetForegroundWindow' D$hMainWindow
    ret
 ____________________________________________________________________________________________

DebugOutputString:

    arguments @pStringName, @pString
    pushad

        call LoadDebugWindow

        ; Output text
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, D@pStringName
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, Equal
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, D@pString
        call 'USER32.SendMessageA' D$hDebugWindow &EM_SCROLLCARET 0, 0
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, NewLine

    popad
    return

DebugOutputText:

    arguments @pText
    pushad

        call LoadDebugWindow

        ; Output text
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, D@pText
        call 'USER32.SendMessageA' D$hDebugWindow &EM_SCROLLCARET 0, 0
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, NewLine

    popad
    return

DebugOutputWrite:

    pushad

        push eax

            ; Address
            [@Address: B$ '$0000', 0]
            mov edi @Address+4, ecx 4
            std
            L0: mov eax edx
                and eax 0F | mov al B$HexTable+eax
                stosb
                shr edx 4 | dec ecx | jnz L0<
            cld
        
        pop edx
        
        ; Data
        [@Data: B$ '$00', 0]
        mov edi @Data+2, ecx 2
        std
        L0: mov eax edx
            and eax 0F | mov al B$HexTable+eax
            stosb
            shr edx 4 | dec ecx | jnz L0<
        cld

        call LoadDebugWindow

        ; Output text
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, @Address
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, Equal
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, @Data
        call 'USER32.SendMessageA' D$hDebugWindow &EM_SCROLLCARET 0, 0
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, NewLine

    popad
    ret

[HexTable: '0123456789ABCDEF']
DebugOutputHex:

    arguments @pHexName, @Number
    pushad

        call LoadDebugWindow

        ; Convert to hex
        [@HexString: B$ '$00000000', 0]
        mov edx D@Number, edi @HexString, ecx 8
        ifNotFlag edx 0FFFF0000, mov ecx 4
        ifNotFlag edx 0FFFFFF00, mov ecx 2
        add edi ecx
        mov B$edi+1 0
        std
        L0: mov eax edx
            and eax 0F | mov al B$HexTable+eax
            stosb
            shr edx 4 | dec ecx | jnz L0<
        cld

        ; Output text
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, D@pHexName
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, Equal
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, @HexString
        call 'USER32.SendMessageA' D$hDebugWindow &EM_SCROLLCARET 0, 0
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, NewLine

    popad
    return

DebugOutputDec:

    arguments @pDecName, @Number
    pushad

        call LoadDebugWindow

        ; Convert to decimal
        [@DecString: B$ '0000000000000', 0]
        mov eax D@Number, edi @DecString
        mov dl 0FF | push edx
        mov ecx 10
    L0: mov edx 0 | div ecx
        push edx | on eax > 0, L0<
    L0: pop eax
        if. al != 0FF
            add al '0'
            stosb
            jmp L0<
        endif
        mov B$edi 0

        ; Output text
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, D@pDecName
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, Equal
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, @DecString
        call 'USER32.SendMessageA' D$hDebugWindow &EM_SCROLLCARET 0, 0
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, NewLine

    popad
    return

DebugOutputError:

    pushad

        [@pMessage: ?]
        call 'Kernel32.GetLastError'
        call 'Kernel32.FormatMessageA' &FORMAT_MESSAGE_ALLOCATE_BUFFER+&FORMAT_MESSAGE_FROM_SYSTEM,
                                       &NULL,
                                       eax,
                                       &NULL,
                                       @pMessage,
                                       &NULL,
                                       &NULL
        call LoadDebugWindow

        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, D@pMessage
        call 'USER32.SendMessageA' D$hDebugWindow &EM_SCROLLCARET 0, 0
        call 'USER32.SendMessageA' D$hDebugWindow &EM_REPLACESEL &FALSE, NewLine
        call 'Kernel32.LocalFree' D$@pMessage

    popad
    ret
____________________________________________________________________________________________
; Emulation debugging
____________________________________________________________________________________________

DrawNameTable:

    mov esi NameTable, edi VideoBuffer
    mov edx 28
L1: pushad | mov ecx 32 | L0: lodsb | stosb | stosb | stosb | stosb | loop L0< | popad | add edi 0100 | add esi 32
    pushad | mov ecx 32 | L0: lodsb | stosb | stosb | stosb | stosb | loop L0< | popad | add edi 0100 | add esi 32
    pushad | mov ecx 32 | L0: lodsb | stosb | stosb | stosb | stosb | loop L0< | popad | add edi 0100 | add esi 32
    pushad | mov ecx 32 | L0: lodsb | stosb | stosb | stosb | stosb | loop L0< | popad | add edi 0100 | add esi 32
    dec edx | jnz L1<
    ret

[Emulate:
 @SoundSquare1:  &TRUE
 @SoundSquare2:  &TRUE
 @SoundTriangle: &TRUE
 @SoundNoise:    &TRUE
 @SoundDMC:      &TRUE
 @Background:    &TRUE
 @Sprites:       &TRUE
 @SpriteZero:    &FALSE]
;;
____________________________________________________________________________________________
TITLE Video
____________________________________________________________________________________________
; Init/shutdown
____________________________________________________________________________________________

VideoInit:

    ; IDirectDraw interface
    call 'DDraw.DirectDrawCreate' &NULL, lpdd, &NULL
    dxErrorCheck 'IDirectDraw creation failed.'
    if D$lpdd = &NULL, ret

    ; Set cooperative level
    dxCall lpdd,SetCooperativeLevel D$hMainWindow, &DDSCL_NORMAL
    dxErrorCheck 'Could not set cooperative level.'

    ; Create primary surface
    call ZeroMemory ddsd (SIZE_DDSD shl 2)
    mov D$ddsd@dwSize (SIZE_DDSD shl 2)
    mov D$ddsd@ddsCaps &DDSCAPS_PRIMARYSURFACE
    mov D$ddsd@dwFlags &DDSD_CAPS
    dxCall lpdd,CreateSurface ddsd, lpddsPrimary, &NULL
    dxErrorCheck 'Primary surface creation failed.'
    if D$lpddsPrimary = &NULL, ret

    ; Create second surface
    call ZeroMemory ddsd (SIZE_DDSD shl 2)
    mov D$ddsd@dwSize (SIZE_DDSD shl 2)
    mov D$ddsd@ddsCaps &DDSCAPS_OFFSCREENPLAIN
    mov D$ddsd@dwFlags &DDSD_CAPS+&DDSD_WIDTH+&DDSD_HEIGHT
    mov D$ddsd@dwWidth  WIDTH_VIDEO_BUFFER
    mov D$ddsd@dwHeight HEIGHT_VIDEO_BUFFER
    dxCall lpdd,CreateSurface ddsd, lpddsSecond, &NULL
    dxErrorCheck 'Secondary surface creation failed.'
    if D$lpddsSecond = &NULL, ret

    ; Attach a clipper to D$hMainWindow
    dxCall lpdd,CreateClipper &NULL, lpddclipper, &NULL
    dxErrorCheck 'Could not create clipper.'
    if. D$lpddclipper != &NULL
        dxCall lpddclipper,SetHWnd &NULL, D$hMainWindow
        dxErrorCheck 'Could not set clipping window.'
        dxCall lpddsPrimary,SetClipper D$lpddclipper
        dxErrorCheck 'Could not attach clipper to surface.'
        dxRelease lpddclipper
        dxErrorCheck 'Could not release clipper on initialization.'
    endif

    ; Retrieve color depth
    call ZeroMemory ddsd (SIZE_DDSD shl 2)
    mov D$ddsd@ddpfPixelFormat.dwSize 32
    dxCall lpddsPrimary,GetPixelFormat ddsd@ddpfPixelFormat
    dxErrorCheck 'Could not retrieve pixel format for primary surface.'
    copy D$ddsd@ddpfPixelFormat.dwRGBbitCount D$ColorDepth      ; <-- Very readable name ;-) (34 letters)
    if. D$ColorDepth = 15 | msgError '15-bit color depth not supported' | endif

    ; Arrange 16-bit palette (0.8.8.8 DWORDs  -->  5.6.5 WORDs)
    call TranslatePalette

    ; Clear screen
    call UpdateClientRect
    call ClearVideo
    call FlushVideo
    ret

VideoShutDown:

    ; Release interfaces
    dxRelease lpddsSecond  | dxErrorCheck 'Second surface release failed.'
    dxRelease lpddsPrimary | dxErrorCheck 'Primary surface release failed.'
    dxRelease lpdd         | dxErrorCheck 'IDirectDraw release failed.'
    ret

____________________________________________________________________________________________
; Video output
____________________________________________________________________________________________

VideoOutput: ; Nes palette index in al

    ; Plot pixel
    mov ebx D$VideoCursor | inc D$VideoCursor
    mov B$VideoBuffer+ebx al

    ; Buffer full?
    on D$VideoCursor >= LENGTH_VIDEO_BUFFER, FlushVideo
    ret

FlushVideo:

    mov D$VideoCursor 0
    if D$lpddsSecond = &NULL, ret

    ; Lock secondary surface
    call ZeroMemory ddsd (SIZE_DDSD shl 2) | mov D$ddsd@dwSize SIZE_DDSD+SIZE_DDSD+SIZE_DDSD+SIZE_DDSD
    dxCall lpddsSecond,Lock &NULL, ddsd, &DDLOCK_SURFACEMEMORYPTR+&DDLOCK_WAIT, &NULL
    dxErrorCheck 'Surface lock failed.'

    ; Copy screen data to surface
    mov esi VideoBuffer, edi D$ddsd@lpSurface, edx HEIGHT_VIDEO_BUFFER
L0: mov ecx WIDTH_VIDEO_BUFFER

    push edi

        if. D$ColorDepth = 32 | L1: movzx eax B$esi | inc esi | mov eax D$eax*4+Palette32 | stosd | dec ecx | jnz L1< | else
        if. D$ColorDepth = 24 | L1: movzx eax B$esi | inc esi | mov eax D$eax*4+Palette32 | stosw | shr eax 16 | stosb | dec ecx | jnz L1< | else
        if. D$ColorDepth = 16 | L1: movzx eax B$esi | inc esi | mov  ax W$eax*2+Palette16 | stosw | dec ecx | jnz L1< | else
        if. D$ColorDepth = 15 | L1: movzx eax B$esi | inc esi | mov  ax W$eax*2+Palette15 | stosw | dec ecx | jnz L1< | else
        if. D$ColorDepth = 8  | L1: movzx eax B$esi | inc esi |                           | stosb | dec ecx | jnz L1< | endif

    pop edi

    add edi D$ddsd@lPitch
    dec edx | jnz L0<<

    ; Unlock secondary, blit it to primary
    dxCall lpddsSecond,Unlock &NULL
    call UpdateVideo
    ret

UpdateVideo:

    ; Blit enabled?
    if D$Window@Minimized = &TRUE, ret
    if D$Window@Existing = &FALSE, ret
    if D$lpddsPrimary = &NULL, ret
    if D$lpddsSecond = &NULL, ret

    ; Blit secondary to primary
    dxCall lpddsPrimary,Blt ClientRect, D$lpddsSecond, SourceRect, &DDBLT_ASYNC, &NULL
    dxErrorCheck 'Could not blit NES surface.'
    ret

ClearVideo:

    call ZeroMemory VideoBuffer LENGTH_VIDEO_BUFFER

    push D$VideoCursor

        call FlushVideo

    pop D$VideoCursor
    ret

____________________________________________________________________________________________
; Screenshot
____________________________________________________________________________________________

; Save '.\saves\[ROMName]\snapXXXX.bmp'
[ScreenshotFilename:  B$ 'snap', 0]
[ScreenshotExtension: B$ '.bmp', 0]
SaveScreenShot: call SaveNumberedROMFile ScreenshotFilename, ScreenshotExtension | ret

SaveBMP:

    ; BMP file header
    mov edi SaveData
    call CopyMemory BitmapFileHeader, edi, 14          | add edi 14
    call CopyMemory BitmapInfoHeader, edi, 40          | add edi 40
    call CopyMemory Palette32,        edi, (040 shl 2) | add edi (040 shl 2)

    ; Copy line by line in reversed order
    mov esi VideoBuffer+LENGTH_VIDEO_BUFFER-0900
    mov ecx HEIGHT_VIDEO_BUFFER-17
L0: call CopyMemory esi, edi, WIDTH_VIDEO_BUFFER
    sub esi WIDTH_VIDEO_BUFFER
    add edi WIDTH_VIDEO_BUFFER
    dec ecx | jnz L0<

    ; Set size of saved file
    sub edi SaveData | mov D$SaveSize edi
    ret

____________________________________________________________________________________________
; Fullscreen/windowed mode
____________________________________________________________________________________________

ToggleFullscreen:

    if. B$DisplayMode = DM_WINDOWED
        call SetFullscreen
    else
        call SetWindowed
    endif
    ret

SetFullscreen:

    ; Main window not created yet? (Happens on 'run maximized')
    if D$hMainWindow = &NULL, copy D$hWindow D$hMainWindow

    ; Set fullscreen style
    mov B$DisplayMode DM_FULLSCREEN
    call 'User32.CheckMenuItem' D$hMenu, IDM_FULLSCREEN, &MF_BYCOMMAND+&MF_CHECKED
    call 'User32.SetWindowLongA' D$hMainWindow, &GWL_STYLE, FULLSCREEN_STYLE
    winErrorCheck 'Failed to set fullscreen style.'

    ; Resize window to screen borders, put it on top
    push &NULL
    call 'User32.GetSystemMetrics' &SM_CYSCREEN | push eax
    call 'User32.GetSystemMetrics' &SM_CXSCREEN | push eax
    call 'User32.SetWindowPos' D$hMainWindow, &HWND_TOPMOST, 0, 0 ; width, height, flags

    ; Hide cursor?
    call UpdateCursor
    ret

SetWindowed:

    ; Main window not created yet?
    if D$hMainWindow = &NULL, copy D$hWindow D$hMainWindow

    ; Set windowed style
    mov B$DisplayMode DM_WINDOWED
    call 'User32.CheckMenuItem' D$hMenu, IDM_FULLSCREEN, &MF_BYCOMMAND+&MF_UNCHECKED
    call 'User32.SetWindowLongA' D$hMainWindow, &GWL_STYLE, WINDOW_STYLE
    winErrorCheck 'Failed to set windowed style.'

    ; Resize window to saved size
    mov eax D$WindowRect+8  | sub eax D$WindowRect+0
    mov edx D$WindowRect+12 | sub edx D$WindowRect+4
    call 'User32.SetWindowPos' D$hMainWindow, &HWND_NOTOPMOST, D$WindowRect, D$WindowRect+4, eax, edx, &NULL

    ; Show cursor
    call UpdateCursor
    ret

____________________________________________________________________________________________
; Palette
____________________________________________________________________________________________

; Save '.\saves\[ROMName]\palette.pal'
[PaletteFilename: B$ 'palette.pal',  0]
SavePalette: call SaveROMFile PaletteFilename | ret

SavePAL:

    ; Copy from 32-bit palette
    mov esi Palette32, edi SaveData, ecx 040
L0: lodsd
    mov dx ax
    shr eax 16 | stosb
    mov al dh  | stosb
    mov al dl  | stosb
    dec ecx | jnz L0<

    ; Set size to 192 bytes
    mov D$SaveSize 0C0
    ret

LoadPAL:

    ; Either 256 RGB values (stupid) or 64 (like the NES)
    if D$FileSize != 0300,
        if D$FileSize != 0C0, ret

    ; RRRRRRRR GGGGGGGG BBBBBBBB --> 00000000 RRRRRRRR GGGGGGGG BBBBBBBB
    mov esi D$pFile, edi Palette32, ecx 040
L0: mov eax 0
    lodsb | shl eax 8
    lodsb | shl eax 8
    lodsb | stosd
    dec ecx | jnz L0<

    ; Translate to other color depths
    call TranslatePalette
    ret
____________________________________________________________________________________________

TranslatePalette:

    ; Translate to 16-bit palette
    if. D$ColorDepth = 16
        ; 00000000 RRRRRRRR GGGGGGGG BBBBBBBB --> RRRRRGGG GGGBBBBB
        mov esi Palette32, edi Palette16
    L0: lodsb | movzx edx al
        lodsb | movzx ecx al
        lodsb | movzx ebx al
        lodsb
        shr ebx 3
        shr ecx 2
        shr edx 3
        and bl 01F | shl ebx 11
        and cl 03F | shl ecx 5
        and dl 01F

        mov eax ebx
        or  eax ecx
        or  eax edx

        stosw
        on edi < Palette16+080, L0<<
    endif

    ; Translate to 16-bit palette
    if. D$ColorDepth = 15
        ; 00000000 RRRRRRRR GGGGGGGG BBBBBBBB --> xRRRRRGG GGGBBBBB
        mov esi Palette32, edi Palette15
    L0: lodsb | movzx edx al
        lodsb | movzx ecx al
        lodsb | movzx ebx al
        lodsb
        shr ebx 3
        shr ecx 3
        shr edx 3
        and bl 01F | shl ebx 10
        and cl 01F | shl ecx 5
        and dl 01F

        mov eax ebx
        or  eax ecx
        or  eax edx

        stosw
        on edi < Palette15+080, L0<<
    endif

    ; Translate to 8-bit palette
    if. D$ColorDepth = 8

        ; 00000000 RRRRRRRR GGGGGGGG BBBBBBBB --> ???????????????
        mov esi Palette32, edi Palette8, ecx 040
    L0: movsd ; Some translation process is supposed to take place here... ?
        dec ecx | jnz L0<

        ; Create a new palette and attach it to primary surface
        if D$lpdd = &NULL, break
        dxCall lpdd,CreatePalette &DDPCAPS_8BIT+&DDPCAPS_INITIALIZE+&DDPCAPS_ALLOW256, Palette8, lpddpal, &NULL
        dxErrorCheck 'Could not create palette'
        if D$lpddpal = &NULL, break
        if D$lpddsPrimary = &NULL, break
        dxCall lpddsPrimary,SetPalette D$lpddpal
        dxErrorCheck 'Could not set palette'
        dxRelease lpddpal

    endif
    ret

____________________________________________________________________________________________
; Update client area coordinates
____________________________________________________________________________________________

UpdateClientRect:

    if D$Window@Minimized = &TRUE, ret

    ; Get new coordinates
    call 'User32.GetClientRect'  D$hMainWindow, ClientRect
    call 'User32.ClientToScreen' D$hMainWindow, ClientRect
    call 'User32.ClientToScreen' D$hMainWindow, ClientRect+8

    ; Do not stretch to screen if fullscreen
    if. D$DisplayMode = DM_FULLSCREEN

        ; New width --> eax
        mov eax D$ClientRect+12
        mov ecx D$ScreenWidth  | mul ecx
        mov ecx D$ScreenHeight | div ecx

        ; Adjust x coordinates
        mov edx D$ClientRect+8 | sub edx eax
        shr edx 1 | add eax edx
        mov D$ClientRect+0 edx
        mov D$ClientRect+8 eax

    endif

    ; Adjust top border if menu is present
    call 'User32.GetMenu' D$hMainWindow
    if. eax != &NULL
        call 'User32.GetSystemMetrics' &SM_CYMENU
        sub D$ClientRect+4 eax
    endif
    ret

____________________________________________________________________________________________
; DirectDraw stuff
____________________________________________________________________________________________

; Interface pointers
[lpdd:          ?
 lpddsPrimary:  ?
 lpddsSecond:   ?
 lpddclipper:   ?
 lpddpal:       ?]

; Input buffer
[VideoCursor:      ?
 VideoBuffer:   B$ ? #LENGTH_VIDEO_BUFFER]
[LENGTH_VIDEO_BUFFER <(HEIGHT_VIDEO_BUFFER shl 8)>]
[WIDTH_VIDEO_BUFFER  256
 HEIGHT_VIDEO_BUFFER 241]

; Output buffer
[ClientRect: ? #4
 ColorDepth: ?]
[SourceRect: 0, NTSC_SCREEN_TOP, 0100, NTSC_SCREEN_BOTTOM]

; Surface description
[SIZE_DDSD 27]
[ddsd:
 @dwSize:                               ?
 @dwFlags:                              ?
 @dwHeight:                             ?
 @dwWidth:                              ?
 @lPitch:                               ?
 @dwBackBufferCount:                    ?
 @dwRefreshRate:                        ?
 @dwAlphaBitDepth:                      ?
 @dwReserved:                           ?
 @lpSurface:                            ?
 @ddckCKDestOverlay:                    ? ?
 @ddckCKDestBlt:                        ? ?
 @ddckCKSrcOverlay:                     ? ?
 @ddckCKSrcBlt:                         ? ?
 @ddpfPixelFormat:
 @ddpfPixelFormat.dwSize:               ?
 @ddpfPixelFormat.dwFlags:              ?
 @ddpfPixelFormat.dwFourCC:             ?
 @ddpfPixelFormat.dwRGBBitCount:        ?
 @ddpfPixelFormat.dwRBitMask:           ?
 @ddpfPixelFormat.dwGBitMask:           ?
 @ddpfPixelFormat.dwBBitMask:           ?
 @ddpfPixelFormat.dwRGBAlphaBitMask:    ?
 @ddsCaps:                              ?]

; NES Palette
[Palette32: D$ ? #040   ; 00000000 RRRRRRRR GGGGGGGG BBBBBBBB
 Palette16: W$ ? #040   ;                   RRRRRGGG GGGBBBBB
 Palette15: W$ ? #040   ;                   xRRRRRGG GGGBBBBB
 Palette8:  D$ ? #0100] ; 00000000 RRRRRRRR GGGGGGGG BBBBBBBB

____________________________________________________________________________________________
; Screenshot data
____________________________________________________________________________________________

[BitmapFileHeader:
 @bfType:           W$ 'BM'
 @bfSize:           D$ 14+40+(040 shl 2)+(LENGTH_VIDEO_BUFFER-(17 shl 8))
 @bfReserved1:      W$ 0
 @bfReserved2:      W$ 0
 @bfOffBits:        D$ 14+40+(040 shl 2)

 BitmapInfoHeader:
 @biSize:           D$ 40
 @biWidth:          D$ WIDTH_VIDEO_BUFFER
 @biHeight:         D$ (HEIGHT_VIDEO_BUFFER-17)
 @biPlanes:         W$ 1
 @biBitCount:       W$ 8
 @biCompression:    D$ &BI_RGB
 @biSizeImage:      D$ 0
 @biXPelsPerMeter:  D$ 0
 @biYPelsPerMeter:  D$ 0
 @biClrUsed:        D$ 040
 @biClrImportant:   D$ 040]

____________________________________________________________________________________________
; DirectDraw vtables
____________________________________________________________________________________________

; IUnknown
[QueryInterface          0
 AddRef                  4
 Release                 8

; IDirectDraw
 Compact                12
 CreateClipper          16
 CreatePalette          20
 CreateSurface          24
 DuplicateSurface       28
 EnumDisplayModes       32
 EnumSurfaces           36
 FlipToGDISurface       40
 GetCaps                44
 GetDisplayMode         48
 GetFourCCCodes         52
 GetGDISurface          56
 GetMonitorFrequency    60
 GetScanLine            64
 GetVerticalBlankStatus 68
 Initialize             72
 RestoreDisplayMode     76
 SetCooperativeLevel    80
 SetDisplayMode         84
 WaitForVerticalBlank   88

; IDirectDrawSurface
 AddAttachedSurface     12
 AddOverlayDirtyRect    16
 Blt                    20
 BltBatch               24
 BltFast                28
 DeleteAttachedSurface  32
 EnumAttachedSurfaces   36
 EnumOverlayZOrders     40
 Flip                   44
 GetAttachedSurface     48
 GetBltStatus           52
 sGetCaps               56
 GetClipper             60
 GetColorKey            64
 GetDC                  68
 GetFlipStatus          72
 GetOverlayPosition     76
 GetPalette             80
 GetPixelFormat         84
 GetSurfaceDesc         88
 sInitialize            92
 IsLost                 96
 Lock                  100
 ReleaseDC             104
 Restore               108
 SetClipper            112
 SetColorKey           116
 SetOverlayPosition    120
 SetPalette            124
 Unlock                128
 UpdateOverlay         132
 UpdateOverlayDisplay  136
 UpdateOverlayZOrder   140

; IDirectDrawClipper
 GetClipList           12
 GetHWnd               16
 cInitialize           20
 IsClipListChanged     24
 SetClipList           28
 SetHWnd               32]
____________________________________________________________________________________________
TITLE Audio

____________________________________________________________________________________________
; Init/shutdown
____________________________________________________________________________________________

AudioInit:

    ; IDirectSound interface
    call 'DSound.DirectSoundCreate' &NULL, lpds, &NULL
    dxErrorCheck 'Could not create DirectSound interface.'
    if D$lpds = &NULL, ret

    ; Set cooperative level
    dxCall lpds,dsSetCooperativeLevel D$hMainWindow, &DSSCL_NORMAL
    dxErrorCheck 'Could not set cooperative level.'

    ; Create a secondary sound buffer
    dxCall lpds,CreateSoundBuffer dsbd, lpdsb, &NULL
    dxErrorCheck 'Could not create sound buffer.'
    if D$lpdsb = &NULL, ret

    ; Start playing buffer
    call PlayAudio
    ret

AudioShutdown:

    ; Release interfaces
    dxRelease lpdsb | dxErrorCheck 'Could not release DirectSound buffer.'
    dxRelease lpds  | dxErrorCheck 'Could not release DirectSound interface.'
    ret

____________________________________________________________________________________________
; Sound output
____________________________________________________________________________________________

AudioOutput: ; 8-bit sample in al

    ; Write byte to buffer
    mov edx D$AudioCursor
    inc D$AudioCursor
    mov B$edx+AudioBuffer al

    ; Buffer full?
    on D$AudioCursor >= SOUND_BUFFER_LENGTH, FlushAudio
    ret

FlushAudio:

    ; Reset cursor
    mov D$AudioCursor 0
    if B$AudioActive = &FALSE, ret
    if D$lpdsb =  &NULL, ret

    ; Position of the block to write -> WriteCursor
    inc D$nDSBuffer | and D$nDSBuffer 07
    mov ecx D$nDSBuffer
    mov eax SOUND_BUFFER_LENGTH
    mul ecx
    mov D$WriteCursor eax

    ; Wait for block to finish playing
L0: dxCall lpdsb,GetCurrentPosition Temp, &NULL
    dxErrorCheck 'Could not retrieve sound position.'
    mov eax D$Temp, edx 0, ecx SOUND_BUFFER_LENGTH
    div ecx
    if. eax = D$nDSBuffer
        ; Give CPU time to other applications
;        call 'Kernel32.SwitchToThread'
        call 'Kernel32.Sleep' 0
        jmp L0<
;;
        on eax > 0, L0<
        ; Give CPU time to message loop
        call CheckMessageQueue
        on eax = 0, L0<
        call 'User32.PostQuitMessage' 0 | jmp L0<
;;
    endif

    ; Lock output buffer
    dxCall lpdsb,dsbLock D$WriteCursor, SOUND_BUFFER_LENGTH, pAudio1, lAudio1, pAudio2, lAudio2, 0
    dxErrorCheck 'Sound buffer lock failed.'

    ; Write block #1
    mov esi AudioBuffer
    mov edi D$pAudio1
    mov ecx D$lAudio1
    rep movsb

    ; Write block #2 (most probably not necessary, or something's wrong...)
    if. D$lAudio2 != 0
        mov edi D$pAudio2
        mov ecx D$lAudio2
        rep movsb
    endif

    ; Unlock output buffer
    dxCall lpdsb,dsbUnlock D$pAudio1, D$lAudio1, D$pAudio2, D$lAudio2
    dxErrorCheck 'Could not unlock sound buffer.'
    ret

PlayAudio:

    if B$AudioActive = &TRUE, ret
    if D$lpdsb = &NULL, ret

    ; Start playing buffer
    mov D$AudioActive &TRUE
    dxCall lpdsb,Play 0, 0, &DSBPLAY_LOOPING
    dxErrorCheck 'Could not play audio.'
    ret

StopAudio:

    if B$AudioActive = &FALSE, ret
    if D$lpdsb = &NULL, ret

    ; Stop playing buffer
    call ClearAudio
    mov D$AudioActive &FALSE
    dxCall lpdsb,Stop
    dxErrorCheck 'Could not stop playing audio.'
    ret

ClearAudio:

    if D$lpdsb = &NULL, ret

    ; Lock output buffer
    dxCall lpdsb,dsbLock 0, DSB_LENGTH, pAudio1, lAudio1, pAudio2, lAudio2, &DSBLOCK_ENTIREBUFFER
    dxErrorCheck 'Sound buffer lock failed on ClearAudio.'

    ; Clear it
    mov edi D$pAudio1
    if. edi != &NULL
        mov al B$edi
        mov ecx DSB_LENGTH
        rep stosb
    endif

    ; Unlock
    dxCall lpdsb,dsbUnlock D$pAudio1, D$lAudio1, D$pAudio2, D$lAudio2
    dxErrorCheck 'Could not unlock sound buffer.'
    ret

____________________________________________________________________________________________
; Data
____________________________________________________________________________________________

; Interface pointers
[lpds:  ?
 lpdsb: ?]

; Input buffer
[AudioBuffer:   ? #SOUND_BUFFER_LENGTH
 AudioCursor:   ?]

; Output buffer
[AudioActive:   ?
 nDSBuffer:     ?
 WriteCursor:   ?
 pAudio1:       ?
 lAudio1:       ?
 pAudio2:       ?
 lAudio2:       ?]

; Equates
[SAMPLE_RATE          44100
 SOUND_BUFFER_LENGTH  734]

; WaveFormatEx
[WAVEFORMATEX:
 @wFormatTag:        W$ &WAVE_FORMAT_PCM
 @nChannels:         W$ 1
 @nSamplesPerSec:    D$ SAMPLE_RATE
 @nAvgBytesPerSec:   D$ SAMPLE_RATE
 @nBlockAlign:       W$ 1
 @nBitsPerSample:    W$ 8
 @cbSize:            W$ 0]

; DirectSoundBuffer description
[DSB_LENGTH <(SOUND_BUFFER_LENGTH shl 3)>]
[dsbd:
 @dwSize:         len
 @dwFlags:        &DSBCAPS_GETCURRENTPOSITION2
 @dwBufferBytes:  DSB_LENGTH
 @dwReserved:     0
 @lpwfxFormat:    WAVEFORMATEX]

____________________________________________________________________________________________
; DirectSound vtables
____________________________________________________________________________________________

; IDirectSound
[CreateSoundBuffer      12
 dsGetCaps              16
 DuplicateSoundBuffer   20
 dsSetCooperativeLevel  24
 dsCompact              28
 GetSpeakerConfig       32
 SetSpeakerConfig       36
 dsInitialize           40

; IDirectSoundBuffer
 dsbGetCaps             12
 GetCurrentPosition     16
 GetFormat              20
 GetVolume              24
 GetPan                 28
 GetFrequency           32
 GetStatus              36
 dsbInitialize          40
 dsbLock                44
 Play                   48
 SetCurrentPosition     52
 SetFormat              56
 SetVolume              60
 SetPan                 64
 SetFrequency           68
 Stop                   72
 dsbUnlock              76
 dsbRestore             80]

____________________________________________________________________________________________
TITLE Input

____________________________________________________________________________________________
; Init/shutdown
____________________________________________________________________________________________

InputInit:

    ; IDirectInput interface
    call 'DInput.DirectInputCreateA' D$hInstance, &DIRECTINPUT_VERSION, lpdi, &NULL
    dxErrorCheck 'Could not create DirectInput'
    if D$lpdi = &NULL, ret

    ; Init devices
    call Init_rgodfKeyboard
    call InitDevices
    ret

InputShutdown:

    ; Release devices
L0: if. D$nDevices > 0
        dec D$nDevices
        mov edx D$nDevices | dxCall lpdid+edx*4,UnAcquire
        mov edx D$nDevices | dxRelease lpdid+edx*4
        jmp L0<
    endif

    ; Release IDirectInput interface
    dxRelease lpdi
    ret
____________________________________________________________________________________________

InitDevices:

    ; Enumerate devices
    dxCall lpdi,EnumDevices &DIDEVTYPE_KEYBOARD, DIEnumDevices, &NULL, &DIEDFL_ATTACHEDONLY
    dxErrorCheck 'Could not enumerate keyboards'
    dxCall lpdi,EnumDevices &DIDEVTYPE_JOYSTICK, DIEnumDevices, &NULL, &DIEDFL_ATTACHEDONLY
    dxErrorCheck 'Could not enumerate joysticks'

    ; Set up all devices
    mov ebx 0
L0: if ebx = D$nDevices, ret

    ; Create device
    push ebx

        move edx lpdid+ebx*4
        shl ebx 4 | add ebx GUID
        pushad
            dxCall lpdi,CreateDevice ebx, Temp, &NULL
            dxErrorCheck 'Could not create DirectInput device'
        popad
        dxCall Temp,QueryInterface IID_IDirectInputDevice2, edx
        dxRelease Temp

    pop ebx
    on D$lpdid+ebx*4 = &NULL, L0<<

    ; Set cooperative level
    push ebx

        dxCall lpdid+ebx*4,didSetCooperativeLevel D$hMainWindow, &DISCL_BACKGROUND+&DISCL_NONEXCLUSIVE
        dxErrorCheck 'Could not set device cooperation level'

    pop ebx

    ; Set data format
    push ebx

        if. B$DeviceType+ebx*4 = &DIDEVTYPE_KEYBOARD
            copy D$c_dfDIKeyboard@dwDataSize D$LengthDeviceState+ebx*4
            mov D$pButtonNames+ebx*4 KeyboardNames
            dxCall lpdid+ebx*4,SetDataFormat c_dfDIKeyboard
        else
            copy D$c_dfDIJoystick@dwDataSize D$LengthDeviceState+ebx*4
            mov D$pButtonNames+ebx*4 JoystickNames
            dxCall lpdid+ebx*4,SetDataFormat c_dfDIJoyStick
        endif
        dxErrorCheck 'Could not set data format'

    pop ebx

    ; Joystick-specific properties
    if. B$DeviceType+ebx*4 = &DIDEVTYPE_JOYSTICK

        [DIProperty:
         @dwSize:        ?
         @dwHeaderSize:  ?
         @dwObj:         ?
         @dwHow:         ?
         @lMin: @dwData: ?
         @lMax:          ?]

        ; Set dead zone
        push ebx

            mov D$DIProperty@dwSize       014
            mov D$DIProperty@dwHeaderSize 010
            mov D$DIProperty@dwObj        &NULL
            mov D$DIProperty@dwHow        &DIPH_DEVICE
            mov D$DIProperty@dwData       5000

            dxCall lpdid+ebx*4,SetProperty &DIPROP_DEADZONE, DIProperty
            dxErrorCheck 'Could not set dead zone'

        pop ebx

        ; Set range
        push ebx

            mov D$DIProperty@dwSize       018
            mov D$DIProperty@dwHeaderSize 010
            mov D$DIProperty@dwObj        &NULL
            mov D$DIProperty@dwHow        &DIPH_DEVICE
            mov D$DIProperty@lMin         0-1
            mov D$DIProperty@lMax         0+1

            dxCall lpdid+ebx*4,SetProperty &DIPROP_RANGE, DIProperty
            dxErrorCheck 'Could not set range'

        pop ebx

    endif

    ; Set pointer to button assignments
    mov eax ebx | shl eax 8
    move eax Mappings+eax*4 | mov D$pMappings+ebx*4 eax
    inc ebx | jmp L0<<

DIEnumDevices:

    arguments @pDeviceInstance, @pvRef
    pushad

        ; Save type
        mov ebx D$nDevices | inc D$nDevices
        mov esi D@pDeviceInstance | copy D$esi+36 D$DeviceType+ebx*4

        ; Save GUID
        add esi 4
        mov edi ebx | shl edi 4 | add edi GUID
        mov ecx 010 | rep movsb

    popad
    mov eax &DIENUM_CONTINUE
    return

____________________________________________________________________________________________
; Get input
____________________________________________________________________________________________

[pDeviceState: ?]

GetDeviceStates:

    ; Get all device states
    mov edx DeviceStates
    mov ebx 0
L0: if. ebx < D$nDevices

        on D$lpdid+ebx*4 = &NULL, S0>>

        ; Acquire device
        pushad
            dxCall lpdid+ebx*4, Acquire
            dxErrorCheck 'Could not acquire device'
        popad

        ; Poll device
        pushad
            dxCall lpdid+ebx*4, Poll
            dxErrorCheck 'Could not poll device'
        popad

        ; Get device state
        pushad
            dxCall lpdid+ebx*4,GetDeviceState D$LengthDeviceState+ebx*4, edx
            dxErrorCheck 'Could not get joystick state'
        popad

        ; Next device
    S0: add edx D$LengthDeviceState+ebx*4
        inc ebx | jmp L0<<

    endif

    ; Convert data from certain joystick input objects
    call ConvertDeviceStates
    ret

ConvertDeviceStates:

    mov edx DeviceStates
    mov ebx 0
L0: if. ebx < D$nDevices

        on B$DeviceType+ebx*4 != &DIDEVTYPE_JOYSTICK, S0>>
        mov esi edx, edi esi

        ; Convert 3+3 axes (axis value --> pressed buttons)
        mov ecx 3+3
    L1: lodsd
        if.. eax = 0-1 | mov eax 080   | else..
        if.. eax = 0+1 | mov eax 08000 | endif..
        stosd
        dec ecx | jnz L1<

        ; Eliminate 2 sliders <------- todo: conversion
        lodsd | mov eax 0 | stosd
        lodsd | mov eax 0 | stosd

        ; Convert 4 POVs (angle value --> pressed buttons)
        mov ecx 4
    L1: lodsd
        if.. ax < 2250+0000  | mov eax 0______80 | else..  ; up
        if.. ax < 2250+4500  | mov eax 0____8080 | else..  ; up+right
        if.. ax < 2250+9000  | mov eax 0____8000 | else..  ; right
        if.. ax < 2250+13500 | mov eax 0__808000 | else..  ; right+down
        if.. ax < 2250+18000 | mov eax 0__800000 | else..  ; down
        if.. ax < 2250+22500 | mov eax 080800000 | else..  ; down+left
        if.. ax < 2250+27000 | mov eax 080000000 | else..  ; left
        if.. ax < 2250+31500 | mov eax 080000080 | else..  ; left+up
        if.. ax < 0000+36000 | mov eax 0______80 | else..  ; up
        if.. ax = 0FFFF      | mov eax 000000000 | endif.. ; center
        stosd
        dec ecx | jnz L1<<

        ; Next device
    S0: add edx D$LengthDeviceState+ebx*4
        inc ebx | jmp L0<<

    endif
    ret

____________________________________________________________________________________________

; Assembles all device states into a single input status array
TranslateInput:

    ; Save previous state
    call CopyMemory InputArray, OldInputArray, LENGTH_INPUTARRAY
    mov edi InputArray, al BUTTON_UP, ecx LENGTH_INPUTARRAY | rep stosb

    ; Loop through all devices
    mov edx DeviceStates
    mov ebx 0
L0: if. ebx < D$nDevices

        ; Loop through all buttons
        mov ecx 0
    L1: if.. ecx < D$LengthDeviceState

            ; Button down?
            test B$edx+ecx 080 | jz S0>

            ; Find mappings for this button
            mov esi D$pMappings+ebx*4
        L2: lodsw | on ax = 0, S0>
            lodsw | on ax != cx, L2<

            ; Set function = down
            movzx eax W$esi-4
            mov B$InputArray+eax BUTTON_DOWN
            jmp L2<

            ; Next button
        S0: inc ecx | jmp L1<
        endif..

        ; Next device
        add edx D$LengthDeviceState
        inc ebx | jmp L0<<

    endif

    ; Compare with old status, set additional flags
    mov ebx 0
L0: ifFlag  B$OldInputArray+ebx BUTTON_DOWN,
        ifFlag B$InputArray+ebx BUTTON_UP,
            or B$InputArray+ebx BUTTON_RELEASED
    ifFlag  B$OldInputArray+ebx BUTTON_UP,
        ifFlag B$InputArray+ebx BUTTON_DOWN,
            or B$InputArray+ebx BUTTON_PRESSED
    inc ebx | on ebx < LENGTH_INPUTARRAY, L0<
    ret

____________________________________________________________________________________________
; Handle input
____________________________________________________________________________________________

HandleInput:

    ; NES
    ifFlag B$InputArray+TOGGLE_POWER       BUTTON_PRESSED, call MenuPower
    ifFlag B$InputArray+RESET              BUTTON_PRESSED, call MenuReset
    ifFlag B$InputArray+QUICK_SAVE         BUTTON_PRESSED, call SaveQuick
    ifFlag B$InputArray+QUICK_LOAD         BUTTON_PRESSED, call LoadQuick
    ifFlag B$InputArray+SAVE_DIALOG        BUTTON_PRESSED, call MenuSaveGame
    ifFlag B$InputArray+LOAD_DIALOG        BUTTON_PRESSED, call MenuLoadGame

    ; Nessie
    ifFlag. B$InputArray+TOGGLE_PAUSE      BUTTON_PRESSED
        xor B$Paused &TRUE
        call ClearAudio
    endif
    ifFlag B$InputArray+FAST_FORWARD       BUTTON_PRESSED,  call StopAudio
    ifFlag B$InputArray+FAST_FORWARD       BUTTON_RELEASED, call PlayAudio
    ifFlag B$InputArray+CLOSE              BUTTON_PRESSED,  call 'User32.DestroyWindow' D$hMainWindow
    ifFlag B$InputArray+FULLSCREEN         BUTTON_PRESSED,  call ToggleFullscreen
    ifFlag B$InputArray+TAKE_SCREENSHOT    BUTTON_PRESSED,  call SaveScreenshot

    ; Autofire
    call HandleAutoFire
    ret

HandleAutoFire:

    ; Button down?
    ifFlag. B$InputArray+PAD1_AUTOFIRE_A BUTTON_DOWN | inc B$AutoFire+00 | else | mov B$AutoFire+00 0 | endif
    ifFlag. B$InputArray+PAD1_AUTOFIRE_B BUTTON_DOWN | inc B$AutoFire+01 | else | mov B$AutoFire+01 0 | endif
    ifFlag. B$InputArray+PAD2_AUTOFIRE_A BUTTON_DOWN | inc B$AutoFire+02 | else | mov B$AutoFire+02 0 | endif
    ifFlag. B$InputArray+PAD2_AUTOFIRE_B BUTTON_DOWN | inc B$AutoFire+03 | else | mov B$AutoFire+03 0 | endif
    ifFlag. B$InputArray+PAD3_AUTOFIRE_A BUTTON_DOWN | inc B$AutoFire+04 | else | mov B$AutoFire+04 0 | endif
    ifFlag. B$InputArray+PAD3_AUTOFIRE_B BUTTON_DOWN | inc B$AutoFire+05 | else | mov B$AutoFire+05 0 | endif
    ifFlag. B$InputArray+PAD4_AUTOFIRE_A BUTTON_DOWN | inc B$AutoFire+06 | else | mov B$AutoFire+06 0 | endif
    ifFlag. B$InputArray+PAD4_AUTOFIRE_B BUTTON_DOWN | inc B$AutoFire+07 | else | mov B$AutoFire+07 0 | endif

    ; Reset autofire cycle?
    mov al B$AutoFireDelay
    if al <= B$AutoFire+00, mov B$AutoFire+00 0
    if al <= B$AutoFire+01, mov B$AutoFire+01 0
    if al <= B$AutoFire+02, mov B$AutoFire+02 0
    if al <= B$AutoFire+03, mov B$AutoFire+03 0
    if al <= B$AutoFire+04, mov B$AutoFire+04 0
    if al <= B$AutoFire+05, mov B$AutoFire+05 0
    if al <= B$AutoFire+06, mov B$AutoFire+06 0
    if al <= B$AutoFire+07, mov B$AutoFire+07 0

    ; Button auto-down?
    mov al B$AutoFireDelay  | shr al 1
    if al >= B$AutoFire+00, if B$AutoFire+00 > 0, mov B$InputArray+PAD1_A BUTTON_DOWN
    if al >= B$AutoFire+01, if B$AutoFire+01 > 0, mov B$InputArray+PAD1_B BUTTON_DOWN
    if al >= B$AutoFire+02, if B$AutoFire+02 > 0, mov B$InputArray+PAD2_A BUTTON_DOWN
    if al >= B$AutoFire+03, if B$AutoFire+03 > 0, mov B$InputArray+PAD2_B BUTTON_DOWN
    if al >= B$AutoFire+04, if B$AutoFire+04 > 0, mov B$InputArray+PAD3_A BUTTON_DOWN
    if al >= B$AutoFire+05, if B$AutoFire+05 > 0, mov B$InputArray+PAD3_B BUTTON_DOWN
    if al >= B$AutoFire+06, if B$AutoFire+06 > 0, mov B$InputArray+PAD4_A BUTTON_DOWN
    if al >= B$AutoFire+07, if B$AutoFire+07 > 0, mov B$InputArray+PAD4_B BUTTON_DOWN
    ret

____________________________________________________________________________________________
; Add/remove button assignments
____________________________________________________________________________________________

AssignFunction:

    arguments @Device, @Button, @Function

    ; Scan through button assignments
    mov ebx D@Device
    mov esi D$pMappings+ebx*4
L0: lodsd
    if. eax = 0
        ; Put new function at the end
        copy W@Function W$esi-4
        copy W@Button   W$esi-2
        mov D$esi 0
    else
        ; Function already assigned?
        on ax != W@Function, L0<
        mov ax W$esi-2 | on ax != W@Button, L0<
    endif
    return

UnAssignFunction:

    arguments @Function

    ; Scan through all devices
    mov ebx 0
L0: if. ebx < D$nDevices

        ; Find function
        mov esi D$pMappings+ebx*4
    L1: lodsd
        if.. eax > 0

            on ax != W@Function, L1<

            ; Remove it
            pushad
                move edi esi-4
            L2: movsd | on D$edi-4 != 0, L2<
            popad
            sub esi 4
            jmp L1<

        endif..

        ; Next device
        inc ebx | jmp L0<

    endif
    return

____________________________________________________________________________________________
; Input array
____________________________________________________________________________________________

; Button assignments
[Mappings: ? #(NUM_DEVICES shl 8)] ; W$ FunctionID, W$ DIK_****

; Input array
[LENGTH_INPUTARRAY 0100]
[OldInputArray: B$ ? #LENGTH_INPUT_ARRAY]
[InputArray:    B$ ? #LENGTH_INPUT_ARRAY]

; Input array flags
[BUTTON_DOWN            080
 BUTTON_UP              040
 BUTTON_CHANGED         BUTTON_RELEASED+BUTTON_PRESSED
 BUTTON_RELEASED        020
 BUTTON_PRESSED         010
 BUTTON_LOCKED_DOWN     08
 BUTTON_TOGGLED         BUTTON_PRESSED]

; Input array indexes
enumerate 1,

    ; Nessie
    CLOSE, TOGGLE_PAUSE, FAST_FORWARD,
    FULLSCREEN, TAKE_SCREENSHOT,

    ; NES
    TOGGLE_POWER, RESET,
    QUICK_SAVE, QUICK_LOAD, SAVE_DIALOG, LOAD_DIALOG,

    ; NES pads
    PAD1_A, PAD1_B, PAD1_SELECT, PAD1_START, PAD1_UP, PAD1_DOWN, PAD1_LEFT, PAD1_RIGHT, PAD1_AUTOFIRE_A, PAD1_AUTOFIRE_B,
    PAD2_A, PAD2_B, PAD2_SELECT, PAD2_START, PAD2_UP, PAD2_DOWN, PAD2_LEFT, PAD2_RIGHT, PAD2_AUTOFIRE_A, PAD2_AUTOFIRE_B,
    PAD3_A, PAD3_B, PAD3_SELECT, PAD3_START, PAD3_UP, PAD3_DOWN, PAD3_LEFT, PAD3_RIGHT, PAD3_AUTOFIRE_A, PAD3_AUTOFIRE_B,
    PAD4_A, PAD4_B, PAD4_SELECT, PAD4_START, PAD4_UP, PAD4_DOWN, PAD4_LEFT, PAD4_RIGHT, PAD4_AUTOFIRE_A, PAD4_AUTOFIRE_B,

    OPEN_DIALOG

____________________________________________________________________________________________
; DirectInput device data
____________________________________________________________________________________________

; Interface pointer
[lpdi: ?]

; Device data
[NUM_DEVICES 20]
[nDevices:          ?
 DeviceStates:      ? #(NUM_DEVICES shl 8)
 GUID:              ? #(NUM_DEVICES shl 2)
 lpdid:             ? #NUM_DEVICES
 DeviceType:        ? #NUM_DEVICES
 LengthDeviceState: ? #NUM_DEVICES
 pMappings:         ? #NUM_DEVICES
 pButtonNames:      ? #NUM_DEVICES]

____________________________________________________________________________________________
; Input settings (autofire)
____________________________________________________________________________________________

[AutoFire: B$ ? #8]
[LENGTH_INPUTSETTINGS <(1 shl 2)>]
[InputSettings:
 AutoFireDelay: ?

 Port1:         ?
 Port2:         ?
 Port3:         ?
 Port4:         ?]

[CONTROLLER_GAMEPAD     01
 CONTROLLER_ZAPPER      02
 CONTROLLER_UNCONNECTED 080]

____________________________________________________________________________________________
; Vtables
____________________________________________________________________________________________

; IDirectInput
[CreateDevice           12
 EnumDevices            16
 GetDeviceStatus        20
 RunControlPanel        24
 diInitialize           28]

; IDirectInputDevice
[GetCapabilities        12
 EnumObjects            16
 GetProperty            20
 SetProperty            24
 Acquire                28
 Unacquire              32
 GetDeviceState         36
 GetDeviceData          40
 SetDataFormat          44
 SetEventNotification   48
 didSetCooperativeLevel 52
 GetObjectInfo          56
 GetDeviceInfo          60
 didRunControlPanel     64
 didInitialize          68
; IDirectInputDevice2
 CreateEffect           72
 EnumEffects            76
 GetEffectInfo          80
 GetForceFeedbackState  84
 SendForceFeedbackCommand 88
 EnumCreatedEffectObjects 92
 Escape                 96
 Poll                   100
 SendDeviceData         104]

[IID_IDirectInputDevice2:
 D$ 05944E682
 W$ 0C92E, 011CF
 B$ 0BF, 0C7, 044, 045, 053, 054, 000, 000]

____________________________________________________________________________________________
; Data format
____________________________________________________________________________________________

Init_rgodfKeyboard:

    mov ecx 0
    mov edi rgodfKeyboard
L0: mov eax GUID_Key                            | stosd
    mov eax ecx                                 | stosd
    shl eax 8 | or eax &DIDFT_BUTTON+080000000  | stosd
    mov eax &NULL                               | stosd
    inc cl | jnz L0<
    ret

; Keyboard data format
[c_dfDIKeyboard:
 @dwSize:     018
 @dwObjSize:  010
 @dwFlags:    &DIDF_RELAXIS
 @dwDataSize: 0100
 @dwNumObjs:  0100
 @rgodf:      rgodfKeyboard]
; GUID_Key, (button #), (DIDFT_BUTTON | 080000000 | DIDFT_MAKEINSTANCE(button #)), NULL
[rgodfKeyboard: ? #0400]

; Joystick data format
[c_dfDIJoyStick:
 @dwSize:       018
 @dwObjSize:    010
 @dwFlags:      &DIDF_ABSAXIS
 @dwDataSize:   050
 @dwNumObjs:    020
 @rgodf:        rgodfJoystick]
[rgodfJoystick:
 GUID_XAxis     00      &DIDFT_ANYINSTANCE+080000000+&DIDFT_AXIS    &DIDOI_ASPECTPOSITION
 GUID_YAxis     04      &DIDFT_ANYINSTANCE+080000000+&DIDFT_AXIS    &DIDOI_ASPECTPOSITION
 GUID_ZAxis     08      &DIDFT_ANYINSTANCE+080000000+&DIDFT_AXIS    &DIDOI_ASPECTPOSITION
 GUID_RxAxis    0C      &DIDFT_ANYINSTANCE+080000000+&DIDFT_AXIS    &DIDOI_ASPECTPOSITION
 GUID_RyAxis    010     &DIDFT_ANYINSTANCE+080000000+&DIDFT_AXIS    &DIDOI_ASPECTPOSITION
 GUID_RzAxis    014     &DIDFT_ANYINSTANCE+080000000+&DIDFT_AXIS    &DIDOI_ASPECTPOSITION
 GUID_Slider    018     &DIDFT_ANYINSTANCE+080000000+&DIDFT_AXIS    &DIDOI_ASPECTPOSITION
 GUID_Slider    01C     &DIDFT_ANYINSTANCE+080000000+&DIDFT_AXIS    &DIDOI_ASPECTPOSITION
 GUID_POV       020     &DIDFT_ANYINSTANCE+080000000+&DIDFT_POV     0
 GUID_POV       024     &DIDFT_ANYINSTANCE+080000000+&DIDFT_POV     0
 GUID_POV       028     &DIDFT_ANYINSTANCE+080000000+&DIDFT_POV     0
 GUID_POV       02C     &DIDFT_ANYINSTANCE+080000000+&DIDFT_POV     0
 &NULL          030     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          031     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          032     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          033     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          034     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          035     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          036     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          037     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          038     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          039     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          03A     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          03B     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          03C     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          03D     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          03E     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          03F     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          040     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          041     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          042     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          043     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          044     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          045     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          046     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          047     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          048     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          049     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          04A     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          04B     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          04C     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          04D     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          04E     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0
 &NULL          04F     &DIDFT_ANYINSTANCE+080000000+&DIDFT_BUTTON  0]

; Input object GUIDs
[GUID_XAxis:   D$ 0A36D02E0, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_YAxis:   D$ 0A36D02E1, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_ZAxis:   D$ 0A36D02E2, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_RxAxis:  D$ 0A36D02F4, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_RyAxis:  D$ 0A36D02F5, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_RzAxis:  D$ 0A36D02E3, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_Slider:  D$ 0A36D02E4, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_Button:  D$ 0A36D02F0, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_Key:     D$ 055728220, W$ 0D33C,011CF, B$ 0BF,0C7,044,045,053,054,000,000
 GUID_POV:     D$ 0A36D02F2, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000]
;GUID_Unknown: D$ 0A36D02F3, W$ 0C9F3,011CF, B$ 0BF,0C7,044,045,053,054,000,000

____________________________________________________________________________________________
; Input object names
____________________________________________________________________________________________

[JoystickNames: B$

 ; Axes
 'lX-',    0    'lX+',   0      '...', 0        '...', 0
 'lY-',    0    'lY+',   0      '...', 0        '...', 0
 'lZ-',    0    'lZ+',   0      '...', 0        '...', 0
 'lRX-',   0    'lRX+',  0      '...', 0        '...', 0
 'lRY-',   0    'lRY+',  0      '...', 0        '...', 0
 'lRZ-',   0    'lRZ+',  0      '...', 0        '...', 0

 ; Sliders
 '...',    0    '...',   0      '...', 0        '...', 0
 '...',    0    '...',   0      '...', 0        '...', 0

 ; POVs
 'POV Up',  0   'POV Right',  0 'POV Down',  0  'POV Left',  0
 'POV2 Up', 0   'POV2 Right', 0 'POV2 Down', 0  'POV2 Left', 0
 'POV3 Up', 0   'POV3 Right', 0 'POV3 Down', 0  'POV3 Left', 0
 'POV4 Up', 0   'POV4 Right', 0 'POV4 Down', 0  'POV4 Left', 0

 ; Buttons
 'Button 0',  0 'Button 1',  0  'Button 2',  0  'Button 3',  0
 'Button 4',  0 'Button 5',  0  'Button 6',  0  'Button 7',  0
 'Button 8',  0 'Button 9',  0  'Button 10', 0  'Button 11', 0
 'Button 12', 0 'Button 13', 0  'Button 14', 0  'Button 15', 0
 'Button 16', 0 'Button 17', 0  'Button 18', 0  'Button 19', 0
 'Button 20', 0 'Button 21', 0  'Button 22', 0  'Button 23', 0
 'Button 24', 0 'Button 25', 0  'Button 26', 0  'Button 27', 0
 'Button 28', 0 'Button 29', 0  'Button 30', 0  'Button 31', 0

KeyboardNames: B$
; 00
'...', 0
'Escape', 0
'1', 0
'2', 0
'3', 0
'4', 0
'5', 0
'6', 0
'7', 0
'8', 0
'9', 0
'0', 0
'-', 0
'=', 0
'Backspace', 0
'Tab', 0
; 10
'Q', 0
'W', 0
'E', 0
'R', 0
'T', 0
'Y', 0
'U', 0
'I', 0
'O', 0
'P', 0
'[', 0
']', 0
'Return', 0
'Left control', 0
'A', 0
'S', 0
; 20
'D', 0
'F', 0
'G', 0
'H', 0
'J', 0
'K', 0
'L', 0
';', 0
"'", 0
167, 0
'Left shift', 0
'\', 0
'Z', 0
'X', 0
'C', 0
'V', 0
; 30
'B', 0
'N', 0
'M', 0
',', 0
'.', 0
'/', 0
'Right shift', 0
'Numpad *', 0
'Left alt', 0
'Space', 0
'Caps lock', 0
'F1', 0
'F2', 0
'F3', 0
'F4', 0
'F5', 0
; 40
'F6', 0
'F7', 0
'F8', 0
'F9', 0
'F10', 0
'Num lock', 0
'Scroll lock', 0
'Numpad 7', 0
'Numpad 8', 0
'Numpad 9', 0
'Numpad -', 0
'Numpad 4', 0
'Numpad 5', 0
'Numpad 6', 0
'Numpad +', 0
'Numpad 1', 0
; 50
'Numpad 2', 0
'Numpad 3', 0
'Numpad 0', 0
'Numpad .', 0
'...', 0 ; 54
'...', 0 ; 55
'<', 0
'F11', 0
'F12', 0
'...', 0 ; 59
'...', 0 ; 5A
'...', 0 ; 5B
'...', 0 ; 5C
'...', 0 ; 5D
'...', 0 ; 5E
'...', 0 ; 5F
; 60
'...', 0 ; 60
'...', 0 ; 61
'...', 0 ; 62
'...', 0 ; 63
'F13', 0
'F14', 0
'F15', 0
'...', 0 ; 67
'...', 0 ; 68
'...', 0 ; 69
'...', 0 ; 6A
'...', 0 ; 6B
'...', 0 ; 6C
'...', 0 ; 6D
'...', 0 ; 6E
'...', 0 ; 6F
; 70
'Kana', 0
'...', 0 ; 71
'...', 0 ; 72
'Cedilha', 0
'...', 0 ; 74
'...', 0 ; 75
'...', 0 ; 76
'...', 0 ; 77
'...', 0 ; 78
'Convert', 0
'...', 0 ; 7A
'No convert', 0
'...', 0 ; 7C
'Yen', 0
'Cedilha', 0
'...', 0 ; 7F
; 80
'...', 0 ; 80
'...', 0 ; 81
'...', 0 ; 82
'...', 0 ; 83
'...', 0 ; 84
'...', 0 ; 85
'...', 0 ; 86
'...', 0 ; 87
'...', 0 ; 88
'...', 0 ; 89
'...', 0 ; 8A
'...', 0 ; 8B
'...', 0 ; 8C
'Numpad =', 0
'...', 0 ; 8E
'...', 0 ; 8F
; 90
'Previous Track', 0
'AT', 0
':', 0
'_', 0
'Kanji', 0
'Stop', 0
'AX', 0
'[Unlabeled]', 0
'...', 0 ; 98
'Next Track', 0
'...', 0 ; 9A
'...', 0 ; 9B
'Numpad Enter', 0
'Right Control', 0
'...', 0 ; 9E
'...', 0 ; 9F
; A0
'Mute', 0
'Calculator', 0
'Play/Pause', 0
'...', 0 ; A3
'Media Stop', 0
'...', 0 ; A5
'...', 0 ; A6
'...', 0 ; A7
'...', 0 ; A8
'...', 0 ; A9
'...', 0 ; AA
'...', 0 ; AB
'...', 0 ; AC
'...', 0 ; AD
'Volume Down', 0
'...', 0 ; AF
; B0
'Volume Up', 0
'...', 0 ; B1
'Home', 0
'Numpad ,', 0
'...', 0 ; B4
'Numpad /', 0
'...', 0 ; B6
'Print screen', 0
'Right alt', 0
'...', 0 ; B9
'...', 0 ; BA
'...', 0 ; BB
'...', 0 ; BC
'...', 0 ; BD
'...', 0 ; BE
'...', 0 ; BF
; C0
'...', 0 ; C0
'...', 0 ; C1
'...', 0 ; C2
'...', 0 ; C3
'...', 0 ; C4
'Pause', 0
'...', 0 ; C6
'Home', 0
'Up', 0
'Page up', 0
'...', 0 ; CA
'Left', 0
'...', 0 ; CC
'Right', 0
'...', 0 ; CE
'End', 0
; D0
'Down', 0
'Page down', 0
'Insert', 0
'Delete', 0
'...', 0 ; D4
'...', 0 ; D5
'...', 0 ; D6
'...', 0 ; D7
'...', 0 ; D8
'...', 0 ; D9
'...', 0 ; DA
'Left Windows Key', 0
'Right Windows Key', 0
'Application Menu', 0
'Power', 0
'Sleep', 0
; E0
'...', 0 ; E0
'...', 0 ; E1
'...', 0 ; E2
'Wake', 0
'...', 0 ; E4
'Web Search', 0
'Web Favorites', 0
'Web Refresh', 0
'Web Stop', 0
'Web Forward', 0
'Web Back', 0
'My Computer', 0
'Mail', 0
'Media Select', 0
'...', 0  ; EE
'...', 0] ; EF
____________________________________________________________________________________________
TITLE InputDialog
____________________________________________________________________________________________
; Input dialog
____________________________________________________________________________________________

InputDialogProc:

    arguments @hDialog, @Message, @wParam, @lParam
    pushad

        ; Message?
               if. D@Message = &WM_CLOSE      | call @Cancel
        else | if. D@Message = &WM_INITDIALOG | call @Init
        else | if. D@Message = &WM_COMMAND    | call @Command
        else | if. D@Message = &WM_TIMER      | call @Timer
        else | mov eax &FALSE | endif

        mov D$esp+01C eax

    popad
    return

@Init:
;;
    mov eax D$AutofireDelay
    mov ecx 60   | mul ecx
    mov ecx 1000 | div ecx
    call 'User32.SetTimer' D@hDialog, IDT_AUTOFIRE, eax, &NULL
;;
    ; Get handles
  . D@hDeviceList   = 'User32.GetDlgItem' D@hDialog, IDC_LISTBOX+0
  . D@hFunctionList = 'User32.GetDlgItem' D@hDialog, IDC_LISTBOX+1
  . D@hMappingList  = 'User32.GetDlgItem' D@hDialog, IDC_LISTBOX+2

    ; Autofire trackbar
  . D@hAutofire = 'User32.GetDlgItem' D@hDialog, IDC_TRACKBAR+0
    call 'User32.SendMessageA' D@hAutofire, &TBM_SETRANGE, &FALSE, (AUTOFIRE_MAXIMUM shl 16)+AUTOFIRE_MINIMUM
    call 'User32.SendMessageA' D@hAutofire, &TBM_SETTHUMBLENGTH, 2, 0
    mov eax AUTOFIRE_MAXIMUM+AUTOFIRE_MINIMUM | sub eax D$AutoFireDelay
    call 'User32.SendMessageA' D@hAutofire, &TBM_SETPOS, &TRUE, eax

    ; Zapper checkboxes
    mov eax 0 | cmp B$Port1 CONTROLLER_ZAPPER | sete al | call 'User32.CheckDlgButton' D@hDialog, IDC_CHECKBOX+0, eax
    mov eax 0 | cmp B$Port2 CONTROLLER_ZAPPER | sete al | call 'User32.CheckDlgButton' D@hDialog, IDC_CHECKBOX+1, eax

    ; Put some text in the list boxes
    call @ListDevices
    call @ListFunctions
    mov eax &TRUE
    ret

@Command:

    call @CancelBind

if W@wParam = IDC_TRACKBAR+0,outhex D@wParam

    ; Escape pressed
    on W@wParam = &IDCANCEL, @Cancel

    ; OK/Cancel
    on W@wParam = IDC_OK,     @SaveAndExit
    on W@wParam = IDC_CANCEL, @Cancel
    ; Set
    if W@wParam = IDC_BUTTON+0, call @DoBind D@nSelectedFunction
    ; Clear
    if. W@wParam = IDC_BUTTON+1
        call GetFunctionID D@nSelectedDevice, D@nSelectedFunction
        call UnAssignFunction eax
        call @ListFunctions
    endif
    ; Set all
    if W@wParam = IDC_BUTTON+2, call @BindAllFunctions
    ; Clear all
    if. W@wParam = IDC_BUTTON+3
        call 'User32.SendMessageA' D@hFunctionList, &LB_GETCOUNT, 0, 0
    L0: if.. eax > 0
            dec eax
            push eax
                call GetFunctionID D@nSelectedDevice, eax
                call UnAssignFunction eax
            pop eax | jmp L0<
        endif..
        call @ListFunctions
    endif

    ; Select changed
    if. W@wParam+2 = &CBN_SELCHANGE
        if W@wParam = IDC_LISTBOX+0, . D@nSelectedDevice   = 'User32.SendMessageA' D@hDeviceList,   &LB_GETCURSEL, 0, 0
        if W@wParam = IDC_LISTBOX+1, . D@nSelectedFunction = 'User32.SendMessageA' D@hFunctionList, &LB_GETCURSEL, 0, 0
        call @ListFunctions
    endif

    ; Double-click
    if W@wParam+2 = &LBN_DBLCLK,
        if W@wParam = IDC_LISTBOX+1,
            call @DoBind D@nSelectedFunction

    mov eax &TRUE
    ret

@SaveAndExit:

    call @KillDialog
    call SaveConfiguration
    mov eax &TRUE
    ret

@Cancel:

    ; Restore everything, kill dialog
    call @KillDialog
    call LoadConfiguration
    mov eax &TRUE
    ret

@KillDialog:

    call @CancelBind
    call 'User32.KillTimer' D@hDialog, IDT_AUTOFIRE

    ; Get trackbar value
    call 'User32.SendMessageA' D@hAutofire, &TBM_GETPOS, 0, 0
    mov edx AUTOFIRE_MAXIMUM+AUTOFIRE_MINIMUM | sub edx eax
    mov D$AutoFireDelay edx

    ; Zapper checkboxes
    call 'User32.IsDlgButtonChecked' D@hDialog, IDC_CHECKBOX+0 | call SetZapper Port1, eax
    call 'User32.IsDlgButtonChecked' D@hDialog, IDC_CHECKBOX+1 | call SetZapper Port2, eax

    ; Exit
    call 'User32.EndDialog' D@hDialog, 1
    ret

@Timer:

    ; Waiting for button press?
    if D@wParam = IDT_PRESSNEWBUTTON,
        if B@WaitingForPress = &TRUE,
            call @WaitForPress

    ; Toggle autofire?
    if. D@wParam = IDT_AUTOFIRE
        call 'User32.GetWindowLongA' D@hAutofire, &GWL_STYLE | xor eax &TBS_NOTHUMB
        call 'User32.SetWindowLongA' D@hAutofire, &GWL_STYLE, eax
    endif
    mov eax &TRUE
    ret
____________________________________________________________________________________________

@ListDevices:

    ; Fill list
    call 'User32.SendMessageA' D@hDeviceList, &LB_RESETCONTENT, 0, 0
    mov edi @DeviceNames, al 0, ecx 0-1
L0: if. B$edi > 0
        pushad
            call 'User32.SendMessageA' D@hDeviceList, &LB_ADDSTRING, 0, edi
        popad
        repne scasb
        jmp L0<
    endif

    ; Set current selection
    call 'User32.SendMessageA' D@hDeviceList, &LB_SETCURSEL, D@nSelectedDevice, 0
    ret

@ListFunctions:

    ; Reset
    call 'User32.SendMessageA' D@hFunctionList, &LB_RESETCONTENT, 0, 0
    if D@nSelectedFunction = &LB_ERR, mov D@nSelectedFunction 0

    ; Insert function names
    call GetFunctionNames D@nSelectedDevice
    mov edi eax, al 0, ecx 0-1
L0: if. B$edi > 0
        pushad
            call 'User32.SendMessageA' D@hFunctionList, &LB_ADDSTRING, 0, edi
        popad
        repne scasb
        jmp L0<
    endif

    ; Select function
L0: call 'User32.SendMessageA' D@hFunctionList, &LB_SETCURSEL, D@nSelectedFunction, 0
    if. eax = &LB_ERR
        call 'User32.SendMessageA' D@hFunctionList, &LB_GETCOUNT, D@nSelectedFunction, 0
        dec eax | mov D@nSelectedFunction eax
        on eax != &LB_ERR, L0<
    endif

    ; List their mappings
    call @ListMappings
    call @AdjustMappingScroll
    ret

@ListMappings:

    ; ecx = list index, eax = number of items
    call 'User32.SendMessageA' D@hMappingList, &LB_RESETCONTENT, 0, 0
    call 'User32.SendMessageA' D@hFunctionList, &LB_GETCOUNT, 0, 0
    mov ecx 0
L0: pushad

        mov D@Buttonname 0

        ; Scan through all devices
        mov ebx 0
    L1: if. ebx < D$nDevices
            mov esi D$pMappings+ebx*4

            ; Find mapping for this function
        L2: call GetFunctionID D@nSelectedDevice ecx | mov edx eax
        L3: lodsd
            on eax = 0, S0>
            on ax != dx, L3<

            ; Find button name
            push ecx
                movzx edx W$esi-2
                mov al 0, edi D$pButtonNames+ebx*4, ecx 0-1
            L3: dec edx | js C0> | repne scasb | jmp L3<
        C0: pop ecx

            ; Add it
            [@Buttonname:      B$ ? #&MAX_PATH]
            [@ButtonSeparator: B$ ', ', 0]
            call AddString @Buttonname @ButtonSeparator
            call AddString @Buttonname edi
            jmp L2<

        S0: inc ebx | jmp L1<<
        endif

        ; Add string
        call 'User32.SendMessageA' D@hMappingList, &LB_ADDSTRING, 0, @Buttonname+2

    popad
    inc ecx | on ecx < eax, L0<<
    ret

____________________________________________________________________________________________

@AdjustMappingScroll:

    [@hDC:      ?
     @hFontOld: ?
     @hFontNew: ?
     @nCount:   ?
     @nLength:  ?
     @Size:     ? ?
     @Text:     ? #&MAX_PATH]

    mov D@nLength 0
    pushad

        ; Retrieve data
      . D@hDC = 'User32.GetDC' D@hMappingList
      . D@hFontNew = 'User32.SendMessageA' D@hMappingList, &WM_GETFONT, &NULL, &NULL
      . D@hFontOld = 'GDI32.SelectObject'  D@hDC, D@hFontNew

        ; Find longest string
      . D@nCount = 'User32.SendMessageA' D@hFunctionList, &LB_GETCOUNT, 0, 0
    L0: if. D@nCount > 0
            dec D@nCount
            call 'User32.SendMessageA' D@hMappingList, &LB_GETTEXTLEN, D@nCount, &NULL | inc eax
            push eax
                call 'User32.SendMessageA' D@hMappingList, &LB_GETTEXT, D@nCount, @Text
            pop eax
            call 'GDI32.GetTextExtentPoint32A' D@hDC, @Text, eax, @Size
            mov eax D@Size
            if eax > D@nLength, mov D@nLength eax
            jmp L0<
        endif

        ; Restore data, set new width
    S0: call 'GDI32.SelectObject' D@hDC, D@hFontOld
        call 'User32.ReleaseDC' D@hMappingList, D@hDC
        call 'User32.SendMessageA' D@hMappingList, &LB_SETHORIZONTALEXTENT, D@nLength, 0

    popad
    call @ListMappings
    ret
____________________________________________________________________________________________

[@PressNewButton: B$ 'Press new button...', 0]
[@WaitingForPress:  ?
 @nFunction:        ?
 @nFunctionsToBind: ?]

@BindAllFunctions:

    ; Get number of functions, start binding first
  . D@nFunctionsToBind = 'User32.SendMessageA' D@hFunctionList, &LB_GETCOUNT, 0, 0
    if eax > 0, call @DoBind 0
    ret

@DoBind:

    pop eax, D@nFunction | push eax
    if D@nFunction = &LB_ERR, ret
    pushad

        call 'User32.SetFocus' 0

        ; Copy to old device state
        mov esi DeviceStates, edi @OldDeviceStates, ecx (NUM_DEVICES shl 10)
        rep movsb

        ; Move the list box cursor to this function - only necessary on [set all]
        call 'User32.SendMessageA' D@hFunctionList, &LB_SETCURSEL, D@nFunction, 0
        copy D@nFunction D@nSelectedFunction
        if D@nFunctionsToBind = 0, mov D@nFunctionsToBind 1

        ; 'Press new button...'
        call 'User32.SendMessageA' D@hMappingList, &LB_DELETESTRING, D@nFunction, 0
        call 'User32.SendMessageA' D@hMappingList, &LB_INSERTSTRING, D@nFunction, @PressNewButton

        ; Start waiting for button press
        call 'User32.SetTimer' D@hDialog, IDT_PRESSNEWBUTTON, 20, &NULL
        mov D@WaitingForPress &TRUE

    popad
    ret

@CancelBind:

    ; Cancel all binds
    if. D@WaitingForPress = &TRUE
        call 'User32.SetFocus' D@hDialog
        call 'User32.KillTimer' D@hDialog, IDT_PRESSNEWBUTTON
        mov D@WaitingForPress &FALSE
        mov D@nFunctionsToBind 0
        call @ListFunctions
    endif
    ret

@EndBind:

    ; End current bind
    call @ListFunctions
    dec D@nFunctionsToBind | jz @CancelBind | js @CancelBind
    inc D@nFunction
    call @DoBind D@nFunction
    ret

@WaitForPress:

    ; Get device state
    [@OldDeviceStates: ? #(NUM_DEVICES shl 8)]
    call GetDeviceStates

    ; Scan device states for changes
    mov esi DeviceStates, edi @OldDeviceStates
    mov ebx 0
L0: if. ebx < D$nDevices

        mov ecx 0
    L1: if.. ecx < D$LengthDeviceState+ebx*4
            ifFlag B$esi+ecx BUTTON_DOWN,
                ifNotFlag B$edi+ecx BUTTON_DOWN,
                    jmp Q0>
            inc ecx | jmp L1<
        endif..
        add esi ecx
        add edi ecx
        inc ebx | jmp L0<

    endif

    ; Copy to old device state
    mov esi DeviceStates, edi @OldDeviceStates, ecx (NUM_DEVICES shl 8)
    rep movsb
    ret

    ; A button was pressed!
Q0: mov D@WaitingForPress &FALSE
    call GetFunctionID D@nSelectedDevice, D@nFunction
    call AssignFunction ebx, ecx, eax
    call @EndBind
    ret
____________________________________________________________________________________________

[AUTOFIRE_MINIMUM 2
 AUTOFIRE_MAXIMUM 60]

____________________________________________________________________________________________

[@nSelectedDevice:   ?
 @nSelectedFunction: ?
 @hDeviceList:       ?
 @hFunctionList:     ?
 @hMappingList:      ?
 @hAutofire:         ?]

____________________________________________________________________________________________

[@DeviceNames: B$
    'Pad 1',  0
    'Pad 2',  0
    'Pad 3',  0
    'Pad 4',  0
    '-----',  0
    'NES',    0
    'TechNES', 0, 0]

____________________________________________________________________________________________

[PadNames: B$
    'A',         0
    'B',         0
    'Select',    0
    'Start',     0
    'Up',        0
    'Down',      0
    'Left',      0
    'Right'      0
    'Autofire A' 0
    'Autofire B' 0, 0

 NESNames:
    'Power',       0
    'Reset',       0
    'Quick save',  0
    'Quick load',  0
    'Save dialog', 0
    'Load dialog', 0, 0

 NessieNames:
    'Close',             0,
    'Pause',             0,
    'Fast forward',      0,
    'Fullscreen', 0,
    'Screenshot',        0, 0

 NoNames: 0, 0]

[pFunctionNames:
 PadNames
 PadNames
 PadNames
 PadNames
 NoNames
 NESNames
 NessieNames]

[nFunctionIDs:
 PAD1_A
 PAD2_A
 PAD3_A
 PAD4_A
 &NULL
 TOGGLE_POWER
 CLOSE]

GetFunctionNames:

    arguments @nDevice
    pushad
        mov ebx D@nDevice
        copy D$pFunctionNames+ebx*4 D@nDevice
    popad
    mov eax D@nDevice
    return

GetFunctionID:

    arguments @nDevice, @nFunction
    pushad
        mov eax D@nDevice
        mov eax D$nFunctionIDs+eax*4
        add D@nFunction eax
    popad
    mov eax D@nFunction
    return
____________________________________________________________________________________________
TITLE ToDo
;;

* Before release:
- Set version number
- Remove debug window
- Ignore or remove illegal opcodes
- update change log

* Now
- VRC7
- Namco106
- ROM config
- DISCL_FOREGROUND but still have a working input dialog

* Later
- VS
- tooltips on close button disrupts sound (WM_NCHITTEST?)
- save windowsize on exit
- sound filter
- Triple buffering and vsync

* Even later
- Game genie comments
- support joystick sliders
- input device number in mappinglist ('j0-b10')
- load dialog
  - descriptions instead of file names in list box
  - sort list box (newest games first)
- UNIF
- FDS
- complete database with only full CRC
- Resize window when switching PAL <--> NTSC
- Netplay using kailleraclient.dll
- NSF
- Support RAR & 7zip archives with unrar.dll & 7zxa.dll
? PC10

* Bugs
- Wrong noise frequency in 'Batman - Return of the Joker' when taking damage
- battletoads locks up on 2nd stage
- square when entering pipes in smb1
- Back to the future 2 & 3 has BG flicker

- 8-bit colors are screwed up
- Binding Alt in input dialog opens system menu
- resizing window so that it's higher than screen --> no longer locked ratio

* NES
- Colour emphasis
- Family keyboard
- Arkanoid paddle
- Power Pad
- $4011 triangle volume control
- perfect DMC
- First 2048 cycles for APU
? NMI on $2000.7 write
? VRAM buffer = last write/last read?
? Palette reads return what?

* Mappers
? Mappers #245 and #74 are supposed to be different from MMC3
? Mapper #6 IRQs are never used
? Mapper #16 is supposed to use EEPROM (???)
? Mapper #17 has some mirroring thing (Check $4500)
? Mapper #85 weird writes in 'Lagrange point'
? Mapper #222 IRQ counter
? Gimmick never uses the noise channel
? No ROMs for mapper 198
____________________________________________________________________________________________
 
 
 




