TITLE PPU

; Emulates the Picture Processing Unit.
; Read200x/Write200x handle PPU register access.
; PPUSynchronize makes the PPU catch up in time with the CPU.
; PPUReset resets all PPU regs, but not any CiRAM or CRAM.

____________________________________________________________________________________________
; Memory
____________________________________________________________________________________________
[DoPPURead  | push ebx | call D$PPUReadTable+eax*4 | pop ebx]
[DoPPUWrite | push ebx | call D$PPUWriteTable+edx*4 | pop ebx]

ReadVRAM:

    on eax < 02000, ReadCROM
    on eax < 03000, ReadCiRAM
    on eax < 03F00, ReadHighCiRAM
    on eax < 04000, ReadPalette
    msgError 'Invalid PPU read address.'
    ret

ReadPlainCRAM:

    movzx eax B$CRAM+eax
    ret

ReadCRAM:

    movzx ebx ah | and ebx 01C
    and eax 03FF | add eax D$IndexCRAM+ebx
    movzx eax B$CRAM+eax
    ret

ReadCROM:

    on D$Cartridge@SizeCROM = 0, ReadCRAM

    movzx ebx ah | and ebx 01C
    and eax 03FF | add eax D$IndexCROM+ebx
    add eax D$pCROM
    movzx eax B$eax
    ret

ReadCiROM:

    movzx ebx ah | and ebx 0C
    and eax 03FF | add eax D$IndexCiROM+ebx
    add eax D$pCROM
    movzx eax B$eax
    ret

ReadCiRAM:

    movzx ebx ah | and ebx 0C
    and eax 03FF | add eax D$IndexCiRAM+ebx
    movzx eax B$NameTable+eax
    ret

ReadHighCiRAM:

    ; $3000-$3EFF
    and eax 02FFF
    DoPPURead
    ret

ReadPalette:

    and eax 01F
    movzx eax B$eax+Palette
    ret
____________________________________________________________________________________________

WriteVRAM:

    on edx < 02000, WriteCRAM
    on edx < 03000, WriteCiRAM
    on edx < 03F00, WriteHighCiRAM
    on edx < 04000, WritePalette
    msgError 'Invalid PPU write address.'
    ret

WritePlainCRAM:

    mov B$CRAM+edx al
    ret

WriteCRAM:

    if. D$Cartridge@SizeCRAM > 0
        movzx ebx dh | and ebx 01C
        and edx 03FF | add edx D$IndexCRAM+ebx
        mov B$CRAM+edx al
    endif
    ret

WriteCiRAM:

    movzx ebx dh | and ebx 0C
    and edx 03FF | add edx D$IndexCiRAM+ebx
    mov B$NameTable+edx al
    ret

WriteHighCiRAM:

    ; $3000-$3EFF
    and edx 02FFF
    DoPPUWrite
    ret

WritePalette:

    and edx 01F
    and eax 03F

    ; Color #0?
    ifNotFlag. edx 0F
        mov edx 0
    L0: mov B$Palette+edx*4 al | inc edx | on edx < 8, L0<
    endif

    ; Write to palette
    ifFlag edx 3, mov B$Palette+edx al
    ret

____________________________________________________________________________________________
; Reset
____________________________________________________________________________________________

PPUReset:

    ; Zero out all data
    call ZeroMemory PPU       (SIZE_PPU_REGS shl 2)
    call ZeroMemory Rendering (SIZE_RENDERING shl 2)
    mov D$HSyncHook &NULL

    ; Begin at VBlank
    call StartVBlank

    ; Set up memory mapping
    PPURead  00000, 01FFF, ReadCROM
    PPURead  02000, 02FFF, ReadCiRAM
    PPURead  03000, 03EFF, ReadHighCiRAM
    PPURead  03F00, 03FFF, ReadPalette
    PPUWrite 00000, 01FFF, WriteCRAM
    PPUWrite 02000, 02FFF, WriteCiRAM
    PPUWrite 03000, 03EFF, WriteHighCiRAM
    PPUWrite 03F00, 03FFF, WritePalette
    ret

____________________________________________________________________________________________
; Register reads
____________________________________________________________________________________________

Read200x:
    call PPUSynchronize

    ; Write-only registers
    movzx eax B$VRAMBuffer
    ret

Write2002:
    call PPUSynchronize

    ; Reset latch
    and B$Status 0FF-IN_VBLANK
    mov D$FlipFlop 0
    ret

Read2002:
    call PPUSynchronize

    ; Return PPU status
    movzx eax B$VRAMBuffer
    and al 01F
    or al B$Status

    ; Reset latch
    and B$Status 0FF-IN_VBLANK
    mov D$FlipFlop 0
    ret

Read2004:
    call PPUSynchronize

    ; Read from OAM
    movzx eax B$OAMAddress
    inc B$OAMAddress
    if B$OAMLatch < 8, movzx eax B$OAMLatch
    inc B$OAMLatch
    movzx eax B$eax+OAM
    ret

Read2007:
    call PPUSynchronize

    ; Read from VRAM
    mov eax D$VRAMAddress | and eax 03FFF
    DoPPURead
    xchg al B$VRAMBuffer

ifFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE,
    if D$PPUCycle < 341, ret

    ; Increase VRAM address
    ifFlag. B$Control VERTICAL_WRITE
        add D$VRAMAddress 020
    else
        inc D$VRAMAddress
    endif
    and D$VRAMAddress 07FFF
    ret

____________________________________________________________________________________________
; Register writes
____________________________________________________________________________________________

Write2000:
    call PPUSynchronize

    ; Save register
    mov B$Control al

    ; Update TempAddress
    and eax NAME_TABLE_SELECT
    shl eax, 10
    and D$TempAddress 07FFF-0C00
    or D$TempAddress eax
    ret

Write2001:
    call PPUSynchronize

    ; Save register
    mov B$Mask al
    ; Update masks
    mov D$PixelMask 0
    mov D$ClipMask 0
    ifFlag al BG_VISIBLE,      mov B$PixelMask 0FF
    ifFlag al SPRITES_VISIBLE, mov B$PixelMask+1 0FF
    ifFlag al BG_CLIPPING,     mov B$ClipMask 0FF
    ifFlag al SPRITE_CLIPPING, mov B$ClipMask+1 0FF
    ret

Write2003:
    call PPUSynchronize

    ; Save new OAM address
    mov B$OAMAddress al
    and al 7 | mov B$OAMLatch al
    ret

Write2004:
    call PPUSynchronize

    ; Write
    if. B$OAMLatch < 8
        movzx edx B$OAMLatch
        mov B$edx+OAM al
    else
        movzx edx B$OAMAddress
        if edx >= 8, mov B$edx+OAM al
    endif

    ; Increase address
    inc B$OAMAddress
    inc B$OAMLatch
    ret

Write2005:
    call PPUSynchronize

    ; Horizontal scroll
    if. D$FlipFlop = 0

        mov D$XOffset eax
        and D$XOffset 7

        shr eax, 3  | and eax 01F
        and D$TempAddress 07FFF-01F
        or  D$TempAddress eax

    ; Vertical scroll
    else
        push eax
            shl eax, 12 | and eax 07000
            and D$TempAddress 07FFF-07000
            or  D$TempAddress eax
        pop eax

        shl eax, 2  | and eax 03E0
        and D$TempAddress 07FFF-03E0
        or  D$TempAddress eax
    endif

    ; Toggle
    xor D$FlipFlop 1
    ret

Write2006:

    call PPUSynchronize

    ; Write MSB
    if. D$FlipFlop = 0
        and al 03F
        mov B$TempAddress+1 al

    ; Write LSB, update address
    else
        mov B$TempAddress al
        copy D$TempAddress D$VRAMAddress
    endif

    ; Toggle
    xor B$FlipFlop 1
    ret

Write2007:
    call PPUSynchronize

    ; Write to VRAM
    mov edx D$VRAMAddress | and edx 03FFF
    DoPPUWrite

ifFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE,
    if D$PPUCycle < 341, ret

    ; Increase VRAM address
    ifFlag. B$Control VERTICAL_WRITE
        add W$VRAMAddress 020
    else
        inc W$VRAMAddress
    endif
    and W$VRAMAddress 07FFF
    ret

Write4014:

    ; MSB of 16-bit address
    shl eax 8

    ; Do DMA
    and B$OAMAddress 0FC
    and B$OAMLatch 0FC
L0: push eax
        doRead
        call Write2004
        cpuTick 2
    pop eax
    inc al | jnz L0<
    ret

____________________________________________________________________________________________
; Mirroring
____________________________________________________________________________________________

[mirror | #=1 | call SetMirroring #1]
enumerate 0,
    ONE_SCREEN_2000,
    ONE_SCREEN_2400,
    HORIZONTAL,
    VERTICAL,
    FOUR_SCREEN,
    MIRRORING_0001

SetMirroring:

    call PPUSynchronize
    arguments @Mirroring

    if D@Mirroring = MIRRORING_0001,  setNameTables 0000, 0000, 0000, 0400
    if D@Mirroring = FOUR_SCREEN,     setNameTables 0000, 0400, 0800, 0C00
    if D@Mirroring = ONE_SCREEN_2000, setNameTables 0000, 0000, 0000, 0000
    if D@Mirroring = ONE_SCREEN_2400, setNameTables 0400, 0400, 0400, 0400

    if. D$Cartridge@Mirroring != FOUR_SCREEN_MIRRORING
        if D@Mirroring = HORIZONTAL,  setNameTables 0000, 0000, 0400, 0400
        if D@Mirroring = VERTICAL,    setNameTables 0000, 0400, 0000, 0400
    endif
    return

[SetNameTables | #=4
    mov D$IndexCiRAM+00 #1
    mov D$IndexCiRAM+04 #2
    mov D$IndexCiRAM+08 #3
    mov D$IndexCiRAM+0C #4]

____________________________________________________________________________________________
; Rendering
____________________________________________________________________________________________

PPUSynchronize:

    [@Busy: ?]
    if B$@Busy = &TRUE, ret ; For MMC2 CROM switch during rendering
    mov B$@Busy &TRUE

    pushad

        ; PPU really behind?
        test D$CPUClock 0FFFF_FFFF | js Q0>
        mov eax D$PPUClock | on eax >= D$CPUClock, Q0>

        ; Run PPU until synchronized
    L0: mov eax D$PPUCycle
        inc D$PPUCycle
        push eax
            call D$PPUJumpTable+eax*4
        pop eax
        call D$AddressUpdateTable+eax*4

        ; Loop
        mov eax D$PPUClock | add eax D$PPUCycles | mov D$PPUClock eax
        on eax < D$CPUClock, L0<

Q0: popad
    mov B$@Busy &FALSE
    ret

____________________________________________________________________________________________
; Empty scanline (#241), VBlank.
____________________________________________________________________________________________

IdleEmptyLine:

    ; Continue idle
    mov D$PPUCycle 341
    ; Count down
    dec D$EmptyLineCounter
    jz StartVBlank
    ret

IdleVBlank:

    ; Continue idle
    mov D$PPUCycle 342

    ; MMC5
    if. D$HSyncHook != &NULL
        mov eax D$VBlankCounter
        mov edx 0, ecx 341
        div ecx
        if edx = 0, call D$HSyncHook
    endif

    ; Count down
    dec D$VBlankCounter
    jz EndVBlank
    ret

StartVBlank:

    ; Start waiting in VBlank
    mov D$PPUCycle 342
    copy D$PPUCyclesInVBlank D$VBlankCounter

    ; Adjust regs
    or B$Status IN_VBLANK
    ifFlag B$Control NMI_ON_VBLANK, SetIRQ PPU, IRQ_NMI
    mov D$OAMAddress 0
    mov D$OAMLatch 0

    ; First VBlank cycle
    call IdleVBlank
    ret

EndVBlank:

    if D$HSyncHook != &NULL, call D$HSyncHook

;mov D$Divide42 32
;mov D$A13 0
    if D$VideoCursor != 0, call FlushVideo
    mov D$PPUCycle 0
    mov D$nScanline 0
    mov B$Status 0
    ret

____________________________________________________________________________________________
; Render one pixel
____________________________________________________________________________________________

RenderPixel:

    ; Fetch pixels
    mov esi D$nPixel  |              mov dh B$SpritePipeline+esi
    add esi D$XOffset | and esi 0F | mov dl B$BackgroundPipeline+esi

    ; Clipping
    if D$nPixel < 8, and edx D$ClipMask

    ; GFX visible?
    and edx D$PixelMask

    ; Sprite #0?
    ifFlag. dh 040
        ifFlag dh 3,
            ifFlag dl 3,
                or B$Status SPRITE_ZERO
;;
        if B$Emulate@SpriteZero = &FALSE, break
        mov dh B$APUClock
        or dh 3
;;
    endif

    ; Output sprite
    ifFlag. dh 3
;        on B$Emulate@Sprites = &FALSE, I8>
        movzx eax dh | or eax 010 | and eax 01F
        ifFlag dl 03,
            ifFlag dh 080,
                jmp S0>

    ; Output background
    else
S0:     ;if B$Emulate@Background = &FALSE, mov dl 0
        movzx eax dl
        and eax 0F
    endif

    ; Output pixel
    movzx eax B$eax+Palette
    ifFlag B$Mask MONOCHROME, and al 030
    call VideoOutput
    inc D$nPixel
    ret

____________________________________________________________________________________________
; Cycle 0 - 255
____________________________________________________________________________________________
GetNameTable:
    call ReadNameTable
    call UpdateBackgroundPipeline
    call FindSpritesInRange
    call RenderPixel
    ret

GetAttribute:
    call ReadAttribute
    call RenderPixel
    ret

GetPattern:

    ; MMC5 or normal?
    if. D$Cartridge@Mapper = 5
        call DoMMC5Graphics
    else
        call ReadPattern
    endif

    call FindSpritesInRange
    call RenderPixel
    ret

FindSpritesInRange:

    ; Determine sprite height
    ifFlag. B$Control DOUBLE_SPRITE_SIZE
        mov edx 15
    else
        mov edx 7
    endif

    ; Get OAM data
    ; YYYYYYYY, PATTERN#, VHPxxxAA, XXXXXXXX
    mov ebx D$nPixel | shr ebx 2
    mov eax D$ebx*4+OAM

    ; Out of screen?
    if al >= 240, ret

    ; Calculate new Y
    not al | add al B$nScanline

    ; Out of range?
    if al > dl, ret

    ; Sprite memory full?
    if. D$nSpritesFound < 8
        ; Modify Y to new result
        and al dl

        ; Invert vertical?
        ifFlag eax 080_0000, xor al dl

        ; Sprite #0?
        and eax 0FFFF_FFFF-010_0000
        if ebx = 0, or eax 010_0000

        ; Store in temporary memory
        ; xxxxYYYY, PATTERN#, xHPZxxAA, XXXXXXXX
        mov ebx D$nSpritesFound | inc D$nSpritesFound
        mov D$ebx*4+SpriteMemory eax

        if D$nSpritesFound = 8, or B$Status EIGHT_SPRITES
    endif
    ret
____________________________________________________________________________________________
; Cycle 255
____________________________________________________________________________________________
LineAddress:

    call ZeroMemory SpritePipeline, 0100

    ; GFX visible?
    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Scroll Y
    add W$VRAMAddress 01000 | jns L0>
    and W$VRAMAddress 0FFF
    mov ax W$VRAMAddress | and ax 03E0 | shr ax 5
           if. ax = 29 | and W$VRAMAddress 0FFFF-03E0 | xor W$VRAMAddress 0800
    else | if. ax = 31 | and W$VRAMAddress 0FFFF-03E0
    else | add W$VRAMAddress 020
    endif

L0: ; Update VRAM address(X)
    mov ax W$TempAddress | and ax 00_0000_0100_0001_1111
    and W$VRAMAddress      (07FFF-00_0000_0100_0001_1111)
    or W$VRAMAddress ax

    ; Update VRAM address(Y) at end of dummy scanline
    if. D$nScanline = 0
        copy W$TempAddress W$VRAMAddress
    endif
    ret

____________________________________________________________________________________________
; Cycle 256 - 319
____________________________________________________________________________________________
GetSprite:

    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Any sprites left to fetch?
    mov ebx D$nSprite
    if. ebx = D$nSpritesFound
        mov eax 0
        DoPPURead
        DoPPURead
        ret
    endif
    inc D$nSprite

    ; Get sprite info
    ; xxxxYYYY, PATTERN#, xHPZxxAA, XXXXXXXX
S0: mov ebx D$ebx*4+SpriteMemory

    ; Pattern address -> eax
    ; (Horrible code, but it works)
    mov eax ebx
    ifFlag. B$Control DOUBLE_SPRITE_SIZE
        ; 8x16 sprites
        mov ecx 0
        shr ah 1 | rcl ecx 13
        shl al 5
        rcl ah 1
        shr al 1
        shr ax 4
        and eax 0FFF | or eax ecx
    else
        ; 8x8 sprites
        and al 7
        shl al 4 | shr ax 4
        and eax 0FFF
        ifFlag B$Control SPRITE_ADDRESS_1000, or eax 01000
    endif

    ; Patterns --> dx
    push eax  | DoPPURead | mov dl al | pop eax
    or  eax 8 | DoPPURead | mov dh al

    ; xHPZxxAA --> bl
    ; XXXXXXXX --> bh
    shr ebx 16

    ; Get x coordinate
    movzx edi bh
    add edi SpritePipeline
    mov ecx 8

    ; PZxxAAxx --> bl
    shl bl 2 | jc S0>


    ; - No horizontal flip -
    ; Extract pixel-> al
L1: mov al 0
    shl dh, 1 | rcl al, 1
    shl dl, 1 | rcl al, 1
    ; Add attribute, store
    ifFlag B$edi 3, mov al 0
    if. al != 0
        or al bl
        stosb
    else
        inc edi
    endif
    dec ecx | jnz L1<
    ret

    ; - Horizontal flip -
S0:
    ; Extract pixel-> al
L1: mov al 0
    shr dh, 1 | rcl al, 1
    shr dl, 1 | rcl al, 1
    ; Add attribute, store
    ifFlag B$edi 3, mov al 0
    if. al != 0
        or al bl
        stosb
    else
        inc edi
    endif
    dec ecx | jnz L1<
    ret

____________________________________________________________________________________________
; Cycles 320 - 335
____________________________________________________________________________________________
PreNameTable:
    call ReadNameTable
    call UpdateBackgroundPipeline
    call SimRender
    ret

PreAttribute:
    call ReadAttribute
    call SimRender
    ret

PrePattern:

    ; MMC5 or normal?
    if. D$Cartridge@Mapper = 5
        call DoMMC5Graphics
    else
        call ReadPattern
    endif

    call SimRender
    ret

SimRender: inc D$nPixel | ret

____________________________________________________________________________________________
; Cycle 336 - 339
____________________________________________________________________________________________
DummyFetch: call ReadNameTable | ret
DoNothing:                       ret
____________________________________________________________________________________________
; Cycle 340
____________________________________________________________________________________________
HBlank:

    if D$HSyncHook != &NULL, call D$HSyncHook

    mov D$nPixel 0
    call ZeroMemory SpriteMemory 32
    mov D$nSpritesFound 0
    mov D$nSprite 0
    mov D$PPUCycle 0
    and B$Status 0FF-EIGHT_SPRITES

    inc D$nScanline
    if B$nScanline < 241, ret

    mov D$PPUCycle 341
    mov D$EmptyLineCounter NUM_CYCLES_LINE
    ret
____________________________________________________________________________________________
; Pipeline
____________________________________________________________________________________________
; Sets the pipelined pixels according to W$Patterns
UpdateBackgroundPipeline:

    ; Set up edi
    mov edi D$nPixel
    add edi 8+6 ; Last pixels of next tile
    and edi 0F | add edi BackgroundPipeline

    ; Set up cx/dx (even/odd pixels)
    mov dx W$Patterns | mov cx dx
    and dx 00_01010101_10101010
    and cx 00_10101010_01010101
    shr dl 1 | shl dh 1
    or dl ch | or dh cl

    ; Extract pixel data
    mov ecx D$Attribute
    std
        mov ax dx | and ax 0303 | jz L0> | ifFlag al 3, or al cl | ifFlag ah 3, or ah cl | L0: | stosw | shr dx 2
        mov ax dx | and ax 0303 | jz L0> | ifFlag al 3, or al cl | ifFlag ah 3, or ah cl | L0: | stosw | shr dx 2
        mov ax dx | and ax 0303 | jz L0> | ifFlag al 3, or al cl | ifFlag ah 3, or ah cl | L0: | stosw | shr dx 2
        mov ax dx | and ax 0303 | jz L0> | ifFlag al 3, or al cl | ifFlag ah 3, or ah cl | L0: | stosw | shr dx 2
    cld
    ret
____________________________________________________________________________________________
; Memory reads
____________________________________________________________________________________________

ReadNameTable:

    ; GFX visible?
    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Read name table
    mov eax D$VRAMAddress
    and eax 0FFF | or eax 02000
    DoPPURead
    mov B$NTByte al
    ret

ReadAttribute:

    ; GFX visible?
    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Read attribute
    mov eax 023C0
    mov edx D$VRAMAddress  | shr edx 4 | and dx 038  | or eax edx
    mov edx D$VRAMAddress  | shr edx 2 | and dx 7    | or eax edx
    mov edx D$VRAMAddress  |             and dx 0C00 | or eax edx
    DoPPURead

    ; Shift attribute
    ifFlag W$VRAMAddress 040, shr al 4
    ifFlag W$VRAMAddress 02,  shr al 2
    and al 3 | shl al 2
    mov B$Attribute al
    ret

ReadPattern:

    ; GFX visible?
    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret

    ; Get pattern address
    movzx eax B$NTByte | shl eax 4
    ifFlag B$Control BG_ADDRESS_1000, or eax 01000
    mov dx W$VRAMAddress | shr dx 12 | or ax dx

    ; Get pattern #1
    push eax
        DoPPURead
        mov B$Patterns al
    pop eax

    ; Get pattern #2
    add eax 8
    DoPPURead
    mov B$Patterns+1 al

    ; No BG pixels
    ifNotFlag B$Mask BG_VISIBLE, mov D$Patterns 0
    ret

____________________________________________________________________________________________
; Address updates
____________________________________________________________________________________________

TileAddress:

    ifNotFlag B$Mask BG_VISIBLE+SPRITES_VISIBLE, ret
    ; Next tile
    inc W$VRAMAddress
    ifNotFlag. W$VRAMAddress 01F
        sub W$VRAMAddress 020
        xor W$VRAMAddress 0400
    endif
    ret
____________________________________________________________________________________________
; PPU data
____________________________________________________________________________________________
[LENGTH_PPU      <(SIZE_PPU shl 2)>]
[SIZE_PPU        SIZE_PPU_REGS+SIZE_PPU_MEMORY
 SIZE_PPU_REGS   6+3+1
 SIZE_PPU_MEMORY SIZE_OAM+SIZE_NAMETABLE+SIZE_PALETTE+SIZE_CRAM
 SIZE_OAM        040
 SIZE_NAMETABLE  0400
 SIZE_PALETTE    08
 SIZE_CRAM       02000] ; Extra big for CRAM switching (mappers #13 and #96)
[PPU:
 ; Registers
 Control:      ?
 Mask:         ?
 Status:       ?
 OAMAddress:   ?
 OAMLatch:     ?
 VRAMAddress:  ?

 ; Latches
 TempAddress:  ?
 VRAMBuffer:   ?
 FlipFlop:     ?

 PPUClock:     ?

 ; Memory
 PPUMemory:
 OAM:          ? #SIZE_OAM
 NameTable:    ? #SIZE_NAMETABLE
 Palette:      ? #SIZE_PALETTE
 CRAM:         ? #SIZE_CRAM]

; Register flags
[NAME_TABLE_SELECT   03  ; $2000
 VERTICAL_WRITE      04
 SPRITE_ADDRESS_1000 08
 BG_ADDRESS_1000     010
 DOUBLE_SPRITE_SIZE  020
 NMI_ON_VBLANK       080

 MONOCHROME          01  ; $2001
 BG_CLIPPING         02
 SPRITE_CLIPPING     04
 BG_VISIBLE          08
 SPRITES_VISIBLE     010
 COLOR_MASK          0E0

 EIGHT_SPRITES       020 ; $2002
 SPRITE_ZERO         040
 IN_VBLANK           080]
____________________________________________________________________________________________
; Rendering data
____________________________________________________________________________________________
[LENGTH_RENDERING         <(SIZE_RENDERING shl 2)>]
[SIZE_RENDERING           5+2+04E+014]

[Rendering:
 nScanline:            ?
 nPixel:               ?
 PPUCycle:             ?
 VBlankCounter:        ?
 EmptyLineCounter:     ?

 PixelMask:            ?
 ClipMask:             ?

 nSprite:              ?
 nSpritesFound:        ?
 SpriteMemory:         ? #8
 SpritePipeline:       ? #044

 BackgroundRenderer:
 XOffset:              ?
 NTByte:               ?
 Attribute:            ?
 Patterns:             ?
 BackgroundPipeline:   ? #010]

[NUM_CYCLES_LINE   341]
____________________________________________________________________________________________
; PPU jump table
____________________________________________________________________________________________
[PPUJumpTable:
; Background (0 - 255)
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel

GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel

GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel

GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel
GetNameTable   RenderPixel     GetAttribute    RenderPixel     GetPattern      RenderPixel     RenderPixel     RenderPixel

; Sprites (256 - 319)
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
DummyFetch     DoNothing       DoNothing       DoNothing       GetSprite       DoNothing       DoNothing       DoNothing
; Background for next line (320 - 335)
PreNameTable   SimRender       PreAttribute    SimRender       PrePattern      SimRender       SimRender       SimRender
PreNameTable   SimRender       PreAttribute    SimRender       PrePattern      SimRender       SimRender       SimRender
; Get garbage, HBlank (336 - 340)
DummyFetch     DoNothing       DummyFetch      DoNothing       HBlank
; Do nothing (Empty scanline)
IdleEmptyLine
; Do nothing (VBlank)
IdleVBlank]

[AddressUpdateTable:
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       LineAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       TileAddress

 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing
 DoNothing     DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing       DoNothing]
_____________________________________________________________________________________________
