TITLE IRQCounter
; ____________________________________________________________________________________________
;
;                               CPU cycle counters
;
; ____________________________________________________________________________________________
____________________________________________________________________________________________
; Mapper #083, Dragon Ball Z 4-in-1
; (Probably a bunch of different mappers jammed togehter)

; $5000-$5FFF: RRRRRRRR - Register
; $8000,$B000: --SSPPPP - select ROM segment
;                       - swap 16k PROM at $8000 (SSPPPP)
;                       - swap 16k PROM at $C000 (SS1111)
; $8100: i-----MM - IRQ mode
;                 - mirror VERTICAL/HORIZONTAL/$2000/$2400
; $8200: IIIIIIII - LSB of IRQ counter
; $8201: IIIIIIII - MSB of IRQ counter
;                 - IRQ disable/enable (i = 0/1)
; $8300: PPPPPPPP - swap 8k PROM at $8000
; $8301: PPPPPPPP - swap 8k PROM at $A000
; $8302: PPPPPPPP - swap 8k PROM at $C000
; $8310: CCCCCCCC - swap 1k CROM at $0000 (SSCCCCCCCC)
; $8311: CCCCCCCC - swap 1k CROM at $0400 (SSCCCCCCCC)
; $8312: CCCCCCCC - swap 1k CROM at $0800 (SSCCCCCCCC)
; $8313: CCCCCCCC - swap 1k CROM at $0C00 (SSCCCCCCCC)
; $8314: CCCCCCCC - swap 1k CROM at $1000 (SSCCCCCCCC)
; $8315: CCCCCCCC - swap 1k CROM at $1400 (SSCCCCCCCC)
; $8316: CCCCCCCC - swap 1k CROM at $1800 (SSCCCCCCCC)
; $8317: CCCCCCCC - swap 1k CROM at $1C00 (SSCCCCCCCC)
____________________________________________________________________________________________

Mapper083:

    setIRQCounter IRQCountDown, &NULL

    ; Memory mapping
    CPUWrite 05000, 05FFF, @5000_5FFF
    CPURead  05000, 05FFF, @Read5000_5FFF
    ; PROM segment
    CPUWrite 08000, 08000, @8000
    CPUWrite 0B000, 0B000, @8000
    ; Mirroring
    CPUWrite 08100, 08100, @8100
    ; IRQ counter
    CPUWrite 08200, 08200, @8200
    CPUWrite 08201, 08201, @8201
    ; 8k PROM
    CPUWrite 08300, 08300, @8300
    CPUWrite 08301, 08301, @8301
    CPUWrite 08302, 08302, @8302
    ; 1k CROM
    CPUWrite 08310, 08310, @8310
    CPUWrite 08311, 08311, @8311
    CPUWrite 08312, 08312, @8312
    CPUWrite 08313, 08313, @8313
    CPUWrite 08314, 08314, @8314
    CPUWrite 08315, 08315, @8315
    CPUWrite 08316, 08316, @8316
    CPUWrite 08317, 08317, @8317
    ; 2k CROM
    if. D$Cartridge@PROMCRC32 = 01461_D1F8 ; World heroes 2
        CPUWrite 08310, 08310, @8310_2k
        CPUWrite 08311, 08311, @8311_2k
        CPUWrite 08316, 08316, @8316_2k
        CPUWrite 08317, 08317, @8317_2k
    endif

    ; Last bank
    swap PROM, 16k, 0C000, 0F
    ret

; RRRRRRRR
@5000_5FFF:     mov D$Register+08 eax | ret
@Read5000_5FFF: mov eax D$Register+08 | ret

@8000:

    ; --SSPPPP
    swap PROM, 16k, 08000, eax
    and eax 030 | or eax 0F
    swap PROM, 16k, 0C000, eax

    ; --SS----
    and eax 030 | shl eax 4
    mov D$Register+00 eax
    ret

@8100:

    ; I-------
    test al 080 | setnz B$Register+04

    ; ------MM
    and al 03
    if al = 0, mirror VERTICAL
    if al = 1, mirror HORIZONTAL
    if al = 2, mirror ONE_SCREEN_2000
    if al = 3, mirror ONE_SCREEN_2400
    ret
____________________________________________________________________________________________

; PPPPPPPP
@8300: swap PROM, 8k, 08000, eax | ret
@8301: swap PROM, 8k, 0A000, eax | ret
@8302: swap PROM, 8k, 0C000, eax | ret
; CCCCCCCC
@8310:    or eax D$Register+00 | swap CROM, 1k,  0000, eax | ret
@8311:    or eax D$Register+00 | swap CROM, 1k,  0400, eax | ret
@8312:    or eax D$Register+00 | swap CROM, 1k,  0800, eax | ret
@8313:    or eax D$Register+00 | swap CROM, 1k,  0C00, eax | ret
@8314:    or eax D$Register+00 | swap CROM, 1k, 01000, eax | ret
@8315:    or eax D$Register+00 | swap CROM, 1k, 01400, eax | ret
@8316:    or eax D$Register+00 | swap CROM, 1k, 01800, eax | ret
@8317:    or eax D$Register+00 | swap CROM, 1k, 01C00, eax | ret
; CCCCCCCC
@8310_2k: or eax D$Register+00 | swap CROM, 2k,  0000, eax | ret
@8311_2k: or eax D$Register+00 | swap CROM, 2k,  0800, eax | ret
@8316_2k: or eax D$Register+00 | swap CROM, 2k, 01000, eax | ret
@8317_2k: or eax D$Register+00 | swap CROM, 2k, 01800, eax | ret
____________________________________________________________________________________________

@8200:

    call CartridgeSynchronize

    ; IIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    mov B$IRQCounter+00 al
    ret

@8201:

    call CartridgeSynchronize

    ; IIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    mov B$IRQCounter+01 al
    copy D$Register+04 D$IRQEnabled
    ret
____________________________________________________________________________________________
; Mapper #042, Mario Baby

; $6000-$7FFF: swappable PROM
; (mask $E003)
; $E000: ----PPPP - swap 8k ExROM at $6000
; $E001: ----M--- - mirror VERTICAL/HORIZONTAL (0/1)
; $E002: ------I- - disable/enable IRQ (0/1)

; $40xx: ????????
____________________________________________________________________________________________

Mapper042:

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL
    mov D$IRQLatch 06000

    ; Memory mapping
    CPURead  06000, 07FFF, @ReadExROM
    mov ecx 0E000
L0: push ecx

        and ecx 0E003
        mov eax WriteVoid

        if ecx = 0E000, mov eax @E000
        if ecx = 0E001, mov eax @E001
        if ecx = 0E002, mov eax @E002

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<

    ; Last bank
    swap PROM, 32k, 08000, LAST_BANK
    ret

@E000:

    ; ----PPPP
    and eax 0F | shl eax 3
    mov edx 0 | div D$Cartridge@SizePROM
    shl edx 10 | mov D$Register edx
    ret

@E001:

    ; ----M---
    ifNotFlag. al 08
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

@E002:

    call CartridgeSynchronize

    ; ------I-
    ClearIRQ IRQ_CARTRIDGE
    test al 02 | setnz B$IRQEnabled
    if B$IRQEnabled = &FALSE, copy D$IRQLatch D$IRQCounter
    ret
____________________________________________________________________________________________

@ReadExROM:

    and eax 01FFF
    add eax D$Register
    add eax D$pPROM
    movzx eax B$eax
    ret

____________________________________________________________________________________________
; Mapper #019, Namcot 106

; (mask $F800)
; $5000: IIIIIIII - 8 LSB of IRQ counter
; $5800: iIIIIIII - 7 MSB of IRQ counter, IRQ enable
; $8000: CCCCCCCC - swap 1k CROM at $0000
; $8800: CCCCCCCC - swap 1k CROM at $0400
; $9000: CCCCCCCC - swap 1k CROM at $0800
; $9800: CCCCCCCC - swap 1k CROM at $0C00
; $A000: CCCCCCCC - swap 1k CROM at $1000
; $A800: CCCCCCCC - swap 1k CROM at $1400
; $B000: CCCCCCCC - swap 1k CROM at $1800
; $B800: CCCCCCCC - swap 1k CROM at $1C00
; (CROM bank > $E0 --> use non-swappable CRAM)
; $C000: CCCCCCCC - swap 1k CiROM at $1000
; $C800: CCCCCCCC - swap 1k CiROM at $1400
; $D000: CCCCCCCC - swap 1k CiROM at $1800
; $D800: CCCCCCCC - swap 1k CiROM at $1C00
; (CiROM bank > $E0 --> use CiRAM based on bit 0)
; $E000: --PPPPPP - swap 8k PROM at $8000
; $E800: CcPPPPPP - swap 8k PROM at $A000
;                 - (c = 1) Force CROM at $0000
;                 - (C = 1) Force CROM at $1000
; $F000: --PPPPPP - swap 8k PROM at $C000
____________________________________________________________________________________________

Mapper019:

    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    ; Extra memory/extra sound
    CPUWrite 04800, 04FFF, @4800
    CPURead  04800, 04FFF, @Read4800
    CPUWrite 0F800, 0FFFF, @F800
    call Namco106Sound
    ; IRQ
    CPUWrite 05000, 057FF, @5000
    CPUWrite 05800, 05FFF, @5800
    CPURead  05000, 057FF, @Read5000
    CPURead  05800, 05FFF, @Read5800
    ; CROM
    CPUWrite 08000, 087FF, @8000
    CPUWrite 08800, 08FFF, @8800
    CPUWrite 09000, 097FF, @9000
    CPUWrite 09800, 09FFF, @9800
    CPUWrite 0A000, 0A7FF, @A000
    CPUWrite 0A800, 0AFFF, @A800
    CPUWrite 0B000, 0B7FF, @B000
    CPUWrite 0B800, 0BFFF, @B800
    ; CiROM
    on D$Cartridge@PROMCRC32 = 0B62A_7B71, S0> ; Family Circuit '91
    on D$Cartridge@PROMCRC32 = 01494_2C06, S0> ; Wagan Land 3
        CPUWrite 0C000, 0C7FF, @C000
        CPUWrite 0C800, 0CFFF, @C800
        CPUWrite 0D000, 0D7FF, @D000
        CPUWrite 0D800, 0DFFF, @D800
    S0:
    ; PROM
    CPUWrite 0E000, 0E7FF, @E000
    CPUWrite 0E800, 0EFFF, @E800
    CPUWrite 0F000, 0F7FF, @F000

    ; PPU settings
    if. D$Cartridge@SizeCROM > 0

        ; Override address line
        PPUWrite 00000, 01FFF, @WriteCROM
        PPURead  00000, 01FFF, @ReadCROM

        ; CiROM, but not for these two games
        on D$Cartridge@PROMCRC32 = 0B62A_7B71, S0> ; Family Circuit '91
        on D$Cartridge@PROMCRC32 = 01494_2C06, S0> ; Wagan Land 3
        PPUWrite 02000, 02FFF, @WriteCiROM
        PPURead  02000, 02FFF, @ReadCiROM

        ; Enable 8k CRAM
    S0: mov D$Cartridge@SizeCRAM 8
        swap CRAM, 8k, 00000, 0

        ; Set all banks to CROM
        mov D$Bank+00 01010101
        mov D$Bank+04 01010101

    endif

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; CCCCCCCC
@8000: swap CROM,  1k,  0000, eax | cmp al 0E0 | setb B$Bank+00 | ifFlag B$Register 040, mov B$Bank+00 &TRUE | ret
@8800: swap CROM,  1k,  0400, eax | cmp al 0E0 | setb B$Bank+01 | ifFlag B$Register 040, mov B$Bank+01 &TRUE | ret
@9000: swap CROM,  1k,  0800, eax | cmp al 0E0 | setb B$Bank+02 | ifFlag B$Register 040, mov B$Bank+02 &TRUE | ret
@9800: swap CROM,  1k,  0C00, eax | cmp al 0E0 | setb B$Bank+03 | ifFlag B$Register 040, mov B$Bank+03 &TRUE | ret
@A000: swap CROM,  1k, 01000, eax | cmp al 0E0 | setb B$Bank+04 | ifFlag B$Register 080, mov B$Bank+04 &TRUE | ret
@A800: swap CROM,  1k, 01400, eax | cmp al 0E0 | setb B$Bank+05 | ifFlag B$Register 080, mov B$Bank+05 &TRUE | ret
@B000: swap CROM,  1k, 01800, eax | cmp al 0E0 | setb B$Bank+06 | ifFlag B$Register 080, mov B$Bank+06 &TRUE | ret
@B800: swap CROM,  1k, 01C00, eax | cmp al 0E0 | setb B$Bank+07 | ifFlag B$Register 080, mov B$Bank+07 &TRUE | ret
; CCCCCCCC
@C000: swap CiROM, 1k, 02000, eax | cmp al 0E0 | setb B$Bank+08 | and al 01 | shl eax 10 | mov D$IndexCiRAM+00 eax | ret
@C800: swap CiROM, 1k, 02400, eax | cmp al 0E0 | setb B$Bank+09 | and al 01 | shl eax 10 | mov D$IndexCiRAM+04 eax | ret
@D000: swap CiROM, 1k, 02800, eax | cmp al 0E0 | setb B$Bank+0A | and al 01 | shl eax 10 | mov D$IndexCiRAM+08 eax | ret
@D800: swap CiROM, 1k, 02C00, eax | cmp al 0E0 | setb B$Bank+0B | and al 01 | shl eax 10 | mov D$IndexCiRAM+0C eax | ret
; PPPPPPPP
@E000: swap PROM,  8k, 08000, eax | ret; if (pRomCrc == 0x14942C06UL) // Wagan Land 3  ppu.SetMirroring( (data & 0x40) ? MIRROR_VERTICAL : MIRROR_ZERO );
@E800: mov B$Register al | swap PROM,  8k, 0A000, eax | ret
@F000: swap PROM,  8k, 0C000, eax | ret
____________________________________________________________________________________________

@WriteCROM:

    mov ebx edx | shr ebx 10
    on B$Bank+ebx = &FALSE, WritePlainCRAM
    ret

@WriteCiROM:

    mov ebx edx | shr ebx 10
    on B$Bank+ebx = &FALSE, WriteCiRAM
    ret

@ReadCROM:

    mov ebx eax | shr ebx 10
    on B$Bank+ebx = &FALSE, ReadPlainCRAM
    jmp ReadCROM

@ReadCiROM:

    mov ebx eax | shr ebx 10
    on B$Bank+ebx = &FALSE, ReadCiRAM
    jmp ReadCiROM
____________________________________________________________________________________________

[ExRAM CRAM+02000]
@F800: mov D$Register+04 eax | ret

@4800:

    mov edx D$Register+04
    and edx 07F | mov B$ExRAM+edx al
    ifFlag. B$Register+04 080
        inc B$Register+04
        or  B$Register+04 080
    endif

    on eax = 0329F_FF7A, S0> ; Battle Fleet
    on eax = 09653_3999, S0> ; Dokuganryuu Masamune
    on eax = 0429F_D177, S0> ; Famista '90
    on eax = 0DD45_4208, S0> ; Hydlide 3 - Yami Kara no Houmonsha
    on eax = 0B1B9_E187, S0> ; Kaijuu Monogatari
    on eax = 0AF15_338F, S0> ; Mindseeker
    call Namco106Sound@4800
S0: ret

@Read4800:

    mov eax D$Register+04
    and eax 07F | mov al B$ExRAM+eax
    ifFlag. B$Register+04 080
        inc B$Register+04
        or  B$Register+04 080
    endif
    ret

____________________________________________________________________________________________

@5000:

    call CartridgeSynchronize

    ; IIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    not al
    mov B$IRQCounter+00 al
    ret

@5800:

    call CartridgeSynchronize

    ; iIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    test al 080 | setnz B$IRQEnabled
    not al | and al 07F
    mov B$IRQCounter+01 al
    ret

@Read5000:

    call CartridgeSynchronize

    movzx eax B$IRQCounter+00 | not al
    ret

@Read5800:

    call CartridgeSynchronize

    movzx eax B$IRQCounter+01 | not al
    ret
____________________________________________________________________________________________
; Mapper #018, Jaleco SS8806

; $8000: xxxxPPPP (xxxx3210) - swap 8k PROM at $8000
; $8001: xxxxPPPP (xxxx7654) - swap 8k PROM at $8000
; $8002: xxxxPPPP (xxxx3210) - swap 8k PROM at $A000
; $8003: xxxxPPPP (xxxx7654) - swap 8k PROM at $A000
; $9000: xxxxPPPP (xxxx3210) - swap 8k PROM at $C000
; $9001: xxxxPPPP (xxxx7654) - swap 8k PROM at $C000

; $A000: xxxxCCCC (xxxx3210) - swap 1k CROM at $0000
; $A001: xxxxCCCC (xxxx7654) - swap 1k CROM at $0000
; $A002: xxxxCCCC (xxxx3210) - swap 1k CROM at $0400
; $A003: xxxxCCCC (xxxx7654) - swap 1k CROM at $0400
; $B000: xxxxCCCC (xxxx3210) - swap 1k CROM at $0800
; $B001: xxxxCCCC (xxxx7654) - swap 1k CROM at $0800
; $B002: xxxxCCCC (xxxx3210) - swap 1k CROM at $0C00
; $B003: xxxxCCCC (xxxx7654) - swap 1k CROM at $0C00
; $C000: xxxxCCCC (xxxx3210) - swap 1k CROM at $1000
; $C001: xxxxCCCC (xxxx7654) - swap 1k CROM at $1000
; $C002: xxxxCCCC (xxxx3210) - swap 1k CROM at $1400
; $C003: xxxxCCCC (xxxx7654) - swap 1k CROM at $1400
; $D000: xxxxCCCC (xxxx3210) - swap 1k CROM at $1800
; $D001: xxxxCCCC (xxxx7654) - swap 1k CROM at $1800
; $D002: xxxxCCCC (xxxx3210) - swap 1k CROM at $1C00
; $D003: xxxxCCCC (xxxx7654) - swap 1k CROM at $1C00

; $E000: xxxxIIII (xxxx3210) - 1st nibble of IRQ counter latch
; $E001: xxxxIIII (xxxx7654) - 2nd nibble of IRQ counter latch
; $E002: xxxxIIII (xxxxBA98) - 3rd nibble of IRQ counter latch
; $E003: xxxxIIII (xxxxFEDC) - 4th nibble of IRQ counter latch

; $F000: xxxxxxxx - Reset IRQ counter
; $F001: xxxxxxxI - disable/enable IRQ (0/1)
; $F002: xxxxxxMM - mirror HORIZONTAL/VERTICAL/$2000/$2000 (0/1/2/3)
____________________________________________________________________________________________

Mapper018:

    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        and ecx 0F003
        mov eax WriteVoid

        ; PROM
        if ecx = 08000, mov eax @8000
        if ecx = 08001, mov eax @8001
        if ecx = 08002, mov eax @8002
        if ecx = 08003, mov eax @8003
        if ecx = 09000, mov eax @9000
        if ecx = 09001, mov eax @9001
        ; CROM
        if ecx = 0A000, mov eax @A000
        if ecx = 0A001, mov eax @A001
        if ecx = 0A002, mov eax @A002
        if ecx = 0A003, mov eax @A003
        if ecx = 0B000, mov eax @B000
        if ecx = 0B001, mov eax @B001
        if ecx = 0B002, mov eax @B002
        if ecx = 0B003, mov eax @B003
        if ecx = 0C000, mov eax @C000
        if ecx = 0C001, mov eax @C001
        if ecx = 0C002, mov eax @C002
        if ecx = 0C003, mov eax @C003
        if ecx = 0D000, mov eax @D000
        if ecx = 0D001, mov eax @D001
        if ecx = 0D002, mov eax @D002
        if ecx = 0D003, mov eax @D003
        ; IRQ
        if ecx = 0E000, mov eax @E000
        if ecx = 0E001, mov eax @E001
        if ecx = 0E002, mov eax @E002
        if ecx = 0E003, mov eax @E003
        if ecx = 0F000, mov eax @F000
        if ecx = 0F001, mov eax @F001
        ; Mirroring
        if ecx = 0F002, mov eax @F002

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Set up banks
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; xxxxPPPP
@8000: and eax 0F |           | and D$Bank+00  0F0 | or D$Bank+00  eax | swap PROM, 8k, 08000, D$Bank+00 | ret
@8001: and eax 0F | shl eax 4 | and D$Bank+00   0F | or D$Bank+00  eax | swap PROM, 8k, 08000, D$Bank+00 | ret
@8002: and eax 0F |           | and D$Bank+04  0F0 | or D$Bank+04  eax | swap PROM, 8k, 0A000, D$Bank+04 | ret
@8003: and eax 0F | shl eax 4 | and D$Bank+04   0F | or D$Bank+04  eax | swap PROM, 8k, 0A000, D$Bank+04 | ret
@9000: and eax 0F |           | and D$Bank+08  0F0 | or D$Bank+08  eax | swap PROM, 8k, 0C000, D$Bank+08 | ret
@9001: and eax 0F | shl eax 4 | and D$Bank+08   0F | or D$Bank+08  eax | swap PROM, 8k, 0C000, D$Bank+08 | ret
;9002: and eax 0F |           | and D$Bank+0C  0F0 | or D$Bank+0C  eax | swap PROM, 8k, 0E000, D$Bank+0C | ret
;9003: and eax 0F | shl eax 4 | and D$Bank+0C   0F | or D$Bank+0C  eax | swap PROM, 8k, 0E000, D$Bank+0C | ret


; xxxxCCCC
@A000: and eax 0F |           | and D$Bank+010 0F0 | or D$Bank+010 eax | swap CROM, 1k,  0000, D$Bank+010 | ret
@A001: and eax 0F | shl eax 4 | and D$Bank+010  0F | or D$Bank+010 eax | swap CROM, 1k,  0000, D$Bank+010 | ret
@A002: and eax 0F |           | and D$Bank+014 0F0 | or D$Bank+014 eax | swap CROM, 1k,  0400, D$Bank+014 | ret
@A003: and eax 0F | shl eax 4 | and D$Bank+014  0F | or D$Bank+014 eax | swap CROM, 1k,  0400, D$Bank+014 | ret
@B000: and eax 0F |           | and D$Bank+018 0F0 | or D$Bank+018 eax | swap CROM, 1k,  0800, D$Bank+018 | ret
@B001: and eax 0F | shl eax 4 | and D$Bank+018  0F | or D$Bank+018 eax | swap CROM, 1k,  0800, D$Bank+018 | ret
@B002: and eax 0F |           | and D$Bank+01C 0F0 | or D$Bank+01C eax | swap CROM, 1k,  0C00, D$Bank+01C | ret
@B003: and eax 0F | shl eax 4 | and D$Bank+01C  0F | or D$Bank+01C eax | swap CROM, 1k,  0C00, D$Bank+01C | ret
@C000: and eax 0F |           | and D$Bank+020 0F0 | or D$Bank+020 eax | swap CROM, 1k, 01000, D$Bank+020 | ret
@C001: and eax 0F | shl eax 4 | and D$Bank+020  0F | or D$Bank+020 eax | swap CROM, 1k, 01000, D$Bank+020 | ret
@C002: and eax 0F |           | and D$Bank+024 0F0 | or D$Bank+024 eax | swap CROM, 1k, 01400, D$Bank+024 | ret
@C003: and eax 0F | shl eax 4 | and D$Bank+024  0F | or D$Bank+024 eax | swap CROM, 1k, 01400, D$Bank+024 | ret
@D000: and eax 0F |           | and D$Bank+028 0F0 | or D$Bank+028 eax | swap CROM, 1k, 01800, D$Bank+028 | ret
@D001: and eax 0F | shl eax 4 | and D$Bank+028  0F | or D$Bank+028 eax | swap CROM, 1k, 01800, D$Bank+028 | ret
@D002: and eax 0F |           | and D$Bank+02C 0F0 | or D$Bank+02C eax | swap CROM, 1k, 01C00, D$Bank+02C | ret
@D003: and eax 0F | shl eax 4 | and D$Bank+02C  0F | or D$Bank+02C eax | swap CROM, 1k, 01C00, D$Bank+02C | ret

@F002:

    ; xxxxxxMM
    and eax 3
    if eax = 0, mirror HORIZONTAL
    if eax = 1, mirror VERTICAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2000
    ret
____________________________________________________________________________________________

; xxxxIIII
@E000: call CartridgeSynchronize | and eax 0F | shl eax 00 | and D$IRQLatch 0FFF0 | or D$IRQLatch eax | ret
@E001: call CartridgeSynchronize | and eax 0F | shl eax 04 | and D$IRQLatch 0FF0F | or D$IRQLatch eax | ret
@E002: call CartridgeSynchronize | and eax 0F | shl eax 08 | and D$IRQLatch 0F0FF | or D$IRQLatch eax | ret
@E003: call CartridgeSynchronize | and eax 0F | shl eax 0C | and D$IRQLatch  0FFF | or D$IRQLatch eax | ret

@F000:

    call CartridgeSynchronize

    ; xxxxxxxI
    copy D$IRQLatch D$IRQCounter
    ret

@F001:

    call CartridgeSynchronize

    ; xxxxxxxI
    ClearIRQ IRQ_CARTRIDGE
    test al 01 | setnz B$IRQEnabled
    ret
____________________________________________________________________________________________
; Mapper #017, FFE F8xxx

; $42FE: xxxMxxxx - mirror $2000/$2400 (0/1)
; $42FF: xxxMxxxx - mirror VERTICAL/HORIZONTAL (0/1)
; $4501: xxxxxxxx - disable IRQ
; $4502: IIIIIIII - LSB of IRQ counter
; $4503: IIIIIIII - MSB of IRQ counter
; $4504: PPPPPPPP - swap 8k PROM at $8000
; $4505: PPPPPPPP - swap 8k PROM at $A000
; $4506: PPPPPPPP - swap 8k PROM at $C000
; $4507: PPPPPPPP - swap 8k PROM at $E000
; $4510: CCCCCCCC - swap 1k CROM at $0000
; $4511: CCCCCCCC - swap 1k CROM at $0400
; $4512: CCCCCCCC - swap 1k CROM at $0800
; $4513: CCCCCCCC - swap 1k CROM at $0C00
; $4514: CCCCCCCC - swap 1k CROM at $1000
; $4515: CCCCCCCC - swap 1k CROM at $1400
; $4516: CCCCCCCC - swap 1k CROM at $1800
; $4517: CCCCCCCC - swap 1k CROM at $1C00
____________________________________________________________________________________________

Mapper017:

    setIRQCounter IRQCountdown @IRQ

    ; Set ports
    CPUWrite 042FE, 042FE, @42FE
    CPUWrite 042FF, 042FF, @42FF
    CPUWrite 04500, 04500, @4500
    CPUWrite 04501, 04501, @4501
    CPUWrite 04502, 04502, @4502
    CPUWrite 04503, 04503, @4503
    CPUWrite 04504, 04504, @4504
    CPUWrite 04505, 04505, @4505
    CPUWrite 04506, 04506, @4506
    CPUWrite 04507, 04507, @4507
    CPUWrite 04510, 04510, @4510
    CPUWrite 04511, 04511, @4511
    CPUWrite 04512, 04512, @4512
    CPUWrite 04513, 04513, @4513
    CPUWrite 04514, 04514, @4514
    CPUWrite 04515, 04515, @4515
    CPUWrite 04516, 04516, @4516
    CPUWrite 04517, 04517, @4517

    ; Set IRQ counter length
    mov D@IRQLength 010000
    mov eax D$Cartridge@PROMCRC32
    if eax != 057BA_F095, ; Doki! Doki! Yuuenchi
    if eax != 0E641_38EC, ; Dragon Ball Z 2 - Gekishin Freeza!!
    if eax != 0C7A4_583E, ; Dragon Ball Z 3 - Ressen Jinzou Ningen
    if eax != 0FF2C_8EE4, ; Dynamite Batman
    if eax != 0CB7E_529D, ; SD Gundam Gaiden - Knight Gundam Monogatari 2 - Hikari no Kishi
    if eax != 08F3F_8B1F, ; Spartan X2
    if eax != 0A304_7263, ; Spartan X2
    if eax != 0C529_C604, ; Spartan X2
        mov D@IRQLength 0D000

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@4504: swap PROM, 8k, 08000, eax | ret
@4505: swap PROM, 8k, 0A000, eax | ret
@4506: swap PROM, 8k, 0C000, eax | ret
@4507: swap PROM, 8k, 0E000, eax | ret

; CCCCCCCC
@4510: swap CROM, 1k,  0000, eax | ret
@4511: swap CROM, 1k,  0400, eax | ret
@4512: swap CROM, 1k,  0800, eax | ret
@4513: swap CROM, 1k,  0C00, eax | ret
@4514: swap CROM, 1k, 01000, eax | ret
@4515: swap CROM, 1k, 01400, eax | ret
@4516: swap CROM, 1k, 01800, eax | ret
@4517: swap CROM, 1k, 01C00, eax | ret

@4500:
;outwrite
test al al | setnz B$IRQEnabled
    ret

@42FE:
;outwrite

    ; xxxMxxxx
    ifNotFlag. eax 010
        mirror ONE_SCREEN_2000
    else
        mirror ONE_SCREEN_2400
    endif
    ret

@42FF:
;outwrite

    ; xxxMxxxx
shr eax 4
and eax 3
if al = 2, mirror VERTICAL
if al = 3, mirror HORIZONTAL
if al = 0, mirror ONE_SCREEN_2000
if al = 1, mirror ONE_SCREEN_2400
ret

    ifNotFlag. eax 020
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

@4501:

    call CartridgeSynchronize

    ; Disable IRQs
    ClearIRQ IRQ_CARTRIDGE
    mov B$IRQEnabled &FALSE
    ret

@4502:

    call CartridgeSynchronize

    ; IIIIIIII
    mov B$IRQLatch al
    ret

@4503:

    call CartridgeSynchronize

    ; IIIIIIII
    mov B$IRQLatch+1 al

    ; Enable IRQs
    ClearIRQ IRQ_CARTRIDGE
    mov B$IRQEnabled &TRUE
    mov eax D@IRQLength | sub eax D$IRQLatch
    mov D$IRQCounter eax
    ret

@IRQ:

    ; IRQ!
    copy D@IRQLength D$IRQCounter
    mov D$IRQEnabled &FALSE
    SetIRQ Cartridge, IRQ_CARTRIDGE
    ret

[@IRQLength: ?]
____________________________________________________________________________________________
; Mapper #016, Bandai

; xxx0: CCCCCCCC - swap 1k  CROM at $0000
; xxx1: CCCCCCCC - swap 1k  CROM at $0400
; xxx2: CCCCCCCC - swap 1k  CROM at $0800
; xxx3: CCCCCCCC - swap 1k  CROM at $0C00
; xxx4: CCCCCCCC - swap 1k  CROM at $1000
; xxx5: CCCCCCCC - swap 1k  CROM at $1400
; xxx6: CCCCCCCC - swap 1k  CROM at $1800
; xxx7: CCCCCCCC - swap 1k  CROM at $1C00
; xxx8: PPPPPPPP - swap 16k PROM at $8000
; xxx9: xxxxxxMM - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; xxxA: xxxxxxxI - IRQ enable/disable (1/0)
; xxxB: IIIIIIII - LSB of IRQ counter latch
; xxxC: IIIIIIII - MSB of IRQ counter latch
; xxxD: DDDDDDDD - EEPROM (?)
____________________________________________________________________________________________

Mapper016:

    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    mov ecx 06000
L0: push ecx

        mov eax WriteVoid
        and ecx 0F

        [@WriteTable: @xxx0 @xxx1 @xxx2 @xxx3 @xxx4 @xxx5 @xxx6     @xxx7
                      @xxx8 @xxx9 @xxxA @xxxB @xxxC @xxxD WriteVoid WriteVoid]

        mov eax D@WriteTable+ecx*4

        ; Don't swap CRAM
        if ecx < 08,
            if D$Cartridge@SizeCROM = 0,
                mov eax WriteVoid

    pop ecx
    CPUWrite ecx, ecx, eax
    inc ecx | on ecx <= 0FFFF, L0<<

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; CCCCCCCC
@xxx0: swap CROM, 1k,   0000, eax | ret
@xxx1: swap CROM, 1k,   0400, eax | ret
@xxx2: swap CROM, 1k,   0800, eax | ret
@xxx3: swap CROM, 1k,   0C00, eax | ret
@xxx4: swap CROM, 1k,  01000, eax | ret
@xxx5: swap CROM, 1k,  01400, eax | ret
@xxx6: swap CROM, 1k,  01800, eax | ret
@xxx7: swap CROM, 1k,  01C00, eax | ret

; PPPPPPPP
@xxx8: swap PROM, 16k, 08000, eax | ret

@xxx9:

    ; xxxxxxMM
    and eax 3
    if eax = 0, mirror VERTICAL
    if eax = 1, mirror HORIZONTAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2400
    ret

@xxxA:

    call CartridgeSynchronize

    ; xxxxxxxI
    ClearIRQ IRQ_CARTRIDGE
    test al 01 | setnz B$IRQEnabled
    if B$IRQEnabled = &TRUE, copy D$IRQLatch D$IRQCounter
    ret

; IIIIIIII
@xxxB: call CartridgeSynchronize | mov B$IRQLatch+00 al | ret
@xxxC: call CartridgeSynchronize | mov B$IRQLatch+01 al | ret

; EEPROM (?)
@xxxD: ret

____________________________________________________________________________________________
; Mapper #050, SMB2j rev A

; $4020: xxxxxxxI - IRQ disabled/enabled (0/1)
; $4120: xxxxPPPP - swap 8k PROM at $C000
____________________________________________________________________________________________

Mapper050:

    setIRQCounter IRQCountdown, &NULL
    mov D$IRQLatch 01000

    ; Set ports
    CPURead  06000, 07FFF, @ReadWROM
    mov ecx 04020
L0: push ecx

        mov eax WriteVoid
        and ecx 0E160
        if ecx = 04020, mov eax @4020
        if ecx = 04120, mov eax @4120

    pop ecx
    CPUWrite ecx, ecx, eax
    inc ecx | on ecx < 06000, L0<

    ; Swap in PROM banks
    swap PROM, 32k, 08000, 2
    ret

@ReadWROM:

    ; Read from 8k PROM bank #F
    and eax 01FFF
    add eax (0F shl 13)
    add eax D$pPROM
    movzx eax B$eax
    ret

@4020:

    ; xxxxPPPP (xxxx3102)
    mov edx eax
    and edx 8                          ; 3xxx
    ifFlag eax 01, or edx 4            ; x2xx
    and eax 7 | shr eax 1 | or edx eax ; xx10
    swap PROM, 8k, 0C000, edx
    ret

@4120:

    call CartridgeSynchronize
    ClearIRQ IRQ_CARTRIDGE

    ; xxxxxxxI
    test al 01 | setnz B$IRQEnabled
    if B$IRQEnabled = &TRUE, mov D$IRQCounter 01000
    ret
____________________________________________________________________________________________
; Mapper #40, SMB2j

; $8000-$9FFF: xxxxxxxx - disable IRQ counter
; $A000-$BFFF: xxxxxxxx - enable IRQ counter
; $E000-$FFFF: xxxxxPPP - swap 8k PROM at $C000
____________________________________________________________________________________________

Mapper040:

    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    CPURead  06000, 07FFF, @ReadWROM
    CPUWrite 08000, 09FFF, @8000_9FFF
    CPUWrite 0A000, 0BFFF, @A000_BFFF
    CPUWrite 0E000, 0FFFF, @E000_FFFF
CPUWrite 04020, 04021, @4020_4021

    ; Swap PROM banks
    swap PROM, 32k, 08000, 1
    ret

; ????????
@4020_4021:ret

@ReadWROM:

    ; Read from 8k PROM bank #6
    add eax D$pPROM
    sub eax 06000
    movzx eax B$eax+0C000
    ret

@E000_FFFF:

    ; xxxxxPPP
    swap PROM, 8k, 0C000, eax
    ret

@8000_9FFF:

    call CartridgeSynchronize

    ; Disable IRQ
    ClearIRQ IRQ_CARTRIDGE
    mov D$IRQEnabled &FALSE
    ret

@A000_BFFF:

    call CartridgeSynchronize

    ; Enable IRQ
    ClearIRQ IRQ_CARTRIDGE
    mov D$IRQEnabled &TRUE
    mov D$IRQCounter 4096
    ret
____________________________________________________________________________________________
; Mapper #065, Irem H3001

; $8000: PPPPPPPP - swap 8k PROM at $8000
; $A000: PPPPPPPP - swap 8k PROM at $A000
; $C000: PPPPPPPP - swap 8k PROM at $C000
; $9000: xMxxxxxx - mirror HORIZONTAL/VERTICAL (0/1)
; $9003: Ixxxxxxx - disable/enable IRQ (0/1)
; $9004: xxxxxxxx - update IRQ counter
; $9005: IIIIIIII - high byte of IRQ latch
; $9006: IIIIIIII - low  byte of IRQ latch
; $B000: CCCCCCCC - swap 1k CROM at $0000
; $B001: CCCCCCCC - swap 1k CROM at $0400
; $B002: CCCCCCCC - swap 1k CROM at $0800
; $B003: CCCCCCCC - swap 1k CROM at $0C00
; $B004: CCCCCCCC - swap 1k CROM at $1000
; $B005: CCCCCCCC - swap 1k CROM at $1400
; $B006: CCCCCCCC - swap 1k CROM at $1800
; $B007: CCCCCCCC - swap 1k CROM at $1C00
____________________________________________________________________________________________

Mapper065:

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 0A000, 0A000, @A000
    CPUWrite 0C000, 0C000, @C000
    CPUWrite 09000, 09000, @9000
    CPUWrite 09003, 09003, @9003
    CPUWrite 09004, 09004, @9004
    CPUWrite 09005, 09005, @9005
    CPUWrite 09006, 09006, @9006
    CPUWrite 0B000, 0B000, @B000
    CPUWrite 0B001, 0B001, @B001
    CPUWrite 0B002, 0B002, @B002
    CPUWrite 0B003, 0B003, @B003
    CPUWrite 0B004, 0B004, @B004
    CPUWrite 0B005, 0B005, @B005
    CPUWrite 0B006, 0B006, @B006
    CPUWrite 0B007, 0B007, @B007

    ; IRQ counter and mirroring hack
    if. D$Cartridge@PROMCRC32 = 0E30B_7F64 ; Kaiketsu Yanchamaru 3
        CPUWrite 09000, 09004, WriteVoid
        CPUWrite 09001, 09001, @9001
        CPUWrite 09005, 09005, @Hack9005
        CPUWrite 09006, 09006, @Hack9006
    endif

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@8000: swap PROM, 8k, 08000, eax | ret
@A000: swap PROM, 8k, 0A000, eax | ret
@C000: swap PROM, 8k, 0C000, eax | ret

; CCCCCCCC
@B000: swap CROM, 1k,  0000, eax | ret
@B001: swap CROM, 1k,  0400, eax | ret
@B002: swap CROM, 1k,  0800, eax | ret
@B003: swap CROM, 1k,  0C00, eax | ret
@B004: swap CROM, 1k, 01000, eax | ret
@B005: swap CROM, 1k, 01400, eax | ret
@B006: swap CROM, 1k, 01800, eax | ret
@B007: swap CROM, 1k, 01C00, eax | ret

@9000:

    ; xMxxxxxx
    ifNotFlag. eax 040
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif
    ret

@9001:

    ; Mxxxxxxx
    ifNotFlag. eax 080
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

@9003:

    call CartridgeSynchronize

    ; Ixxxxxxx
    ClearIRQ IRQ_CARTRIDGE
    test eax 080
    setnz B$IRQEnabled
    ret

@9004:

    call CartridgeSynchronize

    ; xxxxxxxx
    copy D$IRQLatch D$IRQCounter
    ret

; IIIIIIII
@9005: call CartridgeSynchronize | mov B$IRQLatch+01 al | ret
@9006: call CartridgeSynchronize | mov B$IRQLatch+00 al | ret

@Hack9005:

    call CartridgeSynchronize

    ClearIRQ IRQ_CARTRIDGE
    test eax eax | setnz B$IRQEnabled
    shl al 1 | mov D$IRQCounter eax
    ret

@Hack9006:

    call CartridgeSynchronize

    mov B$IRQEnabled &TRUE
    ret
____________________________________________________________________________________________
; Mapper #069, Sunsoft FME-7

; $8000-$9FFF: xxxxNNNN - command number
; $A000-$BFFF: ABCDEFGH - (command = 0) swap 1k CROM at $0000
;                       - (command = 1) swap 1k CROM at $0400
;                       - (command = 2) swap 1k CROM at $0800
;                       - (command = 3) swap 1k CROM at $0C00
;                       - (command = 4) swap 1k CROM at $1000
;                       - (command = 5) swap 1k CROM at $1400
;                       - (command = 6) swap 1k CROM at $1800
;                       - (command = 7) swap 1k CROM at $1C00
;                       - (command = 8) swap 8k PROM at $6000
;                       - (command = 9) swap 8k PROM at $8000
;                       - (command = A) swap 8k PROM at $A000
;                       - (command = B) swap 8k PROM at $C000
;                       - (command = C) mirror VERTICAL/HORIZONTAL/$2000/$2400 (GH = 0/1/2/3)
;                       - (command = D) IRQ disabled/enabled (= 0 / != 0)
;                       - (command = E) low  byte of IRQ counter
;                       - (command = F) high byte of IRQ counter
; $C000-$DFFF: External sound
; $E000-$FFFF: External sound
____________________________________________________________________________________________

Mapper069:

    ; Sound
    call FME7Sound

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL
    mov D$IRQLatch 0FFFF

    ; Memory mapping
    CPUWrite 08000, 09FFF, @8000_9FFF
    CPUWrite 0A000, 0BFFF, @A000_BFFF
    CPUWrite 06000, 07FFF, @WriteWRAM
    CPURead  06000, 07FFF, @ReadWRAM

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000_9FFF:

    ; xxxxNNNN
    and eax 0F
    mov D$Command eax
    ret

@A000_BFFF:

    mov edx D$Command

    ; CCCCCCCC
    if. dl = 00 | swap CROM, 1k,  0000, eax | else
    if. dl = 01 | swap CROM, 1k,  0400, eax | else
    if. dl = 02 | swap CROM, 1k,  0800, eax | else
    if. dl = 03 | swap CROM, 1k,  0C00, eax | else
    if. dl = 04 | swap CROM, 1k, 01000, eax | else
    if. dl = 05 | swap CROM, 1k, 01400, eax | else
    if. dl = 06 | swap CROM, 1k, 01800, eax | else
    if. dl = 07 | swap CROM, 1k, 01C00, eax | else

    ; PPPPPPPP
    if. dl = 08

        ifFlag.. eax 040

            ; Use WRAM
            ifFlag eax 080, mov B$WRAMEnabled &TRUE

        else..

            ; Use PROM
            mov B$WRAMEnabled &FALSE
            shl eax 3
            mov edx 0 | div D$Cartridge@SizePROM
            shl edx 10
            mov D$Bank edx

        endif..
    else
    if. dl = 09 | swap PROM, 8k, 08000, eax | else
    if. dl = 0A | swap PROM, 8k, 0A000, eax | else
    if. dl = 0B | swap PROM, 8k, 0C000, eax | else

    ; ------MM
    if. dl = 0C | and al 03
        if al = 0, mirror VERTICAL
        if al = 1, mirror HORIZONTAL
        if al = 2, mirror ONE_SCREEN_2000
        if al = 3, mirror ONE_SCREEN_2400
    else

    ; IIIIIIII
    if. dl = 0D | call CartridgeSynchronize | ClearIRQ IRQ_CARTRIDGE | test al al | setnz B$IRQEnabled | else
    if. dl = 0E | call CartridgeSynchronize | mov B$IRQCounter+0 al | else
    if. dl = 0F | call CartridgeSynchronize | mov B$IRQCounter+1 al | endif
    ret

@ReadWRAM:

    on B$WRAMEnabled = &TRUE, ReadWRAM
    sub eax 06000
    add eax D$Bank
    add eax D$pPROM
    movzx eax B$eax
    ret

@WriteWRAM:

    on B$WRAMEnabled = &TRUE, WriteWRAM
    ret
____________________________________________________________________________________________
; Mapper #067, Sunsoft #3

; $8800-$8FFF: CCCCCCCC - swap 2k CROM at $0000
; $9800-$9FFF: CCCCCCCC - swap 2k CROM at $0800
; $A800-$AFFF: CCCCCCCC - swap 2k CROM at $1000
; $B800-$BFFF: CCCCCCCC - swap 2k CROM at $1800
; $C000-$CFFF: IIIIIIII - high/low byte of IRQ counter (first/second write)
; $D800-$DFFF: xxxIxxxx - disable/enable IRQ (0/1)
; $E800-$EFFF: xxxxxxMM - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; $F800-$FFFF: PPPPPPPP - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper067:

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL
    mov D$IRQLatch 0FFFF

    ; Memory mapping
    CPUWrite 08800, 08FFF, @8800_8FFF
    CPUWrite 09800, 09FFF, @9800_9FFF
    CPUWrite 0A800, 0AFFF, @A800_AFFF
    CPUWrite 0B800, 0BFFF, @B800_BFFF
    CPUWrite 0C000, 0CFFF, @C000_CFFF
    CPUWrite 0D800, 0DFFF, @D800_DFFF
    CPUWrite 0E800, 0EFFF, @E800_EFFF
    CPUWrite 0F800, 0FFFF, @F800_FFFF

    ; Last banks
    swap PROM, 16k, 0C000, LAST_BANK
    swap CROM, 4k,  01000, LAST_BANK
    ret

; CCCCCCCC
@8800_8FFF: swap CROM, 2k,   0000, eax | ret
@9800_9FFF: swap CROM, 2k,   0800, eax | ret
@A800_AFFF: swap CROM, 2k,  01000, eax | ret
@B800_BFFF: swap CROM, 2k,  01800, eax | ret

@E800_EFFF:

    ; xxxxxxMM
    and eax 03
    if eax = 0, mirror VERTICAL
    if eax = 1, mirror HORIZONTAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2400
    ret

; PPPPPPPP
@F800_FFFF: swap PROM, 16k, 08000, eax | ret

@C000_CFFF:

    call CartridgeSynchronize

    ; Set up mask
    mov edx 0FF00
    xor B$Register &TRUE | jz F0>

    ; 1st write - high byte
    xchg al ah
    xchg dl dh

    ; IIIIIIII
F0: and D$IRQCounter edx
    or  D$IRQCounter eax
    ret

@D800_DFFF:

    call CartridgeSynchronize

    ; xxxIxxxx
    ClearIRQ IRQ_CARTRIDGE
    test al 010 | setnz B$IRQEnabled
    mov B$Register 0
    ret
____________________________________________________________________________________________
; Mapper #073, Konami VRC3

; $8000-$8FFF: xxxxIIII - bits 3210 of IRQ counter
; $9000-$9FFF: xxxxIIII - bits 7654 of IRQ counter
; $A000-$AFFF: xxxxIIII - bits BA98 of IRQ counter
; $B000-$BFFF: xxxxIIII - bits FEDC of IRQ counter
; $C000-$CFFF: xxxxxxIx - disable/enable IRQ (0/1)
; $F000-$FFFF: PPPPPPPP - swap 16k PROM at $8000
____________________________________________________________________________________________

Mapper073:

    setIRQCounter IRQCount, @IRQ
    mov D$IRQLatch 0

    ; Memory mapping
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 09000, 09FFF, @9000_9FFF
    CPUWrite 0A000, 0AFFF, @A000_AFFF
    CPUWrite 0B000, 0BFFF, @B000_BFFF
    CPUWrite 0C000, 0CFFF, @C000_CFFF
    CPUWrite 0D000, 0DFFF, @D000_DFFF
    CPUWrite 0F000, 0FFFF, @F000_FFFF

    ; Load last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; ????????
@D000_DFFF: ret

; PPPPPPPP
@F000_FFFF: swap PROM, 16k, 08000, eax | ret

; xxxxIIII
@8000_8FFF: and eax 0F |            | and D$IRQCounter 0FFF0 | or D$IRQCounter eax | ret
@9000_9FFF: and eax 0F | shl eax 4  | and D$IRQCounter 0FF0F | or D$IRQCounter eax | ret
@A000_AFFF: and eax 0F | shl eax 8  | and D$IRQCounter 0F0FF | or D$IRQCounter eax | ret
@B000_BFFF: and eax 0F | shl eax 12 | and D$IRQCounter  0FFF | or D$IRQCounter eax | ret

@C000_CFFF:

    call CartridgeSynchronize

    ; xxxxxxIx
    ClearIRQ IRQ_CARTRIDGE
    test al 02 | setnz B$IRQEnabled
    ret

@IRQ:

    ; Increase IRQ counter
    add D$IRQCounter eax
    on D$IRQCounter <= 0FFFF, Q0>

    ; Trigger IRQ
    mov D$IRQEnabled &FALSE
    copy D$IRQLatch D$IRQCounter
    SetIRQ Cartridge, IRQ_CARTRIDGE
Q0: ret

; ____________________________________________________________________________________________
;
;                             PPU read line counters
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #117, San Guo Zhi 4 - Chi Bi Feng Yun (Sangokushi 4)

; $8000: PPPPPPPP - swap 8k PROM at $8000
; $8001: PPPPPPPP - swap 8k PROM at $A000
; $8002: PPPPPPPP - swap 8k PROM at $C000
; $8003: PPPPPPPP - swap 8k PROM at $E000
; $A000: CCCCCCCC - swap 1k CROM at $0000
; $A001: CCCCCCCC - swap 1k CROM at $0400
; $A002: CCCCCCCC - swap 1k CROM at $0800
; $A003: CCCCCCCC - swap 1k CROM at $0C00
; $A004: CCCCCCCC - swap 1k CROM at $1000
; $A005: CCCCCCCC - swap 1k CROM at $1400
; $A006: CCCCCCCC - swap 1k CROM at $1800
; $A007: CCCCCCCC - swap 1k CROM at $1C00
; $C001: IIIIIIII - IRQ counter latch
; $C002: -------- - Clear IRQ line
; $C003: -------- - Reset IRQ
; $E000: -------I - disable/enable IRQ (0/1)
____________________________________________________________________________________________

Mapper117:

    setIRQCounter IRQHookPPU, @IRQ

    ; Memory mapping
    CPUWrite 08000, 08000, @8000
    CPUWrite 08001, 08001, @8001
    CPUWrite 08002, 08002, @8002
    CPUWrite 08003, 08003, @8003
    CPUWrite 0A000, 0A000, @A000
    CPUWrite 0A001, 0A001, @A001
    CPUWrite 0A002, 0A002, @A002
    CPUWrite 0A003, 0A003, @A003
    CPUWrite 0A004, 0A004, @A004
    CPUWrite 0A005, 0A005, @A005
    CPUWrite 0A006, 0A006, @A006
    CPUWrite 0A007, 0A007, @A007
    CPUWrite 0C001, 0C001, @C001
    CPUWrite 0C002, 0C002, @C002
    CPUWrite 0C003, 0C003, @C003
    CPUWrite 0E000, 0E000, @E000

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@8000: swap PROM, 8k, 08000, eax | ret
@8001: swap PROM, 8k, 0A000, eax | ret
@8002: swap PROM, 8k, 0C000, eax | ret
@8003: swap PROM, 8k, 0E000, eax | ret
; CCCCCCCC
@A000: swap CROM, 1k,  0000, eax | ret
@A001: swap CROM, 1k,  0400, eax | ret
@A002: swap CROM, 1k,  0800, eax | ret
@A003: swap CROM, 1k,  0C00, eax | ret
@A004: swap CROM, 1k, 01000, eax | ret
@A005: swap CROM, 1k, 01400, eax | ret
@A006: swap CROM, 1k, 01800, eax | ret
@A007: swap CROM, 1k, 01C00, eax | ret


@C001:

    call CartridgeSynchronize

    ; IIIIIIII
    mov D$IRQLatch eax
    ret

@C002:

    call CartridgeSynchronize

    ; --------
    ClearIRQ IRQ_CARTRIDGE
    ret

@C003:

    call CartridgeSynchronize

    ; --------
    copy D$IRQLatch D$IRQCounter
    mov B$Register 01
    cmp D$Register 0101 | sete B$IRQEnabled
    ret

@E000:

    call CartridgeSynchronize

    ; -------I
    ClearIRQ IRQ_CARTRIDGE
    test al 01 | setnz B$Register+1
    cmp D$Register 0101 | sete B$IRQEnabled
    ret

@IRQ:

    mov B$Register 00
    mov B$IRQEnabled &FALSE
    copy D$IRQLatch D$IRQCounter
    SetIRQ PPU, IRQ_CARTRIDGE
    ret

____________________________________________________________________________________________
; Mapper #033, Taito TC0190

; $8000: xxPPPPPP - swap 8k PROM at $8000
; $8001: xxPPPPPP - swap 8k PROM at $A000
; $8002: CCCCCCCC - swap 2k CROM at $0000
; $8003: CCCCCCCC - swap 2k CROM at $0800
; $A000: CCCCCCCC - swap 1k CROM at $1000
; $A001: CCCCCCCC - swap 1k CROM at $1400
; $A002: CCCCCCCC - swap 1k CROM at $1800
; $A003: CCCCCCCC - swap 1k CROM at $1C00
; $C000: IIIIIIII - IRQ latch
; $C001: xxxxxxxx - Update IRQ counter
; $C002: xxxxxxxx - Disable IRQ
; $C003: xxxxxxxx - ENable IRQ
; $E000: xMxxxxxx - mirror VERTICAL/HORIZONTAL (0/1) (some games use $8000 for this)
____________________________________________________________________________________________

Mapper033:

    setIRQCounter IRQHookPPU, &NULL

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        mov eax WriteVoid
        and ecx 0F003
        if cx = 08000, mov eax @8000
        if cx = 08001, mov eax @8001
        if cx = 08002, mov eax @8002
        if cx = 08003, mov eax @8003
        if cx = 0A000, mov eax @A000
        if cx = 0A001, mov eax @A001
        if cx = 0A002, mov eax @A002
        if cx = 0A003, mov eax @A003
        if cx = 0C000, mov eax @C000
        if cx = 0C001, mov eax @C001
        if cx = 0C002, mov eax @C002
        if cx = 0C003, mov eax @C003
        if cx = 0E000, mov eax @E000

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000:

    ; xxPPPPPP
    swap PROM, 8k, 08000, eax
    if B$Register = &TRUE, ret
    ifNotFlag. eax 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

; xxPPPPPP
@8001: swap PROM, 8k, 0A000, eax | ret

; CCCCCCCC
@8002: swap CROM, 2k,  0000, eax | ret
@8003: swap CROM, 2k,  0800, eax | ret
@A000: swap CROM, 1k, 01000, eax | ret
@A001: swap CROM, 1k, 01400, eax | ret
@A002: swap CROM, 1k, 01800, eax | ret
@A003: swap CROM, 1k, 01C00, eax | ret

; IIIIIIII
@C000: neg al | jmp MMC3@IRQLatch
; xxxxxxxx
@C001: call MMC3@ResetIRQ | mov D$Divide42 35 | ret
@C002: jmp MMC3@EnableIRQ
@C003: jmp MMC3@DisableIRQ

@E000:

    ; xMxxxxxx
    mov B$Register &TRUE
    ifNotFlag. eax 040
        mirror VERTICAL
    else
        mirror HORIZONTAL
    endif
    ret

____________________________________________________________________________________________
; Mapper #091, PC-HK-SF3

; $6000: CCCCCCCC - swap 2k CROM at $0000
; $6001: CCCCCCCC - swap 2k CROM at $0800
; $6002: CCCCCCCC - swap 2k CROM at $1000
; $6003: CCCCCCCC - swap 2k CROM at $1800

; $7000: PPPPPPPP - swap 8k PROM at $8000
; $7001: PPPPPPPP - swap 8k CROM at $A000
; $7002: IIIIIIII - IRQ counter
; $7003: IIIIIIII - Disable/Enable IRQ (0/1)
____________________________________________________________________________________________

Mapper091:

    setIRQCounter IRQHookPPU, &NULL

    ; Memory mapping
    mov ecx 06000
L0: push ecx

        mov eax WriteVoid
        and ecx 0F003

        if ecx = 06000, mov eax @6000
        if ecx = 06001, mov eax @6001
        if ecx = 06002, mov eax @6002
        if ecx = 06003, mov eax @6003
        if ecx = 07000, mov eax @7000
        if ecx = 07001, mov eax @7001
        if ecx = 07002, mov eax @7002
        if ecx = 07003, mov eax @7003

    pop ecx
    CPUWrite ecx, ecx, eax
    inc ecx | on ecx < 08000, L0<<

    ; Load last bank
    swap PROM, 16k, 08000, LAST_BANK
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; CCCCCCCC
@6000: swap CROM, 2k,  0000, eax | ret
@6001: swap CROM, 2k,  0800, eax | ret
@6002: swap CROM, 2k, 01000, eax | ret
@6003: swap CROM, 2k, 01800, eax | ret

; PPPPPPPP
@7000: swap PROM, 8k, 08000, eax | ret
@7001: swap PROM, 8k, 0A000, eax | ret

; IIIIIIII
@7002: mov D$IRQEnabled &FALSE | mov D$IRQCounter 7 | ClearIRQ IRQ_CARTRIDGE | ret
@7003: mov B$IRQEnabled &TRUE  | ret

; ____________________________________________________________________________________________
;
;                               Scanline counters
;
; ____________________________________________________________________________________________
____________________________________________________________________________________________
; Mappers #024 and #026, VRC6

; (mask $F003)
; $8000: PPPPPPPP - swap 16k PROM at $8000
; $A000: \
; $A001:  \
; $A002:   \ - - > VRC6 sound channels
; $B000:   /
; $B001:  /
; $B002: /
; $B003: xxxxMMxx - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; $C000: PPPPPPPP - swap 9k  PROM at $C000
; $D000: CCCCCCCC - swap 1k CROM at $0000
; $D001: CCCCCCCC - swap 1k CROM at $0400
; $D002: CCCCCCCC - swap 1k CROM at $0800
; $D003: CCCCCCCC - swap 1k CROM at $0C00
; $E000: CCCCCCCC - swap 1k CROM at $1000
; $E001: CCCCCCCC - swap 1k CROM at $1400
; $E002: CCCCCCCC - swap 1k CROM at $1800
; $E003: CCCCCCCC - swap 1k CROM at $1C00
; $F000: IIIIIIII - IRQ counter latch
; $F001: xxxxxxIi - disable/enable IRQ (I = 0/1)
; $F002: xxxxxxxx - disable/enable IRQ (i = 0/1)
____________________________________________________________________________________________

Mapper024:
Mapper026:

    call VRC6
    setIRQCounter IRQCountdown, @IRQ

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        ; Mapper #026 has A0 and A1 reversed
        if. D$Cartridge@Mapper = 26
            mov eax ecx   | and eax 01 | shl eax 1
            mov edx ecx   | and edx 02 | shr edx 1
            and ecx 0FFFC | or ecx eax | or ecx edx
        endif

        and ecx 0F003
        mov eax WriteVoid

        if ecx = 08000, mov eax @8000
        if ecx = 09000, mov eax VRC6@9000
        if ecx = 09001, mov eax VRC6@9001
        if ecx = 09002, mov eax VRC6@9002
        if ecx = 0A000, mov eax VRC6@A000
        if ecx = 0A001, mov eax VRC6@A001
        if ecx = 0A002, mov eax VRC6@A002
        if ecx = 0B000, mov eax VRC6@B000
        if ecx = 0B001, mov eax VRC6@B001
        if ecx = 0B002, mov eax VRC6@B002
        if ecx = 0B003, mov eax @B003
        if ecx = 0C000, mov eax @C000
        if ecx = 0D000, mov eax @D000
        if ecx = 0D001, mov eax @D001
        if ecx = 0D002, mov eax @D002
        if ecx = 0D003, mov eax @D003
        if ecx = 0E000, mov eax @E000
        if ecx = 0E001, mov eax @E001
        if ecx = 0E002, mov eax @E002
        if ecx = 0E003, mov eax @E003
        if ecx = 0F000, mov eax @F000
        if ecx = 0F001, mov eax @F001
        if ecx = 0F002, mov eax @F002

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; PPPPPPPP
@8000: swap PROM, 16k, 08000, eax | ret
@C000: swap PROM, 8k,  0C000, eax | ret

; CCCCCCCC
@D000: swap CROM, 1k,  0000, eax | ret
@D001: swap CROM, 1k,  0400, eax | ret
@D002: swap CROM, 1k,  0800, eax | ret
@D003: swap CROM, 1k,  0C00, eax | ret
@E000: swap CROM, 1k, 01000, eax | ret
@E001: swap CROM, 1k, 01400, eax | ret
@E002: swap CROM, 1k, 01800, eax | ret
@E003: swap CROM, 1k, 01C00, eax | ret

@B003:

    ; xxxxMMxx
    shr al 2 | and al 03
    if al = 0, mirror VERTICAL
    if al = 1, mirror HORIZONTAL
    if al = 2, mirror ONE_SCREEN_2000
    if al = 3, mirror ONE_SCREEN_2400
    ret

@F000:

    call CartridgeSynchronize

    ; IIIIIIII
    not al | inc eax
    mul D$Mul113 | div D$Div113
    mov D$IRQLatch eax
    ret

@F001:

    call CartridgeSynchronize

    ; xxxxxxIi
    ClearIRQ IRQ_CARTRIDGE
    test al 02 | setnz B$IRQEnabled
    test al 01 | setnz B$Register
    copy D$IRQLatch D$IRQCounter
    ret

@F002:

    call CartridgeSynchronize

    ; xxxxxxxx
    ClearIRQ IRQ_CARTRIDGE
    copy D$Register D$IRQEnabled
    ret

@IRQ:

;    mov B$IRQEnabled &FALSE
    mov eax D$IRQLatch
    add D$IRQCounter eax
    SetIRQ Cartridge, IRQ_CARTRIDGE
    ret
____________________________________________________________________________________________
; Mapper #085, VRC7
____________________________________________________________________________________________

Mapper085:

    ; IRQ counter
    setIRQCounter IRQCountdown, &NULL

    ; Memory mapping
    mov ecx 08000
L0: push ecx

        mov eax WriteVoid
        and ecx 0F038
        if cx = 08000, mov eax @8000
        if cx = 08008, mov eax @8010 | if cx = 08010, mov eax @8010
        if cx = 09000, mov eax @9000
    if cx = 09010, mov eax @9010
    if cx = 09030, mov eax @9030
        if cx = 0A000, mov eax @A000
        if cx = 0A008, mov eax @A010 | if cx = 0A010, mov eax @A010
        if cx = 0B000, mov eax @B000
        if cx = 0B008, mov eax @B010 | if cx = 0B010, mov eax @B010
        if cx = 0C000, mov eax @C000
        if cx = 0C008, mov eax @C010 | if cx = 0C010, mov eax @C010
        if cx = 0D000, mov eax @D000
        if cx = 0D008, mov eax @D010 | if cx = 0D010, mov eax @D010
        if cx = 0E000, mov eax @E000
        if cx = 0E008, mov eax @E010 | if cx = 0E010, mov eax @E010
        if cx = 0F000, mov eax @F000
        if cx = 0F008, mov eax @F010 | if cx = 0F010, mov eax @F010

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Last bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

; ????????
@9010:
@9030:ret

; PPPPPPPP
@8000: swap PROM, 8k, 08000, eax | ret
@8010: swap PROM, 8k, 0A000, eax | ret
@9000: swap PROM, 8k, 0C000, eax | ret

; CCCCCCCC
@A000: swap CROM, 1k,  0000, eax | ret
@A010: swap CROM, 1k,  0400, eax | ret
@B000: swap CROM, 1k,  0800, eax | ret
@B010: swap CROM, 1k,  0C00, eax | ret
@C000: swap CROM, 1k, 01000, eax | ret
@C010: swap CROM, 1k, 01400, eax | ret
@D000: swap CROM, 1k, 01800, eax | ret
@D010: swap CROM, 1k, 01C00, eax | ret

@E000:

    ; xxxxxxMM
    and al 03
    if al = 0, mirror VERTICAL
    if al = 1, mirror HORIZONTAL
    if al = 2, mirror ONE_SCREEN_2000
    if al = 3, mirror ONE_SCREEN_2400
    ret

@E010:

    call CartridgeSynchronize

    ; IIIIIIII
    not al | inc eax
    mul D$Mul113 | div D$Div113
    mov D$IRQLatch eax
    ret

@F000:

    call CartridgeSynchronize

    ; xxxxxxEE
    ClearIRQ IRQ_CARTRIDGE
    and eax 03 | mov D$Register eax
    test eax 02 | setnz B$IRQEnabled
    if B$IRQEnabled = &TRUE, copy D$IRQLatch D$IRQCounter
    ret

@F010:

    call CartridgeSynchronize

    ; xxxxxxxx
    ClearIRQ IRQ_CARTRIDGE
    copy D$Register D$IRQEnabled
    ret

____________________________________________________________________________________________
; Mapper #025, Konami VRC4

; --> Mapper #021, with a different mask and slightly different PROM swapping
____________________________________________________________________________________________

Mapper025:

    ; Almost mapper #021
    call Mapper021

    ; Memory mapping
    CPUWrite 08000, 08FFF, @8000_8FFF
    CPUWrite 0A000, 0AFFF, Mapper021@A000

    mov edx 0F00F
    mov ecx 08000
L0: push ecx

        mov eax &NULL
        and ecx 0F00F
        if cx = 09000, mov eax Mapper021@9000
        if cx = 0B000, mov eax Mapper021@B000
        if cx = 0C000, mov eax Mapper021@C000
        if cx = 0D000, mov eax Mapper021@D000
        if cx = 0E000, mov eax Mapper021@E000
        if cx = 0F000, mov eax Mapper021@F000

        if cx = 09001, mov eax Mapper025@9004 | if cx = 09004, mov eax Mapper025@9004
        if cx = 0B001, mov eax Mapper021@B004 | if cx = 0B004, mov eax Mapper021@B004
        if cx = 0B002, mov eax Mapper021@B002 | if cx = 0B008, mov eax Mapper021@B002
        if cx = 0B003, mov eax Mapper021@B006 | if cx = 0B00C, mov eax Mapper021@B006
        if cx = 0C001, mov eax Mapper021@C004 | if cx = 0C004, mov eax Mapper021@C004
        if cx = 0C002, mov eax Mapper021@C002 | if cx = 0C008, mov eax Mapper021@C002
        if cx = 0C003, mov eax Mapper021@C006 | if cx = 0C00C, mov eax Mapper021@C006
        if cx = 0D001, mov eax Mapper021@D004 | if cx = 0D004, mov eax Mapper021@D004
        if cx = 0D002, mov eax Mapper021@D002 | if cx = 0D008, mov eax Mapper021@D002
        if cx = 0D003, mov eax Mapper021@D006 | if cx = 0D00C, mov eax Mapper021@D006
        if cx = 0E001, mov eax Mapper021@E004 | if cx = 0E004, mov eax Mapper021@E004
        if cx = 0E002, mov eax Mapper021@E002 | if cx = 0E008, mov eax Mapper021@E002
        if cx = 0E003, mov eax Mapper021@E006 | if cx = 0E00C, mov eax Mapper021@E006
        if cx = 0F001, mov eax Mapper021@F004 | if cx = 0F004, mov eax Mapper021@F004
        if cx = 0F002, mov eax Mapper021@F002 | if cx = 0F008, mov eax Mapper021@F002
        if cx = 0F003, mov eax Mapper021@F006 | if cx = 0F00C, mov eax Mapper021@F006

    pop ecx
    if eax != &NULL, CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    mov D$Bank+020 0
    mov D$Bank+024 LAST_BANK-1
    ret

@8000_8FFF:

    if. B$Register+04 = &FALSE
        mov D$Bank+020 eax
        swap PROM, 8k, 08000, eax
    else
        mov D$Bank+024 eax
        swap PROM, 8k, 0C000, eax
    endif
    ret

@9004:

    test eax 02 | setnz al
    xchg al B$Register+04
    if. al != B$Register+04
        push D$Bank+020 | push D$Bank+024
        pop  D$Bank+020 | pop  D$Bank+024
        swap PROM, 8k, 08000, D$Bank+020
        swap PROM, 8k, 0C000, D$Bank+024
    endif
    ret
____________________________________________________________________________________________
; Mapper #023, Konami VRC2 type B

; --> Mapper #021, with a different mask
____________________________________________________________________________________________

Mapper023:

    ; Almost mapper #021
    call Mapper021

    ; Address mask
    mov edx 0FFFF
    if D$Cartridge@PROMCRC32 = 09379_4634, mov edx 0F00C ; Akumajou Special - Boku Dracula Kun
    if D$Cartridge@PROMCRC32 = 0C782_9DAE, mov edx 0F00C ; Akumajou Special - Boku Dracula Kun (t.eng)
    if D$Cartridge@PROMCRC32 = 06A50_B553, mov edx 0F00C ; Akumajou Special - Boku Dracula Kun (b)
    if D$Cartridge@PROMCRC32 = 08C62_37FD, mov edx 0F00C ; Kaiketsu Yanchamaru 2 - Karakuri Land

    if D$Cartridge@PROMCRC32 = 0E4CE_EAD1, mov edx 0F00C ; Parodius

    ; Set ports
    mov ecx 08000
L0: push ecx

        mov eax &NULL
        and ecx edx

        if cx = 08000, mov eax Mapper021@8000 | if cx = 08004, mov eax Mapper021@8000
        if cx = 08008, mov eax Mapper021@8000 | if cx = 0800C, mov eax Mapper021@8000
;        if cx = 09008, mov eax Mapper021@9002
        if cx = 0A000, mov eax Mapper021@A000 | if cx = 0A004, mov eax Mapper021@A000
        if cx = 0A008, mov eax Mapper021@A000 | if cx = 0A00C, mov eax Mapper021@A000

        if cx = 0B001, mov eax Mapper021@B002 | if cx = 0B004, mov eax Mapper021@B002
        if cx = 0B002, mov eax Mapper021@B004 | if cx = 0B008, mov eax Mapper021@B004
        if cx = 0B003, mov eax Mapper021@B006 | if cx = 0B00C, mov eax Mapper021@B006
        if cx = 0C001, mov eax Mapper021@C002 | if cx = 0C004, mov eax Mapper021@C002
        if cx = 0C002, mov eax Mapper021@C004 | if cx = 0C008, mov eax Mapper021@C004
        if cx = 0C003, mov eax Mapper021@C006 | if cx = 0C00C, mov eax Mapper021@C006
        if cx = 0D001, mov eax Mapper021@D002 | if cx = 0D004, mov eax Mapper021@D002
        if cx = 0D002, mov eax Mapper021@D004 | if cx = 0D008, mov eax Mapper021@D004
        if cx = 0D003, mov eax Mapper021@D006 | if cx = 0D00C, mov eax Mapper021@D006
        if cx = 0E001, mov eax Mapper021@E002 | if cx = 0E004, mov eax Mapper021@E002
        if cx = 0E002, mov eax Mapper021@E004 | if cx = 0E008, mov eax Mapper021@E004
        if cx = 0E003, mov eax Mapper021@E006 | if cx = 0E00C, mov eax Mapper021@E006

        if cx = 09000, mov eax Mapper021@9000
        if cx = 09008, mov eax Mapper021@9002
        if cx = 0B000, mov eax Mapper021@B000
        if cx = 0C000, mov eax Mapper021@C000
        if cx = 0D000, mov eax Mapper021@D000
        if cx = 0E000, mov eax Mapper021@E000
        if cx = 0F000, mov eax Mapper021@F000
        if cx = 0F004, mov eax Mapper021@F002
        if cx = 0F008, mov eax Mapper021@F004
        if cx = 0F00C, mov eax Mapper021@F006

    pop ecx
    if eax != &NULL, CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<
    ret
____________________________________________________________________________________________
; Mapper #021, Konami VRC4 2B

; $8000:               PPPPPPPP - swap 8k PROM at $8000 or $C000
; $9000:               xxxxxxMM - mirror VERTICAL/HORIZONTAL/$2000/$2400 (0/1/2/3)
; $9002: $9080:        xxxxxxSx - ensable swapping for $8000/$C000 (0/1)
; $A000:               PPPPPPPP - swap 8k PROM at $C000
;        $B000:        xxxxCCCC (xxxx3210) - swap 1k CROM at $0000
;        $B002: $B040: xxxxCCCC (xxxx7654) - swap 1k CROM at $0000
; $B001: $B004: $B080: xxxxCCCC (xxxx3210) - swap 1k CROM at $0400
; $B003: $B006: $B0C0: xxxxCCCC (xxxx7654) - swap 1k CROM at $0400
;        $C000:        xxxxCCCC (xxxx3210) - swap 1k CROM at $0800
;        $C002: $C040: xxxxCCCC (xxxx7654) - swap 1k CROM at $0800
; $C001: $C004: $C080: xxxxCCCC (xxxx3210) - swap 1k CROM at $0C00
; $C003: $C006: $C0C0: xxxxCCCC (xxxx7654) - swap 1k CROM at $0C00
;        $D000:        xxxxCCCC (xxxx3210) - swap 1k CROM at $1000
;        $D002: $D040: xxxxCCCC (xxxx7654) - swap 1k CROM at $1000
; $D001: $D004: $D080: xxxxCCCC (xxxx3210) - swap 1k CROM at $1400
; $D003: $D006: $D0C0: xxxxCCCC (xxxx7654) - swap 1k CROM at $1400
;        $E000:        xxxxCCCC (xxxx3210) - swap 1k CROM at $1800
;        $E002: $E040: xxxxCCCC (xxxx7654) - swap 1k CROM at $1800
; $E001: $E004: $E080: xxxxCCCC (xxxx3210) - swap 1k CROM at $1C00
; $E003: $E006: $E0C0: xxxxCCCC (xxxx7654) - swap 1k CROM at $1C00

;        $F000:        xxxxIIII (xxxx3210) - IRQ counter latch
;        $F002: $F040: xxxxIIII (xxxx7654) - IRQ counter latch
; $F001: $F004: $F080: xxxxxxIi - i = IRQ enable latch
;                               - I = IRQ enable
; $F003: $F006: $F0C0: xxxxxxxx - disable/enable IRQ depending on IRQ enable latch
____________________________________________________________________________________________

Mapper021:

    setIRQCounter IRQCount, @IRQ

    ; Set ports
    mov ecx 08000
L0: push ecx

        and ecx 0F0CF
        mov eax WriteVoid

        if cx = 08000, mov eax @8000
        if cx = 09000, mov eax @9000
        if cx = 09002, mov eax @9002 | if ecx = 09080, mov eax @9002
        if cx = 0A000, mov eax @A000

        if cl = 040, mov cl 02
        if cl = 01,  mov cl 04
        if cl = 080, mov cl 04
        if cl = 03,  mov cl 06
        if cl = 0C0, mov cl 06

        if cx = 0B000, mov eax @B000
        if cx = 0B002, mov eax @B002
        if cx = 0B004, mov eax @B004
        if cx = 0B006, mov eax @B006
        if cx = 0C000, mov eax @C000
        if cx = 0C002, mov eax @C002
        if cx = 0C004, mov eax @C004
        if cx = 0C006, mov eax @C006
        if cx = 0D000, mov eax @D000
        if cx = 0D002, mov eax @D002
        if cx = 0D004, mov eax @D004
        if cx = 0D006, mov eax @D006
        if cx = 0E000, mov eax @E000
        if cx = 0E002, mov eax @E002
        if cx = 0E004, mov eax @E004
        if cx = 0E006, mov eax @E006

        if cx = 0F000, mov eax @F000
        if cx = 0F002, mov eax @F002
        if cx = 0F004, mov eax @F004
        if cx = 0F006, mov eax @F006

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Load last PROM bank
    swap PROM, 16k, 0C000, LAST_BANK
    ret

@8000:

    ; PPPPPPPP
    if. B$Register+04 = &FALSE
        swap PROM, 8k, 08000, eax
    else
        swap PROM, 8k, 0C000, eax
    endif
    ret

; PPPPPPPP
@A000: swap PROM, 8k, 0A000, eax | ret

@9000:

    ; xxxxxxMM
    and eax 03
    if eax = 0, mirror VERTICAL
    if eax = 1, mirror HORIZONTAL
    if eax = 2, mirror ONE_SCREEN_2000
    if eax = 3, mirror ONE_SCREEN_2400
    ret

@9002:

    ; xxxxxxSx
    test eax 02 | setnz B$Register+04
    ret

; xxxxCCCC
@B000: and eax 0F |           | and D$Bank+00  0F0 | or D$Bank+00  eax | swap CROM, 1k,  0000, D$Bank+00  | ret
@B002: and eax 0F | shl eax 4 | and D$Bank+00  0F  | or D$Bank+00  eax | swap CROM, 1k,  0000, D$Bank+00  | ret
@B004: and eax 0F |           | and D$Bank+04  0F0 | or D$Bank+04  eax | swap CROM, 1k,  0400, D$Bank+04  | ret
@B006: and eax 0F | shl eax 4 | and D$Bank+04  0F  | or D$Bank+04  eax | swap CROM, 1k,  0400, D$Bank+04  | ret
@C000: and eax 0F |           | and D$Bank+08  0F0 | or D$Bank+08  eax | swap CROM, 1k,  0800, D$Bank+08  | ret
@C002: and eax 0F | shl eax 4 | and D$Bank+08  0F  | or D$Bank+08  eax | swap CROM, 1k,  0800, D$Bank+08  | ret
@C004: and eax 0F |           | and D$Bank+0C  0F0 | or D$Bank+0C  eax | swap CROM, 1k,  0C00, D$Bank+0C  | ret
@C006: and eax 0F | shl eax 4 | and D$Bank+0C  0F  | or D$Bank+0C  eax | swap CROM, 1k,  0C00, D$Bank+0C  | ret
@D000: and eax 0F |           | and D$Bank+010 0F0 | or D$Bank+010 eax | swap CROM, 1k, 01000, D$Bank+010 | ret
@D002: and eax 0F | shl eax 4 | and D$Bank+010 0F  | or D$Bank+010 eax | swap CROM, 1k, 01000, D$Bank+010 | ret
@D004: and eax 0F |           | and D$Bank+014 0F0 | or D$Bank+014 eax | swap CROM, 1k, 01400, D$Bank+014 | ret
@D006: and eax 0F | shl eax 4 | and D$Bank+014 0F  | or D$Bank+014 eax | swap CROM, 1k, 01400, D$Bank+014 | ret
@E000: and eax 0F |           | and D$Bank+018 0F0 | or D$Bank+018 eax | swap CROM, 1k, 01800, D$Bank+018 | ret
@E002: and eax 0F | shl eax 4 | and D$Bank+018 0F  | or D$Bank+018 eax | swap CROM, 1k, 01800, D$Bank+018 | ret
@E004: and eax 0F |           | and D$Bank+01C 0F0 | or D$Bank+01C eax | swap CROM, 1k, 01C00, D$Bank+01C | ret
@E006: and eax 0F | shl eax 4 | and D$Bank+01C 0F  | or D$Bank+01C eax | swap CROM, 1k, 01C00, D$Bank+01C | ret

; xxxxIIII
@F000: and eax 0F |           | and D$IRQLatch 0F0 | or D$IRQLatch eax | ret
@F002: and eax 0F | shl eax 4 | and D$IRQLatch  0F | or D$IRQLatch eax | ret

@F004:

    call CartridgeSynchronize

    ; xxxxxxIi
    ClearIRQ IRQ_CARTRIDGE
    test eax 01 | setnz B$Register
    test eax 02 | setnz B$IRQEnabled
    if. B$IRQEnabled = &TRUE
        mov eax D$IRQLatch
        not al
        mul D$Mul113 | div D$Div113
        mov D$IRQCounter eax
    endif
    ret

@F006:

    call CartridgeSynchronize

    ; xxxxxxxx
    ClearIRQ IRQ_CARTRIDGE
    copy D$Register D$IRQEnabled
    ret

@IRQ:

    ; Count
    sub D$IRQCounter eax | jns Q0>
    ; IRQ latch
    mov eax D$IRQLatch
    not al
    mul D$Mul113 | div D$Div113
    mov D$IRQCounter eax
    ; IRQ
    SetIRQ Cartridge, IRQ_CARTRIDGE
Q0: ret

; ____________________________________________________________________________________________
;
;                               Other IRQ counters
;
; ____________________________________________________________________________________________

____________________________________________________________________________________________
; Mapper #222, Dragon Ninja

; $8000: PPPPPPPP - swap 8k PROM at $8000
; $A000: PPPPPPPP - swap 8k PROM at $A000
; $B000: CCCCCCCC - swap 1k CROM at $0000
; $B002: CCCCCCCC - swap 1k CROM at $0400
; $C000: CCCCCCCC - swap 1k CROM at $0800
; $C002: CCCCCCCC - swap 1k CROM at $0C00
; $D000: CCCCCCCC - swap 1k CROM at $1000
; $D002: CCCCCCCC - swap 1k CROM at $1400
; $E000: CCCCCCCC - swap 1k CROM at $1800
; $E002: CCCCCCCC - swap 1k CROM at $1C00
; $F00x: Some IRQ thing
____________________________________________________________________________________________

Mapper222:

    setIRQCounter IRQCountdown, &NULL

    ; Set ports
CPUWrite 05000, 05000, @5000
    mov ecx 08000
L0: push ecx

        and ecx 0F003
        mov eax WriteVoid
        if ecx = 08000, mov eax @8000
        if ecx = 09000, mov eax @9000
        if ecx = 09002, mov eax @9002
        if ecx = 0A000, mov eax @A000
        if ecx = 0B000, mov eax @B000
        if ecx = 0B002, mov eax @B002
        if ecx = 0C000, mov eax @C000
        if ecx = 0C002, mov eax @C002
        if ecx = 0D000, mov eax @D000
        if ecx = 0D002, mov eax @D002
        if ecx = 0E000, mov eax @E000
        if ecx = 0E002, mov eax @E002
        if ecx = 0F000, mov eax @F000
        if ecx = 0F002, mov eax @F002

ifFlag ecx 1, mov eax @1

    pop ecx
    CPUWrite ecx, ecx, eax
    inc cx | jnz L0<<

    ; Swap in last bank
    swap PROM, 16k, 0C000, LAST_BANK

    ; Vertical mirroring
    mirror VERTICAL
@1:
    ret

@8000: swap PROM, 8k, 08000, eax | ret
@A000: swap PROM, 8k, 0A000, eax | ret
@B000: swap CROM, 1k,  0000, eax | ret
@B002: swap CROM, 1k,  0400, eax | ret
@C000: swap CROM, 1k,  0800, eax | ret
@C002: swap CROM, 1k,  0C00, eax | ret
@D000: swap CROM, 1k, 01000, eax | ret
@D002: swap CROM, 1k, 01400, eax | ret
@E000: swap CROM, 1k, 01800, eax | ret
@E002: swap CROM, 1k, 01C00, eax | ret

@9002:

@9000:

    ret

@F000:
@F002:

    call CartridgeSynchronize

    ; IIIIIIII
    ClearIRQ IRQ_CARTRIDGE
    not al
    mul D$Mul113 | div D$Div113
    mov D$IRQCounter eax
    ret

@5000:

    call CartridgeSynchronize

    test al al | setnz B$IRQEnabled
    ret

____________________________________________________________________________________________
; Mapper #006, FFE F4xxx
____________________________________________________________________________________________

Mapper006:

    ; 32k of CRAM
    if D$Cartridge@SizeCROM = 0, mov D$Cartridge@SizeCRAM 32

    ; Memory mapping
    CPUWrite 042FE, 042FE, @42FE
    CPUWrite 042FF, 042FF, @42FF
    CPUWrite 04501, 04501, @4501
    CPUWrite 04502, 04502, @4502
    CPUWrite 04503, 04503, @4503
    CPUWrite 08000, 0FFFF, @8000_FFFF

    ; Set PROM bank
    swap PROM, 16k, 0C000, 7
    ret

@42FE:
    ifFlag. eax 010
        mirror ONE_SCREEN_2400
    else
        mirror ONE_SCREEN_2000
    endif
    ret

@42FF:
    ifFlag. eax 010
        mirror HORIZONTAL
    else
        mirror VERTICAL
    endif
    ret

@8000_FFFF:

    mov edx eax | and eax 03 | shr edx 2 | and edx 0F
    swap PROM, 16k, 08000, edx
    swap CROM, 8k,  00000, eax
    ret

@4501:
@4502:
@4503:

    ; Some ROM tries to use the IRQ counter (haven't found one that does yet)
    ret
____________________________________________________________________________________________
