TITLE Cartridge

; Sets up the cartridge hardware.
; Can use either CRC32 and the database resource, or the iNES header.

____________________________________________________________________________________________
; Synchronize the cartridge hardware (IRQ counters)
____________________________________________________________________________________________

[setIRQCounter | #=2
 call InstallIRQCounter #1 #2]
InstallIRQCounter:

    arguments @Type, @Callback
    copy D@Type D$IRQType
    copy D@Callback D$IRQHook

    if. D$IRQType = IRQHookPPU

        ; Monitor read line
        PPURead 00000, 01FFF, ReadA13Low
        PPURead 02000, 02FFF, ReadA13High

        ; Make sure the PPU is synchronized at all times
        mov D$IRQType PPUSynchronize

    else

        if D$IRQHook = &NULL, mov D$IRQHook TriggerIRQ

    endif
    return
____________________________________________________________________________________________

IRQHookPPU:ret

ReadA13Low:

    ; Memorize A13, read
    mov D$A13 0
    jmp ReadCROM

ReadA13High:

    ; A13?
    on D$A13 = 1, ReadCiRAM
    ; Divide-by-42
    dec B$Divide42 | jnz S0>
    mov B$Divide42 42

    ; IRQ counter
mov D$ResetIRQ 0
    on B$IRQEnabled = &FALSE, S0>
    dec D$IRQCounter | jns S0>

    ; IRQ!
    if. D$IRQHook = &NULL
mov B$IRQEnabled 0
        SetIRQ PPU, IRQ_CARTRIDGE
        copy D$IRQLatch D$IRQCounter
    else
        call D$IRQHook
    endif

    ; Memorize A13, read
S0: mov D$A13 1
    jmp ReadCiRAM
____________________________________________________________________________________________
[Mul113: 113+113+113+2
 Div113: 3]
IRQCountdown:

    ; Count down
    on B$IRQEnabled = &FALSE, Q0>
    sub D$IRQCounter eax | jns Q0>

    ; Adjust clock, call IRQ hook
    mov eax D$IRQCounter
    imul D$CPUCycles
    push D$CartridgeClock
        add D$CartridgeClock eax
        call D$IRQHook
    pop D$CartridgeClock
Q0: ret
____________________________________________________________________________________________

IRQCount:

    if B$IRQEnabled = &TRUE, jmp D$IRQHook
    ret
____________________________________________________________________________________________

TriggerIRQ:

    mov B$IRQEnabled &FALSE
    copy D$IRQLatch D$IRQCounter
    SetIRQ Cartridge, IRQ_CARTRIDGE
    ret
____________________________________________________________________________________________

CartridgeSynchronize:

    pushad

        ; Behind?
        mov eax D$CPUClock | on eax <= D$CartridgeClock, Q0>>

        sub eax D$CartridgeClock
        mov edx 0 | div D$CPUCycles
        test eax eax | jz Q0>>
        copy D$CPUClock D$CartridgeClock

        ; DMC DMA?
        if. D$DmcIRQCounter > 0
            push eax
                mov eax D$DmcIRQCounter, edx 0
                div D$DMC+ConvertedTimerLatch
                if edx = 0, cpuTick 2
            pop eax
        endif

        ; DNC IRQ?
        if. D$DmcIRQCounter > 0
            sub D$DmcIRQCounter eax | js F0> | jz F0> | break | F0:
            ; Stop
            mov D$DmcIRQCounter 0
            ; IRQ?
            if D$DMC+DMCEnableIRQ = &TRUE, SetIRQ Cartridge, IRQ_DMC
        endif

        ; Frame IRQ?
        if. D$FrameIRQCounter > 0
            sub D$FrameIRQCounter eax | js F0> | jz F0> | break | F0:
            mov D$FrameIRQCounter 29830
            SetIRQ Cartridge, IRQ_FRAME
        endif

        ; IRQ counter
        if D$IRQType != &NULL, call D$IRQType

Q0: popad
    ret

____________________________________________________________________________________________
; Memory mapping
____________________________________________________________________________________________

[CPURead  | call MapAddress CPUReadTable,  #1, #2, #3]
[CPUWrite | call MapAddress CPUWriteTable, #1, #2, #3]
[PPURead  | call MapAddress PPUReadTable,  #1, #2, #3]
[PPUWrite | call MapAddress PPUWriteTable, #1, #2, #3]

MapAddress:

    arguments @pTable, @Start, @End, @Label

    pushad

        mov edi D$@Start | shl edi 2 | add edi D$@pTable
        mov ecx D$@End | sub ecx D$@Start | inc ecx
        mov eax D$@Label
        rep stosd

    popad
    return

____________________________________________________________________________________________
; Reset
____________________________________________________________________________________________

[HSyncHook: ?]
[IRQHook: ?]
[IRQType: ?]

CartridgeReset:

    call ZeroMemory MapperData LENGTH_MAPPER_DATA-LENGTH_HARDRESET_DATA

    ; Got WRAM?
    if D$Cartridge@SizeWRAM = 0, mov D$Cartridge@SizeWRAM 8

    ; Nullify any hooks
    mov D$IRQType &NULL
    mov D$IRQHook &NULL
    mov D$HSyncHook &NULL

    ; Set PPU mirroring
    if D$Cartridge@Mirroring = HORIZONTAL_MIRRORING,  mirror HORIZONTAL
    if D$Cartridge@Mirroring = VERTICAL_MIRRORING,    mirror VERTICAL
    if D$Cartridge@Mirroring = FOUR_SCREEN_MIRRORING, mirror FOUR_SCREEN

    ; Default - swap in first PROM banks
    swap PROM, 16k, 08000, 0
    swap PROM, 16k, 0C000, 1

    ; Default - swap in first CROM bank
    swap CROM, 8k,  00000, 0

    ; Load trainer
    if D$Cartridge@Trainer = &TRUE, call CopyMemory D$pTrainer, WRAM+01000, 0200

    ; VsSystem?
    if B$Cartridge@VsSystem = &TRUE, call ResetVsSystem

    ; Reset mapper
    mov eax D$Cartridge@Mapper
    call D$eax*4+MapperTable
    ret

____________________________________________________________________________________________
; Load a NES ROM
____________________________________________________________________________________________

LoadNES:

    ; Unload previous cartridge
    call UnloadNES
    mov D$GameGenieCodes &INVALID_HANDLE_VALUE
    copy D$pFile D$pImage | mov D$pFile &NULL

    ; File ID?
    mov eax D$pImage
    if. D$eax != 01A53454E
        closeMessage 'Bad file format.'
        ret
    endif

    ; Insert cartridge
    mov B$ROMTitle 0 | call AddString ROMTitle FileTitle
    mov B$ROMDirectory 0
    call AddString ROMDirectory, FilesDirectory
    call AddString ROMDirectory, ROMTitle
    call AddString ROMDirectory, BackSlash
    call CartridgeGetInfo

    ; - Validate -
    ; Mo PROM?
    if. D$Cartridge@SizePROM < 16
        closeMessage 'No program data.'
        ret
    endif
    ; Mapper supported?
    mov eax D$Cartridge@Mapper
    if. D$MapperTable+eax*4 = &NULL
        closeMessage 'Mapper not supported yet.'
        ret
    endif
    mov B$WindowTitle 0
    call AddString WindowTitle D$Cartridge@pGamename
    ; Bad dump?
    if. B$Cartridge@BadDump = &TRUE
        call AddString WindowTitle BadDump
    endif
    ; Wrong size?
    mov eax D$FileSize | sub eax 010 | shr eax 10
    mov edx D$Cartridge@SizePROM | add edx D$Cartridge@SizeCROM
    if. eax != edx
        call AddString WindowTitle WrongSize
    endif
    ; Set window text
    call 'User32.SendMessageA' D$hMainWindow, &WM_SETTEXT, &NULL, WindowTitle

    ; - Preferences -
    ; Set mode (NTSC/PAL)
    if. B$Preferences@PreferNTSC = &TRUE
        call MenuNTSC
    else
        call MenuPAL
    endif
    if B$Cartridge@PAL = &TRUE,
        if B$Cartridge@NTSC = &FALSE,
            call MenuPAL
    if B$Cartridge@NTSC = &TRUE,
        if B$Cartridge@PAL = &FALSE,
            call MenuNTSC
    ; Power ON
    if B$Preferences@AutoPowerOn = &TRUE, call SetPower POWER_ON
    ; Remove menu?
    if. B$Preferences@AutoHideMenu = &TRUE
        call 'User32.SetMenu' D$hMainWindow, &NULL | winErrorCheck 'Could not set menu.'
    endif


    ; Palette
    [PALExtension: B$ '*.pal', 0]
    push D$pFile
        copy D$pPalette D$pFile | mov D$FileSize 0C0 | call LoadPAL
    pop D$pFile
    call LoadFirstFile NessieDirectory, PALExtension
    call LoadFirstFile ROMDirectory, PALExtension

    ; IPS patch
    [IPSExtension: B$ '*.ips', 0]
    call LoadFirstFile ROMDirectory, IPSExtension

    ; Configuration
    call LoadConfiguration

    ; Zapper?
    mov eax D$Cartridge@PROMCRC32 | mov D$EndZapperTable eax
    mov edi ZapperTable, ecx 0-1 | repne scasd
    [ZapperTable:
    ZapperPort1Port2:
    0BC1D_CE96 ; Chiller (Unl) (U)
    0639C_B8B8 ; Chiller (HES)
    0BE1B_F10C ; Chiller [o1]
    ZapperPort2:
    0FBFC_6A6C ; Adventures of Bayou Billy, The (E)
    0CB27_5051 ; Adventures of Bayou Billy, The (U)
    0FB69_C131 ; Baby Boomer (Unl) (U)
    0F264_1AD0 ; Barker Bill's Trick Shooting (U)
    090CA_616D ; Duck Hunt (JUE)
    0EEB7_A62B ; Duck Hunt (JUE) [p1]
    07521_7873 ; Duck Hunt (JUE) [p2]
    041CF_3616 ; Duck Hunt (JUE) [h1]
    0F428_6D0F ; Duck Hunt (JUE) [h2]
    0D947_6F2E ; Duck Hunt (JUE) [h3]
    0AD6C_DF29 ; Duck Hunt (JUE) [b1]
    059E3_343F ; Freedom Force (U)
    0242A_270C ; Gotcha! (U)
    0D83C_671A ; Gotcha! (U) [p1]
    07B5B_D2DE ; Gumshoe (UE)
    08D21_13BC ; Gumshoe (UE) (bad)
    0255B_129C ; Gun Sight (J)
    08963_AE6E ; Hogan's Alley (JU)
    0E5ED_CDE3 ; Hogan's Alley (JU) [p1]
    0DCCD_F578 ; Hogan's Alley (JU) [p2]
    0818F_DB2B ; Hogan's Alley (JU) [o1]
    0A9A1_07B5 ; Hogan's Alley (JU) [p2][o1]
    02716_BD4B ; Hogan's Alley (JU) [h1]
    0B092_DD8E ; Hogan's Alley (JU) [b1]
    051D2_112F ; Laser Invasion (U)
     0A86_6C94 ; Lone Ranger, The (U)
    0E4C0_4EEA ; Mad City (J)
    09EEF_47AA ; Mechanized Attack (U)
    084AB_73C1 ; Mechanized Attack (U) (bad)
               ; Mechanized attach [b2] and [b5] won't run anyway
    0C2DB_7551 ; Shooting Range (U) (bad)
    0885A_8A6D ; Shooting Range (U)
    08106_9812 ; Super Mario Bros / Duck Hunt (U)
    05D37_3F6A ; Super Mario Bros / Duck Hunt (U) [b1]
    0E8F8_F7A5 ; Super Mario Bros / Duck Hunt (E)
    0D4F0_18F5 ; Super Mario Bros / Duck Hunt / Track Meet
    0163E_86C0 ; To The Earth (U)
    03899_60DB ; Wild Gunman (U)
     0D3C_F705 ; Wild Gunman (U) [!]
    04A60_A644 ; Wild Gunman (U) [p1]
    0ED58_8f00 ; VS Duck Hunt
    017AE_56BE ; VS Freedom Force
    0FF51_35A3 ; VS Hogan's Alley
    ZapperNoPort:
    EndZapperTable: 0]
    sub edi 4
    if. edi >= ZapperNoPort
        call SetZapper Port1, &FALSE
        call SetZapper Port2, &FALSE
    else
    if. edi >= ZapperPort2
        call SetZapper Port1, &FALSE
        call SetZapper Port2, &TRUE
    else
    if. edi >= ZapperPort1Port2
        call SetZapper Port1, &TRUE
        call SetZapper Port2, &TRUE
    endif

    ; Enable NES menu and Close
    call 'User32.EnableMenuItem' D$hMenu, 1, &MF_BYPOSITION+&MF_ENABLED
    call 'User32.EnableMenuItem' D$hMenu, IDM_CLOSE, &MF_BYCOMMAND+&MF_ENABLED
    call 'User32.DrawMenuBar' D$hMainWindow
    ret

UnloadNES:


    if D$pImage = &NULL, ret
    call 'User32.SendMessageA' D$hMainWindow, &WM_SETTEXT, &NULL, Classname
    call SetPower POWER_OFF
    call 'Kernel32.GlobalFree' D$pImage | mov D$pImage &NULL
    call ZeroMemory Cartridge (SIZE_CARTRIDGE shl 2)

    ; Disable NES menu and Close
    call 'User32.EnableMenuItem' D$hMenu, 1, &MF_BYPOSITION+&MF_GRAYED
    call 'User32.EnableMenuItem' D$hMenu, IDM_CLOSE, &MF_BYCOMMAND+&MF_GRAYED
    call 'User32.DrawMenuBar' D$hMainWindow
    ret

____________________________________________________________________________________________
; Cartridge info
____________________________________________________________________________________________

CartridgeGetInfo:

    ; Read that crappy iNES header
    call CopyMemory D$pImage, INESHeader, LENGTH_INES_HEADER

    ; File CRC (no iNES interference)
    mov ebx D$pImage, ecx D$FileSize
    add ebx LENGTH_INES_HEADER                              ; Header
    and ecx 0FFFF_E200 | ifFlag ecx 0200, add ebx 0200      ; Trainer
    and ecx 0FFFF_E000                                      ; 8k aligned
  . D$Cartridge@CRC32 = CRC32 ebx, ecx

    ; PROM CRC (based on iNES data)
    mov eax D$pImage | add eax LENGTH_INES_HEADER           ; Skip header
    ifFlag B$INESHeader@ROMControl1 04, add eax 0200        ; Skip trainer
    movzx ecx B$INESHeader@nPROMBanks | shl ecx 14          ; Get PROM size from header
    if ecx < D$Filesize,                                    ; Do not allow size overflow
        . D$Cartridge@PROMCRC32 = CRC32 eax, ecx

    ; Query database
    call DatabaseFind D$Cartridge@CRC32 D$pCRC32
    if esi = &NULL, call DatabaseFind D$Cartridge@PROMCRC32 D$pPROMCRC32
    ; Get cartridge info (esi = data, edi = name)
    if. esi = &NULL
        call UseINES
    else
        call UseDatabase
    endif

    ; If battery but no WRAM, default to 8k WRAM
    if D$Cartridge@SizeWRAM = 0,
        if D$Cartridge@Battery = &TRUE,
            mov D$Cartridge@SizeWRAM 8

    ; pTrainer
    mov eax D$pImage
    add eax LENGTH_INES_HEADER
    if. B$Cartridge@Trainer = &TRUE
        mov D$pTrainer eax
        add eax 0200
    else
        mov D$pTrainer &NULL
    endif

    ; pPROM
    mov D$pPROM eax

    ; pCROM
    if. D$Cartridge@SizeCROM > 0
        mov edx D$Cartridge@SizePROM | shl edx 10 | add eax edx
        mov D$pCROM eax
    else ; Default to 8k of CRAM
        mov D$Cartridge@SizeCRAM 8
        mov D$pCROM &NULL
    endif
    ret

UseDatabase:  ; edi = name, esi = data

    ; Save game name
    mov D$Cartridge@pGamename edi

    ; Get cartridge info from database
    lodsb | movzx eax al | shl eax 4                 ; 16k PROM banks
            mov D$Cartridge@SizePROM eax
    lodsb | movzx eax al | shl eax 3                 ; 8k CROM banks
            mov D$Cartridge@SizeCROM eax
    lodsb | movzx eax al | shl eax 3                 ; 8k WRAM banks
            mov D$Cartridge@SizeWRAM eax
    lodsb | mov B$Cartridge@Mapper al                ; Mapper
    lodsw                                            ; Various stuff
    shr ax 1 | setc B$Cartridge@PAL
    shr ax 1 | setc B$Cartridge@NTSC
    shr ax 1 | setc B$Cartridge@VsSystem
    shr ax 1 | setc B$Cartridge@Playchoice
                mov B$Cartridge@Mirroring al
                and B$Cartridge@Mirroring 3
    shr ax 3 | setc B$Cartridge@Battery
    shr ax 1 | setc B$Cartridge@Trainer
    shr ax 1 | setc B$Cartridge@BadDump
    shr ax 1 | setc B$Cartridge@Hacked
    shr ax 1 | setc B$Cartridge@Translated
    shr ax 1 | setc B$Cartridge@Unlicensed
    shr ax 1 | setc B$Cartridge@Bootleg
    ret

UseINES:

    ; Get name (file title)
    mov D$Cartridge@pGamename FileTitle

    ; PROM banks
    movzx eax B$INESHeader@nPROMBanks
    shl eax 4 | mov D$Cartridge@SizePROM eax

    ; CROM banks
    movzx eax B$INESHeader@nCROMBanks
    shl eax 3 | mov D$Cartridge@SizeCROM eax

    ; Mapper
    mov ax W$INESHeader@ROMControl1
    shr ah, 4 | shr ax, 4 | and eax 0FF
    mov D$Cartridge@Mapper eax

    ; Mirroring
    mov B$Cartridge@Mirroring HORIZONTAL_MIRRORING
    ifFlag B$INESHeader@ROMControl1 01, mov B$Cartridge@Mirroring VERTICAL_MIRRORING
    ifFlag B$INESHeader@ROMControl1 08, mov B$Cartridge@Mirroring FOUR_SCREEN_MIRRORING
    test B$INESHeader@ROMControl1 04 | setnz B$Cartridge@Trainer
    test B$INESHeader@ROMControl1 02 | setnz B$Cartridge@Battery


    ; Default - activate 8k WRAM at $6000
    mov D$Cartridge@SizeWRAM 8
    ret
____________________________________________________________________________________________
; Database
____________________________________________________________________________________________

DatabaseFind:

    arguments @CRC, @pSearch

    ; Database in use?
    mov esi &NULL
    on B$Preferences@UseDatabase = &FALSE, Q0>>
    on D@CRC = &NULL, Q0>>

    ; Scan through
    mov edi D@pSearch, eax D@CRC, ecx D$nROMs
    repne scasd | jecxz Q0>
    sub ecx D$nROMs | not ecx

    ; Data pointer
    move eax ecx*8
    add eax D$pData | mov esi eax

    ; Name pointer
    movzx edx W$esi+6
    mov edi D$pNames, al 0, ecx 0-1
L0: dec edx | js Q0>
    repne scasb | jmp L0<

Q0: return
____________________________________________________________________________________________
; ROM name
____________________________________________________________________________________________

[ROMTitle:  B$ ? #&MAX_PATH]
[BadDump:   B$ ' (Bad dump)',   0
 WrongSize: B$ ' (Wrong size)', 0]
____________________________________________________________________________________________
; Cartridge data
____________________________________________________________________________________________

; iNES header
[LENGTH_INES_HEADER 010]
[INESHeader:
 @ID:           D$ ?
 @nPROMBanks:   B$ ?
 @nCROMBanks:   B$ ?
 @ROMControl1:  B$ ?
 @ROMControl2:  B$ ?
 @Unused:       B$ ? #8]

; Pointers to image
[pImage:        ?
 pTrainer:      ?
 pPROM:         ?
 pCROM:         ?]

; Cartridge data
[SIZE_CARTRIDGE 21]
[Cartridge:
 @CRC32:        ?
 @PROMCRC32:    ?

 @SizePROM:     ?
 @SizeCROM:     ?
 @SizeCRAM:     ?
 @SizeWRAM:     ?

 @Mapper:       ?

 @PAL:          ?
 @NTSC:         ?
 @VsSystem:     ?
 @Playchoice:   ?
 @Mirroring:    ?
 @Battery:      ?
 @Trainer:      ?
 @BadDump:      ?
 @Hacked:       ?
 @Translated:   ?
 @Unlicensed:   ?
 @Bootleg:      ?

 @pGamename:    ?]

[HORIZONTAL_MIRRORING  00
 VERTICAL_MIRRORING    01
 FOUR_SCREEN_MIRRORING 02]
____________________________________________________________________________________________
