TITLE CPU

; This is used from the NES title to emulate the Motorola 6502.
; SetIRQ [IRQ_NMI/IRQ_RESET/...] - triggers an interrupt
; ClearIRQ [IRQ_NMI/IRQ_RESET/...] - clears an interrupt
; HandleInterrupts - handles any pending interrupts (should be called after every instruction)
; To execute an opcode, see RunOneFrame

____________________________________________________________________________________________
; Interrupts
____________________________________________________________________________________________

[SetIRQ   | or  D$IRQLine #2 | copy D$#1Clock D$IRQClock | call CheckIRQLine]
[ClearIRQ | and D$IRQLine 0FF-#1 | call CheckIRQLine]
[IRQ            IRQ_FRAME+IRQ_DMC+IRQ_CARTRIDGE
 IRQ_RESET      080
 IRQ_NMI        040
 IRQ_FRAME      020
 IRQ_DMC        010
 IRQ_CARTRIDGE   08]

HandleInterrupts:

    ; Reset?
    ifFlag. D$IRQLine IRQ_RESET
        call DoReset
        mov D$IRQLine 0
        call CheckIRQLine
        ret
    endif

    ; 6502 pipelining! If next opcode has already been fetched, execute that one first
    mov eax D$CPUClock | sub eax D$IRQClock | js Q0>>
    if. eax <= D$CPUCycles
        ; Execute
        getNextByte
        copy D$InstructionTable+eax*4 D$Instruction
        call D$AddressingTable+eax*4
    endif
    mov D$IRQClock 0-100

    ; NMI?
    ifFlag. D$IRQLine IRQ_NMI
        call DoNMI
        ClearIRQ IRQ_NMI
        call CheckIRQLine
        ret
    endif

    ; IRQ?
    ifFlag D$IRQLine IRQ, call DoIRQ
    call CheckIRQLine
Q0: ret

CheckIRQLine:

    test D$IRQLine IRQ_RESET+IRQ_NMI | setnz B$CheckIRQStatus
    ifFlag B$IRQLine IRQ,
        if B$Flag.I = &FALSE,
            mov B$CheckIRQStatus &TRUE
    ret


InterruptSequence:

    ; Do interrupt
    cpuTick 7
    pushWord W$PC
    call PackFlags | pushByte al
    ret

DoIRQ:

    ; IRQs enabled?
    if. B$Flag.I = &FALSE
        ;ClearIRQ IRQ_CARTRIDGE
        call InterruptSequence
        mov B$Flag.I &TRUE
        mov B$Flag.D &FALSE
        ; Update PC
        mov eax IRQ_VECTOR
        getWrappedWordEax
        mov D$PC eax
        mov B$Flag.I &TRUE
        mov B$Flag.D &FALSE
    endif
    ret

DoNMI:

    call InterruptSequence

    ; Update PC
    mov B$Flag.I &TRUE
    mov B$Flag.D &FALSE
    mov eax NMI_VECTOR
    getWrappedWordEax
    mov D$PC eax
    ret

DoReset:

    ; Reset other
    call ClearAudio
    call FlushVideo | call ClearVideo
    ; Reset NES

    call ResetMemoryMapping
    call APUReset
    call PPUReset
    call CartridgeReset
    call ResetGameGenie
    call ZeroMemory Ports LENGTH_PORTS | mov B$Zapper 08

    ; Reset CPU
    mov eax RESET_VECTOR | getWrappedWordEax | mov D$PC eax
    mov B$S 0FF
    mov B$Flag.I &TRUE
    mov B$Flag.D &FALSE
    ret

____________________________________________________________________________________________
; Flags
____________________________________________________________________________________________

[BREAK_FLAG  010
 UNUSED_FLAG 020]
[saveSZCV | sets B$Flag.S | setz B$Flag.Z | setc B$Flag.C | seto B$Flag.V]
[saveSZC  | sets B$Flag.S | setz B$Flag.Z | setc B$Flag.C]
[saveSZ   | sets B$Flag.S | setz B$Flag.Z]

PackFlags:

    mov al B$Flag.S | shl al 1
    or  al B$Flag.V | shl al 3
    or  al B$Flag.D | shl al 1
    or  al B$Flag.I | shl al 1
    or  al B$Flag.Z | shl al 1
    or  al B$Flag.C
    or  al UNUSED_FLAG

    call CheckIRQLine
    ret

UnpackFlags:

    shr al 1 | setc B$Flag.C
    shr al 1 | setc B$Flag.Z
    shr al 1 | setc B$Flag.I
    shr al 1 | setc B$Flag.D
    shr al 3 | setc B$Flag.V
    shr al 1 | setc B$Flag.S
    ret

____________________________________________________________________________________________
; Memory access
____________________________________________________________________________________________

; Memory read
;[doRead             | mov D$Temp eax | call D$eax*4+CPUReadTable | mov B$CPUBus al | ifFlag eax 0FFFF_FF00, int 3]
[doRead             | call D$eax*4+CPUReadTable | mov B$CPUBus al]
[getWrappedWordEax  | mov edx eax | doRead | xchg eax edx | inc al | doRead | mov ah al | mov al dl] ; Both bytes from same page
[getNextByte        | movzx eax W$PC | doRead | inc W$PC]
[getNextWord        | getNextByte | mov dl al | getNextByte | mov ah al | mov al dl]
ReadVoid:    | outhex eax | movzx eax B$CPUBus     | ret
ReadRAM:       movzx eax B$eax+RAM    | ret
ReadMirrorRAM: and eax 07FF  | doRead | ret
ReadMirrorPPU: and eax 02007 | doRead | ret
ReadWRAM:      and eax 01FFF | movzx eax B$eax+WRAM | ret

ReadPROM:

    mov ebx eax | shr ebx 13 | and ebx 3
    and eax 01FFF
    add eax D$IndexPROM+ebx*4
    add eax D$pPROM
    movzx eax B$eax
    ret

; Memory write
[doWrite | call D$edx*4+CPUWriteTable]
WriteVoid:       outWrite | ret
WriteRAM:        mov B$edx+RAM al | ret
WriteMirrorRAM:  and edx 07FF  | doWrite | ret
WriteMirrorPPU:  and edx 02007 | doWrite | ret
WriteWRAM:       and edx 01FFF | mov B$edx+WRAM al | ret
____________________________________________________________________________________________
; Stack
____________________________________________________________________________________________
[pushByte | mov al #1 | movzx edx B$S | mov  B$edx+RAM+0100 al | dec B$S]
[pullByte | inc B$S   | movzx edx B$S | mov  al B$edx+RAM+0100 | mov #1 al]
[pushWord | mov bx #1 | pushByte bh | pushByte bl]
[pullWord | pullByte bl | pullByte bh | mov #1 bx]
____________________________________________________________________________________________
; Addressing modes
____________________________________________________________________________________________
[doReadCycle  |             doRead  | cpuTick 1]
[doWriteCycle | cpuTick 1 | doWrite] ; Must tick before or Dizzy the adventurer will shake
;      Save address, read data,     write back unmodified data,    do operation,        write data,    done
Read:                doReadCycle |                                 call D$Instruction                | ret
RMW:   mov edx eax | doReadCycle | pushad | doWriteCycle | popad | call D$Instruction | doWriteCycle | ret
Write: mov edx eax |                                               call D$Instruction | doWriteCycle | ret

; Zero page
ZP_RMW:   cpuTick 2 | getNextByte | jmp RMW
ZP_W:     cpuTick 2 | getNextByte | jmp Write
ZP_R:     cpuTick 2 | getNextByte | jmp Read
ZPX_RMW:  cpuTick 3 | getNextByte | add al B$X | jmp RMW
ZPX_W:    cpuTick 3 | getNextByte | add al B$X | jmp Write
ZPX_R:    cpuTick 3 | getNextByte | add al B$X | jmp Read
ZPY_RMW:  cpuTick 3 | getNextByte | add al B$Y | jmp RMW
ZPY_W:    cpuTick 3 | getNextByte | add al B$Y | jmp Write
ZPY_R:    cpuTick 3 | getNextByte | add al B$Y | jmp Read

; Absolute
Abs_RMW:  cpuTick 3 | getNextWord | jmp RMW
Abs_W:    cpuTick 3 | getNextWord | jmp Write
Abs_R:    cpuTick 3 | getNextWord | jmp Read
AbsX_RMW: cpuTick 4 | getNextWord | mov dx W$X |  add ax dx | jmp RMW
AbsX_W:   cpuTick 4 | getNextWord | mov dx W$X |  add ax dx | jmp Write
AbsX_R:   cpuTick 3 | getNextWord | mov dx W$X | xadd ax dx | if ah != dh, cpuTick 1 | jmp Read
AbsY_RMW: cpuTick 4 | getNextWord | mov dx W$Y |  add ax dx | jmp RMW
AbsY_W:   cpuTick 4 | getNextWord | mov dx W$Y |  add ax dx | jmp Write
AbsY_R:   cpuTick 3 | getNextWord | mov dx W$Y | xadd ax dx | if ah != dh, cpuTick 1 | jmp Read

; Indirect
IndX_RMW: cpuTick 5 | getNextByte | add al B$X | getWrappedWordEax | jmp RMW
IndX_W:   cpuTick 5 | getNextByte | add al B$X | getWrappedWordEax | jmp Write
IndX_R:   cpuTick 5 | getNextByte | add al B$X | getWrappedWordEax | jmp Read
IndY_RMW: cpuTick 5 | getNextByte | getWrappedWordEax | mov dx W$Y |  add ax dx | jmp RMW
IndY_W:   cpuTick 5 | getNextByte | getWrappedWordEax | mov dx W$Y |  add ax dx | jmp Write
IndY_R:   cpuTick 4 | getNextByte | getWrappedWordEax | mov dx W$Y | xadd ax dx | if ah != dh, cpuTick 1 | jmp Read

; Misc
Imm:      cpuTick 1 | movzx eax W$PC | inc W$PC | jmp Read
Acc:      cpuTick 2 | mov eax D$A | call D$Instruction | mov D$A eax | ret
Imp:      cpuTick 2 | call D$Instruction | ret
Spec:     call D$Instruction | ret
Rel:      cpuTick 2 | call D$Instruction
          ; Don't branch!
          je L0> | inc W$PC | ret | L0:
          ; Branch!
          getNextByte | cbw | add ax W$PC
          cpuTick 1
          if ah != B$PC+1, cpuTick 1
          mov W$PC ax
          ret
____________________________________________________________________________________________
; Instructions
____________________________________________________________________________________________

; Read
ADC: shr B$Flag.C 1 |       adc B$A al       | saveSZCV | ret
SBC: shr B$Flag.C 1 | cmc | sbb B$A al | cmc | saveSZCV | ret
AND: and B$A al | saveSZ | ret
EOR: xor B$A al | saveSZ | ret
ORA:  or B$A al | saveSZ | ret
LDA: mov B$A al | add al 0 | saveSZ | ret
LDX: mov B$X al | add al 0 | saveSZ | ret
LDY: mov B$Y al | add al 0 | saveSZ | ret
CMP: cmp B$A al | cmc | saveSZC | ret
CPX: cmp B$X al | cmc | saveSZC | ret
CPY: cmp B$Y al | cmc | saveSZC | ret
BIT: test al B$A | setz  B$Flag.Z
     test al 040 | setnz B$Flag.V
     test al 080 | setnz B$Flag.S
     ret

; Write
STA: mov eax D$A | ret
STX: mov eax D$X | ret
STY: mov eax D$Y | ret

; Read-Modify-Write
ASL: shl al 1 | saveSZC | ret
LSR: shr al 1 | saveSZC | ret
ROL: shr B$Flag.C 1 | rcl al, 1 | setc B$Flag.C | add al 0 | saveSZ | ret
ROR: shr B$Flag.C 1 | rcr al, 1 | setc B$Flag.C | add al 0 | saveSZ | ret
INC: inc al | saveSZ | ret
DEC: dec al | saveSZ | ret

; Special
NOP: ret
JMPi:cpuTick 5 | getNextWord | getWrappedWordEax | mov W$PC ax | ret
JMP: cpuTick 3 | getNextWord | mov W$PC ax | ret
JSR: cpuTick 6 | getNextWord | xchg ax W$PC | dec ax | pushWord ax | ret
RTI: cpuTick 6 | pullByte al | call UnpackFlags | pullWord W$PC | ret
RTS: cpuTick 6 | pullWord W$PC | inc W$PC | ret
PHA: cpuTick 3 | pushByte B$A | ret
PLA: cpuTick 4 | pullByte B$A | add B$A 0 | saveSZ | ret
PHP: cpuTick 3 | call PackFlags | or al BREAK_FLAG | pushByte al | ret
PLP: cpuTick 4 | pullByte al | call UnpackFlags | ret
BRK: cpuTick 7
     inc W$PC | pushWord W$PC
     call PackFlags | or al BREAK_FLAG | pushByte al
     mov eax BRK_VECTOR | getWrappedWordEax | mov D$PC eax
     mov B$Flag.I &TRUE
     ret

; Branch
BCS: cmp B$Flag.C &TRUE  | ret
BEQ: cmp B$Flag.Z &TRUE  | ret
BMI: cmp B$Flag.S &TRUE  | ret
BVS: cmp B$Flag.V &TRUE  | ret
BCC: cmp B$Flag.C &FALSE | ret
BNE: cmp B$Flag.Z &FALSE | ret
BPL: cmp B$Flag.S &FALSE | ret
BVC: cmp B$Flag.V &FALSE | ret

; Implied
SEC: mov B$Flag.C &TRUE  | ret
SED: mov B$Flag.D &TRUE  | ret
SEI: mov B$Flag.I &TRUE  | ret
CLC: mov B$Flag.C &FALSE | ret
CLD: mov B$Flag.D &FALSE | ret
CLI: mov B$Flag.I &FALSE | ret
CLV: mov B$Flag.V &FALSE | ret
DEX: dec B$X | saveSZ | ret
DEY: dec B$Y | saveSZ | ret
INX: inc B$X | saveSZ | ret
INY: inc B$Y | saveSZ | ret
TAX: copy D$A D$X | add B$X 0 | saveSZ | ret
TSX: copy D$S D$X | add B$X 0 | saveSZ | ret
TXA: copy D$X D$A | add B$A 0 | saveSZ | ret
TYA: copy D$Y D$A | add B$A 0 | saveSZ | ret
TAY: copy D$A D$Y | add B$Y 0 | saveSZ | ret
TXS: copy D$X D$S | ret

____________________________________________________________________________________________
; Illegal opcodes
____________________________________________________________________________________________
LAX:SHY:SHX:SBX:ANE:LXA:ARR: ; Read
KIL:ANC:ASR:LAS: ; ?
SHA:SAX:SHS: ; Write
SLO:SRE:ISB:RRA:DCP:RLA:;RMW
;;
    pushad
        outhex D$PC
        closeMessage 'Illegal opcode.'
        copy D$CyclesPerFrame D$CPUClock
    popad
;;
    ret

____________________________________________________________________________________________
; CPU data
____________________________________________________________________________________________
[cpuTick | mov ecx D$CPUCycles | imul ecx #1 | add D$CPUClock ecx]
[ppuTick | mov ecx D$PPUCycles | imul ecx #1 | add D$CPUClock ecx]
[Instruction: ?]

; Sizes
[LENGTH_CPU      <(SIZE_CPU shl 2)>]
[SIZE_CPU        SIZE_CPU_REGS+SIZE_CPU_MEMORY]
[SIZE_CPU_REGS   11+5]
[SIZE_CPU_MEMORY SIZE_RAM+SIZE_WRAM]
[SIZE_RAM        0200]
[SIZE_WRAM       02000]

; 6502 vectors
[NMI_VECTOR 0FFFA   RESET_VECTOR 0FFFC   IRQ_VECTOR 0FFFE   BRK_VECTOR 0FFFE]

; 6502 registers
[CPU:
 PC: ?
 X:  ?
 Y:  ?
 A:  ?
 S:  ?
 Flag.C: ?
 Flag.Z: ?
 Flag.I: ?
 Flag.D: ?
 Flag.V: ?
 Flag.S: ?

; Variables
 IRQClock:       ?
 IRQLine:        ?
 CheckIRQStatus: ?
 CPUClock:       ?
 CPUBus:         ?

; Memory
 CPUMemory:
 RAM:   ? #SIZE_RAM
 WRAM:  ? #SIZE_WRAM]

____________________________________________________________________________________________
; Tables
____________________________________________________________________________________________

; Instructions
[InstructionTable:
 BRK ORA KIL SLO NOP ORA ASL SLO _ PHP ORA ASL ANC NOP  ORA ASL SLO _ BPL ORA KIL SLO NOP ORA ASL SLO _ CLC ORA NOP SLO NOP ORA ASL SLO
 JSR AND KIL RLA BIT AND ROL RLA _ PLP AND ROL ANC BIT  AND ROL RLA _ BMI AND KIL RLA NOP AND ROL RLA _ SEC AND NOP RLA NOP AND ROL RLA
 RTI EOR KIL SRE NOP EOR LSR SRE _ PHA EOR LSR ASR JMP  EOR LSR SRE _ BVC EOR KIL SRE NOP EOR LSR SRE _ CLI EOR NOP SRE NOP EOR LSR SRE
 RTS ADC KIL RRA NOP ADC ROR RRA _ PLA ADC ROR ARR JMPi ADC ROR RRA _ BVS ADC KIL RRA NOP ADC ROR RRA _ SEI ADC NOP RRA NOP ADC ROR RRA
 NOP STA NOP SAX STY STA STX SAX _ DEY NOP TXA ANE STY  STA STX SAX _ BCC STA KIL SHA STY STA STX SAX _ TYA STA TXS SHS SHY STA SHX SHA
 LDY LDA LDX LAX LDY LDA LDX LAX _ TAY LDA TAX LXA LDY  LDA LDX LAX _ BCS LDA KIL LAX LDY LDA LDX LAX _ CLV LDA TSX LAS LDY LDA LDX LAX
 CPY CMP NOP DCP CPY CMP DEC DCP _ INY CMP DEX SBX CPY  CMP DEC DCP _ BNE CMP KIL DCP NOP CMP DEC DCP _ CLD CMP NOP DCP NOP CMP DEC DCP
 CPX SBC NOP ISB CPX SBC INC ISB _ INX SBC NOP SBC CPX  SBC INC ISB _ BEQ SBC KIL ISB NOP SBC INC ISB _ SED SBC NOP ISB NOP SBC INC ISB

; Addressing modes
 AddressingTable:
 Spec IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Spec Imm Acc Imm Abs_R Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Spec IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Spec Imm Acc Imm Abs_R Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Spec IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Spec Imm Acc Imm Spec  Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Spec IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Spec Imm Acc Imm Spec  Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Imm  IndX_W Imm IndX_W   ZP_W ZP_W ZP_W   ZP_W    Imp  Imm Imp Imm Abs_W Abs_W Abs_W   Abs_W    Rel IndY_W Imp IndY_W   ZPX_W ZPX_W ZPY_W   ZPY_W    Imp AbsY_W Imp AbsY_W   AbsX_W AbsX_W AbsY_W   AbsY_W
 Imm  IndX_R Imm IndX_R   ZP_R ZP_R ZP_R   ZP_R    Imp  Imm Imp Imm Abs_R Abs_R Abs_R   Abs_R    Rel IndY_R Imp IndY_R   ZPX_R ZPX_R ZPY_R   ZPY_R    Imp AbsY_R Imp AbsY_R   AbsX_R AbsX_R AbsY_R   AbsY_R
 Imm  IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Imp  Imm Imp Imm Abs_R Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW
 Imm  IndX_R Imm IndX_RMW ZP_R ZP_R ZP_RMW ZP_RMW  Imp  Imm Imp Imm Abs_R Abs_R Abs_RMW Abs_RMW  Rel IndY_R Imp IndY_RMW ZPX_R ZPX_R ZPX_RMW ZPX_RMW  Imp AbsY_R Imp AbsY_RMW AbsX_R AbsX_R AbsX_RMW AbsX_RMW]
____________________________________________________________________________________________
