TITLE APU

; Emulates the Audio Processing Unit.
; Read40xx/Write40xx handle APU register access.
; APUSynchronize makes the APU catch up in time with the CPU.
; APUReset resets the regs.

____________________________________________________________________________________________
; APU synchronization
____________________________________________________________________________________________
APUSynchronize:

    pushad

        ; Is APU behind?
        mov eax D$APUClock | on eax >= D$CPUClock, Q0>
        copy D$CPUClock D$APUClock

        ; How much?
        neg eax | add eax D$CPUClock
        mov edx 0 | div D$CPUCycles

        ; How many samples is that?
        mov edx 0
        add eax D$SampleCounter
        div D$CyclesPerSample
        mov D$SampleCounter edx

        ; Output samples
        test eax eax | jz Q0>
    L0: push eax
            call GetSample | call AudioOutput
        pop eax
        dec eax | jnz L0<

Q0: popad
    ret

____________________________________________________________________________________________
; Output
____________________________________________________________________________________________

GetSample:

    ; Clock all timers
    mov eax D$CyclesPerSample
    add D$Square1+Timer eax
    add D$Square2+Timer eax
    add D$Triangle+Timer eax
    add D$Noise+Timer eax
    if B$DMC+LengthEnabled = &TRUE, add D$DMC+Timer eax
    if D$TimerHook != &NULL, call D$TimerHook

    ; Frame counter
    sub D$FrameCounter eax | jns S0> | call FrameCounterZero | S0:

    ; Get channel amplitudes
    mov edi 0
    call SampleSquare Square1 | add edi D$Square1+Output
    call SampleSquare Square2 | add edi D$Square2+Output
    call SampleTriangle       | add edi D$Triangle+Output
    call SampleNoise          | add edi D$Noise+Output
    call SampleDMC            | mov eax D$DMC+Output | shr eax 2 | add edi eax
    if D$SampleHook != &NULL, call D$SampleHook

    mov eax edi
    ret

____________________________________________________________________________________________
; Signal generation
____________________________________________________________________________________________

SampleSquare:

    pop eax ebx | push eax

or D$ebx+TimerLatch 01

    ; Count steps
    mov eax D$ebx+Timer, edx 0
    div D$ebx+TimerLatch
    mov D$ebx+Timer edx

    if B$ebx+Active = &FALSE, ret

    ; Wave high or low?
    add eax D$ebx+DutyCycleCounter | and eax 0F
    mov D$ebx+DutyCycleCounter eax
    movzx edx B$ebx+DutyCycleType
    shr al 1 | and eax 07

; Send decay or volume?
if. B$ebx+DecayDisabled = &TRUE
    copy W$ebx+Volume W$ebx+Output
else
    copy W$ebx+DecayCounter W$ebx+Output
endif

    if B$DutyCycleTable+edx*8+eax = 0, mov B$ebx+Output 0
    ret

SampleTriangle:

or D$Triangle+TimerLatch 01

    ; Count steps
    mov eax D$Triangle+Timer, edx 0
    div D$Triangle+TimerLatch
    mov D$Triangle+Timer edx
    if B$Triangle+Active = &FALSE, ret

    ; Triangle step
    add al B$Triangle+Step
    mov B$Triangle+Step al
    ifFlag al 010, xor al 0F | and al 0F
    mov B$Triangle+Output al
    ret

SampleNoise:

or D$Noise+TimerLatch 01

    ; Count steps
    mov eax D$Noise+Timer, edx 0
    div D$Noise+TimerLatch
    mov D$Noise+Timer edx
    if B$Noise+Active = &FALSE, ret
    if eax = 0, ret

; Send decay or volume?
if. B$Noise+DecayDisabled = &TRUE
    copy W$Noise+Volume W$Noise+Output
else
    copy W$Noise+DecayCounter W$Noise+Output
endif
shr B$Noise+Output 1

; Shift reg
if B$Noise+RandomMode93 = &FALSE
    mov dx 02000
else
    mov dx 0100
endif

    mov ecx eax
L0: test W$Noise+ShiftRegister 04000 | setnz al
    ifFlag W$Noise+ShiftRegister dx, xor al 1
    shr al 1 | rcl W$Noise+ShiftRegister 1
    dec ecx | jnz L0<

    ifFlag W$Noise+ShiftRegister 04000, mov B$Noise+Output 0
    ret
____________________________________________________________________________________________
; Frame counter
____________________________________________________________________________________________
Write4017:

    call APUSynchronize
    ClearIRQ IRQ_FRAME

    mov B$Reg4017 al
    ; Short: 4, 0,1,2,3, 0,1...
    ; Long:  0,1,2,3,4, 0,1...
    mov B$Update 0
    ifFlag. B$Reg4017 LONG_FRAMES
        mov D$FrameCounter 0
        call FrameCounterZero
    else
        mov D$FrameCounter 7457
    endif

    ; Frame IRQ
    ifNotFlag. B$Reg4017 0C0
        mov D$FrameIRQCounter 29830
    else
        mov D$FrameIRQCounter 0
    endif
    ret

FrameCounterZero:

    ; Reset counter
    add D$FrameCounter 7457

    ; Which update?
    if B$Update = 0, call Clock240Hz
    if B$Update = 1, call Clock120Hz
    if B$Update = 2, call Clock240Hz
    if B$Update = 3, call Clock120Hz

    ; Final update?
    inc B$Update
    ifFlag. B$Reg4017 LONG_FRAMES
        if B$Update >= 5, mov B$Update 0
    else
        if B$Update >= 4, mov B$Update 0
    endif
Q0: ret

Clock120Hz:

    inc D$FrameCounter

    ; Sweep units
    call ClockSweepUnit SQUARE1
    call ClockSweepUnit SQUARE2

    ; Length counters
    call ClockLengthCounter SQUARE1
    call ClockLengthCounter SQUARE2
    call ClockLengthCounter Triangle
    call ClockLengthCounter Noise

    if D$FrameCounterHook120Hz != &NULL, call D$FrameCounterHook120Hz
    call Clock240Hz
    ret

Clock240Hz:

    ; Decay units, linear counter
    call ClockEnvelopeDecay SQUARE1
    call ClockEnvelopeDecay SQUARE2
    call ClockEnvelopeDecay Noise
    call ClockLinearCounter

    if D$FrameCounterHook240Hz != &NULL, call D$FrameCounterHook240Hz
    ret
____________________________________________________________________________________________

ClockEnvelopeDecay:

    pop eax ebx | push eax

    ; Decay clock
    if. B$ebx+DecayDivider != 0
        dec B$ebx+DecayDivider
        ret
    endif
    mov al B$ebx+DecayRate | mov B$ebx+DecayDivider al

    ; Decrease decay volume
    if. B$ebx+DecayCounter != 0
        dec B$ebx+DecayCounter
    else
        if B$ebx+DecayLooping = &TRUE, mov B$ebx+DecayCounter 0F
    endif
    ret

ClockSweepUnit:

    pop eax ebx | push eax

    ; Channel active?
    if B$ebx+Active = &FALSE, ret
    if B$ebx+SweepEnabled = &FALSE, ret

    ; Sweep clock
    if. B$ebx+SweepDivider > 0
        dec B$ebx+SweepDivider
        ret
    endif
    mov al B$ebx+SweepRate | mov B$ebx+SweepDivider al

    ; Shift
    mov eax D$ebx+TimerLatch
    mov cl B$ebx+ShiftAmount
    if cl = 0, ret
    shr eax cl

    ; Update wavelength
    if. B$ebx+SweepNegate = &TRUE
        ; Decrease wavelength
        if ebx = SQUARE1, not eax
        if ebx = SQUARE2, neg eax
    endif
    add eax D$ebx+TimerLatch
    if. eax >= D$ebx+MaxWavelength | mov B$ebx+Active &FALSE | ret | endif
;    if. eax >= 07FF | mov B$ebx+Active &FALSE | ret | endif
    if. eax <= MIN_SQUARE_WAVELENGTH | mov B$ebx+Active &FALSE | ret | endif
    mov D$ebx+TimerLatch eax
    ret

ClockLengthCounter:

    pop eax ebx | push eax

    ; Count down length counter
    if B$ebx+LengthCounter = 0, mov B$ebx+Active &FALSE
    if B$ebx+LengthHalted = &FALSE,
        if B$ebx+LengthCounter != 0,
            dec B$ebx+LengthCounter
    ret

ClockLinearCounter:

    if. B$Triangle+LinearCounter =  0
        mov B$Triangle+Active &FALSE
    else
        if B$Triangle+LinearCounterHalt = &FALSE, dec B$Triangle+LinearCounter
    endif

    if B$Triangle+LinearCounterControl = &FALSE, mov B$Triangle+LinearCounterHalt &FALSE
    ret
____________________________________________________________________________________________
; Length counters
____________________________________________________________________________________________

Write4015:

    call APUSynchronize

    ; Kill length counters?
    ifNotFlag al  SQUARE1_ENABLED, mov B$Square1+LengthCounter  0
    ifNotFlag al  SQUARE2_ENABLED, mov B$Square2+LengthCounter  0
    ifNotFlag al TRIANGLE_ENABLED, mov B$Triangle+LengthCounter 0
    ifNotFlag al    NOISE_ENABLED, mov B$Noise+LengthCounter    0

    ; DMC
    ifFlag. al DMC_ENABLED
        if B$DMC+LengthEnabled = &FALSE, call ResetDMC
    else
        mov B$DMC+LengthEnabled &FALSE
        mov D$DmcIrqCounter 0
    endif

    ; Enable channels?
    shr al 1 | setc B$Square1+LengthEnabled
    shr al 1 | setc B$Square2+LengthEnabled
    shr al 1 | setc B$Triangle+LengthEnabled
    shr al 1 | setc B$Noise+LengthEnabled
    call ChannelsActive

    ; Reset DMC IRQ status
    ClearIRQ IRQ_DMC
    ret

Read4015:

    call APUSynchronize

    ; Length counter non-zero?
    mov eax 0
    if B$Square1+LengthCounter  > 0, or al SQUARE1_ENABLED
    if B$Square2+LengthCounter  > 0, or al SQUARE2_ENABLED
    if B$Triangle+LengthCounter > 0, or al TRIANGLE_ENABLED
    if B$Noise+LengthCounter    > 0, or al NOISE_ENABLED
    if B$DMC+LengthEnabled      > 0, or al DMC_ENABLED

    ; IRQ status
    ifFlag D$IRQLine IRQ_FRAME, or al 040
    ifFlag D$IRQLine IRQ_DMC,   or al 080

    ; Reset frame IRQ status
    ClearIRQ IRQ_FRAME
    ret
____________________________________________________________________________________________

ChannelsActive:

    call SquareActive Square1
    call SquareActive Square2
    call TriangleActive
    call NoiseActive
    ret

SquareActive:

    pop eax, ebx | push eax

    mov edx D$ebx+MaxWavelength
    mov B$ebx+Active &FALSE

    if B$ebx+LengthEnabled = &TRUE,
        if B$ebx+LengthCounter != 0,
            if W$ebx+TimerLatch >= MIN_SQUARE_WAVELENGTH,
;                if W$ebx+TimerLatch <= MAX_SQUARE_WAVELENGTH,
                if D$ebx+TimerLatch < edx,
                    mov B$ebx+Active &TRUE
    ret

TriangleActive:

    mov B$Triangle+Active &FALSE
    if B$Triangle+LengthEnabled = &TRUE,
        if B$Triangle+LengthCounter != 0,
            if B$Triangle+LinearCounter != 0,
                    if W$Triangle+TimerLatch >= MIN_TRIANGLE_WAVELENGTH,
                        mov B$Triangle+Active &TRUE
    ret

NoiseActive:

    mov B$Noise+Active &FALSE
    if B$Noise+LengthEnabled = &TRUE,
        if B$Noise+LengthCounter != 0,
            mov B$Noise+Active &TRUE
    ret

____________________________________________________________________________________________
; Reset
____________________________________________________________________________________________

APUReset:

    [TimerHook: ?
     SampleHook: ?
     FrameCounterHook120Hz: ?
     FrameCounterHook240Hz: ?]
    call ZeroMemory TimerHook 010

    call ZeroMemory APU (SIZE_APU shl 2)
    mov B$Noise+ShiftRegister 1
    mov B$Reg4017 DISABLE_FRAME_IRQ
    mov W$Square1+MaxWavelength MAX_SQUARE_WAVELENGTH
    mov W$Square2+MaxWavelength MAX_SQUARE_WAVELENGTH
    ret
____________________________________________________________________________________________
; Register writes
____________________________________________________________________________________________

; Square
____________________________________________________________________________________________

Write4000: call WriteSquareReg0 Square1 | ret
Write4001: call WriteSquareReg1 Square1 | ret
Write4002: call WriteSquareReg2 Square1 | ret
Write4003: call WriteSquareReg3 Square1 | ret
Write4004: call WriteSquareReg0 Square2 | ret
Write4005: call WriteSquareReg1 Square2 | ret
Write4006: call WriteSquareReg2 Square2 | ret
Write4007: call WriteSquareReg3 Square2 | ret

; ddle vvvv
WriteSquareReg0:

    pop edx, ebx | push edx
    call APUSynchronize

    ; --le ----
    test al 020 | setnz B$ebx+LengthHalted
    test al 010 | setnz B$ebx+DecayDisabled
    ; ---- vvvv
    mov B$ebx+Volume al
    and B$ebx+Volume 0F
    ; dd-- ----
    mov B$ebx+DutyCycleType al
    shr B$ebx+DutyCycleType 6
    ret

; errr nsss
WriteSquareReg1:

    pop edx, ebx | push edx
    call APUSynchronize

    ; ---- -sss
    mov B$ebx+ShiftAmount al
    and B$ebx+ShiftAmount 07
    ; ---- n---
    test al 08  | setnz B$ebx+SweepNegate
    ; -rrr ----
    mov B$ebx+SweepRate al
    shr B$ebx+SweepRate 4
    and B$ebx+SweepRate 07
    ; e--- ----
    test al 080 | setnz B$ebx+SweepEnabled

    ; Reset sweep divider
    mov al B$ebx+SweepRate | mov B$ebx+SweepDivider al

    ; Max wavelength
;;
    [SquareMax: 0400, 0556, 0667, 071D, 0788, 07C2, 07E1, 07F1]
    movzx eax B$ebx+SweepRate | xor al 07 | mov ax W$SquareMax+eax*4
    if B$ebx+SweepEnabled = &FALSE, mov ax MAX_SQUARE_WAVELENGTH
    if B$ebx+SweepNegate  = &TRUE,  mov ax MAX_SQUARE_WAVELENGTH
    mov W$ebx+MaxWavelength ax
;;
    call SquareActive ebx
    ret

; wwww wwww
WriteSquareReg2:

    pop edx, ebx | push edx
    call APUSynchronize

    ; wwww wwww
    mov B$ebx+TimerLatch al

    call SquareActive ebx
    ret

; llll lwww
WriteSquareReg3:

    pop edx, ebx | push edx
    call APUSynchronize

    ; ---- -www
    mov B$ebx+TimerLatch+1 al
    and B$ebx+TimerLatch+1 07
    ; llll l---
    if. B$ebx+LengthEnabled = &TRUE
        shr eax 3 | and eax 01F
        mov al B$eax+LengthCounterTable | mov B$ebx+LengthCounter al
    endif

    ; Reset duty cycle counter and decay counter
    mov B$ebx+DutyCycleCounter 0
    mov B$ebx+DecayCounter 0F
    mov al B$ebx+DecayRate | mov B$ebx+DecayDivider al

    ; Channel active?
    call SquareActive ebx
    ret

; Triangle
____________________________________________________________________________________________

; hlll llll
Write4008:

    call APUSynchronize

    ; h--- ----
    test eax 080 | setnz B$Triangle+LengthHalted
    ; -lll llll
    and al 07F
    mov B$Triangle+LinearCounterLoad al
    if B$Triangle+LinearCounterHalt = &TRUE, mov B$Triangle+LinearCounter al

    call TriangleActive
    ret

; tttt tttt
Write400A:

    call APUSynchronize

    ; tttt tttt
    mov B$Triangle+TimerLatch al

    call TriangleActive
    ret

; llll lttt
Write400B:

    call APUSynchronize

    ; ---- -ttt
    mov B$Triangle+TimerLatch+1 al
    and B$Triangle+TimerLatch+1 07
    ; llll l---
    if. B$Triangle+LengthEnabled = &TRUE
        shr eax 3 | and eax 01F
        mov al B$eax+LengthCounterTable | mov B$Triangle+LengthCounter al
    endif

    ; Halt linear counter
    mov B$Triangle+LinearCounterHalt &TRUE
    mov al B$Triangle+LinearCounterLoad | mov B$Triangle+LinearCounter al

    call TriangleActive
    ret

; Noise
____________________________________________________________________________________________

; --ld vvvv
Write400C:

    call APUSynchronize

    ; ---- vvvv
    mov B$Noise+Volume al | and B$Noise+Volume 0F
    ; ---d ----
    test al 010 | setnz B$Noise+DecayDisabled
    ; --l- ----
    test al 020 | setnz B$Noise+LengthHalted

    call NoiseActive
    ret

; m--- wwww
Write400E:

    call APUSynchronize

    ; m--- ----
    test al 080 | setnz B$Noise+RandomMode93
    ; --- wwww
    and eax 0F
    copy W$NoiseWaveLengthTable+eax*2 W$Noise+TimerLatch

    call NoiseActive
    ret

; llll l---
Write400F:

    call APUSynchronize

    ; llll l---
    if. B$Noise+LengthEnabled = &TRUE
        shr eax 3 | and eax 01F
        mov al B$eax+LengthCounterTable | mov B$Noise+LengthCounter al
    endif

    ; Reset decay
    mov B$Noise+DecayCounter 0F
    mov al B$Noise+DecayRate | mov B$Noise+DecayDivider al

    call NoiseActive
    ret
____________________________________________________________________________________________
; APU data
____________________________________________________________________________________________

; Frame counter
[DISABLE_FRAME_IRQ  040
 FRAME_IRQ          040
 LONG_FRAMES        080]

; Length counter
[SQUARE1_ENABLED    01
 SQUARE2_ENABLED    02
 TRIANGLE_ENABLED   04
 NOISE_ENABLED      08
 DMC_ENABLED        010]
[LengthCounterTable: B$
  0A  0FE
 014   02
 028   04
 050   06
 0A0   08
 03C   0A
  0E   0C
 01A   0E
  0C   010
 018   012
 030   014
 060   016
 0C0   018
 048   01A
 010   01C
 020   01E]

; Square
[DutyCycleTable: B$
 000 0FF 000 000 000 000 000 000
 000 0FF 0FF 000 000 000 000 000
 000 0FF 0FF 0FF 0FF 000 000 000
 0FF 000 000 0FF 0FF 0FF 0FF 0FF]
[MIN_SQUARE_WAVELENGTH   8
 MAX_SQUARE_WAVELENGTH   07B7]

; Triangle
[MIN_TRIANGLE_WAVELENGTH 4]

; Noise
[NoiseWaveLengthTable: W$ 02 04 08 010   020 030 040 050   065 07F 0BE 0FE   017D 01FC 03F9 07F2]

; APU/channels
[LENGTH_APU     <(SIZE_APU shl 2)>]
[SIZE_APU       <7+(10 shl 5)>]
[APU:
 APUClock:              ?
 SampleCounter:         ?
 FrameCounter:          ?
 Update:                ?
 Reg4017:               ?
 FrameIRQCounter:       ?
 DmcIRQCounter:         ?

____________________________________________________________________________________________

 Channels:
 Square1:     ? #020
 Square2:     ? #020
 Triangle:    ? #020
 Noise:       ? #020
 DMC:         ? #020

 ExChannel:
 ExChannel0:  ? #020
 ExChannel1:  ? #020
 ExChannel2:  ? #020
 ExChannel3:  ? #020
 ExChannel4:  ? #020]

enumerate4 0,

    ; Common
    Timer,
    TimerLatch,
    Output, Active,
    LengthHalted, LengthEnabled, LengthCounter,

    ; Square/Noise
    DecayRate, DecayCounter, DecayDivider, DecayDisabled,

    ; Square
    DutyCycleType, DutyCycleCounter,
    SweepNegate, SweepEnabled, SweepDivider, SweepRate, ShiftAmount,
    MaxWaveLength, MaxWaveLengthHigh

    ; Triangle
enumerate4 030,
    LinearCounter, LinearCounterLoad, LinearCounterHalt,
    Step, ChangeMode

    ; Noise
enumerate4 030,
    RandomMode93, ShiftRegister, ShiftRegisterHigh

    ; DMC
enumerate4 030,
    LengthCounterLatch, DMCLoop, nBits,
    pSample, pSampleLatch, Sample, DMCEnableIRQ,
    ConvertedTimer, ConvertedTimerLatch

[Volume               DecayRate
 LinearCounterControl LengthHalted
 DecayLooping         LengthHalted]
____________________________________________________________________________________________
