/**
 * This file is a part of JaC64 - a Java C64 Emulator
 * Main Developer: Joakim Eriksson (Dreamfabric.com)
 * Contact: joakime@sics.se
 * Web: http://www.dreamfabric.com/c64
 * ---------------------------------------------------
 */

package com.dreamfabric.jac64;

/**
 * This mixes four channels of SID sounds (or three)
 *
 * 0 - 2 => 16 bits channels from the SID chip
 *
 * 3 => 8 bits from PSID sample player
 *
 *
 * Created: Fri Oct 28 17:45:19 2005
 *
 * @author Joakim Eriksson
 * @version 1.0
 */

import javax.sound.sampled.*;
import javax.sound.sampled.DataLine.Info;


public class SIDMixerSE extends SIDMixer {

  private SourceDataLine dataLine;
  private SourceDataLine syncDataLine;
  private FloatControl volume;

  byte[] syncBuffer = new byte[4096];

  public SIDMixerSE() {
  }

  /**
   * Creates a new <code>SIDMixer</code> instance.
   *
   *
   */
  public SIDMixerSE(SID6581[] channels, SID sample) {
    init(channels, sample);
  }

  public void init(SID6581[] channels, SID sample) {
    super.init(channels, sample);

    // Allocate Audio resources
    AudioFormat af = new AudioFormat(SID6581.SAMPLE_RATE, 16, 1, true, false);
    DataLine.Info dli =
      new DataLine.Info(SourceDataLine.class, af, DL_BUFFER_SIZE);
    DataLine.Info dli2 =
      new DataLine.Info(SourceDataLine.class, af, SYNCH_BUFFER);
    try {
      dataLine = (SourceDataLine) AudioSystem.getLine(dli);
      syncDataLine = (SourceDataLine) AudioSystem.getLine(dli2);

      if (dataLine == null)
	System.out.println("DataLine: not existing...");
      else {
	System.out.println("Format: " + dataLine.getFormat());
	dataLine.open(dataLine.getFormat(), DL_BUFFER_SIZE);
	syncDataLine.open(syncDataLine.getFormat(), SYNCH_BUFFER);
	volume = (FloatControl)
	  dataLine.getControl(FloatControl.Type.MASTER_GAIN);

 	setMasterVolume(100);

	// Startup the dataline
	dataLine.start();
	syncDataLine.start();

      }
    } catch (Exception e) {
      System.out.println("Problem while getting data line ");
      e.printStackTrace();
    }

    if (NO_SOUND) {
      dataLine = null;
    }

  }

  // -------------------------------------------------------------------
  // Some implementation specifi methods (Abstract)
  // -------------------------------------------------------------------

  public void write(byte[] buffer) {
    if (!fullSpeed && getBufferLevel() > 80.0) {
      dataLine.write(buffer, 0, SID6581.GENLEN * 2);
      syncDataLine.write(syncBuffer, 0, SID6581.GENLEN * 2);
    } else {
      dataLine.write(buffer, 0, SID6581.GENLEN * 2);
    }
  }

  public long getMicros() {
    if (dataLine == null) {
      return System.currentTimeMillis() * 1000;
    }
    return dataLine.getMicrosecondPosition();
  }

  public boolean hasSound() {
    return dataLine != null;
  }

  public int available() {
    return dataLine.available();
  }

  public void setMasterVolume(int v) {
    masterVolume = v;
    if (volume != null) {
      volume.setValue(-10.0f + 0.1f * v);
    }
  }

  public int getMasterVolume() {
    return masterVolume;
  }



  // -------------------------------------------------------------------
  // SE Specifics
  // -------------------------------------------------------------------
  public double getBufferLevel() {
    if (dataLine != null)
      return 100.0 - (100.0 * dataLine.available() / DL_BUFFER_SIZE);
    return 0;
  }

}
