/**
 * This file is a part of JaC64 - a Java C64 Emulator
 * Main Developer: Joakim Eriksson (Dreamfabric.com)
 * Contact: joakime@sics.se
 * Web: http://www.dreamfabric.com/c64
 * ---------------------------------------------------
 * This is the CPU file for Commodore 64 with its
 * ROM files and memory management, etc.
 *
 * @(#)cpu.java	Created date: 99-5-17
 *
 */
package com.dreamfabric.jac64;
import com.dreamfabric.c64utils.*;
import java.io.*;
import java.net.URL;


/**
 * CPU "implements" the C64s 6510 processor in java code.
 * reimplemented from old CPU.java
 *
 * @author  Joakim Eriksson (joakime@sics.se)
 * @version $Revision:$, $Date:$
 */
public class CPU extends MOS6510Core // CPUCore // MOS6510Core
{
  // The IO RAM memory at 0x10000 (just since there is RAM there...)
  public static final int IO_OFFSET = 0x10000 - 0xd000;

  public static final int BASIC_ROM2 = 0x1a000;
  public static final int KERNAL_ROM2 = 0x1e000;
  public static final int CHAR_ROM2 = 0x1d000;

  public static final boolean EMULATE_1541 = true;

  // Defaults for the ROMs
  public boolean basicROM = true;
  public boolean kernalROM = true;
  public boolean charROM = false;
  public boolean ioON = true;

  // The state of the program (runs if running = true)
  public boolean running = true;

  // True if loop should exit after a reset???
  public boolean exit = false;

  private static final long CYCLES_PER_DEBUG = 10000000;
  public static final boolean DEBUG = false;

  private C1541Emu c1541;
  private Loader loader;

  public CPU(IMonitor m, String cb, Loader loader) {
    super(m, cb);
    memory = new int[0x20000];
    this.loader = loader;
    if (EMULATE_1541) {
      IMonitor d = new DefaultIMon(); //new Debugger();
      c1541 = new C1541Emu(d, cb);
      // d.init(c1541);
      // d.setEnabled(true);
    }
  }

  public C1541Emu getDrive() {
    return c1541;
  }

  // Reads the memory with all respect to all flags...
  // Maybe make the read/write a bus which has read/write + configuration!!!
  // => 7 different objects with different $1 configs might be fast!??
  protected final int fetchByte(int adr) {

    chips.updateChips(cycles);

    while (baLowUntil > cycles) {
      chips.updateChips(++cycles);
    }

    // ROM/RAM address fix
    if ((basicROM && ((adr & 0xe000) == 0xa000)) ||
	(kernalROM && ((adr & 0xe000) == 0xe000)) ||
	(charROM && ((adr & 0xf000) == 0xd000))) {
      // Add ROM address for the read!
      adr |= 0x10000;
    }

//     System.out.println("Reading byte at " + Integer.toString(adr, 16) +
// 		       " => " + Integer.toString(memory[adr], 16));


    // Remember the address!
    rindex = adr;
    // Now we have the correct address - perform fetch!
    // If it is an IO read - perform it!
    if (ioON && ((adr & 0x1f000) == 0xd000)) {
      return chips.performRead(adr, cycles);
    } else {
      return memory[adr];
    }
  }

  private void fixRindex(int adr) {
    // ROM/RAM address fix
    if ((basicROM && ((adr & 0xe000) == 0xa000)) ||
	(kernalROM && ((adr & 0xe000) == 0xe000)) ||
	(charROM && ((adr & 0xf000) == 0xd000))) {
      // Add ROM address for the read!
      adr |= 0x10000;
    }
    rindex = adr;
  }

  // A byte is written directly to memory or to ioChips
  protected final void writeByte(int adr, int data) {

    chips.updateChips(cycles);

    // Locking only on fetch byte...

//     System.out.println("Writing byte at: " + Integer.toString(adr, 16)
// 		       + " = " + data);
    if (adr <= 1) {
      memory[adr] = data;
      int p = (memory[0] ^ 0xff) | memory[1];

      kernalROM = ((p & 2) == 2); //Kernal on
      basicROM = ((p & 3) == 3); //Basic on

      charROM = ((p & 3) != 0) && ((p & 4) == 0);
      ioON = ((p & 3) != 0) && ((p & 4) != 0);

//       if ((data & 3) != 0) { //IO is on or char is on...
// 	charROM = (data & 4) == 0;
// 	ioON = !charROM;
//       } else {
// 	// => Read from RAM!
// 	charROM = false;
// 	ioON = false;
//       }

//       System.out.println("Wrote $1 = " + Integer.toString(data, 16));
//       System.out.println(" Basic: " + basicROM);
//       System.out.println("Kernal: " + kernalROM);
//       System.out.println("  Char: " + charROM);
//       System.out.println("   I/O: " + ioON);
    }

    adr &= 0xffff;
    if (ioON && ((adr & 0xf000) == 0xd000)) {
//       System.out.println("IO Write at: " + Integer.toString(adr, 16));
      chips.performWrite(adr, data, cycles);
    } else {
      memory[adr] = data;
    }
  }

  public void patchROM(PatchListener list) {
    this.list = list;

    int pos = 0xf49e | 0x10000;
    memory[pos++] = M6510Ops.JSR;
    memory[pos++] = 0xd2;
    memory[pos++] = 0xf5;

    System.out.println("Patched LOAD at: " + Hex.hex2(pos));
    memory[pos++] = LOAD_FILE;
    memory[pos++] = M6510Ops.RTS;
  }

  public void runBasic() {
    memory[631] = (int) 'R';
    memory[632] = (int) 'U';
    memory[633] = (int) 'N';
    memory[634] = 13;//enter
    memory[198] = 4; //length
  }

  public void enterText(String txt) {
    System.out.println("Entering text into textbuffer: " + txt);
    txt = txt.toUpperCase();
    int len = txt.length();
    int pos = 0;
    for (int i = 0, n = len; i < n; i++) {
      char c = txt.charAt(i);
      if (c == '~') c = 13;
      memory[631 + pos] = c;
      pos++;
      if (pos == 5) {
	memory[198] = pos;
	pos = 0;
	int tries = 5;
	while (tries > 0 && memory[198] > 0) {
	  try {
	    Thread.sleep(50);
	  } catch (Exception e) {
	    e.printStackTrace();
	  }
	  tries--;
	  if (tries == 0) {
	    System.out.println("Buffer still full: " + memory[198]);
	  }
	}
      }
    }
    memory[198] = pos;
    int tries = 5;
    while (tries > 0 && memory[198] > 0) {
      try {
	Thread.sleep(50);
      } catch (Exception e) {
	e.printStackTrace();
      }
      tries--;
      if (tries == 0) {
	System.out.println("Buffer still full: " + memory[198]);
      }
    }
  }

  protected void installROMS() {
    loadROM(loader.getResourceStream("/roms/kernal.c64"), KERNAL_ROM2, 0x2000);
    loadROM(loader.getResourceStream("/roms/basic.c64") , BASIC_ROM2, 0x2000);
    loadROM(loader.getResourceStream("/roms/chargen.c64"), CHAR_ROM2, 0x1000);
  }


  public void run(int address) {
    reset();
    exit = false;
    running = true;
    loop(address);
  }


  // Takes the thread and loops!!!
  public void start() {
    while(!exit) {
      run(0xfce2); // Power UP reset routine!
      if (!exit) {
	monitor.info("Resetting!!!!");
	reset();
      }
    }
  }

  public void stop() {
    // stop completely
    running = false;
    exit = true;
  }

  public void reset() {
    running = false;
    exit = false;
    writeByte(1, 7);
    super.reset();

    if (EMULATE_1541) {
      c1541.reset();
    }
  }


  /**
   * The main emulation <code>loop</code>.
   *
   * @param startAdress an <code>int</code> value that represent the
   * starting address of the emulator
   */
  public void loop(int startAdress) {
    // The processor flags
    pc = startAdress;

    long next_print = cycles + CYCLES_PER_DEBUG;
    // How much should this be???

    monitor.info("Starting CPU at: " + Integer.toHexString(pc) );

    try {
	while (running) {

	  // Debugging?
	  if (monitor.isEnabled()) { // || interruptInExec > 0) {
	    if (baLowUntil <= cycles) {
	      fixRindex(pc); // sets the rindex!

	      monitor.disAssemble(memory,rindex,acc,x,y,
				  (byte)getStatusByte(),interruptInExec,
				  lastInterrupt);
	    }
	  }

	  // Run one instruction!
	  emulateOp();

	  // Also allow the 1541 to run an instruction!
	  if (EMULATE_1541) {
	    c1541.tick(cycles);
	  }

	  nr_ins++;
	  if(next_print < cycles) {
	    long sec = System.currentTimeMillis() - lastMillis;
	    int level = monitor.getLevel();

	    if (DEBUG && level > 1) {
	      monitor.info("--------------------------");
	      monitor.info("Nr ins:" + nr_ins + " sec:" +
			   (sec) + " -> " + ((nr_ins * 1000) / sec) +
			   " ins/s"  + "  " +
			   " clk: " + cycles +
			   " clk/s: "+((CYCLES_PER_DEBUG * 1000)/ sec) +
			   "\n" + ((nr_irq * 1000) / sec));
	      if (level > 2) monitor.disAssemble(memory,rindex,acc,x,y,(byte)getStatusByte(),interruptInExec, lastInterrupt);
	      monitor.info("--------------------------");
	    }
	    nr_irq = 0;
	    nr_ins = 0;
	    lastMillis = System.currentTimeMillis();
	    next_print = cycles + CYCLES_PER_DEBUG;
	  }
	}
    } catch (Exception e) {
      monitor.error("Exception in loop " + pc + " : " + e);
      e.printStackTrace();
      monitor.disAssemble(memory,rindex,acc,x,y,(byte) getStatusByte(),interruptInExec, lastInterrupt);
    }
  }

}
