/**
 * This file is a part of JaC64 - a Java C64 Emulator
 * Main Developer: Joakim Eriksson (Dreamfabric.com)
 * Contact: joakime@sics.se
 * Web: http://www.dreamfabric.com/c64
 * ---------------------------------------------------
 */
package com.dreamfabric.jac64;


/**
 * Describe class CIA here.
 *
 * ciaMemory is used to store what the CPU has written - states, etc.
 * cpuMemory is used to write results from the CIA that can be read
 *           by the CPU (this to simplify some things).
 *
 * Created: Sat Jul 30 05:38:32 2005
 *
 * @author <a href="mailto:Joakim@BOTBOX"></a>
 * @version 1.0
 */
public class CIA {

  public static final boolean TIMER_DEBUG = false; //true; //false;
  public static final boolean WRITE_DEBUG = false; //true;

  public static final int PRA = 0x00;
  public static final int PRB = 0x01;
  public static final int DDRA = 0x02;
  public static final int DDRB = 0x03;
  public static final int TIMALO = 0x04;
  public static final int TIMAHI = 0x05;
  public static final int TIMBLO = 0x06;
  public static final int TIMBHI = 0x07;
  public static final int TODTEN = 0x08;
  public static final int TODSEC = 0x09;
  public static final int TODMIN = 0x0a;
  public static final int TODHRS = 0x0b;

  public static final int CIAICR = 0x0d;

  public static final int CIACRA = 0x0e;
  public static final int CIACRB = 0x0f;

  boolean timersONA = false;
  boolean timersONB = false;

  int ta = 0;
  int tb = 0;
  // For the CPU to read (contains status)
  int ciaicrRead;

  int timerATrigg = 0;
  int timerBTrigg = 0;

  boolean timersIRQ[] = new boolean[2];

  long nextTimersA = 0;
  long nextTimersB = 0;

  // B-Div is set if mode (bit 5,6 of CIACRB) = 10
  public static final int TIMER_B_DIV_MASK = 0x60;
  public static final int TIMER_B_DIV_VAL = 0x40;

  int timerBDiv = 1;

  int latchValue[] = new int[2];

  long nextUpdate = 0;
  public long nextCIAUpdate = 0;

  // The memory of the CIA
  public int[] ciaMemory = new int[16];

  private int[] cpuMemory;
  private int offset;

  public int serialFake = 0;
  private C64Chips chips;
  private int lastTrigg = 0;

  /**
   * Creates a new <code>CIA</code> instance.
   *
   */
  public CIA(int memory[], int offset, C64Chips chips) {
    cpuMemory = memory;
    this.offset = offset;
    this.chips = chips;
  }

  public void reset() {
    timerATrigg = 0;
    timerBTrigg = 0;
    timersONA = false;
    timersONB = false;
  }

  public String ciaID() {
    return offset == 0x10c00 ? "CIA 1" : "CIA 2";
  }

  // Returns 0 if no timer trigged, 1 if timer 1, 2 if timer 2 and 3 if both
  public int updateCIA(long cycles) {
    int trigg = 0;

    // Just pick a number...
    nextCIAUpdate += 37;

    // Timer 1
    if (timersONA && nextTimersA <= cycles) {
      ciaicrRead |= 1; // Timer A count to 0
      timerATrigg = timersIRQ[0] ? 1 : 0;

      if (TIMER_DEBUG && timersIRQ[0]) {
	println("Timer A interrupt: " + cycles + " expected: " +
		nextTimersA + " diff: " + (cycles - nextTimersA));
      }

      // One shot or not?
      if ((ciaMemory[CIACRA] & 0x08) == 0) {
	// Continue! (a few cycles delay before latch, etc)
	nextTimersA = nextTimersA + latchValue[0] + 2;
      } else {
	timersONA = false;
	cpuMemory[offset + CIACRA] &= 0xfe; // clear bit 0 to indicate stop
// 	ciaMemory[0x0e] &= 0xfe;
      }
    }

    // Timer 2
    if (timersONB && nextTimersB <= cycles) {
//       println("TimerB: " +
// 			 Integer.toString(cpuMemory[offset + CIAICR], 16));
      ciaicrRead |= 2; // Timer B count to 0
      timerBTrigg = timersIRQ[1] ? 2 : 0;

      if (TIMER_DEBUG && timersIRQ[1]) {
	println(ciaID() + ": Timer B interrupt: " + cycles + " expected:" +
		nextTimersB + " diff: " + (cycles - nextTimersB));
      }

      if ((ciaMemory[CIACRB] & 0x08) == 0) {
	if ((ciaMemory[CIACRB] & TIMER_B_DIV_MASK) == TIMER_B_DIV_VAL) {
	  // count timer A triggers! => DIV by latchA
	  nextTimersB = nextTimersB + latchValue[1] * latchValue[0];
	} else {
	  nextTimersB = nextTimersB + latchValue[1];
	}
	// some delay...
	nextTimersB += 2;
      } else {
	timersONB = false;
	cpuMemory[offset + CIACRB] &= 0xfe; // clear bit 0 to indicate stop
// 	ciaMemory[0x0f] &= 0xfe;
      }
    }

    // Should be real IRQ triggers, and Clears insead of this...
    trigg += timerATrigg + timerBTrigg;

    // Trigg the stuff...
    if (trigg > 0) {
      ciaicrRead |= 0x80;
      if (TIMER_DEBUG) {
	if (trigg != lastTrigg) {
	  println("Trigger IRQ: " + trigg);
	}
	lastTrigg = trigg;
      }
    }

    if (cycles > nextUpdate) {
      nextUpdate = nextUpdate + 100000;
      int tmp = (cpuMemory[offset + TODTEN] & 0x0f) + 1;
      cpuMemory[offset + TODTEN] = tmp % 10;
      if (tmp > 9) {
	// Maxval == 0x59
	tmp = (cpuMemory[offset + TODSEC] & 0x7f) + 1;
	if ((tmp & 0x0f) > 9)
	  tmp += 0x06;
	if (tmp > 0x59)
	  tmp = 0;
	cpuMemory[offset + TODSEC] = tmp;
	// Wrapped seconds...
	// Minutes inc - max 0x59
	if (tmp == 0) {
	  tmp = (cpuMemory[offset + TODMIN] & 0x7f) + 1;
	  if ((tmp & 0x0f) > 9)
	    tmp += 0x06;
	  if (tmp > 0x59)
	    tmp = 0;
	  cpuMemory[offset + TODMIN] = tmp;

	  // Hours, max 0x12
	  if (tmp == 0) {
	    tmp = (cpuMemory[offset + TODHRS] & 0x1f) + 1;
	    if ((tmp & 0x0f) > 9)
	      tmp += 0x06;

	    if (tmp > 0x11)
	      tmp = 0;

	    // how is hour? 1  - 12  or 0 - 11 ??
	    cpuMemory[offset + TODHRS] = tmp;
	  }
	}
      }
    }

    if (timersONA && nextTimersA < nextCIAUpdate)
      nextCIAUpdate = nextTimersA;
    if (timersONB && nextTimersB < nextCIAUpdate)
      nextCIAUpdate = nextTimersB;

    return trigg;
  }

  public int performRead(int address, long cycles) {
    int val = cpuMemory[address];

    address -= offset;

    //    int val = ciaMemory[address];
    //    val = ciaMemory[address];

    switch(address) {
    case TIMALO:
      if (timersONA) {
	if (nextTimersA > cycles)
	  return (int) (nextTimersA - cycles) & 0xff;
	else
	  return 0; // and update???
      } else {
	return ta & 0xff;
      }
    case TIMAHI:
      if (timersONA) {
	if (nextTimersA > cycles)
	  return (int) (nextTimersA - cycles) >> 8;
	else
	  return 0; // and update???
      } else {
	return ta >> 8;
      }
    case TIMBLO:
      if (timersONB) {
	if (nextTimersB > cycles)
	  return (int) (nextTimersB - cycles) & 0xff;
	else
	  return 0; // and update???
      } else {
	return tb & 0xff;
      }
    case TIMBHI:
      if (timersONB) {
	if (nextTimersB > cycles)
	  return (int) (nextTimersB - cycles) >> 8;
	else
	  return 0; // and update???
      } else {
	return tb >> 8;
      }


    case CIAICR:
      // Clear interrupt register (flags)!
      if (TIMER_DEBUG) println("clear Interrupt register, was: " +
			       Hex.hex2(ciaicrRead));
      val = ciaicrRead;
      ciaicrRead = 0;
      timerATrigg = 0;
      timerBTrigg = 0;

      // Latch off the IRQ/NMI immediately!!!
      if (offset == 0x10c00) {
	chips.clearIRQ(C64Chips.CIA_TIMER_IRQ);
      } else {
	chips.clearNMI(C64Chips.CIA_TIMER_NMI);
      }
    }
    return val;
  }

  public void performWrite(int address, int data, long cycles) {
    address -= offset;

    if (WRITE_DEBUG) println(ciaID() + " Write to :" +
			     Integer.toString(address, 16) + " = " +
			     Integer.toString(data, 16));

    switch (address) {
      //monitor.println("Set Keyboard:" + data);
    case PRA:
      break;
    case PRB:
      break;
    case TIMALO:
      // Update latch value
      latchValue[0] = (latchValue[0] & 0xff00) | data;
      break;
    case TIMAHI:
      latchValue[0] = (latchValue[0] & 0xff) | (data << 8);
//       println(ciaID() +
// 			 ": Timer A latch: " + latchValue[0] + ": " + cycles);
      break;
    case TIMBLO:
      latchValue[1] = (latchValue[1] & 0xff00) | data;
      break;
    case TIMBHI:
      latchValue[1] = (latchValue[1] & 0xff) | (data << 8);
//       println(ciaID() +
// 			 ": Timer B latch: " + latchValue[1] + ": " + cycles);
      break;
      // dc0d - CIAICR - CIA Interrupt Control Register
    case CIAICR:
      boolean val = (data & 0x80) != 0;
      if ((data & 1) == 1) {
	timersIRQ[0] = val;
	if (TIMER_DEBUG) println(" Timer A irq :" + val);
      }
      if ((data & 2) == 2) {
	timersIRQ[1] = val;
	if (TIMER_DEBUG) println(" Timer B irq :" + val);
      }

      // Off with the triggers???
      timerATrigg = 0;
      timerBTrigg = 0;
      ciaicrRead = 0;

      if (val) {
	// Set the 1 bits if val = 1
	ciaMemory[CIAICR] |= data & 0x7f;
      } else {
	// Clear the 1 bits
	ciaMemory[CIAICR] &= ~(data & 0x7f);
      }
      return;
    case CIACRA:
      // bit 4 = force load latch value
      if ((data & 0x10) != 0) {
	latchValue[0] = ciaMemory[4] + (ciaMemory[5] << 8);
	timerATrigg = 0;
      }

      if ((data & 1) == 1) {
	timersONA = true;
	timerATrigg = 0;
	nextTimersA = cycles + latchValue[0];
	if (TIMER_DEBUG) println("Starting timer A: " +
					    latchValue[0] +
					    " OneShot: " +((data & 0x08) > 0)
					    + " triggTime: " + nextTimersA);
      } else {
	timersONA = false;
	timerATrigg = 0;
	ta = (int)(nextTimersA - cycles);
	if (ta < 0) ta = 0;
 	if (TIMER_DEBUG) println("Stopping timer A: ");
      }

      if ((data & 0x02) != 0) {
	println("******* Timer A shows on dc01 b6 - mode: " +
			   (data & 0x04));
      }

      break;

    case CIACRB:
      // bit 4 = force load latch value
      if ((data & 0x10) != 0) {
	latchValue[1] = ciaMemory[6] + (ciaMemory[7] << 8);
	timerBTrigg = 0;
      }

      if ((data & 1) == 1) {
	timersONB = true;
	nextTimersB = cycles + latchValue[1];
	timerBTrigg = 0;
// 	println("Starting timer B: " + latchValue[1]);
      } else {
	timersONB = false;
	timerBTrigg = 0;
	tb = (int) (nextTimersB - cycles);
	if (tb < 0) tb = 0;
	if (TIMER_DEBUG) println("Stopping timer B: ");
      }

      if ((data & 0x02) != 0) {
	println("******* Timer B shows on dc01 b6 - mode: " +
			   (data & 0x04));
      }
      break;
    }

    // Write to the CIA memory but not CPU memory (to always have the
    // by the CPU written config stored...
    ciaMemory[address] = data;

  }

  private void println(String s) {
    System.out.println(ciaID() + ": " + s);
  }

  public void printStatus() {
    System.out.println("--------------------------");
    println(" status");
    System.out.println("Timer A on: " + timersONA);
    System.out.println("Timer A next trigger: " + nextTimersA);
    System.out.println("CIA CRA: " + Hex.hex2(ciaMemory[CIACRA]) + " => " +
		       (((ciaMemory[CIACRA] & 0x08) == 0) ?
		       "cont" : "one-shot"));
    System.out.println("Timer A Interrupt: " + timerATrigg);
    System.out.println("Timer A Latch: " + latchValue[0]);
    System.out.println("Timer B on: " + timersONB);
    System.out.println("Timer B next trigger: " + nextTimersB);
    System.out.println("CIA CRB: " + Hex.hex2(ciaMemory[CIACRB]) + " => " +
		       (((ciaMemory[CIACRB] & 0x08) == 0) ?
		       "cont" : "one-shot"));
    System.out.println("Timer B Interrupt: " + timerBTrigg);
    System.out.println("Timer B Latch: " + latchValue[1]);
    System.out.println("--------------------------");
  }


}
