/**
 * This file is a part of JaC64 - a Java C64 Emulator
 * Main Developer: Joakim Eriksson (Dreamfabric.com)
 * Contact: joakime@sics.se
 * Web: http://www.dreamfabric.com/c64
 * ---------------------------------------------------
 */

package com.dreamfabric.jac64;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.applet.*;
import javax.swing.JPanel;

/**
 * Implements the VIC chip + some other HW
 *
 * @author  Joakim Eriksson (joakime@sics.se) / main developer, still active
 * @author  Jan Blok (jblok@profdata.nl) / co-developer during ~2001
 * @version $Revision: 1.11 $, $Date: 2006/05/02 16:26:26 $
 */

public class C64Screen extends C64Chips implements Observer {
  public static final String version = "1.0 Beta1";

  public static final int SERIAL_ATN = (1 << 3);
  public static final int SERIAL_CLK_OUT = (1 << 4);
  public static final int SERIAL_DATA_OUT = (1 << 5);
  public static final int SERIAL_CLK_IN = (1 << 6);
  public static final int SERIAL_DATA_IN = (1 << 7);

  public static final boolean IRQDEBUG = false;
  public static final boolean SPRITEDEBUG = false;
  public static final boolean IODEBUG = false;
  public static final boolean VIC_MEM_DEBUG = false;
  public static final boolean BAD_LINE_DEBUG = false;
  public static final boolean STATE_DEBUG = false;
  public static final boolean DEBUG_IEC = false;

  public static final int IO_UPDATE = 17;
  // This is PAL speed! - will be called each scan line...

  private static final int VIC_IRQ = 1;
  private static final int CIA_TIMER_IRQ = 2;
  private static final int CIA_TIMER_NMI = 2;

  // This might be solved differently later!!!
  public static final int CYCLES_PER_LINE = VICConstants.SCAN_RATE;

  // ALoow the IO to write in same as RAM
  public static final int IO_OFFSET = CPU.IO_OFFSET;
  public static final boolean SOUND_AVAIABLE = true;

  public static final Color TRANSPARENT_BLACK = new Color(0, 0, 0x40, 0x40);
  public static final Color DARKER_0 = new Color(0, 0, 0x40, 0x20);
  public static final Color LIGHTER_0 = new Color(0xe0, 0xe0, 0xff, 0x30);
  public static final Color DARKER_N = new Color(0, 0, 0x40, 0x70);
  public static final Color LIGHTER_N = new Color(0xe0, 0xe0, 0xff, 0xa0);

  public static final Color LED_ON = new Color(0x60, 0xdf, 0x60, 0xc0);
  public static final Color LED_OFF = new Color(0x20, 0x60, 0x20, 0xc0);
  public static final Color LED_BORDER = new Color(0x40, 0x60, 0x40, 0xa0);


  public static final int LABEL_COUNT = 32;
  private Color[] darks = new Color[LABEL_COUNT];
  private Color[] lites = new Color[LABEL_COUNT];
  private int colIndex = 0;

  // This is the screen widht and height used...
  private final static int SC_WIDTH = 384; //403;
  private final static int SC_HEIGHT = 284;
  private final int SC_XOFFS = 32;
  private final int SC_SPXOFFS = SC_XOFFS - 24;
  private final int FIRST_VISIBLE_VBEAM = 15;
  private final int SC_SPYOFFS = FIRST_VISIBLE_VBEAM + 1;


  private boolean soundOn = true;

  private IMonitor monitor;

  // 20000 us per scan
  private int scanRate = 50;
  private int targetScanTime = 20000;
  private int actualScanTime = 20000;
  private int sidUpdate = 1000;
  private long lastScan = 0;
  private long nextScanLine = 0;
  private long nextIOUpdate = 0;
  private boolean DOUBLE = false;
  private int reset = 100;
  private C64Canvas canvas;

  private int[] memory;

  private Keyboard keyboard;

  SID6581 sid[];
  SIDMixer mixer;
  CIA cia[];
  //  C1541 c1541;
  C1541Chips c1541Chips;

  int iecLines = 0;
  // for disk emulation...
  int cia2PRA = 0;
  int cia2DDRA = 0;

  AudioClip trackSound = null;
  AudioClip motorSound = null;

  private int lastTrack = 0;
  private int lastSector = 0;
  private boolean ledOn = false;
  private boolean motorOn = false;


  // This is an IEC emulation (non ROM based)
  boolean emulateDisk = false; //true; //!CPU.EMULATE_1541; // false;

  private int[] cbmcolor = VICConstants.COLOR_SETS[0];

  // -------------------------------------------------------------------
  // VIC-II variables
  // -------------------------------------------------------------------
  public int vicBank;
  public int charSet;
  public int videoMatrix;
  public int videoMode;

  private int vicBase = 0;
  private boolean badLine = false;
  private int spr0BlockSel;

  // New type of position in video matrix - Video Counter (VIC II docs)
  int vc = 0;
  int vcBase = 0;
  int rc = 0;
  int vmli = 0;
  // The current vBeam pos - 9... => used for keeping track of memory
  // position to write to...
  int vPos = 0;
  int mpos = 0;

  // Cached variables...
  boolean gfxVisible = false;
  boolean paintBorder = false;

  int borderColor = cbmcolor[0];
  int bgColor = cbmcolor[1];
  int irqMask = 0;
  int irqFlags = 0;

  boolean extended = false;
  boolean multiCol = false;
  boolean blankRow = false;
  boolean hideColumn = false;

  int multiColor[] = new int[4];

  // 48 extra for the case of an expanded sprite byte
  int collissionMask[] = new int[SC_WIDTH + 48];

  Sprite sprites[] = new Sprite[8];

  private Color colors[] = null;

  private int horizScroll = 0;
  private int vScroll = 0;


  private Image image;
  private Graphics g2;

  private boolean debug = false;

  // The font is in a copy in "ROM"...
  private int charMemoryIndex = 0;

  // Caching all 40 chars (or whatever) each "bad-line"
  private int[] vicCharCache = new int[40];
  private int[] vicColCache = new int[40];

  // This is the vicMemory where data written exists...
  public int[] vicMemory = new int[0x1000];

  public Image screen = null;
  private Graphics scrGfx = null;
  private MemoryImageSource mis = null;

  // The array to generate the screen in Extra rows for sprite clipping
  // And for clipping when scrolling (smooth)
  int mem[] = new int[SC_WIDTH * (SC_HEIGHT + 10)];

  int rnd = 754;
  String message;
  String tmsg = "";

  int vbeam = 0;
  boolean updating = false;
  boolean displayEnabled = true;
  boolean irqTriggered = false;
  long nextSID = 0;
  long lastLine = 0;
  long firstLine = 0;
  long lastIRQ = 0;

  public C64Screen(IMonitor m, boolean dob) {
    monitor = m;
    DOUBLE = dob;

    setScanRate(50);
    makeColors(darks, DARKER_0, DARKER_N);
    makeColors(lites, LIGHTER_0, LIGHTER_N);
  }

  private void makeColors(Color[] colors, Color c1, Color c2) {
    int a0 = c1.getAlpha();
    int r0 = c1.getRed();
    int g0 = c1.getGreen();
    int b0 = c1.getBlue();
    int an = c2.getAlpha();
    int rn = c2.getRed();
    int gn = c2.getGreen();
    int bn = c2.getBlue();
    int lc = LABEL_COUNT / 2;
    for (int i = 0, n = lc; i < n; i++) {
      colors[i] =
	colors[LABEL_COUNT - i - 1] =
	new Color(((lc - i) * r0 + (i * rn)) / lc,
		  ((lc - i) * g0 + (i * gn)) / lc,
		  ((lc - i) * b0 + (i * bn)) / lc,
		  ((lc - i) * a0 + (i * an)) / lc);
    }
  }

  public void setDouble(boolean dob) {
    DOUBLE = dob;
    image = null;
  }

  public void setColorSet(int c) {
    if (c >= 0 && c < VICConstants.COLOR_SETS.length) {
      cbmcolor = VICConstants.COLOR_SETS[c];
    }
  }

  public SIDMixer getMixer() {
    return mixer;
  }

  public SID6581[] getSIDs() {
    return sid;
  }

  public CIA[] getCIAs() {
    return cia;
  }

  public void setScanRate(double hertz) {
    // Scan time for 10 scans...
    targetScanTime = (int) (1000000 / hertz);
    // Cycles 10000000 / sec, but scan each 65 =>
    // 65 x 312 = 1014000
    scanRate = (int) hertz;
    float diff = 1.0f * VICConstants.SCAN_RATE / 65;
    sidUpdate = (int) (diff * 1000 * hertz / 50);
  }

  public int getScanRate() {
    return (1000000 / targetScanTime);
  }

  public int getActualScanRate() {
    // This should be calculated... if it is too slow it will be
    // shown here
    return (1000000 / actualScanTime);
  }

  public JPanel getScreen() {
    return canvas;
  }

  public boolean ready() {
    return keyboard.ready;
  }

  public void dumpGfxStat() {
    monitor.info("Char MemoryIndex: 0x" +
		       Integer.toString(charMemoryIndex, 16));
    monitor.info("CharSet adr: 0x" +
		       Integer.toString(charSet, 16));
    monitor.info("VideoMode: " + videoMode);
    monitor.info("Vic Bank: 0x" +
		       Integer.toString(vicBank, 16));
    monitor.info("Video Matrix: 0x" +
		       Integer.toString(videoMatrix, 16));

    monitor.info("Text: extended = " + extended +
		 " multicol = " + multiCol);

    monitor.info("24 Rows on? " +
		       (((memory[IO_OFFSET + 0xd011] & 0x08) == 0) ? "yes" : "no"));

    monitor.info("YScroll = " + (memory[IO_OFFSET + 0xd011] & 0x7));
    monitor.info("d011 = " + memory[IO_OFFSET + 0xd011]);

    monitor.info("IRQ Latch: " +
		       Integer.toString(memory[IO_OFFSET + 0xd019], 16));
    monitor.info("IRQ  Mask: " +
		       Integer.toString(memory[IO_OFFSET + 0xd01a], 16));
    monitor.info("IRQ RPos : " +
		       (vicMemory[0x12] + ((vicMemory[0x11] & 0x80) << 1)));

    for (int i = 0, n = 8; i < n; i++) {
      monitor.info("Sprite " + (i + 1) + " pos = " +
			 (memory[IO_OFFSET + 0xd000 + i * 2] +
			  ((memory[IO_OFFSET + 0xd010] & (1 << i)) != 0
			   ? 256 : 0)) + ", " +
			 memory[IO_OFFSET + 0xd000 + i * 2 + 1]);
    }


    monitor.info("IRQFlags: " + getIRQFlags());
    monitor.info("NMIFlags: " + getNMIFlags());

  }

  public void setSoundOn(boolean on) {
    soundOn = on;
    mixer.setSoundOn(on);
  }

  public void setStick(boolean one) {
    keyboard.setStick(one);
  }

  public void registerHotKey(int key, String script, Object o) {
    keyboard.registerHotKey(key, script, o);
  }

  public void setKeyboardEmulation(boolean extended) {
    monitor.info("Keyboard extended: " + extended);

    keyboard.stickExits = !extended;
    keyboard.extendedKeyboardEmulation = extended;
  }

  public void init(CPU cpu) {
    super.init(cpu);

    this.memory = cpu.getMemory();

    c1541Chips = cpu.getDrive().chips;
    c1541Chips.initIEC2(this);
    c1541Chips = cpu.getDrive().chips;
    c1541Chips.setObserver(this);


    for (int i = 0, n = sprites.length; i < n; i++) {
      sprites[i] = new Sprite();
      sprites[i].spriteNo = i;
    }

    cia = new CIA[2];
    cia[0] = new CIA(memory, IO_OFFSET + 0xdc00, this);
    cia[1] = new CIA(memory, IO_OFFSET + 0xdd00, this);
//     c1541 = new C1541(memory);
//     c1541.addObserver(this);

    keyboard = new Keyboard(this, cia[0], memory);
    canvas = new C64Canvas(this, DOUBLE, keyboard);


    if (SOUND_AVAIABLE) {

      try {

	sid = new SID6581[3];
	sid[0] = new SID6581(memory, IO_OFFSET + 0xd400 );
	sid[0].init();

	sid[1] = new SID6581(memory, IO_OFFSET + 0xd400 + 7);
	sid[1].init();

	sid[2] = new SID6581(memory, IO_OFFSET + 0xd400 + 14);
	sid[2].init();

	sid[0].next = sid[2];
	sid[1].next = sid[0];
	sid[2].next = sid[1];

	mixer = new SIDMixerSE(sid, null);

    } catch (Throwable e) {
      monitor.error("Error while initializing SID chip" + e);
      sid = null;
    }
    }

    charMemoryIndex = CPU.CHAR_ROM2;

    for (int i = 0; i < SC_WIDTH * SC_HEIGHT; i++) {
      mem[i] = cbmcolor[6];
    }

    mis = new MemoryImageSource(SC_WIDTH, SC_HEIGHT, mem, 0, SC_WIDTH);

    mis.setAnimated(true);
    mis.setFullBufferUpdates(true);
    screen = canvas.createImage(mis);

    //to fix bug http://developer.java.sun.com/developer/bugParade/bugs/4464723.html
//     setRequestFocusEnabled(true);
//     addMouseListener(new MouseAdapter() {
// 	public void mousePressed(MouseEvent e) {
// 	  requestFocus();
// 	}
//       });
    initUpdate();
  }

  public void update(Object src, Object data) {
    if (src != c1541Chips) {
      // Print some kind of message...
      message = (String) data;
    } else {
      updateDisk(src, data);
    }
  }

  int frq;

  void restoreKey(boolean down) {
    if (down) setNMI(KEYBOARD_NMI);
    else clearNMI(KEYBOARD_NMI);
  }

  // Should be checked up!!!
  private static final int[] IO_ADDRAND = new int[]
    { 0xd03f, 0xd03f, 0xd03f, 0xd03f,
      0xd41f, 0xd41f, 0xd41f, 0xd41f,
      0xd8ff, 0xd9ff, 0xdaff, 0xdbff, // Color ram
      0xdc0f, 0xdd0f, 0xdeff, 0xdfff, // CIA + Expansion...
    };


  private int lastRead = 0;
  public int performRead(int address, long cycles) {
    // dX00 => and address
    // d000 - d3ff => &d063
    int pos = (address >> 8) & 0xf;
    //    monitor.info("Address before: " + address);
    address = address & IO_ADDRAND[pos];

    // Do stuff... - still the old way...
    int val = memory[address + IO_OFFSET];

    switch (address) {
      // Sprite collission registers - zeroed after read!
    case 0xd01f:
    case 0xd01e:
//       if (memory[address] != 0)
      // This means that an interrupt can occur again (e.g. when it is 0)
      val = vicMemory[address & 0xff];

      if (SPRITEDEBUG)
	monitor.info("Reading sprite collission: " +
		     Integer.toString(address, 16) + " => " + val);

      vicMemory[address & 0xff] = 0;
      return val;
    case 0xd019:
      if (SPRITEDEBUG)
	monitor.info("Reading d019: " + memory[address + IO_OFFSET]);
      return irqFlags;
    case 0xd01a:
      return irqMask;
//     case 0xd012:
//       System.out.println("Reading vbeam: " + vbeam + " d012: " +
// 			 memory[address + IO_OFFSET]);
//       break;

    case 0xd41b: //
      return (sid[2].lastSample()) & 0xff;
    case 0xd41c: //
      return sid[2].adsrVol & 0xff;
    case 0xd419:
      // Max resistance - PAD
      memory[IO_OFFSET + 0xd419]++; // = 0xff;
//       monitor.info("/////  READ PAD X");
      break;
    case 0xd41A:
      // Max resistance - PAD
      memory[IO_OFFSET + 0xd41A] = 0xff;
//       monitor.info("/////  READ PAD Y");
      break;
    case 0xdc00:
      return keyboard.readDC00(cpu.lastReadOP);
    case 0xdc01:
      return keyboard.readDC01(cpu.lastReadOP);
    case 0xdd00:
      //       System.out.print("Read dd00 IEC1: ");
      // Try the frodo way... again...
      val = (cia2PRA | ~cia2DDRA) & 0x3f
	| iecLines & c1541Chips.iecLines;

      val &= 0xff;
      return val;
    default:
    }


    if (pos == 0xd) {
      return cia[1].performRead(address + IO_OFFSET, cycles);
    } if (pos == 0xc) {
      return cia[0].performRead(address + IO_OFFSET, cycles);
    }

    return val;
  }


  public void updateCIA(long cycles) {
    int data = 0;
    // Handling of the CIA's
    if (nextIOUpdate < cycles) {
      if ((data = ioTic()) != 0) {
	if ((data & 1) == 1) {
// 		  monitor.info("**** IRQ Interrupt (CIA 1) disableI:" +
// 				     disableInterupt +
// 				     " irqs: " + interruptInExec);
	  setIRQ(CIA_TIMER_IRQ);
	}
	if ((data & 2) == 2) {
// 		monitor.info("**** NMI Interrupt (CIA 2) disableI: " +
// 				   disableInterupt +
// 				   " irqs: " + interruptInExec);
	  setNMI(CIA_TIMER_NMI);
	}
      } else {
	// This is not setting NMILow!!! - later with multiple
	// Sources it can not be like this...
	clearIRQ(CIA_TIMER_IRQ);
	clearNMI(CIA_TIMER_NMI);
      }

      nextIOUpdate += IO_UPDATE;
    }
    if (nextIOUpdate > cia[0].nextCIAUpdate)
      nextIOUpdate = cia[0].nextCIAUpdate;
    if (nextIOUpdate > cia[1].nextCIAUpdate)
      nextIOUpdate = cia[1].nextCIAUpdate;
  }

  public void performWrite(int address, int data, long cycles) {
    int pos = (address >> 8) & 0xf;
    address = address & IO_ADDRAND[pos];

//     monitor.info("Wrote to Chips at " + Integer.toString(address, 16)
// 		       + " = " + Integer.toString(data, 16));

    // Store in the memory given by "CPU"
    memory[address + IO_OFFSET] = data;

//     if (address >= 0xd800 && address < 0xdc00) {
//       int p = address - 0xd800;
//       System.out.println("### Write to color ram: " + (p % 40) + "," + p/40 +
// 			 " = " + data);
//     }


    address = address + IO_OFFSET;

    // The "old" switch

    switch (address) {

  // -------------------------------------------------------------------
  // SID related
  // -------------------------------------------------------------------
    case IO_OFFSET + 0xd404 :
       sid[0].setControl(data, cpu.cycles);
       break;
     case IO_OFFSET + 0xd400 + 5:
       sid[0].setAD(data, cpu.cycles);
       break;
     case IO_OFFSET + 0xd400 + 6:
       sid[0].setSR(data, cpu.cycles);
       break;
    case IO_OFFSET + 0xd402:
    case IO_OFFSET + 0xd403:
      sid[0].updatePulseWidth(cycles);
      break;


     case IO_OFFSET + 0xd40b :
       sid[1].setControl(data, cpu.cycles);
       break;
     case IO_OFFSET + 0xd407 + 5:
       sid[1].setAD(data, cpu.cycles);
       break;
     case IO_OFFSET + 0xd407 + 6:
       sid[1].setSR(data, cpu.cycles);
       break;
    case IO_OFFSET + 0xd402 + 7:
    case IO_OFFSET + 0xd403 + 7:
      sid[1].updatePulseWidth(cycles);
      break;



     case IO_OFFSET + 0xd412 :
       sid[2].setControl(data, cpu.cycles);
       break;
     case IO_OFFSET + 0xd40e + 5:
       sid[2].setAD(data, cpu.cycles);
       break;
     case IO_OFFSET + 0xd40e + 6:
       sid[2].setSR(data, cpu.cycles);
       break;

    case IO_OFFSET + 0xd402 + 14:
    case IO_OFFSET + 0xd403 + 14:
      sid[2].updatePulseWidth(cycles);
      break;


       // Controls for the SID Mixer
     case IO_OFFSET + 0xd415:
       mixer.setFilterCutoffLO(data & 7);
       break;
     case IO_OFFSET + 0xd416:
       mixer.setFilterCutoffHI(data);
       break;
     case IO_OFFSET + 0xd417:
       mixer.setFilterResonance(data >> 4);
       mixer.setFilterOn(data & 0x0f);
       break;
     case IO_OFFSET + 0xd418 :
       mixer.setVolume(data & 0x0f, cpu.cycles);
       mixer.setFilterCtrl(data);
       break;

  // -------------------------------------------------------------------
  // VIC related
  // -------------------------------------------------------------------
    case IO_OFFSET + 0xd000:
    case IO_OFFSET + 0xd002:
    case IO_OFFSET + 0xd004:
    case IO_OFFSET + 0xd006:
    case IO_OFFSET + 0xd008:
    case IO_OFFSET + 0xd00a:
    case IO_OFFSET + 0xd00c:
    case IO_OFFSET + 0xd00e:
      int sprite = (address - IO_OFFSET - 0xd000) >> 1;
      sprites[sprite].x &= 0x100;
      sprites[sprite].x += data;
      break;
    case IO_OFFSET + 0xd010:
      for (int i = 0, m = 1, n = 8; i < n; i++, m = m << 1) {
	sprites[i].x &= 0xff;
	sprites[i].x |= (data & m) != 0 ? 0x100 : 0;
      }
      break;

       // d011 -> high address of raster pos
     case IO_OFFSET + 0xd011 :
       vicMemory[0x11] = data;
 //       monitor.info("Setting blank: " +
 // 			 ((memory[IO_OFFSET + 0xd011] & 0x08) == 0) +
 // 			 " at " + vbeam);

       if (vScroll != (data & 7)) {
	 // update vScroll and badLine!
	 vScroll = data & 0x7;
	 boolean oldBadLine = badLine;
	 badLine =
	   (displayEnabled && vbeam >= 0x30 && vbeam <= 0xf7) &&
	   (vbeam & 0x7) == vScroll;
	 if (BAD_LINE_DEBUG && oldBadLine != badLine)
	   monitor.info("#### BadLC diff@" + vbeam + " => " +
			badLine + " vScroll: " + vScroll +
			" vmli: " + vmli + " vc: " + vc +
			" rc: " + rc + " cyc line: " +
			(cpu.cycles - lastLine) +
			" cyc IRQ: " + (cpu.cycles - lastIRQ));
	 if (oldBadLine != badLine) {
	   System.out.println("Changed badline to: " + badLine + " at " +
			      (cpu.cycles - lastLine));
	 }

       }

       extended = (data & 0x40) != 0;
       blankRow = (data & 0x08) == 0;

       // 000 => normal text, 001 => multicolor text
       // 010 => extended text, 011 => illegal mode...
       // 100 => hires gfx, 101 => multi hires
       // 110, 111 => ?
       videoMode = (extended ? 0x02 : 0)
	 | (multiCol ? 0x01 : 0) | (((data & 0x20) != 0) ? 0x04 : 0x00);

//        System.out.println("Extended set to: " + extended + " at " +
// 			  vbeam + " d011: " + Hex.hex2(data));

       if (VIC_MEM_DEBUG || BAD_LINE_DEBUG) {
	 monitor.info("d011 = " + data + " at " + vbeam +
			    " => YScroll = " + (data & 0x7) +
			    " cyc since line: " + (cpu.cycles-lastLine) +
			    " cyc since IRQ: " + (cpu.cycles-lastIRQ));
       }

       // Restore!
       memory[IO_OFFSET + 0xd011] =
	 (memory[IO_OFFSET + 0xd011] & 0x7f) | ((vbeam & 0x100) >> 1);
       if (IRQDEBUG)
 	 monitor.info("Setting raster position (hi) to: " +
		      (data & 0x80));

       break;

       // d012 -> raster position
     case IO_OFFSET + 0xd012 :
       vicMemory[0x12] = data;
       if (IRQDEBUG)
	 monitor.info("Setting Raster Position (low) to " + data);

       // Restore!
       memory[address] = vbeam & 255;
       break;
    case IO_OFFSET + 0xd015:
      for (int i = 0, m = 1, n = 8; i < n; i++, m = m << 1) {
	sprites[i].enabled = (data & m) != 0;
      }
      break;

    case IO_OFFSET + 0xd016:
      horizScroll = data & 0x7;
      multiCol = (data & 0x10) != 0;
      hideColumn = (data & 0x08) == 0;

      // Set videmode...
      videoMode = (extended ? 0x02 : 0)
	| (multiCol ? 0x01 : 0) | (((memory[IO_OFFSET + 0xd011] & 0x20) != 0)
				   ? 0x04 : 0x00);

//       System.out.println("HorizScroll set to: " + horizScroll + " at "
// 			 + vbeam);

//       System.out.println("MultiColor set to: " + multiCol + " at " + vbeam);
      break;

    case IO_OFFSET + 0xd017:
      for (int i = 0, m = 1, n = 8; i < n; i++, m = m << 1) {
	sprites[i].expandY = (data & m) != 0;
      }
      break;

    case IO_OFFSET + 0xd019 : {
      if ((data & 0x80) != 0) data = 0xff;
      int latchval = 0xff ^ data;
      if (IRQDEBUG)
	monitor.info("Latching VIC-II: " + Integer.toString(data, 16)
		     + " on " + Integer.toString(memory[address], 16) +
		     " 0x19 = " + Integer.toString(vicMemory[0x19], 16) +
		     " latch: " + Integer.toString(latchval, 16));

      irqFlags &= latchval;
      vicMemory[0x19] = memory[address] = irqFlags;
//       monitor.info("After Latch: " + memory[address]);

      // Is this "flagged" off?
      if ((irqMask & 0x0f & irqFlags) == 0) {
	clearIRQ(VIC_IRQ);
      }
    }
      break;
    case IO_OFFSET + 0xd01a:
      irqMask = data;

      // Check if IRQ should trigger or clear!
      if ((irqMask & 0x0f & irqFlags) != 0) {
	irqFlags |= 0x80;
	setIRQ(VIC_IRQ);
      } else {
	clearIRQ(VIC_IRQ);
      }

      if (IRQDEBUG) {
	monitor.info("Changing IRQ mask to: " +
		     Integer.toString(irqMask, 16) + " vbeam: " + vbeam);
      }
      break;

    case IO_OFFSET + 0xd01d:
      for (int i = 0, m = 1, n = 8; i < n; i++, m = m << 1) {
	sprites[i].expandX = (data & m) != 0;
      }
      break;

    case IO_OFFSET + 0xd01b:
//       System.out.println("Set sprite fg priority to: " +
// 			 Integer.toString(data, 2) + " at " + vbeam);
      for (int i = 0, m = 1, n = 8; i < n; i++, m = m << 1) {
	sprites[i].priority = (data & m) != 0;
      }
      break;

    case IO_OFFSET + 0xd01c:
      for (int i = 0, m = 1, n = 8; i < n; i++, m = m << 1) {
	sprites[i].multicolor = (data & m) != 0;
      }
      break;


    case IO_OFFSET + 0xd020:
      borderColor = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd021:
      bgColor = cbmcolor[data & 15];
      for (int i = 0, n = 8; i < n; i++) {
	sprites[i].color[0] = bgColor;
      }
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd025:
      for (int i = 0, n = 8; i < n; i++) {
	sprites[i].color[1] = cbmcolor[data & 15];
      }
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd026:
      for (int i = 0, n = 8; i < n; i++) {
	sprites[i].color[3] = cbmcolor[data & 15];
      }
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd027:
      sprites[0].color[2] = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd028:
      sprites[1].color[2] = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd029:
      sprites[2].color[2] = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd02a:
      sprites[3].color[2] = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd02b:
      sprites[4].color[2] = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd02c:
      sprites[5].color[2] = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd02d:
      sprites[6].color[2] = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
    case IO_OFFSET + 0xd02e:
      sprites[7].color[2] = cbmcolor[data & 15];
      memory[IO_OFFSET + address] |= 0xf0;
      break;
      // CIA 1 & 2 - 'special' addresses
    case IO_OFFSET + 0xdc00:
    case IO_OFFSET + 0xdc01:
    case IO_OFFSET + 0xdc02:
    case IO_OFFSET + 0xdc03:
 //       monitor.println("////////////==> Set Keyboard:" +
 // 		      Integer.toString(address - IO_OFFSET,16) + " = " + data +
 // 		      " => ~" + ((~data) & 0xff));
       cia[0].performWrite(address, data, cpu.cycles);
       keyboard.updateKeyboard();
       break;
    case IO_OFFSET + 0xdd00:

       if (DEBUG_IEC)
	 monitor.info("C64: IEC Write: " + Integer.toHexString(data));

//        if (emulateDisk) {
// 	 c1541.handleDisk(data, cpu.cycles);
//        }

       cia[1].performWrite(address, data, cpu.cycles);
       setVideoMem();

       int old = cia2PRA;
       cia2PRA = data;

       data = ~cia2PRA & cia2DDRA;
       int oldLines = iecLines;
       iecLines = (data << 2) & 0x80	// DATA
	 | (data << 2) & 0x40		// CLK
	 | (data << 1) & 0x10;		// ATN

       if (((oldLines ^ iecLines) & 0x10) != 0) {
	 c1541Chips.atnChanged((iecLines & 0x10) == 0);
       }

       if (DEBUG_IEC) printIECLines();
       break;

    case IO_OFFSET + 0xdd02:
      cia2DDRA = data;
      System.out.println("C64: Wrote to DDRA (IEC): " +
			 Integer.toHexString(data));
      cia[1].performWrite(address, data, cpu.cycles);
      break;

     case IO_OFFSET + 0xd018:
       memory[address] = data;
       setVideoMem();
       break;

    default:
      if (pos == 0xd) {
	cia[1].performWrite(address, data, cycles);
      } if (pos == 0xc) {
	cia[0].performWrite(address, data, cycles);
      }
    }
   }

  private void printIECLines() {
    System.out.print("IEC/F: ");
    if ((iecLines & 0x10) == 0) {
      System.out.print("A1");
    } else {
      System.out.print("A0");
    }

    // The c64 has id = 1
    int sdata = ((iecLines & 0x40) == 0) ? 1 : 0;
    System.out.print(" C" + sdata);
    sdata = ((iecLines & 0x80) == 0) ? 1 : 0;
    System.out.print(" D" + sdata);

    // The 1541 has id = 2
    sdata = ((c1541Chips.iecLines & 0x40) == 0) ? 1 : 0;
    System.out.print(" c" + sdata);
    sdata = ((c1541Chips.iecLines & 0x80) == 0) ? 1 : 0;
    System.out.print(" d" + sdata);

    System.out.println(" => C" +
		       ((iecLines & c1541Chips.iecLines & 0x80) == 0 ? 1 : 0)
		       + " D" +
		       ((iecLines & c1541Chips.iecLines & 0x40) == 0 ? 1 : 0));
  }

  private void setVideoMem() {
    if (VIC_MEM_DEBUG)
       monitor.info("setVideoMem() cycles since line: " +
		    (cpu.cycles - lastLine) +
		    " cycles since IRQ: " + (cpu.cycles-lastIRQ) +
		    " at " + vbeam);

     // Set-up vars for screen rendering
     vicBank = (3 - (memory[IO_OFFSET + 0xdd00] & 3)) << 14;
     charSet = vicBank | (memory[IO_OFFSET + 0xd018] & 0x0e) << 10;
     videoMatrix = vicBank | (memory[IO_OFFSET + 0xd018] & 0xf0) << 6;
     vicBase = vicBank | (memory[IO_OFFSET + 0xd018] & 0x08) << 10;
     spr0BlockSel = 0x03f8 + videoMatrix;




     //monitor.println("--------------------");
     //monitor.println("0xdd00: 0x"+Integer.toHexString(memory[IO_OFFSET + 0xdd00]));
     //monitor.println("0xd018: 0x"+Integer.toHexString(memory[IO_OFFSET + 0xd018]));
     //monitor.println("vicBank: 0x"+Integer.toHexString(vicBank));
     //monitor.println("videoMatrix: 0x"+Integer.toHexString(videoMatrix-vicBank));
     //monitor.println("charSet: 0x"+Integer.toHexString(charSet-vicBank));

     //check if vic not looking at char rom 1, 2, 4, 8
     // This is not correct! Char Rom is not everywhere!!!! - find out!
     if ( (memory[IO_OFFSET + 0xd018] & 0x0c) != 4 ||
	  (vicBank & 0x4000) == 0x4000) {
       charMemoryIndex = charSet;
     } else {
       charMemoryIndex =
	 (((memory[IO_OFFSET + 0xd018] & 0x02) == 0) ? 0 : 0x0800) +
	 CPU.CHAR_ROM2;
     }
   }

   public int ioTic() {
     boolean irq = false;
     boolean nmi = false;

//      c1541.tick(cpu.cycles);

     // CIAs should keep track of next timer so that they can request ticks
     // when needed... - so should the 1541 emulation do!!!

     // Not two interrupts the same time!!! - later this should be possible...
     if (cia[1].updateCIA(cpu.cycles) > 0) {
       nmi = true;
     } else {
       if (cia[0].updateCIA(cpu.cycles) > 0) {
	 irq = true;
       }
     }

     // Trick to get some games to work...???
     // This crashes lots of stuff now when CIAs are more correct
     // Now some games will "stop" again
     // memory[IO_OFFSET + 0xdc0d] ^= 0x08;

     return nmi ? 2 : (irq ? 1 : 0);
   }

   private void initUpdate() {
     vc = 0;
     vcBase = 0;
     vmli = 0;
     //     rc = 0;
     updating = true;

//      // First rendered line will start at cpu.cycles - no at next_scan!
//      firstLine = nextScanLine;

     for (int i = 0; i < 8; i++) {
       sprites[i].nextByte = 0;
       sprites[i].painting = false;
       sprites[i].spriteReg = 0;
     }

     if (colors == null) {
       colors = new Color[16];
       for (int i = 0; i < 16; i++) {
	 colors[i] = new Color(cbmcolor[i]);
       }
     }
     canvas.setBackground(colors[memory[IO_OFFSET + 0xd020] & 15]);
   }

  // -------------------------------------------------------------------
  // Screen rendering!
  // -------------------------------------------------------------------

  // updates the display
  int lastX = 0;
  // for debugging...
  int lastDLine = 0;
  byte vicState = 0;
  int oldDelta = 0;
  int xPos = 0;
  int lastSpriteRead = -2;

  long lastCycle = 0;

  /**
   * <code>updateChips</code> updates the IO, graphics and sound chips
   * of the C64 emulator. Focus is on VIC-II emulation (which is implemented
   * directly in this method.
   *
   * @param cycles a <code>long</code> value representing current
   * cycle count of the CPU
   */

  private int accumulator = 0;
  private int freq = 100;
  private boolean msb_rising = false;
  private int shift_register = 0;

  public void updateChips(long cycles) {

    if (cycles > nextIOUpdate) {
      updateCIA(cycles);
    }

    if (lastCycle + 1 < cycles) {
      System.out.println("More than one cycle passed: " +
			 (cycles - lastCycle) + " at " + cycles + " PC: "
			 + Integer.toHexString(cpu.pc));
    }

    if (lastCycle == cycles) {
      System.out.println("No diff since last update!!!: " +
			 (cycles - lastCycle) + " at " + cycles + " PC: "
			 + Integer.toHexString(cpu.pc));
    }

    lastCycle = cycles;

    // Delta is cycles into the current raster line!
    int delta = (int) (cycles - lastLine);

    // Each cycle is 8 pixels (a byte)
    // Cycle 16 (if first cycle is 0) is the first visible gfx cycle
    // Cycle 12 is first visible border cycle => 12 x 8 = 96
    // Last to "draw" is cycle 59 => 59 * 8 = 472 => 376 visible pixels?

    if (badLine) {
      gfxVisible = true;
    }

    switch (vicState) {
      // Initialization!
    case VICConstants.VS_INIT:
      // Increase the vbeam - rendering is started
      vbeam = (vbeam + 1) % 312;
      vPos = vbeam - (FIRST_VISIBLE_VBEAM + 1);

      if (vbeam == FIRST_VISIBLE_VBEAM) {
	colIndex++;
	if (colIndex >= LABEL_COUNT) colIndex = 0;
	// Display enabled?
	initUpdate();
      }


      // Check for interrupts, etc...
      // Sprite collission interrupts - why only once a line?
      if (((irqMask & 2) != 0) && (vicMemory[0x1f] != 0) &&
	  (irqFlags & 2) == 0) {
	if (SPRITEDEBUG)
	  monitor.info("*** Sprite collission IRQ (d01f): " +
		       vicMemory[0x1f] + " at " + vbeam);
	irqFlags |= 82;
	setIRQ(VIC_IRQ);
      }
      if (((irqMask & 4) != 0) && (vicMemory[0x1e] != 0) &&
	  (irqFlags & 4) == 0) {
	if (SPRITEDEBUG)
	  monitor.info("*** Sprite collission IRQ (d01e): " +
		       vicMemory[0x1e] + " at " + vbeam);
	irqFlags |= 84;
	setIRQ(VIC_IRQ);
      }

      // The IRQ should be before this beam line is painted!??
      // Since it is triggered on cycle 0. In this case the line is already
      // painted (above).
      memory[IO_OFFSET + 0xd012] = vbeam & 0xff;
      memory[IO_OFFSET + 0xd011] = (memory[IO_OFFSET + 0xd011] & 0x7f) |
	((vbeam & 0x100) >> 1);

      // Compare d011 + d012 to the raster position in vicMem
      int irqComp = ((vicMemory[0x11] & 0x80) << 1) + vicMemory[0x12];

      // Not nice... FIX THIS!!!
      if (irqComp > 312) irqComp &= 0xff;

      if ((irqFlags & 1) == 0 && (irqComp == vbeam)) {
	// 	 ((memory[IO_OFFSET + 0xd011] & 0x80) == (vicMemory[0x11] & 0x80))
	// 	 && (memory[IO_OFFSET + 0xd012] == vicMemory[0x12])) {
	irqFlags |= 0x1;
	//        monitor.info("Trigger Raster Interrupt at " + vbeam);

	if ((irqMask & 1) != 0) {
	  irqFlags |= 0x80;
	  irqTriggered = true;
	  setIRQ(VIC_IRQ);
	  lastIRQ = cpu.cycles;
	  if (IRQDEBUG)
	    monitor.info("Generating IRQ at " + vbeam + " req:" +
			 (((vicMemory[0x11] & 0x80) << 1) +
			  vicMemory[0x12]) + " IRQs:" + cpu.interruptInExec
			 + " d019: " + memory[IO_OFFSET + 0xd019] +
			 " flags: " + irqFlags + " delta: " +
			 (cpu.cycles - lastLine));
	}
      } else {
	irqTriggered = false;
      }

      if (vPos < 0 || vPos >= 284) {
	cpu.baLowUntil = 0;
	vicState = VICConstants.VS_FINISH;
	if (STATE_DEBUG)
	  monitor.info("FINISH next at " + vbeam);
	// Jump directly to VS_FINISH and wait for end of line...
	break;
      }

      // Check if display should be enabled...
      if (vbeam == 0x30) {
	displayEnabled = (memory[IO_OFFSET + 0xd011] & 0x10) != 0;
      }

      badLine =
	(displayEnabled && vbeam >= 0x30 && vbeam <= 0xf7) &&
	(vbeam & 0x7) == vScroll;

      if (BAD_LINE_DEBUG && badLine) {
	System.out.println("BAD Line at " + vbeam + " vc: " + vc +
			   " rc: " + rc);
      }

      // remembers the last update X position... should be translated to
      // cycle later?
      lastX = 0;

      // Clear the collission masks each line... - not needed???
      for (int i = 0, n = SC_WIDTH; i < n; i++) {
	collissionMask[i] = 0;
      }

    case VICConstants.VS_SPRITE3:
      // If we should not yet read the sprite data for S3, wait...
      if (delta < 1) {
	vicState = VICConstants.VS_SPRITE3;
	break;
      }

      if (sprites[3].dma) {
	sprites[3].readSpriteData();
      }
      if (sprites[5].dma) {
	cpu.baLowUntil = lastLine + VICConstants.BA_SP5;
      }
    case VICConstants.VS_SPRITE4:
      // If we should not yet read the sprite data for S4, wait...
      if (delta < 3) {
	vicState = VICConstants.VS_SPRITE4;
	break;
      }
      if (sprites[4].dma) {
	sprites[4].readSpriteData();
      }
      if (sprites[6].dma) {
	cpu.baLowUntil = lastLine + VICConstants.BA_SP6;
      }
    case VICConstants.VS_SPRITE5:
      if (delta < 5) {
	vicState = VICConstants.VS_SPRITE5;
	break;
      }
      if (sprites[5].dma) {
	sprites[5].readSpriteData();
      }
      if (sprites[7].dma) {
	cpu.baLowUntil = lastLine + VICConstants.BA_SP7;
      }
    case VICConstants.VS_SPRITE6:
      if (delta < 7) {
	vicState = VICConstants.VS_SPRITE6;
	break;
      }
      if (sprites[6].dma) {
	sprites[6].readSpriteData();
      }

    case VICConstants.VS_SPRITE7:
      if (delta < 9) {
	vicState = VICConstants.VS_SPRITE7;
	break;
      }
      if (sprites[7].dma) {
	sprites[7].readSpriteData();
      }

      // Border management! (at another cycle maybe?)
      if (blankRow) {
	if (vbeam == 247) {
	  paintBorder = true;
	}
      } else {
	if (vbeam == 251) {
	  paintBorder = true;
	}
	if (vbeam == 51) {
	  paintBorder = false;
	  // Reset sprite data to avoid garbage since they are not painted...
	  for (int i = 0, n = 7; i < n; i++) {
	    if (!sprites[i].painting) {
	      sprites[i].lineFinished = true;
	    }
	  }
	}
      }
      // No border after vbeam 55 (ever?)
      if (vbeam == 55) {
	paintBorder = false;
	// Reset sprite data to avoid garbage since they are not painted...
	for (int i = 0, n = 7; i < n; i++) {
	  if (!sprites[i].painting)
	    sprites[i].lineFinished = true;
	}
      }

    case VICConstants.VS_FETCHBADC12:
      if (delta < 11) {
	vicState = VICConstants.VS_FETCHBADC12;
	break;
      }
      if (badLine) {
	cpu.baLowUntil = lastLine + VICConstants.BA_BADLINE;
      }
      // Cycles 11 - 13
    case VICConstants.VS_VCRC:
      if (delta < 13) {
	vicState = VICConstants.VS_VCRC;
	break;
      }

      // calculate mpos before starting the rendering!
      mpos = vPos * SC_WIDTH;
      // Paint border for two cycles here...
      paintBorder(mpos, 16);
      mpos += 16;

      // Set vc, reset vmli...
      vc = vcBase;
      vmli = 0;
      if (badLine) {
	cpu.baLowUntil = lastLine + VICConstants.BA_BADLINE;

	if (BAD_LINE_DEBUG) System.out.println("#### RC = 0 (" + rc + ") at "
					       + vbeam + " vc: " + vc);

	rc = 0;
      }

    case VICConstants.VS_SPRITE_DMAOFF:
      if (delta < 15) {
	vicState = VICConstants.VS_SPRITE_DMAOFF;
	break;
      }

      // And the other two cycles here...
      paintBorder(mpos, 16);
      mpos += 16;

      // Turn off sprite DMA if finished reading!
      for (int i = 0, n = 8; i < n; i++) {
	if (sprites[i].nextByte == 63)
	  sprites[i].dma = false;
      }
      // Now we enter the "painting" of the screen...

    case VICConstants.VS_40CHARSC17:
      if (delta < 16) {
	vicState = VICConstants.VS_40CHARSC17;
	break;
      }

      if (badLine) {
	cpu.baLowUntil = lastLine + VICConstants.BA_BADLINE;
	// Fetch first char into cache! (for the below draw...)
	vicCharCache[0] = memory[videoMatrix + (vcBase & 0x3ff)];
	vicColCache[0] = memory[IO_OFFSET + 0xd800 + (vcBase & 0x3ff)];
      }

      for (int i = 0, n = horizScroll; i < n; i++) {
	mem[mpos + i] = bgColor;
      }
      // Draw one character here!
      drawGraphics(mpos + horizScroll, 1);
      if (hideColumn)
	paintBorder(mpos, 8);
      mpos += 8;
      oldDelta = 16;

      xPos = 40; // Just after one char is painter (32 + 8)

      // Go to draw for a long while... (40 cycles?)
      vicState = VICConstants.VS_DRAWC18_54;

      if (delta == 16) break;

      // Draw the graphics (and fetch if bad line)
    case VICConstants.VS_DRAWC18_54:
      int diff = delta - oldDelta;
      oldDelta = delta;

      if (vmli + diff > 40) {
	diff = 40 - vmli;
      }

      if (badLine) {
	cpu.baLowUntil = lastLine + VICConstants.BA_BADLINE;
	// Fetch a some chars into cache! (for the below draw...)
	for (int i = vmli; i < vmli + diff; i++) {
	  vicCharCache[i] = memory[videoMatrix + ((vcBase + i) & 0x3ff)];
	  vicColCache[i] = memory[IO_OFFSET + 0xd800 + ((vcBase + i) & 0x3ff)];
	}
      }

      // draw the graphics. (should probably handle sprites also??)
      drawGraphics(mpos + horizScroll, diff);
      if (!paintBorder)
	drawSprites(lastX, xPos);

      lastX = xPos;

      xPos += (diff << 3);
      mpos += (diff << 3);

      // Draw until we are at the last vmli - will this always work?
      if (vmli < 40) {
	// 	System.out.println("VMLI: " + vmli + " diff: " + diff +
	// 			   " vbeam: " + vbeam + " xPos: " + xPos);
	if (delta >= 54) {
	  // Then Check if it is time to start up the sprites!
	  // Does not matter in which order this is done ?=
	  int mult = 1;
	  int y = 0;
	  int ypos = vPos + SC_SPYOFFS;

	  for (int i = 0, n = 8; i < n; i++) {
	    Sprite sprite = sprites[i];
	    if (sprite.enabled) {
	      y = memory[IO_OFFSET + 0xd001 + i * 2];
	      // If it is time to start drawing this sprite!
	      if (y == (ypos & 0xff) && (ypos < 270)) {
		sprite.nextByte = 0;
		sprite.dma = true;
		sprite.expFlipFlop = true;
		if (SPRITEDEBUG)
		  System.out.println("Starting painting sprite " + i + " on "
				     + vbeam + " first visible at " + (ypos + 1));
	      }
	    }
	    mult = mult << 1;
	  }
	  if (sprites[0].dma) {
	    cpu.baLowUntil = lastLine + VICConstants.BA_SP0;
	  }
	}
	break;
      }

      // Draw the final sprites... (not if border...)
      if (!paintBorder)
	drawSprites(lastX, xPos);

      // All painter - go on to next step...
    case VICConstants.VS_40CHARSENDC57:
      if (delta < 56) {
	vicState = VICConstants.VS_40CHARSENDC57;
	break;
      }
      if (hideColumn) {
	// Paint the border slighly back...
	paintBorder(mpos - 8, 8);
      }

      // If time to turn of sprite display...
      for (int i = 0, n = 8; i < n; i++) {
	Sprite sprite = sprites[i];
	if (!sprite.dma) {
	  sprite.painting = false;
	  if (SPRITEDEBUG)
	    System.out.println("Stopped painting sprite " +
			       i + " at (after): " + vbeam);
	}
      }

      // Here we should check if sprite dma should start...
      // - probably need to add a dma variable to sprites, and not
      // only use the painting variable for better emulation
      // Bus not available if sp0 or sp1 is painting
      if (sprites[1].dma) {
	cpu.baLowUntil = lastLine + VICConstants.BA_SP1;
      }
      // Paint border, check sprite for display and read sprite 0 data.
    case VICConstants.VS_SPRITE0_RC:
      if (delta < 57) {
	vicState = VICConstants.VS_SPRITE0_RC;
	break;
      }

      for (int i = 0, n = 8; i < n; i++) {
	Sprite sprite = sprites[i];
	if (sprite.dma)
	  sprite.painting = true;
      }

      if (rc == 7) {
	vcBase = vc;
	gfxVisible = false;
	if (BAD_LINE_DEBUG) {
	  monitor.info("#### RC7 ==> vc = " + vc + " at " + vbeam +
		       " delta = " + delta);
	  if (vc == 1000) {
	    monitor.info("--------------- last line ----------------");
	  }
	}
      }

      if (badLine || gfxVisible) {
	rc = (rc + 1) & 7;
	gfxVisible = true;
      }

      paintBorder(mpos, 16);
      mpos += 16;

      if (sprites[0].painting) {
	sprites[0].readSpriteData();
      }

      if (sprites[2].dma) {
	cpu.baLowUntil = lastLine + VICConstants.BA_SP2;
      }
    case VICConstants.VS_SPRITE1:
      if (delta < 59) {
	vicState = VICConstants.VS_SPRITE1;
	break;
      }
      paintBorder(mpos, 16);
      mpos += 16;

      if (sprites[1].painting) {
	sprites[1].readSpriteData();
      }

    case VICConstants.VS_SPRITE2:
      if (delta < 61) {
	vicState = VICConstants.VS_SPRITE2;
	break;
      }
      if (sprites[2].painting) {
	sprites[2].readSpriteData();
      }
      if (sprites[3].dma) {
	cpu.baLowUntil = lastLine + VICConstants.BA_SP3;
      }

      vicState = VICConstants.VS_FINISH;
    // Handle VIC interrupts and SID updates, etc.
    case VICConstants.VS_FINISH:
      // 62 is last cycle for this scan line...
      if (delta >= 62) {
	lastLine += VICConstants.SCAN_RATE;

	// Update screen
	if (updating) {
	  if (vPos == 285) {
 	    mis.newPixels();
 	    canvas.repaint();

	    actualScanTime = (actualScanTime * 9 + (int)
			      ((mixer.getMicros() - lastScan))) / 10;
	    lastScan = mixer.getMicros();
	    updating = false;
	  }
	}

//        monitor.info("IRQMask: " + irqMask + " latch:" + irqLatch);

	// Before finishing this line. check if SID should "sing"
	if (cycles > nextSID) {
	  mixer.updateSound(cycles);
	  nextSID = nextSID + sidUpdate; // Each milli second (1000) at 50 fps
	  if (nextSID < cpu.cycles)
	    nextSID = cpu.cycles + 10;
	}

	// Next time - we initialize next line...
	vicState = VICConstants.VS_INIT;
	if (STATE_DEBUG)
	  monitor.info("INIT next at " + vbeam + " delta = " + delta);

        // Is this needed? - or not?
	if (delta > 62) {
	  updateChips(cycles);
	}
      }
      break;
    }

//       System.out.println("Current VIC State: " + vicState + " vmli: " + vmli
// 			 + " vc: " + vc + " vcBase: " + vcBase +
// 			 " vbeam: " + vbeam + " rc: " + rc + " gfx:"
// 			 + gfxVisible);

    if (IRQDEBUG) {
      if (irqTriggered) {
	if (vPos > 0 && vPos < 284) {
	  int pos = vPos * SC_WIDTH;
	  int col = 0xff809000 | (int) (cycles & 0xff);
	  for (int i = 0, n = SC_WIDTH; i < n; i += 4) {
	    mem[pos + i] = col;
	  }
	}
	}
    }
  }


  private final void paintBorder(int mpos, int nPix) {
    // Draw the border...
    for (int i = 0; i < nPix; i++) {
      mem[mpos++] = borderColor;
    }
  }


  /**
   * <code>drawGraphics</code> - draw the VIC graphics (text/bitmap)
   * Note that sprites are not drawn here... (yet?)
   *
   *
   * @param mpos an <code>int</code> value representing position to
   * draw the graphics from (already fixed with hscroll)
   * @param noChars an <code>int</code> value
   */
  private final void drawGraphics(int mpos, int noChars) {
    if (!gfxVisible || paintBorder) {
      // We know that display is not enabled, and that mpos is already
      // at a correct place, except horizScroll...
      mpos -= horizScroll;
      for (int i = mpos, n = mpos + (noChars << 3); i < n; i++) {
	mem[i] = borderColor;
      }
      // trick to use vmli as a if var even when no gfx.
      vmli += noChars;
      return;
    }

    int maxVMLI = vmli + noChars;
    if (maxVMLI > 40) maxVMLI = 40;
    int collX = (vmli << 3) + horizScroll + SC_XOFFS;

    int position = 0, data = 0, penColor = 0, bgcol = bgColor;

    if ((memory[IO_OFFSET + 0xd011] & 0x20) == 0) {
      int tmp;
      int pcol;

      // This should be in a cache some where...
      if (multiCol) {
	multiColor[0] = bgColor;
	multiColor[1] = cbmcolor[memory[IO_OFFSET + 0xd022] & 15];
	multiColor[2] = cbmcolor[memory[IO_OFFSET + 0xd023] & 15];
      }


      for (; vmli < maxVMLI; collX += 8) {
	penColor = cbmcolor[pcol = vicColCache[vmli] & 15];
	if (extended) {
	  position = charMemoryIndex +
	    (((data = vicCharCache[vmli]) & 0x3f) << 3);
	  bgcol = cbmcolor[memory[IO_OFFSET + 0xd021 + (data >> 6)] & 15];
	} else {
	  position = charMemoryIndex + (vicCharCache[vmli] << 3);
	}

	data = memory[position + rc];

	if (multiCol && pcol > 7) {
	  multiColor[3] = cbmcolor[pcol & 7];
	  for (int pix = 0; pix < 8; pix += 2) {
	    tmp = (data >> pix) & 3;
	    mem[mpos + 6 - pix] = mem[mpos + 7 - pix] = multiColor[tmp];
	    // both 00 and 01 => no collission!?
	    // but what about priority?
	    if (tmp > 0x01) {
	      tmp = 256;
	    } else {
	      tmp = 0;
	    }
	    collissionMask[collX + 7 - pix] =
	      collissionMask[collX + 6 - pix] = tmp;
	  }
	} else {
	  for (int pix = 0; pix < 8; pix++) {
	    if ((data & (1 << pix)) > 0) {
	      mem[mpos + 7 - pix] = penColor;
	      collissionMask[collX + 7 - pix] = 256;
	    } else {
	      mem[mpos + 7 - pix] = bgcol;
	      collissionMask[collX + 7 - pix] = 0;
	    }
	  }
	}

	if (multiCol && extended) {
	  // Illegal mode => all black!
	  for (int pix = 0; pix < 8; pix++) {
	    mem[mpos + 7 - pix] = 0xff000000;
	  }
	}

	if (BAD_LINE_DEBUG && badLine) {
	  for (int pix = 0; pix < 8; pix += 4) {
	    mem[mpos + 7 - pix] = (mem[mpos + 7 - pix] & 0xff7f7f7f) | 0x0fff;
	  }
	}


	mpos += 8;
	vmli++;
	vc++;
      }
    } else {
      // -------------------------------------------------------------------
      // Bitmap mode!
      // -------------------------------------------------------------------
      position = vicBase + (vc & 0x3ff) * 8 + rc;
      if (multiCol) {
	multiColor[0] = bgColor;
      }
      for (; vmli < maxVMLI; vmli++) {
	penColor =
	  cbmcolor[(vicCharCache[vmli] & 0xf0) >> 4];
	bgcol = cbmcolor[vicCharCache[vmli] & 0x0f];

	data = memory[position];

	if (multiCol) {
	  multiColor[1] =
	    cbmcolor[(vicCharCache[vmli] >> 4) & 0x0f];
	  multiColor[2] =
	    cbmcolor[vicCharCache[vmli] & 0x0f];
	  multiColor[3] = cbmcolor[vicColCache[vmli] & 0x0f];

	  // Multicolor
	  int tmp;
	  for (int pix = 0; pix < 8; pix += 2) {
	    mem[mpos + 6 - pix] = mem[mpos + 7 - pix] =
	      multiColor[tmp = (data >> pix) & 3];
	    if (tmp > 0x01) {
	      tmp = 256;
	    } else {
	      tmp = 0;
	    }
	    collissionMask[collX + 7 - pix] =
	      collissionMask[collX + 6 - pix] = tmp;
	  }
	} else {
	  // Non multicolor
	  for (int pix = 0; pix < 8; pix++) {
	    if ((data & (1 << pix)) > 0) {
	      mem[7 - pix + mpos] = penColor;
	      collissionMask[collX + 7 - pix] = 256;
	    } else {
	      mem[7 - pix + mpos] = bgcol;
	      collissionMask[collX + 7 - pix] = 0;
	    }
	  }
	}

	if (extended) {
	  // Illegal mode => all black!
	  for (int pix = 0; pix < 8; pix++) {
	    mem[mpos + 7 - pix] = 0xff000000;
	  }
	}

	if (BAD_LINE_DEBUG && badLine) {
	  for (int pix = 0; pix < 8; pix += 4) {
	    mem[mpos + 7 - pix] = (mem[mpos + 7 - pix] & 0xff3f3f3f) | 0x0fff;
	  }
	}


	position = (position + 8) & 0xffff;
	mpos += 8;
	collX += 8;
	vc++;
      }
    }
  }

  // -------------------------------------------------------------------
  // Sprites...
  // -------------------------------------------------------------------
  private final void drawSprites(int lastX, int xPos) {
    int smult = 0x100;
    int borderPos = SC_XOFFS + (hideColumn ? 8 : 0);

    for (int i = 7; i >= 0; i--) {
      Sprite sprite = sprites[i];
      // Done before the continue...
      smult = smult >> 1;
      if (sprite.lineFinished || !sprite.painting) {
	continue;
      }
      int x = sprite.x + SC_SPXOFFS;
      int mpos = vPos * SC_WIDTH;

      if (x < xPos) {
	// Ok, we should write some data...
	int minX = lastX > x ? lastX : x;
// 	monitor.info("Writing sprite " + i + " first pixel at " +
// 			   minX + " vPos = " + vPos);

	for (int j = minX, m = xPos; j < m; j++) {
	  int c = sprite.getPixel();
	  if (c != 0 && j >= borderPos) {
	    int tmp = (collissionMask[j] |= smult);
	    if (!sprite.priority || (tmp & 0x100) == 0) {
	      mem[mpos + j] = sprite.color[c];
	    }

	    if (tmp != smult) {
	      // If collission with bg then notice!
	      if ((tmp & 0x100) != 0) {
		vicMemory[0x1f] |= smult;
// 		monitor.info("***** Sprite x Bkg collission!");
	      }
	      // If collission with sprite, all colls must
	      // be registered!
	      if ((tmp & 0xff) != smult) {
		vicMemory[0x1e] |= tmp & 0xff;
//  		monitor.info("***** Sprite x Sprite collission: d01e = " +
// 			     vicMemory[0x1e] + " sprite: " + i + " => " +
// 			     smult + " at " + j + "," + vbeam);
	      }
	    }
	  }

	  if (SPRITEDEBUG) {
	    if ((sprite.nextByte == 3) && ((j & 4) == 0)) {
	      mem[mpos + j] = 0xff00ff00;
	    }
	    if ((sprite.nextByte == 63) && ((j & 4) == 0)) {
	      mem[mpos + j] = 0xffff0000;
	    }

	    if (j == x) {
	      mem[mpos + j] = 0xff000000 + sprite.pointer;
	    }

	  }
	}
      }
    }

    memory[IO_OFFSET + 0xd01e] = vicMemory[0x1e];
    memory[IO_OFFSET + 0xd01f] = vicMemory[0x1f];

  }




  public void reset() {
    // Clear a lot of stuff...???
    initUpdate();

    mixer.reset();

    nextSID = cpu.cycles;
    lastLine = cpu.cycles;
    nextIOUpdate = cpu.cycles + 47;
    vicState = VICConstants.VS_INIT;

    for (int i = 0; i < mem.length; i++) mem[i] = 0;

    reset = 100;

//     c1541.reset();
    keyboard.reset();

    motorSound(false);

  }

  public static final int IMG_TOTWIDTH = SC_WIDTH;
  public static final int IMG_TOTHEIGHT = SC_HEIGHT;

  public Image crtImage;

  // Will be called from the c64canvas class
  long repaint = 0;
  public void paint(Graphics g) {

    if (g == null)
      return;
    if (image == null) {
      image = canvas.createImage(IMG_TOTWIDTH, IMG_TOTHEIGHT);
      g2 = image.getGraphics();
      g2.setFont(new Font("Monospaced", Font.PLAIN, 11));
      crtImage = new BufferedImage(IMG_TOTWIDTH * 2, IMG_TOTHEIGHT * 2,
				   BufferedImage.TYPE_INT_ARGB);
      Graphics gcrt = crtImage.getGraphics();
      gcrt.setColor(TRANSPARENT_BLACK);
      for (int i = 0, n = IMG_TOTHEIGHT * 2; i < n; i += 2) {
	gcrt.drawLine(0, i, IMG_TOTWIDTH * 2, i);
      }
    }

    // Why is there transparency?
    g2.drawImage(screen, 0, 0,  null);

    if (reset > 0) {
      g2.setColor(darks[colIndex]);
      int xp = 44;
      if (reset < 44) {
	xp = reset;
      }
      g2.drawString("JaC64 " + version + " - Java C64 - www.jac64.com",
		    xp + 1, 9);
      g2.setColor(lites[colIndex]);
      g2.drawString("JaC64 " + version + " - Java C64 - www.jac64.com",
		    xp, 8);
      reset--;
    } else {
      String msg = "JaC64 ";
      if ((message != null) && (message != "")) {
	msg += message;
      } else {
	colIndex = 0;
      }
      msg += tmsg;

      g2.setColor(darks[colIndex]);
      g2.drawString(msg, 1, 9);
      g2.setColor(lites[colIndex]);
      g2.drawString(msg, 0, 8);

      if (ledOn) {
	g2.setColor(LED_ON);
      } else {
	g2.setColor(LED_OFF);
      }
      g2.fillRect(372, 3, 7, 1);
      g2.setColor(LED_BORDER);
      g2.drawRect(371, 2, 8, 2);

    }

    if (DOUBLE) {
      g.drawImage(image, 0, 0, IMG_TOTWIDTH * 2, IMG_TOTHEIGHT * 2, null);
      //      g.drawImage(crtImage, 0, 0, IMG_TOTWIDTH * 2, IMG_TOTHEIGHT * 2, null);
    } else {
      g.drawImage(image, 0, 0, null);
    }

//     monitor.info("Repaint: " + (System.currentTimeMillis() - repaint));
//     repaint = System.currentTimeMillis();
  }


  // -------------------------------------------------------------------
  // Internal sprite class to handle all data for sprites
  // Just a collection of data registers... so far...
  // -------------------------------------------------------------------
  private class Sprite {

    boolean painting = false; // If sprite is "on" or not (visible)
    boolean dma = false;  // Sprite DMA on/off

    int nextByte;
    int pointer;
    int x;

    int spriteNo;
    // Contains the sprite data to be outshifted
    int spriteReg;

    boolean enabled;
    boolean expFlipFlop;
    boolean multicolor = false;
    boolean expandX = false;
    boolean expandY = false;
    boolean priority = false;
    boolean lineFinished = false;

    int pixelsLeft = 0;
    int currentPixel = 0;

    int[] color = new int[4];

    int getPixel() {
      if (lineFinished) return 0;
      pixelsLeft--;
      if (pixelsLeft > 0) return currentPixel;
      // Indicate finished!
      if (pixelsLeft <= 0 && spriteReg == 0) {
	currentPixel = 0;
	lineFinished = true;
	return 0;
      }

      if (multicolor) {
	// The 23rd and 22nd pixel => data!
	currentPixel = (spriteReg & 0xc00000) >> 22;
	spriteReg = (spriteReg << 2) & 0xffffff;
	pixelsLeft = 2;
      } else {
	// Only the 23rd bit is pixel data!
	currentPixel = (spriteReg & 0x800000) >> 22;
	spriteReg = (spriteReg << 1) & 0xffffff;
	pixelsLeft = 1;
      }
      // Double the number of pixels if expanded!
      if (expandX) {
	pixelsLeft = pixelsLeft << 1;
      }

      return currentPixel;
    }

    void readSpriteData() {
      lastSpriteRead = spriteNo;

      // Read pointer + the three sprite data pointers...
      pointer = vicBank + memory[spr0BlockSel + spriteNo] * 0x40;
      spriteReg = ((memory[pointer + nextByte++] & 0xff) << 16) |
	((memory[pointer + nextByte++] & 0xff)  << 8) |
	memory[pointer + nextByte++];

      // For debugging... seems to be err on other place than the
      // Memoryfetch - since this is also looking very odd...???
//       spriteReg = 0xf0f0f0;

      if (!expandY) expFlipFlop = false;

      if (expFlipFlop) {
	nextByte = nextByte - 3;
      }

      expFlipFlop = !expFlipFlop;
      pixelsLeft = 0;

      lineFinished = false;
    }
  }

  // -------------------------------------------------------------------
  // Observer (1541) - should probably be in C64 screen later...
  // -------------------------------------------------------------------

  public void updateDisk(Object obs, Object msg) {
    if (msg == c1541Chips.HEAD_MOVED) {
      if (lastTrack != c1541Chips.currentTrack) {
	lastTrack = c1541Chips.currentTrack;
	trackSound();
      } else {
	// Head could not move any more... maybe other sound here?
	trackSound();
      }
    }

    // add head beyond here...

    lastSector = c1541Chips.currentSector;

    if (motorOn != c1541Chips.motorOn) {
      motorSound(c1541Chips.motorOn);
    }

    tmsg = " track: " + lastTrack + " / " + lastSector;

    ledOn = c1541Chips.ledOn;
    motorOn = c1541Chips.motorOn;
  }

  private void trackSound() {
    if (trackSound != null) {
      trackSound.play();
    }
  }

  public void motorSound(boolean on) {
    if (motorSound != null) {
      if (on)
	motorSound.loop();
      else
	motorSound.stop();
    }
  }


  public void setSounds(AudioClip track, AudioClip motor) {
    trackSound = track;
    motorSound = motor;
  }


}
