/**
 * This file is a part of JaC64 - a Java C64 Emulator
 * Main Developer: Joakim Eriksson (Dreamfabric.com)
 * Contact: joakime@sics.se
 * Web: http://www.dreamfabric.com/c64
 * ---------------------------------------------------
 */

package com.dreamfabric.jac64;

/**
 * C64Chips - handles IRQs/NMIs for the CPU/IO chips
 * and defines some APIs that is called by CPU
 *
 * Created: Tue Aug 02 08:58:12 2005
 *
 * @author Joakim Eriksson
 * @version 1.0
 */
public abstract class C64Chips {

  public static final boolean DEBUG_INTERRUPS = false;

  // C64 specific names - but... basically just numbers
  public static final int VIC_IRQ = 1;
  public static final int CIA_TIMER_IRQ = 2;
  public static final int KEYBOARD_NMI = 1;
  public static final int CIA_TIMER_NMI = 2;

  // Interrupt management
  // As soon as either of these are != 0 then nmi/irq low!
  private int nmiFlags = 0;
  private int oldNmiFlags = 0;
  private int irqFlags = 0;
  private int oldIrqFlags = 0;
  private Observer observer;

  MOS6510Core cpu;

  public void init(MOS6510Core cpu) {
    this.cpu = cpu;
  }

  public int getNMIFlags() {
    return nmiFlags;
  }

  public int getIRQFlags() {
    return irqFlags;
  }

  public boolean setIRQ(int irq) {
    boolean val = (irqFlags & irq) == 0;
    irqFlags |= irq;
    if (DEBUG_INTERRUPS && oldIrqFlags != 0)
      System.out.println("Setting IRQ! " + irq + " => " + irqFlags);
    if (irqFlags != oldIrqFlags) {
      cpu.setIRQLow(irqFlags != 0);
      oldIrqFlags = irqFlags;
    }
    return val;
  }

  public void clearIRQ(int irq) {
    irqFlags &= ~irq;
    if (DEBUG_INTERRUPS && oldIrqFlags != 0)
      System.out.println("Clearing IRQ! " + irq + " => " + irqFlags);
    if (irqFlags != oldIrqFlags) {
      cpu.setIRQLow(irqFlags != 0);
      oldIrqFlags = irqFlags;
    }
  }

  public boolean setNMI(int nmi) {
    boolean val = (nmiFlags & nmi) == 0;
    nmiFlags |= nmi;
    if (DEBUG_INTERRUPS)
      System.out.println("Setting NMI! " + nmi + " => " + nmiFlags);
    if (nmiFlags != oldNmiFlags) {
      cpu.setNMILow(nmiFlags != 0);
      oldNmiFlags = nmiFlags;
    }
    return val;
  }


  public void clearNMI(int nmi) {
    nmiFlags &= ~nmi;
    if (DEBUG_INTERRUPS && oldNmiFlags != 0)
      System.out.println("Clearing NMI! " + nmi + " => " + nmiFlags);
    if (nmiFlags != oldNmiFlags) {
      cpu.setNMILow(nmiFlags != 0);
      oldNmiFlags = nmiFlags;
    }
  }


  public abstract void reset();

  public abstract int performRead(int address, long cycles);
  public abstract void performWrite(int address, int data, long cycles);
  public abstract void updateChips(long cycles);

  public void setObserver(Observer o) {
    observer = o;
  }

  public void update(Object source, Object data) {
    if (observer != null) {
      observer.update(source, data);
    }
  }
}
