/*
 * Find the index into the env_var array of the variable <var>.  If not
 * found, returns the index of the terminating null entry
 */
static int get_env_index (char *var, char **env_var)
{
  int   i;
  char *temp;

  assert (temp = strdump(var));
  for (i = 0; env_var[i]; i++)
     if (strstr(env_var[i],temp) == env_var[i])
        break;

  free (temp);
  return (i);
}

/*
 * Try to locate the parent environment and fill in the segment address
 * This routine has some OS version dependencies
 * Returns  1  if environment found
 *   0  if environment not found
 */
static int get_env_seg (unsigned *envseg)
{
  unsigned   ppsp;
  union REGS regs;

  regs.x.ax = 0xD44D;        /* 4dos signature value */
  regs.x.bx = 0;
  int86 (0x2F,&regs, &regs); /* TSR multiplex interrput */
  ppsp = regs.x.cx;          /* psp of immediate parent shell */

  if (regs.x.ax == 0x44DD && ppsp == *(unsigned far*)_MK_FP(_psp,0x16))
  {
    /*
     * good, we are running 4dos - AND it is our parent
     * in this case we don't care about dos version
     */
    *envseg = *(unsigned far*) _MK_FP (ppsp, 0x2C);
  }
  else
  {
    /*
     * Follow the pointers back to our parent psp and from there get the
     * parent environment segment address
     * With early command.com (pre 3.3) the lowest level shell did not fill
     * in the environment segment (it is 0) but we can still find the
     * environment since it always follows the lowest level shell directly
     */
    ppsp    = *(unsigned far*) _MK_FP(_psp, 0x16);
    *envseg = *(unsigned far*) _MK_FP(ppsp, 0x2C);
    if ((*envseg == 0) &&
       (_osmajor <= 2 || (_osmajor == 3 && _osminor <= 20)))
    {
     /*
      * we have the pre 3.3 master command.com - have to assume that the
      * environment segment directly follows the program segment
      * so get the length from the MCB and add it to ppsp
      */
      *envseg = ppsp + *(unsigned far*) _MK_FP (ppsp-1, 3) + 1;
    }
  }
  /*
   * Verify that the owner of the envseg matches the ppsp we got above
   * - just to be a little paranoid
   */
  if (*(unsigned far *)_MK_FP((*envseg)-1, 1) != ppsp)
  {
   *envseg = 0;  /* something wrong :-( */
    return (0);
  }
  return (1);
}

/*
 * Getting enviroment variable
 *
 * Copy the parent environment from the segment address given in env_addr,
 * into the memory block pointed to by env_blk.
 * Extract pointers to the start of each variable and build array env_var.
 * Return the length of the environment block, (bytes, always a multiple
 * of 16)
 */
static unsigned get_env_var (unsigned env_addr, char ***env_var, int *count)
{
  static   char *env_blk;
  struct   SREGS segregs;
  unsigned i, len, offset;
  unsigned seglen;
  unsigned DS;

  segread (&segregs);
  DS = segregs.ds;
  seglen = *(unsigned far*) _MK_FP (env_addr-1, 3) * 16;

  if ((env_blk = malloc(seglen)) == NULL)
     fatal ("Cannot allocate memory for environment block", 5);

  movedata (env_addr, 0, DS, (unsigned)&env_blk[0], seglen);
  /*
   * make one pass through, counting variables, and allocate space for the
   * env_var array.  Allow for two extra entries - one to add a new variable
   * and a terminating null.
   */
  i = 0; offset = 0;
  do
  {
   offset += strlen (env_blk+offset) + 1;
   i++;
  }
  while(*(env_blk+offset));

  assert (*env_var = malloc((i+2) * sizeof(char**)));
  *count = i;

  /* load pointers to each of the strings in the array env_var[]
   */
  i = 0; offset = 0;
  do
  {
   len = strlen (env_blk+offset);
   (*env_var)[i] = env_blk + offset;
   offset += len+1;
   i++;
  }
  while (*(env_blk+offset));

  (*env_var)[i] = NULL;    /* terminate with an extra null */
  return (seglen);
}

/*
 * Copy the modified environment from the allocated data area, using the
 * env_var array of pointers, into the memory block pointed to by env_addr.
 */
void put_env_var (unsigned env_addr, unsigned seglen, char **env_var)
{
  struct   SREGS segregs;
  char     *var_addr;
  unsigned offset, len, i;
  unsigned zero = 0;
  unsigned DS;

  segread (&segregs);
  DS = segregs.ds;

  /*
   * Cycle through once to test for environment overflow.  If overflow,
   * nothing is copied to the old environment block
   */
  for (i = 0, offset = 0, len = 0; env_var[i]; offset += len, i++)
  {
    len = strlen (env_var[i]) + 1;
    if (offset+len >= seglen-2)  /* overflow */
       fatal ("Environment overflow - not modified\n", 4);
  }
  /*
   * Now actually copy the strings
   */
  for (i = 0, offset = 0, len = 0; env_var[i]; offset += len, i++)
  {
    len = strlen (env_var[i]) + 1;
    if (len == 1)
    {
      /* this entry was null - skip it */
      len = 0;
      continue;
    }
    var_addr = env_var[i];
    _movedata (DS, (unsigned)var_addr, env_addr, offset, len);
  }

  /* stuff a zero word at the end of the environment block
   */
  _movedata (DS, (unsigned)&zero, env_addr, offset, sizeof(unsigned));
}

/*
 * Setting enviroment variable
 */
int set_env_var (char * variable, char * val)
{
  unsigned env_addr, seglen;
  int      index, i, count;
  char    *value, **env_var;

  if (!get_env_seg(&env_addr))
     fatal ("Cannot locate environment\n", 2);

  seglen = get_env_var (env_addr, &env_var, &count);
  index  = get_env_index (variable, env_var);
  if (index == count)
     env_var[index+1] = NULL;

  /* set the value of the variable to the rest of the arguments
   */
  value = malloc (strlen(variable) + 2);
  strcpy (value, variable);
  strcat (value, "=");
  value = realloc (value, strlen(value) + strlen(val) + 2);
  strcat (value, val);
  strcpy (value-1, '\0');

  env_var[index] = value;
  put_env_var (env_addr, seglen, env_var);
  return (0);
}

